// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.
#include "cylinder.h"
#include "cylmodel.h"
#include GL_INCLUDE    // see platform.h

namespace visual {

double
cylinder::rayIntersect( const vector &camera, const vector &ray)
{
	if (degenerate)
		return 0.0;

	// Ray/cylinder intersection code.  Based on the discussion in
	//   Graphics Gems IV.
	double tin, tout;

	double r = scale.y, x = scale.x*2.0;
	vector Br = camera, R = ray, 
		Bc(mwt[0][3],mwt[1][3],mwt[2][3]), 
		A(mwt[0][0],mwt[1][0],mwt[2][0]);
	vector RC = Br - Bc;
	A = A.norm();

	vector RcA = R.cross(A);
	double RcAm = RcA.mag();
	if (RcAm) {
		vector D = R.cross(A).norm();
		double d = std::fabs(RC.dot(D));

		if (d > r)
			return 0.0;  // ray misses infinite cylinder
		double t = -RC.cross(A).dot(D)/RcAm;
		vector O = D.cross(A).norm();
		double s = std::fabs( std::sqrt(r*r - d*d) / R.dot(O));
		tin = t - s;
		tout = t + s;
	}
	else {    // ray parallel to cylinder
		return 0.0;
		double d = (RC - A*RC.dot(A)).mag();
		if (d > r)
			return 0.0;  // ray outside cylinder

		// ray inside cylinder
		tin = -1e300;
		tout = 1e300;
	}

	double BrdA = RC.dot(A);  // Br*N for the +x endcap
	double e = R.dot(A);      // R*N for the +x endcap
	if (e) {
		double t0 = BrdA / -e;
		double tx = -(BrdA - x) / e;

		if (e < 0.0) {
			// potential exit at x=0
			if (t0 < tout)
				tout = t0;
			// potential entrance at +x
			if (tx > tin)
				tin = tx;
		}
		else {
			// potential entrance at x=0
			if (t0 > tin)
				tin = t0;
			// potential exit at +x
			if (tx < tout)
				tout = tx;
		}
	}
	else {
		if (BrdA < 0.0)
			return 0.0;  // below x=0 endcap
		if (BrdA+scale.x > 0.0)
			return 0.0;  // above +x endcap
	}

	if (tout < tin)
		return 0.0;  // missed cylinder
	else if (tin<0)
		return tout; // inside cylinder
	else
		return tin;  // outside cylinder
}

void
cylinder::glRender( rView& view)
{
	if (degenerate)
		return;

	/*static double brect[] = { 2.0,  1.0,  1.0,
	                            0.0, -1.0, -1.0};
	view.ext_brect(mwt, brect);*/
	view.ext_circle(mwt*vector(0,0,0), wlt[0], scale.y);
	view.ext_circle(mwt*vector(2,0,0), wlt[0], scale.y);

	lighting lt(view.lights, wlt);
	double left = lt.illuminate(-1,0,0);
	double right = lt.illuminate(1,0,0);

	tmatrix mct(mwt, view.wct);

	// Level-of-detail heuristic.
	//   xxx Figure out how this should actually work!
	vector o  = mct*vector(0,0,0) / mct.w(vector(0,0,0));
	vector oy = mct*vector(0,1,0) / mct.w(vector(0,1,0));
	vector oz = mct*vector(0,0,1) / mct.w(vector(0,0,1));
	vector x  = mct*vector(2,0,0) / mct.w(vector(2,0,0));
	vector xy = mct*vector(2,1,0) / mct.w(vector(2,1,0));
	vector xz = mct*vector(2,0,1) / mct.w(vector(2,0,1));
	float size = std::sqrt((o-oy).mag2() 
		+ (o-oz).mag2()
		+ (x-xy).mag2()
		+ (x-xz).mag2());     
	int n = int(size*180);
	if (n<5)
		n=5;
	else if (n>25)
		n=25;

	cyl_model& model = cyl_model::get(n);

	vertex* pr = model.proj;
	float* col = model.color;
	float* mv = model.verts;
	for(int v=0;v<model.nverts;v+=2) {
		double illum = lt.illuminate(0, mv[1], mv[2]);

		// We color two vertices at once, at
		//   opposite ends of the cylinder.
		col[0] = col[4] = color.r*illum;
		col[1] = col[5] = color.g*illum;
		col[2] = col[6] = color.b*illum;
		col[3] = col[7] = 1.0;
		col += 8;

		// Project two vertices.
		//   xxx Optimization: the vertices are (0,y,z) and (2,y,z).
		//       We can avoid most of the work of projecting the 
		//       second, and some of the work of projecting the first.
		mct.project(mv, *pr++); mv+=3;
		mct.project(mv, *pr++); mv+=3;
	}

	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_COLOR_ARRAY);
	glVertexPointer(4, GL_DOUBLE, sizeof(vertex), &model.proj[0].x);
	glColorPointer(4, GL_FLOAT, 4*sizeof(float), model.color);
	glShadeModel(GL_SMOOTH);

	glDrawArrays(GL_TRIANGLE_STRIP, 0, model.nverts);

	glDisableClientState(GL_COLOR_ARRAY);
	glShadeModel(GL_FLAT);

	glVertexPointer(4, GL_DOUBLE, 2*sizeof(vertex), &model.proj[0].x);
	glColor3d(color.r*left, color.g*left, color.b*left);
	glDrawArrays(GL_POLYGON, 0, model.nverts/2);

	glVertexPointer(4, GL_DOUBLE, 2*sizeof(vertex), &model.proj[1].x);
	glColor3d(color.r*right, color.g*right, color.b*right);
	glDrawArrays(GL_POLYGON, 0, model.nverts/2);
}
    
} // !namespace visual
