##########################################################################
#
# TestFileSystem.py: test the FileSystem module
#
#  Python-CDD is a library to make easier to build applications to
#  Custom Debian Distributions.
#  See http://projetos.ossystems.com.br/python-cdd for more information.
#    
# ====================================================================
# Copyright (c) 2002-2005 O.S. Systems.  All rights reserved.
#
# This software is licensed as described in the file COPYING, which
# you should have received as part of this distribution.
#
#########################################################################
# Authors: Otavio Salvador <otavio@ossystems.com.br>
#          Tiago Bortoletto Vaz <tiago@debian-ba.org>

import unittest

from os       import removedirs, unlink, mkdir, rmdir, walk
from os.path  import exists, abspath, join
from gzip     import GzipFile
from bz2      import BZ2File

import sys

from cdd import FileSystem

import logging

class FileSystemTests(unittest.TestCase):
    def __make_junk_file(self, name):
        # Make a test file.
        tmp_filename = join(self.tmpdir, name)
        tmp_file = file(tmp_filename, 'w')
        tmp_file.write("Test with a junk content.")
        tmp_file.close()

    def setUp(self):
        # Do all tests inside of testFileSystem.tmp directory
        self.tmpdir = abspath(sys.path[0]) + '/testFileSystem.tmp'
        self.fs_obj = FileSystem.FileSystem(abspath(sys.path[0]),
                                            'testFileSystem.tmp')

        if not exists(self.tmpdir):
            mkdir(self.tmpdir)

    def tearDown(self):
        # Remove all files inside of self.tmpdir.
        for root, dirs, files in walk(self.tmpdir, topdown=False):
            for name in files:
                unlink(join(root, name))
            for name in dirs:
                rmdir(join(root, name))

        if exists(self.tmpdir):
            rmdir(self.tmpdir)

    def test_1base(self):
        """FileSystem: check if the base definition match"""
        self.assert_(self.tmpdir == self.fs_obj.base(),
                     "Base attribut doesn't match.")

    def test_1base_exists(self):
        """FileSystem: check if we have the base directory for testing"""
        self.failUnless(exists(self.tmpdir))
    
    def test_2create(self):
        """FileSystem: directory creation"""
        self.fs_obj.mkdir('dir1')
        # for test:
        self.fs_obj.mkdir('dir1')
        self.failIf(not exists(self.tmpdir + '/dir1'),
                    "Failed to create 'dir1' directory.")
        # do the test without permission to do that:
        self.fs_objonroot = FileSystem.FileSystem('/')
        self.failUnlessEqual(self.fs_objonroot.mkdir('/monkey/noaccess'),
                             False)
        self.__make_junk_file('dir1/foo')
        self.failUnless(not self.fs_obj.mkdir('dir1/foo'),
                        "Create file with same name of pre-existing file")

    def test_2remove_dir(self):
        """FileSystem: remove a directory"""
        self.fs_obj.mkdir('dir1/dir2/dir3')
        self.fs_obj.rmdir('dir1/dir2/dir3')
        self.failUnless(not exists(self.tmpdir + '/dir1/dir2/dir3'),
                        "Failed to remove dir1/dir2/dir3 directory.")
        self.failUnless(exists(self.tmpdir + '/dir1/dir2'),
                        "Removed dir1/dir2 directory.")

    def test_2remove_dir2(self):
        """FileSystem: remove a leaf directory and empty all intermediate ones"""
        self.fs_obj.mkdir('dir1/dir2/dir3/')
        self.__make_junk_file('dir1/not_empty')
        self.fs_obj.rmdir('dir1/dir2/dir3/', True)

        self.failUnless(not exists(self.tmpdir + '/dir1/dir2/dir3'),
                        "Failed to remove dir1/dir2/dir3 directory.")

        self.failUnless(not exists(self.tmpdir + '/dir1/dir2'),
                        "Failed to remove dir1/dir2 directory.")

        self.failUnless(exists(self.tmpdir + '/dir1'),
                        "Removed dir1/ directory but it wasn't empty.")
        
    def test_2remove_file(self):
        """FileSystem: remove a file"""
        self.__make_junk_file('test_2remove_file')
        self.fs_obj.remove('test_2remove_file')
        self.failUnless(not self.fs_obj.remove('test_2remove_file'))
        self.failUnless(not exists(self.tmpdir + '/test_2remove_file'),
                        'Cannot remove test_2remove_file file.')

    def test_3md5sum(self):
        """FileSystem: md5sum a file"""
        self.__make_junk_file('test_3md5sum')
        self.failUnlessEqual(self.fs_obj.md5sum('test_3md5sum'),
                             '1693f3e5f42f21cb0c78d28443e4b6ce',
                             'Fail to get md5sum of test_3md5sum file.')
        self.failUnlessRaises(IOError, self.fs_obj.md5sum, 'notexist.no')

    def test_3sha1sum(self):
        """FileSystem: sha1sum a file"""
        self.__make_junk_file('test_3sha1sum')
        self.failUnlessEqual(self.fs_obj.sha1sum('test_3sha1sum'),
                             '549642be705d216245262436794515491e93185e',
                             'Fail to get sha1sum of test_3sha1sum file.')
        self.failUnlessRaises(IOError, self.fs_obj.sha1sum, 'notexist.no')
        

    def test_3size(self):
        """FIleSystem: test size method"""
        self.__make_junk_file('test_3size')
        self.failIf(not self.fs_obj.size('test_3size'),
                    'Error Size.')
        self.failUnlessRaises(OSError, self.fs_obj.size, 'monkey.ban')
        
    def test_4uncompress(self):
        """FileSystem: uncompress a file"""
        for class_ref, extension in ((GzipFile, 'gz'), (BZ2File, 'bz2'),
                                     (GzipFile, 'bz2'), (BZ2File, 'gz'),
                                     (open, 'txt')):
            file = class_ref(join(self.tmpdir,
                                'test_3uncompress.' + extension), 'w')
            file.write('Junk content for test_3uncompress.')
            file.close()

            try:
                self.fs_obj.uncompress('test_3uncompress.' + extension)
                if class_ref == open:
                    self.fail("We should receive a IOError Exception.")
            except IOError, msg:
                if class_ref == open:
                    pass
                else:
                    self.failUnless(exists(join(self.tmpdir,
                                           'test_3uncompress')),
                                    'Failed to create test_3uncompress file.')
        
            unzip_file = open(join(self.tmpdir, 'test_3uncompress'), 'r')
            unzip_content = unzip_file.read()
            unzip_file.close()
            self.failUnlessEqual(unzip_content,
                                 'Junk content for test_3uncompress.',
                                 'Uncompressed file with invalid content.')
        self.failUnlessRaises(IOError, self.fs_obj.uncompress, 'noexist.no')
            
    def test_4compress(self):
        """FileSystem: compress a file"""
        for format, class_ref, extension in ((FileSystem.GZIP, GzipFile, 'gz'),
                                             (FileSystem.BZIP2, BZ2File,
                                              'bz2')):
            file = open(join(self.tmpdir, 'test_4compress.txt'), 'w')
            file.write('Junk content for test_4compress.')
            file.close()
            
            try:
                self.fs_obj.compress("test_4compress.txt", format)
            except IOError, msg:
                self.fail(msg)
            
            unzip = class_ref(join(self.tmpdir,
                              'test_4compress.txt.' + extension), 'r')
            self.failUnlessEqual(unzip.read(),
                                 'Junk content for test_4compress.',
                                 'Uncompressed file with invalid content.')
            unzip.close()
        
        self.failUnlessRaises(SyntaxError, self.fs_obj.compress, 'noexist.no',
                              'not')
        self.failUnlessRaises(IOError, self.fs_obj.compress, 'noexist.no')
        
            
    def test_5match(self):
        """FileSystem: match a pattern"""
        self.fs_obj.mkdir('test_5match')
        file = open(join(self.tmpdir, 'test_5match/foo.txt'), 'w')
        file.write('Junk content for test_5match.')
        file.close()
        
        for expression, valid in (('foo\.txt$', [join(self.tmpdir,
                                  'test_5match/foo.txt')]),
                                  ('bar\.txt$', [])):
            if self.fs_obj.match(expression) != valid:
                self.fail("Answer '%s' for expression '%s' is wrong." %
                          (valid , expression))
        
def suite():
    suite = unittest.TestSuite()

    suite.addTest(unittest.makeSuite(FileSystemTests, 'test'))

    return suite

if __name__ == '__main__':
    log = logging.getLogger()
    log_handler = logging.FileHandler(sys.argv[0][:-3] + '.log')
    log.setLevel(logging.DEBUG)
    log.addHandler(log_handler)
    unittest.main(defaultTest='suite')
