########################################################################
# $Header: /var/local/cvsroot/4Suite/Ft/Server/Server/Commands/CommandUtil.py,v 1.9 2005/05/20 20:21:44 jkloth Exp $
"""
Functions used by various commands

Copyright 2003 Fourthought, Inc. (USA).
Detailed license and copyright information: http://4suite.org/COPYRIGHT
Project home, documentation, distributions: http://4suite.org/
"""

import os, sys, sha, getpass, posixpath

from Ft.Server.Server import SCore
from Ft.Server.Common import AclConstants, Schema, ClAuthenticate
from Ft.Server.Server.Lib import ConfigFile, LogUtil

class _PasswordManager(ClAuthenticate.PasswordFileManager):

    fileBaseName = '4ssd'

    def getLogin(self):
        """
        Returns a tuple of (username, password-hash).
        """
        for entry in self.readEntries():
            if len(entry) != 2:
                continue
            return entry
        return (None, None)

    def setLogin(self, username, passwordHash):
        entries = [(username, passwordHash)]
        self.writeEntries(entries)
        return        

PasswordManager = _PasswordManager()

def GetRepository(options, cmdName):
    """
    Given a dictionary of parsed command-line options (see FancyGetOpt),
    reads a repo config file (which may be identified by the 'config-file'
    key in the options dict, and defaults to the value of the environment
    variable FTSERVER_CONFIG_FILE), obtaining the config file's contents
    as a dictionary returned by Ft.Server.Server.Lib.ConfigFile.Read(),
    and creating a properties dictionary that is the value of just one of
    the keys from the original dictionary (the key is either the 'core-id'
    option, the value of environment variable FTSS_CORE_ID, or 'Core').
    If a username and password hash were not found in the options dict,
    prompts for them. Uses username and password hash to obtain a repo
    access object from the local repository (via SCore.GetRepository()).
    Checks ACL in the repo to see if the command with the given name can
    be run by the user, raising an exception if it can't or if the name
    doesn't correspond to a command in the repo. Returns a tuple of:
    username, password hash, config file properties dictionary, repo
    access object.

    Used by pretty much every '4ss' and '4ss_manager' command.
    """
    coreId = options.get('core-id',os.environ.get('FTSS_CORE_ID', 'Core'))
    properties = ConfigFile.Read(options.get('config-file'))
    properties = properties[coreId]

    if 'username' in options:
        userName = options['username']
    else:
        userName = None
        
    if 'password' in options:
        passwdHash = sha.new(options['password']).hexdigest()
    else:
        passwdHash = None

    if userName is None and passwdHash is None:
        # Try and use a stored login
        userName, passwdHash = PasswordManager.getLogin()

    if userName is None:
        userName = ClAuthenticate.GetUserName('4SS Manager Name: ')

    if passwdHash is None:
        password = ClAuthenticate.GetPass('Password for %s: ' % userName)
        passwdHash = ClAuthenticate.HashPasswd(password)

    repo = SCore.GetRepository(userName, passwdHash, LogUtil.NullLogger(),
                               properties, verify=1)

    # Verify that the command can be run
    cmd = repo.getCommand(cmdName)
    if cmd is None:
        raise ValueError("Unknown Command: %s" % cmdName)
    cmd.verifyAcl(AclConstants.EXECUTE_ACCESS,0)

    return userName, passwdHash, properties, repo


