########################################################################
# $Header: /var/local/cvsroot/4Suite/Ft/Server/Client/Commands/Fetch/DocDef.py,v 1.10 2004/09/23 20:53:29 mbrown Exp $
"""
Implementation of '4ss fetch docdef' command
(functions defined here are used by the Ft.Lib.CommandLine framework)

Copyright 2004 Fourthought, Inc. (USA).
Detailed license and copyright information: http://4suite.org/COPYRIGHT
Project home, documentation, distributions: http://4suite.org/
"""

__doc__ = """This command produces a summary of the information in a
Document Definition resource ("DocDef") in a 4Suite repository. \
For XSLT based DocDefs, fetches the XSLT document. For XPath based \
DocDefs, provides a formatted summary. If necessary, you will be \
prompted first for credentials and access info to connect to the \
repository."""

import os, sys

from Ft.Server.Client.Commands import CommandUtil
from Ft.Server.Common import ResourceTypes, ValidationInfo, AclConstants


def Run(options, args):
    repo = CommandUtil.GetRepository(options, '4ss.fetch.docdef')
    if repo is not None:
        try:
            ddRoot = repo.getSystemDocDefContainer()
            dd = ddRoot.fetchResource(args['path'])
            if dd.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT_DEFINITION):
                ddtype = 'XSLT'
            elif dd.isResourceType(ResourceTypes.ResourceType.XPATH_DOCUMENT_DEFINITION):
                ddtype = 'XPath'
            else:
                ddtype = 'unknown type'

            CommandUtil.WritePreamble("Document Definition %s (%s):" % (dd.getPath(), ddtype))
            if ddtype == 'XPath':
                print "Base Definitions: [%s]" % ','.join(map(lambda x: x.getUri(), dd.getBaseDefinitions()))
                print "Namespace mappings:"
                for name, value in dd.getNsMap().items():
                    print "%s --> %s" % (name, value)
                print
                print "RDF Mappings:"
                for mapping in dd.getRdfMaps():
                    print "%s  %s  %s" % (mapping[0],mapping[1],mapping[2])
                print
                print "Read ACL: %s" % dd.getAcl(AclConstants.READ_ACCESS)
                print "Write ACL: %s" % dd.getAcl(AclConstants.WRITE_ACCESS)
                print
                cp = dd.getCreationParams()
                print "Full Text Indexing: %s" % cp.fullTextIndex
                if cp.extModules:
                    print "XSLT Extension Modules:"
                    for e in cp.extModules:
                        print "  %s" % e
                print "Enforce Schema: %s" % cp.enforceSchema
                vi = cp.validationInfo
                if vi.validationType != ValidationInfo.ValidationType.NONE:
                    # FIXME: interpret validation type
                    print "Validation Type: %s" % vi.validationType
            else:
                # XSLT or unknown type: just print it
                content = dd.getContent()
                sys.stdout.write(content)
                if content[-1] != '\n':
                    sys.stderr.write('\n')
                    sys.stderr.flush()
        finally:
            try:
                repo.txRollback()
            except:
                pass

    return


def Register():
    from Ft.Lib.CommandLine import Options, Command, Arguments
    cmd = Command.Command('docdef',
                          'Print info about a Document Definition in a repository',
                          'BASE_XML',
                          __doc__,
                          function = Run,
                          arguments = [Arguments.RequiredArgument('path',
                                                                  'the path of the Document Definition',
                                                                  str),
                                       ],
                          fileName = __file__,
                          )
    return cmd
