/*
 *  ORBit-C++: C++ bindings for ORBit.
 *
 *  Copyright (C) 2000 Andreas Kloeckner
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author:	Andreas Kloeckner <ak@ixion.net>
 *
 *  Purpose:	idl compiler backend main routine
 *
 */




#include <glib.h>
#include <libIDL/IDL.h>
#include <orbit-idl2.h>
#include <stdexcept>
#include <fstream>
#include <string>
#include "pass_gather.hh"
#include "pass_xlate.hh"
#include "pass_stubs.hh"
#include "pass_skels.hh"




extern "C" {
	void idlCPPBackend(OIDL_Output_Tree *tree,OIDL_Run_Info *rinfo);
	extern OIDL_Backend_Info const orbit_idl_backend = {
		    "c++",
		    idlCPPBackend
		};
}




void idlWritePreamble(ostream &ostr) {
	ostr
	<< "// -----------------------------------------------------" << endl
	<< "// generated by orbitcpp, do not edit" << endl
	<< "// -----------------------------------------------------" << endl
	<< endl << endl;
}



void idlCPPBackend(OIDL_Output_Tree *tree,OIDL_Run_Info *rinfo) {
	try {
		// generate base name
		string basename(g_basename(rinfo->input_filename));
		string::size_type dotpos = basename.find('.');

		if (dotpos != string::npos) basename.erase(dotpos,basename.size());

		// state
		IDLCompilerState state(basename,tree->tree);

		ofstream header1((basename + IDL_CPP_HEADER_EXT).c_str());
		ofstream module1((basename + IDL_CPP_MODULE_EXT).c_str());
		idlWritePreamble(header1);
		idlWritePreamble(module1);

		ofstream header2((basename + IDL_CPP_STUB_HEADER_EXT).c_str());
		ofstream module2((basename + IDL_CPP_STUB_MODULE_EXT).c_str());
		idlWritePreamble(header2);
		idlWritePreamble(module2);

		ofstream header3((basename + IDL_CPP_SKEL_HEADER_EXT).c_str());
		ofstream module3((basename + IDL_CPP_SKEL_MODULE_EXT).c_str());
		idlWritePreamble(header3);
		idlWritePreamble(module3);

		IDLPassGather pass0(state);
		IDLPassXlate pass1(state,header1,module1);
		IDLPassStubs pass2(state,header2,module2);
		IDLPassSkels pass3(state,header3,module3);

		pass0.runPass();
		pass1.runPass();
		pass2.runPass();
		pass3.runPass();

		// "Pass" four: write include stuff
		{
			ofstream header((basename + IDL_CPP_MAIN_HEADER_EXT).c_str());
			ofstream module((basename + IDL_CPP_MAIN_MODULE_EXT).c_str());
			idlWritePreamble(header);
			idlWritePreamble(module);
			header
			<< "#include \"" << basename << IDL_CPP_HEADER_EXT << "\"" << endl
			<< "#include \"" << basename << IDL_CPP_STUB_HEADER_EXT << "\"" << endl
			<< "#include \"" << basename << IDL_CPP_SKEL_HEADER_EXT << "\"" << endl;
			module
			<< "#include \"" << basename << IDL_CPP_MODULE_EXT << "\"" << endl
			<< "#include \"" << basename << IDL_CPP_STUB_MODULE_EXT << "\"" << endl
			<< "#include \"" << basename << IDL_CPP_SKEL_MODULE_EXT << "\"" << endl;
		}
	} catch (exception &ex) {
		cerr << ex.what() << endl;
	}
	catch (...) {
		cerr << "C++ code generation failed because of an unknown exception" << endl;
	}
}
