// Copyright (C) 2005 Shai Ayal <shaiay@users.sourceforge.net>
//  
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//  
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//  
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  

#include "prop_handle.h"
#include "octplotapp.h"
#include <set>

//! Adds new value, deleteing the old one
void  HandleScalar::Add(const ocpl::Handle new_handle)
{
  Clear();
  Value.push_back(new_handle);
}

//! adds a new value, first checking that it does not exist
// already to ensure uniquness
void  HandleVectNoOwn::Add(const ocpl::Handle new_handle)
{
  for ( First() ; !IsDone() ; Next() )
    if( CurrentHandle() == new_handle) return;
  
  Value.push_back(new_handle);
}

void  HandleVectNoOwn::Delete(const ocpl::Handle handle) 
{
  for ( First() ; !IsDone() ; Next() ) {
    if( CurrentHandle() == handle) {
      Value.erase(im);
      return;
    }
  }
}
  
void HandleVectNoOwn::AllDraw()
{
  for ( First() ; !IsDone() ; Next() )
    (OctPlotApp::GetObjectD(CurrentHandle()))->draw();
}

void HandleVectNoOwn::SetPrinting(bool set)
{
  for(First() ; !IsDone() ; Next() ) {
    (OctPlotApp::GetObjectD( CurrentHandle()))->SetPrinting(set);
  }
}

void HandleVect::Clear()
{
  for ( First() ; !IsDone() ; Next() )
    delete OctPlotApp::GetObjectD(CurrentHandle());

  HandleVectNoOwn::Clear();
}

void HandleVect::Delete(ocpl::Handle handle)
{
  HandleVectNoOwn::Delete(handle);
  delete OctPlotApp::GetObjectD(handle);
}

void HandleVectNoOwn::Get(ocpl::command& com)
{
  ocpl::Real *val = new ocpl::Real[Value.size()];
  int i=0;
  for (First() ; !IsDone() ; Next() )
    val[i++] = CurrentHandle();
 
  com.argout(0,ocpl::real,Value.size(),1,reinterpret_cast<char*>(val),true);
}

void HandleScalar::Set(ocpl::command& com) 
{
  if(com.argin(2)->id != ocpl::real) {
    ret_error(com,"Value of property must be numerical");
    return;
  }

  if(com.argin(2)->nc*com.argin(2)->nr != 1) {
    ret_error(com,"New value must be a single handle");
    return;
  }

  ocpl::Handle hnd = 
    static_cast
    <ocpl::Handle>
    ((reinterpret_cast<double*>(com.argin(2)->data))[0]);

  if(GetObjectD(hnd)==0) {
    ret_error(com,"New value must be a valid handle");
    return;
  }
  
  Clear();
  Add(hnd);
  com.init_argout(0);
}

void HandleVectNoOwn::Set(ocpl::command& com) 
{
  if(com.argin(2)->id != ocpl::real) {
    ret_error(com,"Value of property must be numerical");
    return;
  }


  // setting a handle vector is only allowed if the new vectro is a
  // permutation of the old vector
  
  // first check that the number of elemnts is similar
  if(com.argin(2)->nc*com.argin(2)->nr != Size()) {
    ret_error(com,"New value must be a permutation of current value");
    return;
  }
   
  // now check if they are a permutation by putting into sets and comparing
  
  typedef std::set<ocpl::Handle> hset ;
  hset os,ns;

  for(int i=0;i<com.argin(2)->nc*com.argin(2)->nr;i++) {
    ocpl::Handle hnd = static_cast<ocpl::Handle>(
      (reinterpret_cast<double*>(com.argin(2)->data))[i]);
    if(GetObjectD(hnd)) {
      ns.insert(hnd);
      os.insert(Value[i]);
    }
    else {
      ret_error(com,"Handle Vector: no such handle");
      return;
    }
  }

  hset::iterator i1,i2;
  for(i1=os.begin(),i2=ns.begin(); ! (i1==os.end() || i2==ns.end()) ; i1++,i2++) {
    if(*i1 != *i2) {
      ret_error(com,"New value must be a permutation of current value");
      return;
    }
  }

  // OK -- it's legal, now let's copy everything into place in the right order
  Value.clear();
  for(int i=0;i<com.argin(2)->nc*com.argin(2)->nr;i++) {
    ocpl::Handle hnd = static_cast<ocpl::Handle>(
      (reinterpret_cast<double*>(com.argin(2)->data))[i]);
    Value.push_back(hnd);
  }

  com.init_argout(0);
}

