/* $Id: protocol.h,v 1.12 2004/09/21 20:45:25 graziano Exp $ */


#ifndef PROTOCOL_H
#define PROTOCOL_H


/*
 * This module defines functions useful in establishing and maintaining
 * connections to other processes on local or remote hosts.  The name is an
 * anachronism.
 */


#include <sys/types.h>    /* pid_t */
#include "dnsutil.h"      /* IPAddress */

#ifdef __cplusplus
extern "C" {
#endif

/* A definition for socket call-back functions (see NotifyOnDisconnection()
 * and NotifyOnNewConnection()). */
typedef void (*SocketFunction)(Socket);
typedef void (*NewConnectionFunction)(Socket ear, Socket newSock);


/*
 * Condition socket #sd# to be reusable (if #reuse# is set) and to use
 * #bufSize# buffers for sending/receiving (if bufSize is not 0). If
 * compiled with non blocking sockets, set the socket to be non-blocking.
 * Returns 0 on failure, otherwise 1 or 2 depending on what the OS sets
 * the socket buffer to (linux sets the buffer to be double what you
 * asked for).
 */
int
ConditionSocket(	Socket sd,
			int reuse,
			int bufSize);

/*
 * Attempts to establish a connection to the server listening to #addr#:#port#.
 * If successful within #timeOut# seconds, returns 1 and sets #sock# to the
 * socket bound to the connection, else returns 0 (and #sock# will be set
 * to NO_SOCKET). If #bufSize# is greater than 0, set the send and
 * receive bffer for the socket to it. A #timeOut# of 0 will disable
 * timeouts while a negative value will use adaptive timeouts.
 *
 * NOTE: too small buffer size could spell troubles.
 */
int
CallAddrBuff(	IPAddress addr, 
		unsigned short port, 
		int bufSize,
		Socket *sock,
		double timeOut);
#define CallAddr(addr, port, sock, timeOut) \
	CallAddrBuff(addr, port, -1, sock, timeOut)


/*
 * Closes connections opened via calls to this module's functions.  Each
 * parameter is a boolean value indicated whether that type of connection
 * should be included in those closed.
 */
void
CloseConnections(int closeEars,
                 int closePipes,
                 int closeSockets);
#define CloseAllConnections() CloseConnections(1, 1, 1)


/*
 * Tests all connections opened via calls to this module's functions and
 * closes any that are no longer connected.  Returns the number of connections
 * closed.
 */
int
CloseDisconnections(void);


/**
 * Tears down #sock#.  If #waitForPeer# is non-zero, the function waits this
 * many seconds for the host on the other end to close the connection and fails
 * if no such close is detected.  If this parameter is zero, the function
 * closes #sock# immediately.  Returns 1 and sets #sock# to NO_SOCKET if
 * successful, else returns 0.
 */
int
CloseSocket(Socket *sock,
            int waitForPeer);
#define DROP_SOCKET(sock) CloseSocket(sock, 0)


/**
 * Removed all records of #sock# from our FD_SETs
 */
void
ClearSocket(Socket sock);

/*
** Spawns a new process, a duplicate of the running one.  Returns 1 if
** successful, 0 otherwise.  Returns in #pid# a 0 to the child process and the
** process id of the child to the parent.  Both processes are given a pair of
** connections in the Socket parameters that can be used to communicate between
** the parent and child.  The parent process should send information to the
** child via #parentToChild# and receive information via #childToParent#; the
** child reads from the former and writes to the latter.  The parameters may be
** NULL indicating that communication is unidirectional (one parameter NULL),
** or that no communication will take place (both NULL).
*/
int
CreateLocalChild(pid_t *pid,
                 Socket *parentToChild,
                 Socket *childToParent);


/*
** Attempts to bind to any port between #startingPort# and #endingPort#,
** inclusive.  If successful, returns 1 and sets #ear# to the bound socket and
** #earPort# to the port to which it is bound, else returns 0.
*/
int
EstablishAnEar(unsigned short startingPort,
               unsigned short endingPort,
               Socket *ear,
               unsigned short *earPort);
/*
 *Attempts to bind to any port between #startingPort# and #endingPort#,
 inclusive. If #bufSize# is greater than 0, the socket's buffer are set
 to that value. If successful, returns 1 and sets #ear# to the bound
 socket and #earPort# to the port to which it is bound, else returns 0.
 */
int
EstablishAnEarBuff(unsigned short startingPort,
		   unsigned short endingPort,
		   Socket *ear,
		   unsigned short *earPort,
		   int bufSize);
#define EstablishAnEar(start, end, ear, port) \
        EstablishAnEarBuff(start, end, ear, port, 0)


/**
 * Checks all connections established via calls to this module for
 * incoming messages.  If one is detected within #timeOut# seconds,
 * returns 1, sets #sd# to the socket containing the message. If no
 * message detected, returns 0. 
 *
 * If #timeOut# is negative, IncomingRequest will poll and returns
 * immediately if no messages are waiting. If #timeOut# is zero, it will
 * block till a messages arrives.
 * 
 * NOTE: you have to use SocketIsAvailable to notify IncomingRequest that
 * the socket is again available (once the socket is been returned from
 * IncomingRequest) */
int 
IncomingRequest(double timeOut,
                Socket *sd);


/**
 * When a socket is returned by IncomingRequest, that socket won't be
 * listen till this function is called.
 * Return 0 upon failure.
 */
int
SocketIsAvailable(Socket sd);

/**
 * Tell NWS that a specific socket is still in use and IncomingRequest
 * shouldn't listen to it.
 */
int
SocketInUse(Socket sd);

/*
** Returns 1 or 0 depending on whether or not #sd# is connected to another
** process.
*/
int
IsOkay(Socket sd);

/**
 * returns 1 or 0 depending on wheter the socket is a pipe or not 
 */
int
IsPipe(Socket sd);


/**
 * Registers a function that should be called whenever a socket is disconnected,
 * either directly via a call to CloseSocket(), or indirectly because the peer
 * termintes the connection.  The function is passed the closed socket after it
 * has been closed. Returns 1 on success, 0 otherwise.
 */
int
NotifyOnDisconnection(SocketFunction notifyFn);


/**
 * Registers a function that should be called whenever a new connection is 
 * made to an ear (and thus a new socket is created). The function is
 * passed the #ear# that received the connection and the in #newSock# the
 * newly created socket. 
 * Returns 1 on success, 0 otherwise.
 */
int
NotifyOnNewConnection(NewConnectionFunction notifyFn);


/*
** Pass socket #sock# along to child #child# -- call after a successful call
** to CreateLocalChild().  The parent process will no longer listen for
** activity on #sock#.  Closing the socket will be put off until after the
** child dies, just in case the parent and child share descriptors.
*/
int
PassSocket(Socket *sock,
           pid_t child);


/**
 * Receives #byteSize# bytes from connection #sd# and stores them in the
 * memory referenced by #bytes#.  The caller must assure that #bytes# is at
 * least #byteSize# bytes long.  Returns the # of bytes succesfully
 * received within #timeOut# seconds. #dontWait# is valid only if
 * nonblocking socket code has been compiled in: if set don't wait for
 * the socket to be ready, but return immediately.
 * If #timeOut# is zero, timeout are disabled alltogether.
 */
int
RecvBytesWait(Socket sd,
              void *byte,
              size_t byteSize,
              double timeOut,
	      int dontWait);
#define RecvBytes(sd, bytes, byteSize, timeOut) \
	RecvBytesWait(sd, bytes, byteSize, timeOut, 0)

/**
 * Sends #bytesSize# bytes from the memory pointed to by #bytes# on
 * connection #sd#.  Returns the # of bytes succesfully sent within
 * #timeOut# seconds. #dontWait# is valid only if nonblocking socket code
 * has been compiled in: if set don't wait for the socket to be ready,
 * but return immediately.
 * If #timeOut# is zero, timeout are disabled alltogether.
 */
int
SendBytesWait(Socket sd,
              const void *bytes,
              size_t byteSize,
              double timeOut,
	      int dontWait);
#define SendBytes(sd, bytes, byteSize, timeOut) \
	SendBytesWait(sd, bytes, byteSize, timeOut, 0)


#ifdef __cplusplus
}
#endif

#endif
