(* $Id: netstream.mli,v 1.2 2000/06/24 20:20:33 gerd Exp $
 * ----------------------------------------------------------------------
 *
 *)


(* A netstream is an input channel that is read block by block. The 
 * fragment of the channel currently loaded into memory is called the
 * current window of the netstream.
 *
 * PICTURE:
 *
 * 0            window_position     current_length                  EOS
 * +------------------+-------------------+--------------------------+
 *                    ====================
 *                     The current window
 *
 * window_length = current_length - window_position
 *
 * There is an automatism that the window has a certain length. If possible,
 * the window is at least twice the block size long, where a "block" is
 * the amount of data that is read from the input channel in one step.
 *
 * (The idea is that you choose as block size the number of bytes you want
 * to analyze at once, and which must be loaded into memory. You can start
 * your analysis at window_position and proceed until window_position +
 * blocksize without having to check whether your window is large enough.
 * Only when the first blocksize bytes of the window are already processed,
 * the window must be enlarged by loading the next block.)
 *
 * If you want that the window becomes larger, you can call 'want' (to
 * enlarge the window to a certain size) or 'want_another_block' (to load
 * just another block from the input channel). Note that this affects only
 * the current window and not future windows.
 *
 * If you do not need the first n bytes of the window anymore, you can
 * call 'move' to move the beginning of the window by n bytes. If the
 * window becomes too small after this operation, it is enlarged until
 * it has twice the block size or until it reaches EOS.
 *)

type t

val create_from_channel : in_channel -> int option -> int -> t
    (* create_from_channel ch maxlength blocksize:
     * The new netstream reads from the channel 'ch'. If maxlength = None,
     * the channel is read until EOF. If maxlength = Some n, at most n bytes
     * are read; i.e. the netstream reads until n bytes have been read or
     * until EOF has been reached, whatever comes first. The blocksize 
     * specifies the number of bytes to read at once.
     *)

val create_from_string : string -> t
    (* Creates a new netstream from a string. The initial window of this
     * netstream is a copy of the passed string.
     *)

val block_size : t -> int
   (* Returns the (immutable) block size. *)

val current_length : t -> int
    (* Returns the number of bytes read so far. *)

val at_eos : t -> bool
    (* True iff EOS (end of stream) is reached, i.e. the last byte of the
     * window is the last byte of the stream.
     *)

val window_position : t -> int
    (* Returns the absolute position of the current window. *)

val window_length : t -> int
    (* Returns the length of the current window. *)

val window : t -> Netbuffer.t
    (* Returns the current window. *)

val move : t -> int -> unit
    (* move s n:
     * Moves the window: The first n bytes of the current window are 
     * discarded. If the window would become smaller than twice the
     * blocksize and if the end of the stream is not yet reached, another
     * block is read from the input channel and appended to the window.
     * 
     * PRECONDITION:
     * - n <= window_length
     *)

val want : t -> int -> unit
    (* want s n:
     * If the window is smaller than n bytes, it is tried to enlarge
     * the window such that it is at least n bytes long. The enlargement
     * is not possible if the stream is not long enough; in this case
     * the window becomes as large as possible.
     *)

val want_another_block : t -> unit
    (* Enlarges the window by another block (if possible i.e. if the stream
     * is long enough).
     *)

val print_stream : t -> unit

(* ======================================================================
 * History:
 * 
 * $Log: netstream.mli,v $
 * Revision 1.2  2000/06/24 20:20:33  gerd
 * 	Added the toploop printer.
 *
 * Revision 1.1  2000/04/15 13:07:48  gerd
 * 	Initial revision.
 *
 * 
 *)
