/* nexp_speakers.c
  
   Network Expect speakers.

   Copyright (C) 2007, 2008, 2009 Eloy Paris

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <unistd.h>
#include <pcap.h>
#include <dnet.h>
#include <errno.h>
#include <string.h>

#include "nexp_speakers.h"
#include "xmalloc.h"
#include "xstrdup.h"
#include "missing.h"

#define SNAPLEN 65535

extern int vflag;

static int spawn_id;

/*
 * Linked list of all active writers. The -o parameter to the
 * send_network command specifies one of these speakers.
 */
static struct nexp_speaker *speakers;

/*
 * Returns a pointer to the listener structure that corresponds to the
 * listener ID passed as a parameter.
 *
 * Returns NULL if no speaker with the specified ID exists.
 */
struct nexp_speaker *
lookup_speaker(const char *speaker_id)
{
    struct nexp_speaker *s;

    for (s = speakers; s; s = s->next)
	if (!strcmp(s->name, speaker_id) )
	    break;

    return s;
}

static int
iface_callback(const struct intf_entry *entry, void *arg _U_)
{
    struct nexp_speaker *s, *new_speaker;
    eth_t *eth;

    if ( (entry->intf_flags & INTF_FLAG_UP) == 0)
	/* Do not create a speaker if the interface is not up. */
	return 0;

    eth = eth_open(entry->intf_name);
    if (!eth) {
	if (vflag)
	    fprintf(stderr,
		    "couldn't create default interface \"%s\" speaker: %s\n",
		    entry->intf_name, strerror(errno) );
    } else {
	/*
	 * Create the speaker.
	 */
	new_speaker = xmalloc(sizeof(struct nexp_speaker) );
	memset(new_speaker, 0, sizeof(struct nexp_speaker) );
	new_speaker->type = SPEAKER_ETHER;
	strlcpy(new_speaker->name, entry->intf_name,
		sizeof(new_speaker->name) );
	new_speaker->_iface.ether = eth;
	new_speaker->_iface.ifname = xstrdup(entry->intf_name);

	/* Add new speaker to linked list of speakers. */
	if (!speakers)
	    speakers = new_speaker;
	else {
	    for (s = speakers; s->next; s = s->next);

	    s->next = new_speaker;
	}
    }

    return 0;
}

/*
 * Default speakers are created for each interface that is up and for
 * IPv4 packets. Sorry, no IPv6 yet (libdnet doesn't support it.)
 */
void
create_default_speakers(void)
{
    struct nexp_speaker *s, *new_speaker;
    intf_t *i;
    ip_t *ip;
#ifdef __linux__
    int ip6;
#endif

    /*
     * Create default IPv4 speaker.
     */
    ip = ip_open();
    if (!ip) {
	if (vflag)
	    fprintf(stderr, "couldn't create default IPv4 speaker: %s\n",
		    strerror(errno) );
    } else {
	new_speaker = xmalloc(sizeof(struct nexp_speaker) );
	memset(new_speaker, 0, sizeof(struct nexp_speaker) );
	new_speaker->type = SPEAKER_IPV4;
	strlcpy(new_speaker->name, DEFAULT_IPV4_SPEAKER_NAME,
		sizeof(new_speaker->name) );
	new_speaker->_ip = ip;

	/* Add new speaker to linked list of speakers. */
	if (!speakers)
	    speakers = new_speaker;
	else {
	    for (s = speakers; s->next; s = s->next);

	    s->next = new_speaker;
	}
    }

#ifdef __linux__
    /*
     * Create default IPv6 speaker.
     */
    ip6 = socket(AF_INET6, SOCK_RAW, IPPROTO_RAW);
    if (ip6 == -1) {
	if (vflag)
	    fprintf(stderr, "couldn't create default IPv6 speaker: %s\n",
		    strerror(errno) );
    } else {
	new_speaker = xmalloc(sizeof(struct nexp_speaker) );
	memset(new_speaker, 0, sizeof(struct nexp_speaker) );
	new_speaker->type = SPEAKER_IPV6;
	strlcpy(new_speaker->name, DEFAULT_IPV6_SPEAKER_NAME,
		sizeof(new_speaker->name) );
	new_speaker->_ip6 = ip6;

	/* Add new speaker to linked list of speakers. */
	if (!speakers)
	    speakers = new_speaker;
	else {
	    for (s = speakers; s->next; s = s->next);

	    s->next = new_speaker;
	}
    }
#endif

    /*
     * Now create layer 2 speakers, i.e. one speaker per interface that is
     * up.
     */
    i = intf_open();
    if (!i) {
	if (vflag)
	    fprintf(stderr, "couldn't create default layer 2 speaker(s)\n");
    } else {
	intf_loop(i, iface_callback, NULL);
	intf_close(i);
    }

    /*
     * Now create a defaul hexadecimal speaker.
     */
    new_speaker = xmalloc(sizeof(struct nexp_speaker) );
    memset(new_speaker, 0, sizeof(struct nexp_speaker) );
    new_speaker->type = SPEAKER_HEX;
    strlcpy(new_speaker->name, DEFAULT_HEX_SPEAKER_NAME,
	    sizeof(new_speaker->name) );

    /* Add new speaker to linked list of speakers. */
    if (!speakers)
	speakers = new_speaker;
    else {
	for (s = speakers; s->next; s = s->next);

	s->next = new_speaker;
    }

    /*
     * Now create a defaul stdout speaker.
     */
    new_speaker = xmalloc(sizeof(struct nexp_speaker) );
    memset(new_speaker, 0, sizeof(struct nexp_speaker) );
    new_speaker->type = SPEAKER_STDOUT;
    strlcpy(new_speaker->name, DEFAULT_STDOUT_SPEAKER_NAME,
	    sizeof(new_speaker->name) );

    /* Add new speaker to linked list of speakers. */
    if (!speakers)
	speakers = new_speaker;
    else {
	for (s = speakers; s->next; s = s->next);

	s->next = new_speaker;
    }

    /* Create a null speaker that just sends packets to a black hole */
    new_speaker = xmalloc(sizeof(struct nexp_speaker) );
    memset(new_speaker, 0, sizeof(struct nexp_speaker) );
    new_speaker->type = SPEAKER_NULL;
    strlcpy(new_speaker->name, DEFAULT_NULL_SPEAKER_NAME,
	    sizeof(new_speaker->name) );

    /* Add new speaker to linked list of speakers. */
    if (!speakers)
	speakers = new_speaker;
    else {
	for (s = speakers; s->next; s = s->next);

	s->next = new_speaker;
    }
}

void
close_speaker(struct nexp_speaker *s)
{
    struct nexp_speaker *previous_s;

    switch (s->type) {
    case SPEAKER_IPV4:
	ip_close(s->_ip);
	break;
#ifdef __linux__
    case SPEAKER_IPV6:
	close(s->_ip6);
	break;
#endif
    case SPEAKER_ETHER:
	eth_close(s->_iface.ether);
	free( (void *) s->_iface.ifname);
	break;
    case SPEAKER_PCAP:
	pcap_close(s->_pcap.pd);
	pcap_dump_close(s->_pcap.pdumper);
	free( (void *) s->_pcap.fname);
	break;
    case SPEAKER_HEX:
    case SPEAKER_STDOUT:
    case SPEAKER_NULL:
	/* These are simple output methods; there's nothing extra to do */
	break;
    case SPEAKER_STREAM:
    case SPEAKER_DGRAM:
	close(s->_socket.fd);
	free(s->_socket.dsthost);
	free(s->_socket.dstport);
	break;
    }

    /* Remove listener from linked list of speakers. */
    if (speakers == s)
	/* Element to remove is at the beginning of the list. */
	speakers = s->next;
    else {
	/* Find linked list item previous to one we want to remove. */
	for (previous_s = speakers;
	     previous_s->next != s;
	     previous_s = previous_s->next);

	previous_s->next = s->next;
    }

    free(s);
}

/*
 * This function is registered with atexit() so all active listeners and
 * speakers are closed when we terminate execution normally.
 */
void
close_speakers(void)
{
    while (speakers) {
#ifdef DEBUG
	printf("Closing speaker %s\n", speakers->name);
#endif
	close_speaker(speakers);
    }
}

void
speakers_info(void)
{
    struct nexp_speaker *s;
    int nspeakers, i;
    static const char *speaker_types_names[] = {
	[SPEAKER_ETHER] = "raw Ethernet frames",
	[SPEAKER_IPV4] = "IPv4 packets (kernel-routed)",
#ifdef __linux__
	[SPEAKER_IPV6] = "IPv6 packets (kernel-routed)",
#endif
	[SPEAKER_HEX] = "hexdump to standard output",
	[SPEAKER_PCAP] = "PCAP file",
	[SPEAKER_STDOUT] = "standard output",
	[SPEAKER_NULL] = "black hole",
	[SPEAKER_STREAM] = "TCP socket",
	[SPEAKER_DGRAM] = "UDP socket"
    };

    printf("\nSpeakers:\n");

    for (s = speakers, nspeakers = 0; s; s = s->next, nspeakers++);

    printf("  Number of speakers: %d\n", nspeakers);

    for (s = speakers, i =0; s; s = s->next, i++) {
	printf("  Speaker #%d:\n", i);
	printf("    Name: %s\n", s->name);
	printf("    Type: %s\n", speaker_types_names[s->type]);
	switch (s->type) {
	case SPEAKER_IPV4:
#ifdef __linux__
	case SPEAKER_IPV6:
#endif
	case SPEAKER_HEX:
	case SPEAKER_STDOUT:
	case SPEAKER_NULL:
	    break;
	case SPEAKER_ETHER:
	    printf("    Interface: %s\n", s->_iface.ifname);
	    break;
	case SPEAKER_PCAP:
	    printf("    Writing to: %s\n", s->_pcap.fname);
	    printf("    Packet capture descriptor at: %p\n", s->_pcap.pd);
	    break;
	case SPEAKER_STREAM:
	case SPEAKER_DGRAM:
	    printf("    File descriptor: %d\n", s->_socket.fd);
	    printf("    Host: %s\n", s->_socket.dsthost);
	    printf("    Port: %s\n", s->_socket.dstport);
	    break;
	}
    }
}

struct nexp_speaker *
nexp_newspeaker(struct nexp_speaker *parms, char *errbuf)
{
    pcap_t *pd;
    pcap_dumper_t *pdumper;
    struct nexp_speaker *s, *new_speaker;

    if (parms->type == SPEAKER_NONE)
	return NULL;

    /* Create new speaker and initialize its fields. */
    new_speaker = xmalloc(sizeof(*new_speaker) );
    memset(new_speaker, 0, sizeof(*new_speaker) );
    new_speaker->type = parms->type;
    snprintf(new_speaker->name, sizeof(new_speaker->name), "nexp%d",
	     spawn_id++); /* Note that the spawn ID advances */

    switch (new_speaker->type) {
    case SPEAKER_IPV4:
	new_speaker->_ip = ip_open();
	if (!new_speaker->_ip) {
	    free(new_speaker);
	    snprintf(errbuf, PCAP_ERRBUF_SIZE, "Can't create raw IPv4 socket");
	    return NULL;
	}
	break;
#ifdef __linux__
    case SPEAKER_IPV6:
	new_speaker->_ip6 = socket(AF_INET6, SOCK_RAW, IPPROTO_RAW);
	if (new_speaker->_ip6 == -1) {
	    free(new_speaker);
	    snprintf(errbuf, PCAP_ERRBUF_SIZE,
		     "Can't create raw IPv6 socket: %s", strerror(errno) );
	    return NULL;
	}
	break;
#endif
    case SPEAKER_ETHER:
	new_speaker->_iface.ether = eth_open(parms->_iface.ifname);
	if (!new_speaker->_iface.ether) {
	    free(new_speaker);
	    snprintf(errbuf, PCAP_ERRBUF_SIZE,
		     "Can't create handle to transmit raw Ethernet frames");
	    return NULL;
	}

	new_speaker->_iface.ifname = xstrdup(parms->_iface.ifname);
	break;
    case SPEAKER_PCAP:
	/*
	 * pcap_dump_open() needs a pcap_t. pcap_open_dead() beats
	 * pcap_open_live() and pcap_open_offline().
	 */
	pd = pcap_open_dead(DLT_EN10MB, SNAPLEN);

	/* Create the output file. */
	pdumper = pcap_dump_open(pd, parms->_pcap.fname);
	if (pdumper == NULL) {
	    free(new_speaker);
	    snprintf(errbuf, PCAP_ERRBUF_SIZE, "pcap_dump_open(): %s\n",
		     pcap_geterr(pd) );
	    pcap_close(pd);
	    return NULL;
	}

	new_speaker->_pcap.pd = pd;
	new_speaker->_pcap.pdumper = pdumper;
	new_speaker->_pcap.fname = xstrdup(parms->_pcap.fname);
	break;
    case SPEAKER_HEX:
    case SPEAKER_STDOUT:
    case SPEAKER_NULL:
	/* These are simple output methods; there's nothing extra to do */
	break;
    case SPEAKER_STREAM:
    case SPEAKER_DGRAM:
	new_speaker->_socket.fd = parms->_socket.fd;
	new_speaker->_socket.dsthost = xstrdup(parms->_socket.dsthost);
	new_speaker->_socket.dstport = xstrdup(parms->_socket.dstport);
	break;
    }

    /* Add new speaker to linked list of speakers. */
    if (!speakers)
	speakers = new_speaker;
    else {
	for (s = speakers; s->next; s = s->next);

	s->next = new_speaker;
    }

    return new_speaker;
}
