#define _LARGEFILE64_SOURCE     /* required for GLIBC to enable stat64 and friends */
#include <sys/types.h>
#include <regex.h>
#include <string.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <pwd.h>

#include "mt.h"
#include "mem.h"
#include "error.h"
#include "utils.h"
#include "color.h"
#include "term.h"
#include "globals.h"

/* "local global" */
char *defaultcscheme = NULL;
char *cur_colorscheme_name = NULL;
char *cur_colorscheme_descr = NULL;
char *cur_filterscheme_name = NULL;
char *cur_filterscheme_descr = NULL;
char first_sort = 1;

void do_load_config(char *file);


int config_yes_no(char *what)
{
	if (what[0] == '1' ||
			strcasecmp(what, "yes") == 0 ||
			strcasecmp(what, "on") == 0)
	{
		return 1;
	}

	return 0;
}

int sort_colorschemes_compare(const void *a, const void *b)
{
	color_scheme *pa = (color_scheme *)a;
	color_scheme *pb = (color_scheme *)b;

	return strcmp(pa -> name, pb -> name);
}

long long int kb_str_to_value(char *field, char *str)
{
	char *mult;
	long long int bytes = atoll(str);
	if (bytes < -1)
		error_exit("%s: value cannot be < -1\n", field);

	mult = &str[strlen(str) - 2];
	if (strcasecmp(mult, "kb") == 0)
		bytes *= 1024;
	else if (strcasecmp(mult, "mb") == 0)
		bytes *= 1024 * 1024;
	else if (strcasecmp(mult, "gb") == 0)
		bytes *= 1024 * 1024 * 1024;

	return bytes;
}

int config_file_entry(char *line)
{
	char *cmd = line, *par = NULL;

	/* remove spaces at the beginning of the line */
	while (isspace(*cmd)) cmd++;

	/* see if there's anything in this line anway */
	if (strlen(cmd) == 0)
		return 0;

	/* skip comments */
	if (cmd[0] == '#' || cmd[0] == ';')
		return 0;

	/* lines are in the format of command:parameter */
	if ((par = find_next_par(cmd)) == NULL)
		error_exit("Malformed configline found: %s (command delimiter (:) missing)\n", cmd);

	if (strcasecmp(cmd, "defaultcscheme") == 0)	/* default colorscheme */
	{
		defaultcscheme = mystrdup(par, "do_load_config defaultcscheme");
	}
	else if (strcmp(cmd, "suppress_empty_lines") == 0)
	{
		suppress_empty_lines = config_yes_no(par);
	}
	else if (strcmp(cmd, "close_closed_windows") == 0)
	{
		do_not_close_closed_windows = !config_yes_no(par);
	}
	else if (strcasecmp(cmd, "follow_filename") == 0)
	{
		default_follow_filename = config_yes_no(par);
	}
	else if (strcasecmp(cmd, "default_linewrap") == 0)
	{
		default_linewrap = par[0];

		if (default_linewrap == 'o')
		{
			char *dummy = find_next_par(par);
			if (dummy)
				default_line_wrap_offset = atoi(dummy);
			else
				error_exit("default_linewrap:o needs a wrap offset parameter\n");
		}
	}
	else if (strcasecmp(cmd, "umask") == 0)
	{
		def_umask = strtol(par, NULL, 0);
	}
	else if (strcasecmp(cmd, "include") == 0)		/* include an other configfile */
	{
		do_load_config(par);
	}
	else if (strcasecmp(cmd, "shell") == 0)		/* shell */
	{
		shell = mystrdup(par, "do_load_config shell");
	}
	else if (strcasecmp(cmd, "statusline_above_data") == 0)		/* ^x */
	{
		statusline_above_data = config_yes_no(par);
	}
	else if (strcasecmp(cmd, "caret_notation") == 0)		/* ^x */
	{
		caret_notation = config_yes_no(par);
	}
	else if (strcasecmp(cmd, "searches_case_insensitive") == 0)
	{
		re_case_insensitive = 1;
	}
	else if (strcasecmp(cmd, "beep_method") == 0)		/* beep_method */
	{
		if (strcasecmp(par, "beep") == 0)
			beep_method = BEEP_BEEP;
		else if (strcasecmp(par, "flash") == 0)
			beep_method = BEEP_FLASH;
		else if (strcasecmp(par, "popup") == 0)
			beep_method = BEEP_POPUP;
		else if (strcasecmp(par, "none") == 0)
			beep_method = BEEP_NONE;
		else
			error_exit("'%s' is an unknown beep method\n");
	}
	else if (strcasecmp(cmd, "beep_popup_length") == 0)		/* beep_method */
	{
		beep_popup_length = atof(par);
		if (beep_popup_length < 0.0)
			error_exit("beep_popup_length must be >= 0.0\n");
	}
	else if (strcasecmp(cmd, "allow_8bit") == 0)		/* shell */
	{
		allow_8bit = config_yes_no(par);
	}
	else if (strcasecmp(cmd, "dsblwm") == 0)		/* default scrollback window line wrap mode */
	{
		no_linewrap = config_yes_no(par);
	}
	else if (strcasecmp(cmd, "warn_closed") == 0)		/* wether to warn if a window got closed */
	{
		warn_closed = config_yes_no(par);
	}
	else if (strcasecmp(cmd, "basename") == 0)
	{
		filename_only = config_yes_no(par);
	}
	else if (strcasecmp(cmd, "bright") == 0)
	{
		bright_colors = config_yes_no(par);
	}
	else if (strcasecmp(cmd, "colorscheme") == 0)	/* name for colorscheme */
	{
		if (use_colors)
		{
			char *dummy;

			myfree(cur_colorscheme_name);
			myfree(cur_colorscheme_descr);

			dummy = find_next_par(par);
			cur_colorscheme_descr = dummy?mystrdup(dummy, "do_load_config: colorscheme description"):NULL;
			cur_colorscheme_name = mystrdup(par, "do_load_config: colorscheme name");
		}
	}
	else if (strcasecmp(cmd, "filterscheme") == 0)
	{
		char *dummy;
		myfree(cur_filterscheme_name);
		myfree(cur_filterscheme_descr);

		dummy = find_next_par(par);
		cur_filterscheme_descr = dummy?mystrdup(dummy, "do_load_config: filterscheme description"):NULL;
		cur_filterscheme_name = mystrdup(par, "filterscheme name");
	}
	else if (strcasecmp(cmd, "rule") == 0)
	{
		int loop;
		char *type = par;
		int use_regex = 0x00;
		char *cmd = NULL;
		char *re_str = find_next_par(par);
		if (!re_str)
			error_exit("missing regular expression in rule-line for scheme %s\n", cur_filterscheme_name);

		for(loop=0; loop<n_fs; loop++)
		{
			if (strcmp(pfs[loop].fs_name, cur_filterscheme_name) == 0)
				break;
		}

		if (loop == n_fs)
		{
			pfs = (filterscheme *)myrealloc(pfs, sizeof(filterscheme) * (n_fs + 1), "do_load_config: list of filterschemes");
			pfs[n_fs].fs_name = mystrdup(cur_filterscheme_name, "fs name");
			pfs[n_fs].fs_desc = mystrdup(cur_filterscheme_descr, "fs descr");
			pfs[n_fs].n_re = 0;
			pfs[n_fs].pre  = NULL;
			n_fs++;
		}

		if (type[0] != 'e')
			error_exit("regular expression type '%s' is not recognized\n", type);

		if (type[1] == 0x00 || type[1] == 'm')
			use_regex = 'm';
		else if (type[1] == 'v')
			use_regex = 'v';
		else if (type[1] == 'c')
			use_regex = 'c';
		else if (type[1] == 'C')
			use_regex = 'C';
		else if (type[1] == 'x')
		{
			char *dummy = find_next_par(re_str);
			if (!dummy)
				error_exit("missing command for rule of type 'ex' for scheme %s\n", cur_filterscheme_name);

			cmd = mystrdup(dummy, "command");
			use_regex = 'x';
		}

		pfs[loop].pre = (re *)myrealloc(pfs[loop].pre, (pfs[loop].n_re + 1) * sizeof(re), "list of regexps for filterscheme");
		memset(&pfs[loop].pre[pfs[loop].n_re], 0x00, sizeof(re));
		pfs[loop].pre[pfs[loop].n_re].use_regex = use_regex;
		pfs[loop].pre[pfs[loop].n_re].regex_str = mystrdup(re_str, "regexp str");
		compile_re(&pfs[loop].pre[pfs[loop].n_re].regex, re_str);
		pfs[loop].pre[pfs[loop].n_re].cmd = cmd;
		pfs[loop].n_re++;
	}
	else if (strcasecmp(cmd, "ts_format") == 0)
	{
		ts_format = mystrdup(par, "do_load_config: timestamp format");
	}
	else if (strcasecmp(cmd, "cnv_ts_format") == 0)
	{
		ts_format = mystrdup(par, "do_load_config: conversion timestamp format");
	}
	else if (strcasecmp(cmd, "statusline_ts_format") == 0)
	{
		statusline_ts_format = mystrdup(par, "do_load_config: statusline timestamp format");
	}
	/* convert:apache:ip4tohost:^([0-9]\{1,3\}\.[0-9]\{1,3\}\.[0-9]\{1,3\}\.[0-9]\{1,3\}) */
	else if (strcasecmp(cmd, "convert") == 0)
	{
		char *conv_name = par;
		char *conv_type = NULL;
		conversion_t type = 0;
		char *conv_re = NULL;
		int loop;

		conv_type = find_next_par(conv_name);
		if (conv_type)
		{
			conv_re = find_next_par(conv_type);
		}

		if (!conv_type || !conv_re)
			error_exit("'convert'-entry malformed: type or regular expression missing\n");

		for(loop=0; loop<n_conversions; loop++)
		{
			if (strcmp(conversions[loop].name, conv_name) == 0)
				break;
		}

		if (loop == n_conversions)
		{
			n_conversions++;
			conversions = (conversion *)myrealloc(conversions, sizeof(conversion) * n_conversions, "list of conversions-sets");
			memset(&conversions[loop], 0x00, sizeof(conversion));
			conversions[loop].name = mystrdup(conv_name, "conversion name");
		}

		if (strcasecmp(conv_type, "ip4tohost") == 0)
			type = CONVTYPE_IP4TOHOST;
		else if (strcasecmp(conv_type, "epochtodate") == 0)
			type = CONVTYPE_EPOCHTODATE;
		else if (strcasecmp(conv_type, "errnotostr") == 0)
			type = CONVTYPE_ERRNO;
		else if (strcasecmp(conv_type, "hextodec") == 0)
			type = CONVTYPE_HEXTODEC;
		else if (strcasecmp(conv_type, "dectohex") == 0)
			type = CONVTYPE_DECTOHEX;
		else if (strcasecmp(conv_type, "tai64todate") == 0)
			type = CONVTYPE_TAI64NTODATE;
		else
			error_exit("convert %s: '%s' is a not recognized conversion type\n", conv_name, conv_type);

		conversions[loop].type = (conversion_t *)myrealloc(conversions[loop].type, sizeof(conversion_t) * (conversions[loop].n + 1), "list of conversion-types per set");
		conversions[loop].regex = (regex_t *)myrealloc(conversions[loop].regex, sizeof(regex_t) * (conversions[loop].n + 1), "list of conversion-re's per set");
		conversions[loop].match_count = (int *)myrealloc(conversions[loop].match_count, sizeof(int) * (conversions[loop].n + 1), "list of matchcounts for conversions");
		conversions[loop].type[conversions[loop].n] = type;
		LOG("conv_re: %p\n", conv_re);
		LOG("conv_re: {%s}\n", conv_re);
		LOG("# n: %d\n", conversions[loop].n);
		compile_re(&conversions[loop].regex[conversions[loop].n], conv_re);
		conversions[loop].match_count[conversions[loop].n] = 0;
		conversions[loop].n++;
	}
	else if (strncmp(cmd, "cs_re", 5) == 0)		/* ColorScheme_RegularExpression */
	{
		if (use_colors)
		{
			char *re = NULL, *val = NULL;
			int sn;

			if (strncmp(cmd, "cs_re_val", 9) == 0)
			{
				val = find_next_par(par);
				if (!val)
					error_exit("cs_re_val...-entry malformed: value missing\n");

				re = find_next_par(val);
			}
			else
				re = find_next_par(par);

			if (re == NULL)
				error_exit("cs_re-entry malformed: color or regular expression missing (%s:%s)\n", cmd, par);

			/* find colorscheme */
			sn = find_colorscheme(cur_colorscheme_name);
			if (sn == -1)
			{
				sn = n_cschemes++;
				cschemes = (color_scheme *)myrealloc(cschemes, n_cschemes * sizeof(color_scheme), "do_load_config: list of colorschemes");

				cschemes[sn].name = mystrdup(cur_colorscheme_name, "do_load_config: do_load_config cs_re");
				cschemes[sn].descr = mystrdup(cur_colorscheme_descr?cur_colorscheme_descr:"", "do_load_config: cscheme descr");
				cschemes[sn].n = 0;
				cschemes[sn].attrs = NULL;
				cschemes[sn].regex = NULL;
				cschemes[sn].flags = NULL;
				cschemes[sn].cmp_value = NULL;
			}

			/* grow/create list */
			cschemes[sn].attrs = (myattr_t *)myrealloc(cschemes[sn].attrs, (cschemes[sn].n + 1) * sizeof(myattr_t), "do_load_config: color for regexp");
			cschemes[sn].regex = (regex_t *)myrealloc(cschemes[sn].regex, (cschemes[sn].n + 1) * sizeof(regex_t), "do_load_config: regexp for colorscheme");
			cschemes[sn].flags = (csreflag_t *)myrealloc(cschemes[sn].flags, (cschemes[sn].n + 1) * sizeof(csreflag_t), "do_load_config: colorscheme match types");
			cschemes[sn].cmp_value = (double *)myrealloc(cschemes[sn].cmp_value, (cschemes[sn].n + 1) * sizeof(double), "do_load_config: colorscheme match compare value");

			/* add to list */
			cschemes[sn].flags[cschemes[sn].n] =
				((strcasecmp(cmd, "cs_re_s") == 0)          ?CSREFLAG_SUB:0)            |
				((strcasecmp(cmd, "cs_re_val_less") == 0)   ?CSREFLAG_CMP_VAL_LESS:0)   |
				((strcasecmp(cmd, "cs_re_val_bigger") == 0) ?CSREFLAG_CMP_VAL_BIGGER:0) |
				((strcasecmp(cmd, "cs_re_val_equal") == 0)  ?CSREFLAG_CMP_VAL_EQUAL:0)
				;

			/* sanity check */
			if (strstr(re, "(") == NULL && strcasecmp(cmd, "cs_re_s") == 0)
				error_exit("cs_re_s without substring selections! ('(' and ')')\n");

			if (val) cschemes[sn].cmp_value[cschemes[sn].n] = atof(val);
			cschemes[sn].attrs[cschemes[sn].n] = parse_attributes(par);
			/* compile regular expression */
			compile_re(&cschemes[sn].regex[cschemes[sn].n], re);

			cschemes[sn].n++;
		}
	}
	else if (strcasecmp(cmd, "markerline_color") == 0)
	{
		markerline_attrs = parse_attributes(par);
	}
	else if (strcasecmp(cmd, "idleline_color") == 0)
	{
		idleline_attrs = parse_attributes(par);
	}
	else if (strcasecmp(cmd, "msgline_color") == 0)
	{
		msgline_attrs = parse_attributes(par);
	}
	else if (strcasecmp(cmd, "changeline_color") == 0)
	{
		changeline_attrs = parse_attributes(par);
	}
	else if (strcasecmp(cmd, "statusline_attrs") == 0)
	{
		statusline_attrs = parse_attributes(par);
	}
	else if (strcasecmp(cmd, "show_subwindow_id") == 0)
	{
		show_subwindow_id = config_yes_no(par);
	}
	else if (strcasecmp(cmd, "markerline_timestamp") == 0)
	{
		timestamp_in_markerline = config_yes_no(par);
	}
	else if (strcasecmp(cmd, "usefilterscheme") == 0)
	{
		char *re, *cur_fs = par;
		int fs;

		if ((re = find_next_par(par)) == NULL)		/* format: cs_re:color:regular expression */
			error_exit("scheme-entry malformed: schemename or regular expression missing (%s:%s)\n", cmd, par);

		/* find colorschemes */
		for(;;)
		{
			char *colon = strchr(cur_fs, ',');
			if (colon) *colon = 0x00;

			fs = find_filterscheme(cur_fs);
			if (fs == -1)
				error_exit("filterscheme '%s' is unknown: you should first define a filterscheme before using it\n", par);

			add_pars_per_file(re, -1, -1, -1, -1, fs);

			if (!colon)
				break;

			cur_fs = colon + 1;
		}
	}
	else if (strcasecmp(cmd, "scheme") == 0)	/* default colorscheme for file */
	{
		if (use_colors)
		{
			char *re;
			int cs;
			char *cur_cs = par;

			if (first_sort)
			{
				first_sort = 0;

				qsort(cschemes, n_cschemes, sizeof(color_scheme), sort_colorschemes_compare);
			}

			if ((re = find_next_par(par)) == NULL)		/* format: cs_re:color:regular expression */
				error_exit("scheme-entry malformed: schemename or regular expression missing (%s:%s)\n", cmd, par);

			/* find colorschemes */
			for(;;)
			{
				char *colon = strchr(cur_cs, ',');
				if (colon) *colon = 0x00;

				cs = find_colorscheme(cur_cs);
				if (cs == -1)
					error_exit("colorscheme '%s' is unknown: you should first define a colorscheme before using it\n", par);

				add_pars_per_file(re, cs, -1, -1, -1, -1);

				if (!colon)
					break;

				cur_cs = colon + 1;
			}
		}
	}
	else if (strcasecmp(cmd, "global_default_nlines") == 0)	/* default number of lines buffered */
	{
		default_maxnlines = atoi(par);
	}
	else if (strcasecmp(cmd, "global_default_nkb") == 0)	/* default kb buffered */
	{
		default_maxbytes = kb_str_to_value(cmd, par);
	}
	else if (strcasecmp(cmd, "default_nlines") == 0)	/* default number of lines buffered */
	{
		char *re;
		int n_lines;

		if ((re = find_next_par(par)) == NULL)		/* format: cs_re:color:regular expression */
			error_exit("scheme-entry malformed: schemename or regular expression missing (%s:%s)\n", cmd, par);

		/* find colorscheme */
		n_lines = atoi(par);
		if (n_lines < -1)
			error_exit("default_nlines: value cannot be < -1\n");
		LOG("default_nlines re: %p\n", re);
		LOG("re: {%s}\n", re);
		add_pars_per_file(re, -1, n_lines, -1, -1, -1);
	}
	else if (strcasecmp(cmd, "default_mark_change") == 0)
	{
		char *re;

		if ((re = find_next_par(par)) == NULL)		/* format: cs_re:color:regular expression */
			error_exit("scheme-entry malformed: scheme name or regular expression missing (%s:%s)\n", cmd, par);

		add_pars_per_file(re, -1, -1, -1, config_yes_no(par), -1);
	}
	else if (strcasecmp(cmd, "default_bytes") == 0)		/* default number of bytes buffered */
	{
		int bytes;
		char *re;

		if ((re = find_next_par(par)) == NULL)		/* format: cs_re:color:regular expression */
			error_exit("scheme-entry malformed: schemename or regular expression missing (%s:%s)\n", cmd, par);

		bytes = kb_str_to_value(cmd, par);

		add_pars_per_file(re, -1, -1, bytes, -1, -1);
	}
	else if (strcasecmp(cmd, "check_mail") == 0)	/* how often to check for mail */
	{
		check_for_mail = atoi(par);
	}
	else if (strcasecmp(cmd, "tab_stop") == 0)	/* length of a tab */
	{
		tab_width = atoi(par);
	}
	else if (strcasecmp(cmd, "bind") == 0)		/* program to execute when user presses a key */
	{
		char *proc;

		if ((proc = find_next_par(par)) == NULL)
			error_exit("bind-entry malformed: key or binding missing (%s:%s)\n", cmd, par);

		if (strlen(par) > 2) error_exit("bind-entry malformed: unknown keyselection (%s:%s)\n", cmd, par);

		keybindings = (keybinding *)myrealloc(keybindings, sizeof(keybinding) * (n_keybindings + 1), "keybindings array");

		if (par[0] == '^')
			keybindings[n_keybindings].key = toupper(par[1]) - 'A' + 1;
		else
			keybindings[n_keybindings].key = par[0];

		keybindings[n_keybindings].command = mystrdup(proc, "do_load_config: keybinding command");
		n_keybindings++;
	}
	else if (strcasecmp(cmd, "titlebar") == 0)	/* do things with xterm title bar */
	{
		set_title = mystrdup(par, "do_load_config: titlebar");
	}
	else if (strcasecmp(cmd, "tail") == 0)	/* where to find tail */
	{
		tail = mystrdup(par, "do_load_config: where to find tail");
	}
	else if (strcasecmp(cmd, "replace_by_markerline") == 0)
	{
		replace_by_markerline = mystrdup(par, "do_load_config: replace_by_markerline");
	}
	else if (strcasecmp(cmd, "abbreviate_filesize") == 0)
	{
		afs = config_yes_no(par);
	}
	else if (strcasecmp(cmd, "popup_refresh_interval") == 0)
	{
		popup_refresh_interval = atoi(par);
		if (popup_refresh_interval < 0)
			error_exit("popup_refresh_interval should be 0 or higher\n");
	}
	else if (strcasecmp(cmd, "global_mark_change") == 0)
	{
		global_marker_of_other_window = config_yes_no(par);
	}
	else if (strcasecmp(cmd, "markerline_char") == 0)
	{
		markerline_char = par[0];
	}
	else if (strcasecmp(cmd, "changeline_char") == 0)
	{
		changeline_char = par[0];
	}
	else if (strcasecmp(cmd, "idleline_char") == 0)
	{
		idleline_char = par[0];
	}
	else if (strcasecmp(cmd, "msgline_char") == 0)
	{
		msgline_char = par[0];
	}
	else if (strcasecmp(cmd, "default_bufferwhat") == 0)
	{
		char what = par[0];

		if (what != 'a' && what != 'f')
			error_exit("default_bufferwhat expects either 'a' or 'f' (got: '%c')\n", what);

		default_bufferwhat = what;
	}
	else
		return -1;

	return 0;
}

/* returns the default color scheme or -1 if none */
void do_load_config(char *file)
{
	FILE *fh;
	int linenr = 0;

	/* given file */
	fh = fopen(file, "r");
	if (fh == NULL)
	{
		if (errno == ENOENT)	/* file does not exist, not an error */
			return;

		error_exit("do_load_config: error loading configfile '%s'\n", file);
	}

	for(;;)
	{
		char read_buffer[4096];
		char *dummy;
		char *cmd = fgets(read_buffer, sizeof(read_buffer) - 1, fh);
		if (!cmd)
			break;

		linenr++;

		/* strip LF */
		dummy = strchr(cmd, '\n');
		if (dummy)
			*dummy = 0x00;
		else
			error_exit("line %d of file '%s' is too long!\n", linenr, file);

		/* LOG("%d: %s\n", linenr, cmdin); */
		if (config_file_entry(cmd) == -1)
			error_exit("Configurationparameter %s is unknown (file: %s, line: %d)\n", read_buffer, file, linenr);
	}
	fclose(fh);

	if (defaultcscheme)
	{
		default_color_scheme = find_colorscheme(defaultcscheme);

		if (default_color_scheme == -1)
			error_exit("Default colorscheme '%s' is not defined! Check multitail.conf\n", defaultcscheme);
	}
}

void load_configfile_wrapper(char *config_file)
{
	/* load configurationfile (if any) */
	if (load_global_config)
		do_load_config(CONFIG_FILE);

	if (config_file)
	{
		do_load_config(config_file);
	}
	else
	{
		int path_max = find_path_max();
		char *path = mymalloc(path_max + 1, "path");
		char *home = getenv("HOME");
		struct passwd *pp = getuserinfo();

		if (home)
			snprintf(path, path_max, "%s/.multitailrc", home);
		else
			snprintf(path, path_max, "%s/.multitailrc", pp -> pw_dir);

		do_load_config(path);

		myfree(path);
	}
}

char load_configfile(char *config_file)
{
	static char config_loaded = 0;

	if (config_loaded == 0)
	{
		/* load configurationfile (if any) */
		load_configfile_wrapper(config_file);

		config_loaded = 1;

		return 1;
	}

	return 0;
}
