#include <unistd.h>
#include <iostream>
#include <memory>
#include <list>
#include <cassert>

#include "Matrix.h"
#include "File.h"


//============================================================================
// This tool is used to convert a version 1 map to version 2.
// There is no documentation except for the source itself. ;-)
//============================================================================


//----------------------------------------------------------------------------
namespace CL_OPT
{
    char **mapFiles = NULL;
}


//----------------------------------------------------------------------------
class Map
{
    //------------------------------------------------------------------------
    typedef Matrix<uint8_t> Tiles;

    struct Layer
    {
        uint8_t category;
        uint8_t background;
        uint8_t alpha;
        uint8_t brightness;

        Tiles tiles;
    };

    typedef std::list<Layer> Layers;

public:
    //------------------------------------------------------------------------
    Map(const char *file);
    ~Map() {}

    //------------------------------------------------------------------------
    inline uint8_t getVersion() const { return m_version; }

    //------------------------------------------------------------------------
    void convert();

    //------------------------------------------------------------------------
    void writeTo(const char *file) const;

private:
    //------------------------------------------------------------------------
    uint8_t m_version;
    uint16_t m_xSize;
    uint16_t m_ySize;
    uint8_t m_numLayers;

    Layers m_layers;
};

//----------------------------------------------------------------------------
Map::Map(const char *file)
{
    File f(file, "r");

    m_version = f.readUint8();
    if (m_version == '\x01')
    {
        m_xSize = f.readUint16();
        m_ySize = f.readUint16();
        m_numLayers = f.readUint8();

        for (unsigned i=0; i<m_numLayers; i++)
        {
            Layer layer;
            layer.category = f.readUint8();
            layer.tiles.resize(m_xSize, m_ySize);

            layer.background =
                (layer.category == 0x00 ||
                 layer.category == 0xff ?
                 0x01 : 0x00);
            layer.alpha = (layer.category == 0xff ? 0x80 : 0xff);
            layer.brightness = 0x80;

            for (unsigned y=0; y<m_ySize; y++)
            {
                for (unsigned x=0; x<m_xSize; x++)
                {
                    layer.tiles.set(x, y, f.readUint8());
                }
            }

            m_layers.push_back(layer);
        }
    }
}

//----------------------------------------------------------------------------
void Map::convert()
{
    m_version = 0x02;

    Layers::iterator iter = m_layers.begin();
    while (iter != m_layers.end())
    {
        Layer &layer = *iter;
        switch (layer.category)
        {
        case 0x00:  // Convert the 'background' tile set.
        {
            Layer clay;
            clay.background = 0x01;
            clay.alpha = 0xff;
            clay.brightness = 0x40;
            clay.category = 0x05;
            clay.tiles.resize(layer.tiles.getXSize(), layer.tiles.getYSize());

            Layer brick;
            brick.background = 0x01;
            brick.alpha = 0xff;
            brick.brightness = 0x40;
            brick.category = 0x01;
            brick.tiles.resize(layer.tiles.getXSize(), layer.tiles.getYSize());

            Layer colors;
            colors.background = 0x01;
            colors.alpha = 0xff;
            colors.brightness = 0x80;
            colors.category = 0x10;
            colors.tiles.resize(layer.tiles.getXSize(), layer.tiles.getYSize());

            Layer clouds;
            clouds.background = 0x01;
            clouds.alpha = 0xff;
            clouds.brightness = 0x80;
            clouds.category = 0x09;
            clouds.tiles.resize(layer.tiles.getXSize(), layer.tiles.getYSize());

            bool addClay = false;
            bool addBrick = false;
            bool addColors = false;
            bool addClouds = false;

            for (size_t y=0; y<layer.tiles.getYSize(); y++)
            {
                for (size_t x=0; x<layer.tiles.getXSize(); x++)
                {
                    switch (layer.tiles.get(x, y))
                    {
                    case 0x00:
                        break;

                    case 0x10:  // brick full
                        brick.tiles.set(x, y, 0x05);
                        addBrick = true;
                        break;
                    case 0x11:  // brick upper left
                        brick.tiles.set(x, y, 0x07);
                        addBrick = true;
                        break;
                    case 0x12:  // brick upper right
                        brick.tiles.set(x, y, 0x09);
                        addBrick = true;
                        break;
                    case 0x13:  // brick bottom row
                        brick.tiles.set(x, y, 0x02);
                        addBrick = true;
                        break;
                    case 0x14:  // brick upper left hole
                        brick.tiles.set(x, y, 0x10);
                        addBrick = true;
                        break;
                    case 0x15:  // brick upper right hole
                        brick.tiles.set(x, y, 0x11);
                        addBrick = true;
                        break;
                    case 0x16:  // brick upper row
                        brick.tiles.set(x, y, 0x08);
                        addBrick = true;
                        break;

                    case 0x50:  // clay full
                        clay.tiles.set(x, y, 0x01);
                        addClay = true;
                        break;
                    case 0x51:  // clay upper left
                        clay.tiles.set(x, y, 0x30);
                        addClay = true;
                        break;
                    case 0x52:  // clay upper right
                        clay.tiles.set(x, y, 0x40);
                        addClay = true;
                        break;
                    case 0x53:  // clay lower right
                        clay.tiles.set(x, y, 0x50);
                        addClay = true;
                        break;
                    case 0x54:  // clay lower left
                        clay.tiles.set(x, y, 0x60);
                        addClay = true;
                        break;
                    case 0x55:  // clay upper row
                        clay.tiles.set(x, y, 0x22);
                        addClay = true;
                        break;
                    case 0x56:  // clay right row
                        clay.tiles.set(x, y, 0x11);
                        addClay = true;
                        break;
                    case 0x57:  // clay bottom row
                        clay.tiles.set(x, y, 0x21);
                        addClay = true;
                        break;
                    case 0x58:  // clay left row
                        clay.tiles.set(x, y, 0x12);
                        addClay = true;
                        break;
                    case 0x59:  // clay lower right hole
                        clay.tiles.set(x, y, 0x90);
                        addClay = true;
                        break;
                    case 0x5a:  // clay lower left hole
                        clay.tiles.set(x, y, 0xa0);
                        addClay = true;
                        break;
                    case 0x5b:  // clay upper left hole
                        clay.tiles.set(x, y, 0x70);
                        addClay = true;
                        break;
                    case 0x5c:  // clay upper right hole
                        clay.tiles.set(x, y, 0x80);
                        addClay = true;
                        break;

                    case 0x80:  // sky
                        colors.tiles.set(x, y, 0x20);
                        addColors = true;
                        break;

                    case 0x90:  // cloud01
                        clouds.tiles.set(x, y, 0x01);
                        addClouds = true;
                        break;
                    case 0x91:  // cloud02
                        clouds.tiles.set(x, y, 0x02);
                        addClouds = true;
                        break;
                    case 0x92:  // cloud03
                        clouds.tiles.set(x, y, 0x03);
                        addClouds = true;
                        break;
                    case 0x93:  // cloud04
                        clouds.tiles.set(x, y, 0x04);
                        addClouds = true;
                        break;
                    case 0x94:  // cloud05
                        clouds.tiles.set(x, y, 0x05);
                        addClouds = true;
                        break;
                    case 0x95:  // cloud06
                        clouds.tiles.set(x, y, 0x06);
                        addClouds = true;
                        break;
                    case 0x96:  // cloud07
                        clouds.tiles.set(x, y, 0x07);
                        addClouds = true;
                        break;
                    case 0x97:  // cloud08
                        clouds.tiles.set(x, y, 0x08);
                        addClouds = true;
                        break;
                    case 0x98:  // cloud09
                        clouds.tiles.set(x, y, 0x09);
                        addClouds = true;
                        break;
                    case 0x99:  // cloud10
                        clouds.tiles.set(x, y, 0x0a);
                        addClouds = true;
                        break;
                    case 0x9a:  // cloud11
                        clouds.tiles.set(x, y, 0x0b);
                        addClouds = true;
                        break;
                    case 0x9b:  // cloud12
                        clouds.tiles.set(x, y, 0x0c);
                        addClouds = true;
                        break;

                    default:
                        assert(false);
                        break;
                    }
                }
            }


            iter = m_layers.erase(iter);
            m_numLayers--;

            if (addClay)
            {
                m_layers.insert(iter, clay);
                m_numLayers++;
            }
            if (addBrick)
            {
                m_layers.insert(iter, brick);
                m_numLayers++;
            }
            if (addColors)
            {
                m_layers.insert(iter, colors);
                m_numLayers++;
            }
            if (addClouds)
            {
                m_layers.insert(iter, clouds);
                m_numLayers++;
            }
        }   break;

        case 0xff:  // Convert the 'transparent' tile set.
            layer.category = 0x10;  // Switch to the 'colors' tile set.
            for (size_t y=0; y<layer.tiles.getYSize(); y++)
            {
                for (size_t x=0; x<layer.tiles.getXSize(); x++)
                {
                    switch (layer.tiles.get(x, y))
                    {
                    case 0x00:  // black
                        break;
                    case 0x01:  // red
                        layer.tiles.set(x, y, 0x08);
                        break;
                    case 0x02:  // yellow
                        layer.tiles.set(x, y, 0x0a);
                        break;
                    case 0x03:  // green
                        layer.tiles.set(x, y, 0x0b);
                        break;
                    case 0x04:  // cyan
                        layer.tiles.set(x, y, 0x0c);
                        break;
                    case 0x05:  // blue
                        layer.tiles.set(x, y, 0x0d);
                        break;
                    case 0x06:  // magenta
                        layer.tiles.set(x, y, 0x0e);
                        break;
                    default:
                        assert(false);
                        break;
                    }
                }
            }
            ++iter;
            break;

        default:
            ++iter;
            break;
        }
    }
}

//----------------------------------------------------------------------------
void Map::writeTo(const char *file) const
{
    File f(file, "w");

    f.writeUint8(m_version);
    f.writeUint16(m_xSize);
    f.writeUint16(m_ySize);
    f.writeUint8(m_numLayers);

    for (Layers::const_iterator iter = m_layers.begin();
         iter != m_layers.end(); ++iter)
    {
        f.writeUint8(iter->category);
        f.writeUint8(iter->background);
        f.writeUint8(iter->alpha);
        f.writeUint8(iter->brightness);
        for (unsigned y=0; y<m_ySize; y++)
        {
            for (unsigned x=0; x<m_xSize; x++)
            {
                f.writeUint8(iter->tiles.get(x, y));
            }
        }
    }
}


//----------------------------------------------------------------------------
void usage()
{
    std::cout << "Usage: convertmap file ...\n\n"
              << "Where options include:\n"
              << "  -h       This help text.\n"
              << "  file     The map file(s) to modify."
              << std::endl;
}

//----------------------------------------------------------------------------
void readCommandLine(int argc, char **argv)
{
    int ch;
    char *argString = "h";

    while ((ch = getopt(argc, argv, argString)) != EOF)
    {
        switch (ch)
        {
        case 'h':
            usage();
            exit(0);
            break;

        default:
            usage();
            exit(-1);
        }
    }

    if (optind < argc)
    {
        CL_OPT::mapFiles = &argv[optind];
    }

    if (CL_OPT::mapFiles == NULL)
    {
        usage();
        exit(-1);
    }
}

//----------------------------------------------------------------------------
void convertMap(const char *file)
{
    std::cout << "Converting '" << file << "' ... " << std::flush;

    Map m(file);

    if (m.getVersion() == '\x02')
    {
        std::cout << "already converted." << std::endl;
        return;
    }

    if (m.getVersion() != '\x01')
    {
        std::cout << "skipping due unknown version." << std::endl;
        return;
    }

    m.convert();

    std::cout << "writing ... " << std::flush;

    std::string bak(file);
    bak.append(".orig");
    File::copy(file, bak.c_str());

    m.writeTo(file);

    std::cout << "done." << std::endl;
}


//----------------------------------------------------------------------------
int main(int argc, char **argv)
{
    readCommandLine(argc, argv);

    for (unsigned i = 0; CL_OPT::mapFiles[i] != NULL; i++)
    {
        convertMap(CL_OPT::mapFiles[i]);
    }

    return 0;
}
