/*
  MeCab -- Yet Another Part-of-Speech and Morphological Analyzer

  $Id: dictionary.h,v 1.16 2006/07/09 13:34:22 taku-ku Exp $;

  Copyright (C) 2001-2006 Taku Kudo <taku@chasen.org>
  Copyright (C) 2004-2006 Nippon Telegraph and Telephone Corporation

*/
#ifndef MECAB_DICTIONARY_H
#define MECAB_DICTIONARY_H

#include "mecab.h"
#include "darts.h"
#include "token.h"
#include "mmap.h"

namespace MeCab {

  class Param;

  class Dictionary {

  private:
    MeCab::Mmap<char>  *dmmap_;
    Token *token_;
    char  *feature_;
    char  *charset_;
    unsigned short version_;
    unsigned short type_;
    unsigned short learn_;
    unsigned int   lexsize_;
    unsigned int   lsize_;
    unsigned int   rsize_;
    whatlog what_;
    Darts::DoubleArray da_;

  public:
    typedef Darts::DoubleArray::result_pair_type result_type;

    bool open(const std::string &);
    void close();

    size_t commonPrefixSearch(const char* key, size_t len,
                              result_type *result,
                              size_t rlen) {
      return da_.commonPrefixSearch(key, result, rlen, len);
    }

    result_type exactMatchSearch(const char* key) {
      result_type n;
      da_.exactMatchSearch(key, n);
      return n;
    }

    bool isCompatible(const Dictionary &d) const {
      return (version_ == d.version_ &&
              learn_   == d.learn_   &&
              lsize_   == d.lsize_   &&
              rsize_   == d.rsize_   &&
              std::strcmp (charset_, d.charset_) == 0);
    }

    const char* charset() const { return const_cast<const char*>(charset_); }
    unsigned short version() const { return version_; }
    size_t  size() const { return static_cast<size_t>(lexsize_); }
    int learn() const { return static_cast<int>(learn_); }
    int type() const { return static_cast<int>(type_);  }
    size_t lsize() const { return static_cast<size_t>(lsize_); }
    size_t rsize() const { return static_cast<size_t>(rsize_); }


    Token *token(result_type &n) const { return token_ + (n.value >> 8) ; }
    size_t token_size(result_type &n) const { return 0xff & n.value; };
    char  *feature(const Token &t) const { return feature_ + t.feature; }

    static bool compile(Param &,
                        const std::vector<std::string> &dics, // inputs
                        const char *matrix_file,
                        const char *matrix_bin_file,
                        const char *left_id_file,
                        const char *right_id_file,
                        const char *rewrite_file,
                        const char* pos_id_file,
                        const char* output);  // outputs

    const char *what()  { return what_.str(); }

    explicit Dictionary(): dmmap_(0), token_(0), feature_(0), charset_(0) {};
    virtual ~Dictionary() { this->close (); }
  };
}
#endif
