/*
 * Copyright (c) 2001, 2002 The XDoclet team
 * All rights reserved.
 */
package xdoclet.modules.ejb.intf;

import org.apache.commons.logging.Log;

import xjavadoc.XClass;
import xjavadoc.XTag;

import xdoclet.XDocletException;
import xdoclet.XDocletMessages;

import xdoclet.modules.ejb.AbstractEjbCodeGeneratorSubTask;
import xdoclet.modules.ejb.XDocletModulesEjbMessages;
import xdoclet.tagshandler.PackageTagsHandler;

import xdoclet.util.LogUtil;
import xdoclet.util.Translator;

/**
 * Generates service endpoint interfaces for JAXPRC beans.
 *
 * @author               Christoph G. Jung (christoph.jung@infor.de)
 * @created              24. Februar 2004
 * @since                22.12.03
 * @version              $Revision: 1.2 $
 * @ant.element          display-name="Service-Endpoint Interface" name="service-endpoint"
 *      parent="xdoclet.modules.ejb.EjbDocletTask"
 * @xdoclet.merge-file   file="service-endpoint-custom.xdt" relates-to="{0}.java" description="A text file containing
 *      custom template and/or java code to include in the service-endpoint interface."
 */

public class ServiceEndpointSubTask extends AbstractEjbCodeGeneratorSubTask
{
    public final static String DEFAULT_SERVICE_ENDPOINT_CLASS_PATTERN = "{0}";

    protected final static String DEFAULT_TEMPLATE_FILE = "resources/service-endpoint.xdt";

    /**
     * A configuration parameter for specifying the service-endpoint interface name pattern. By default the value is
     * used for deciding the service-endpoint interface name. {0} in the value mean current class's symbolic name which
     * for an EJBean is the EJB name.
     *
     * @see   #getServiceEndpointClassPattern()
     */
    protected String serviceEndpointClassPattern;

    /**
     * Describe what the ServiceEndpointSubTask constructor does
     */
    public ServiceEndpointSubTask()
    {
        setTemplateURL(getClass().getResource(DEFAULT_TEMPLATE_FILE));
        setDestinationFile(getServiceEndpointClassPattern() + ".java");
        addOfType("javax.ejb.SessionBean");
    }

    /**
     * Returns the configuration parameter for specifying the service-endpoint interface name pattern. By default the
     * value is used for deciding the service-endpoint interface name. {0} in the value mean current class's symbolic
     * name which for an EJBean is the EJB name. If nothing explicitly specified by user then "{0}" is used by default.
     *
     * @return   The ServiceEndpointClassPattern value
     * @see      #remoteClassPattern
     */
    public String getServiceEndpointClassPattern()
    {
        if (serviceEndpointClassPattern != null) {
            return serviceEndpointClassPattern;
        }
        else {
            return DEFAULT_SERVICE_ENDPOINT_CLASS_PATTERN;
        }
    }

    /**
     * The pattern by which the interfaces are named. {0} designates the EJB name.
     *
     * @param newPattern   The new Pattern value
     * @ant.not-required   No, defaults to {0}
     */
    public void setPattern(String newPattern)
    {
        serviceEndpointClassPattern = newPattern;
    }

    /**
     * Called to validate configuration parameters.
     *
     * @exception XDocletException
     */
    public void validateOptions() throws XDocletException
    {
        super.validateOptions();

        if (getServiceEndpointClassPattern() == null || getServiceEndpointClassPattern().trim().equals("")) {
            throw new XDocletException(Translator.getString(XDocletMessages.class, XDocletMessages.PARAMETER_MISSING_OR_EMPTY, new String[]{"pattern"}));
        }

        if (getServiceEndpointClassPattern().indexOf("{0}") == -1) {
            throw new XDocletException(Translator.getString(XDocletModulesEjbMessages.class, XDocletModulesEjbMessages.PATTERN_HAS_NO_PLACEHOLDER));
        }
    }


    /**
     * Gets the GeneratedFileName attribute of the ServiceEndpointInterfaceSubTask object
     *
     * @param clazz                 Describe what the parameter does
     * @return                      The GeneratedFileName value
     * @exception XDocletException
     */
    protected String getGeneratedFileName(XClass clazz) throws XDocletException
    {
        return PackageTagsHandler.packageNameAsPathFor(InterfaceTagsHandler.getComponentInterface(InterfaceTagsHandler.SERVICE_ENDPOINT, getCurrentClass())) + ".java";
    }

    /**
     * Describe what the method does
     *
     * @param clazz                 Describe what the parameter does
     * @return                      Describe the return value
     * @exception XDocletException
     */
    protected boolean matchesGenerationRules(XClass clazz) throws XDocletException
    {
        if (super.matchesGenerationRules(clazz) == false) {
            return false;
        }

        Log log = LogUtil.getLog(ServiceEndpointSubTask.class, "matchesGenerationRules");

        if (!InterfaceTagsHandler.isServiceEndpointEjb(getCurrentClass())) {
            log.debug("Reject file " + clazz.getQualifiedName() + " because of different view-type");
            return false;
        }

        XTag interfaceTag = getCurrentClass().getDoc().getTag("ejb:interface");

        if (interfaceTag == null) {
            return true;
        }

        String generate = interfaceTag.getAttributeValue("generate");

        if ((generate != null) && (generate.indexOf(InterfaceTagsHandler.SERVICE_ENDPOINT) == -1)) {
            log.debug("Skip service-endpoint interface for " + clazz.getQualifiedName() + " because of generate=" + generate + " flag.");
            return false;
        }

        return true;
    }

    /**
     * Describe what the method does
     *
     * @exception XDocletException
     */
    protected void engineStarted() throws XDocletException
    {
        System.out.println(Translator.getString(XDocletModulesEjbMessages.class, XDocletModulesEjbMessages.GENERATING_SERVICE_ENDPOINT_FOR,
            new String[]{getCurrentClass().getQualifiedName()}));
    }
}
