/***************************************************************************
                  libxbox-eeprom.c - EEPROM  functions for libxbox
                             -------------------
    begin                : Thu Dec 09 23:02:12 BST 2004
    copyright            : (C) 2004 by David Pye
    email                : dmp@davidmpye.dyndns.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#include <stdbool.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "libxbox.h"
#include "libxbox-eeprom.h"

static void xbox_eeprom_crc(unsigned char *crc, unsigned char *data, long dataLen) {
	/* This algorithm was obtained from the XKUtils 0.2 source released by 
	 * TeamAssembly under the GNU GPL.
	 * Specifically, from XKCRC.cpp, method EepromCRC
	 *
	 * Thanks! */
	
	unsigned char* CRC_Data = (unsigned char *)malloc(dataLen+4);
	int pos=0;
	memset(crc,0x00,4);
	memset(CRC_Data,0x00, dataLen+4);
	//Circle shift input data one byte right
	memcpy(CRC_Data + 0x01 , data, dataLen-1);
	memcpy(CRC_Data, data + dataLen-1, 0x01);
             
	for (pos=0; pos<4; ++pos) {
		unsigned short CRCPosVal = 0xFFFF;
		unsigned long l;
		for (l=pos; l<dataLen; l+=4) {
			CRCPosVal -= *(unsigned short*)(&CRC_Data[l]);
		}
		CRCPosVal &= 0xFF00;
		crc[pos] = (unsigned char) (CRCPosVal >> 8);
	}
	free(CRC_Data);
}

bool xbox_manufacturing_info(char *serial, char *info) {
	char *country=NULL;
	memcpy(info, serial, 12);
	//Blank out the identifiable information
	memset(&info[0x01],'X',6);
	switch (atoi(&info[11])) {
		case 2:
			country = "Mexico";
			break;
		case 3:
			country = "Hungary";
			break;
		case 5: 
			country = "China";
			break;
		case 6:
			country = "Taiwan";
			break;
		default:
			country = "Unknown";
			break;
	}
 	info[12]=' ';
	sprintf(&info[13], "Country: %s, Line #%c, Week %c%c, Year 200%c",country,
		info[0x00], info[0x08], info[0x09], info[0x07]);
	return true;
}

bool xbox_calc_eeprom_chk2(EEPROMDATA *eepromdata) {
	xbox_eeprom_crc(eepromdata->Checksum2,eepromdata->SerialNumber,0x28);
	return true;
}

bool xbox_calc_eeprom_chk3(EEPROMDATA *eepromdata) {
	//In the XKUtils, they have the data length that this checksum covers 
	//as 0x60, but that is incorrect. This checksum covers from the start 
	//of TimeZoneBias (0x64) to the end of the DVD Playback kit zone data 
	//(0xBF).  so, 0xBF - 0x64 = 0x5b, not 0x60
	xbox_eeprom_crc(eepromdata->Checksum3, eepromdata->TimeZoneBias,0x5b);
	return true;
}

bool xbox_xbe_region(char *xberegion, XBOX_XBE_REGION *region) {
	bool retval = false;
	switch (*xberegion) {
		case 0x01:
	        	*region = NORTH_AMERICA;
			retval = true;
			break;
		case 0x02:
			*region = JAPAN;
			retval = true;
			break;
		case 0x04:
			*region = EURO_AUSTRALIA;
			retval = true;
			break;
		default:
			*region = INVALID;
			retval = false;
	}
	return retval;
}

bool xbox_video_std(char *VideoStandard, XBOX_VIDEO_STD *std) {
	long data = *(long *)VideoStandard;
	
	switch (data) {
		case 0x00800300:
			*std = VIDEO_PAL;
			break;
		case 0x00400100:
			*std = VIDEO_NTSC;
			break;
		case 0x00400200:
			*std = VIDEO_NTSC_J;
			break;
		default:
			*std = VIDEO_UNKNOWN;
	}

	if (*std == VIDEO_UNKNOWN) return false;
	else return true;
}
