#define PHIDGETS_INTERNAL

#include <phidgets/led64.h>
#include <debug.h>
#include <assert.h>

/* System includes */
#include <time.h>

PhidgetLED64* phidget_new_PhidgetLED64() 
{
	PhidgetLED64 *ret;
		  
	TRACE("creating a new PhidgetLED64 instance...");

	ret = (PhidgetLED64*)malloc(sizeof(PhidgetLED64));
	if(!ret) {
		ERROR("could not allocate memory for PhidgetLED64 instance.");
		return 0;
	}
	ret->leds = PHIDGETS_LED64_NUMLEDS;
	ret->phidget = phidget_new_Phidget();

	return ret;
}

void phidget_delete_PhidgetLED64(PhidgetLED64** const led64)
{
	if (!led64 || !*led64) {
		ERROR("cannot delete NULL PhidgetLED64.");
		return;
	}

	free(*led64);
	*led64 = 0;
}

void phidget_reset_PhidgetLED64(PhidgetLED64* const led64)
{
        unsigned int i;

	if(!led64) {
		ERROR("cannot reset NULL PhidgetLED64.");
	}	

	phidget_reset_Phidget(led64->phidget);

}

phidget_return phidget_led64_open(PhidgetLED64* const led64,
	unsigned int serial, unsigned short retries)
{
	HIDInterfaceMatcher matcher;
	phidget_return ret;
		  
	if(!phidget_is_initialised()) {
		ERROR("cannot open PhidgetLED64 when Phidgets library has not been initialised.");
		return PHIDGET_RET_NOT_INITIALISED;
	}

	if(!led64) {
		ERROR("cannot open NULL PhidgetLED64.");
		return PHIDGET_RET_INVALID_PARAMETER;
	}

	if(phidget_led64_is_opened(led64)) {
		ERROR("cannot open already opened PhidgetLED64.");
		return PHIDGET_RET_DEVICE_ALREADY_OPENED;
	}

	TRACE("opening PhidgetLED64 with serial number %d...", serial);

	matcher.vendor_id = PHIDGETS_USB_VENDORID;
	matcher.product_id = PHIDGETS_USB_PRODUCTID_LED64;

	ret = phidget_open(led64->phidget, 0, &matcher, serial, retries);
	if(ret != PHIDGET_RET_SUCCESS) return ret;

	NOTICE("successfully opened PhidgetLED64 %s...", led64->phidget->id);

	return PHIDGET_RET_SUCCESS;
}

phidget_return phidget_led64_close(PhidgetLED64* const led64)
{
	if(!led64) {
		ERROR("cannot close NULL PhidgetLED64.");
		return PHIDGET_RET_INVALID_PARAMETER;
	}

	if(phidget_led64_is_opened(led64)) {
		phidget_return ret;
		TRACE("closing PhidgetLED64 %s...", led64->phidget->id);
		ret = phidget_close(led64->phidget);
		if (ret != PHIDGET_RET_SUCCESS) return ret;
	}
	else WARNING("attempt to close unopened PhidgetLED64.");

	NOTICE("successfully closed PhidgetLED64 %s.", led64->phidget->id);
	return PHIDGET_RET_SUCCESS;
}

bool phidget_led64_is_opened(PhidgetLED64 const* const led64)
{
	if(!led64) WARNING("attempting to query open status of NULL PhidgetLED64.");
	return led64 && phidget_is_opened(led64->phidget);
}

phidget_return phidget_led64_set_level(PhidgetLED64* const led64,
				       unsigned int led_index, unsigned int led_level)
{
		  
        TRACE("phidget_led64_set_level(%i, %i, %i).", led64, led_index, led_level);
	char packet[PHIDGETS_LED64_SET_PACKETLEN];
	int const PATH[] = 
		{ PHIDGETS_HID_PATH_1, PHIDGETS_HID_PATH_2, PHIDGETS_HID_PATH_LED64_SET };
	int raw_level;
	hid_return ret;
	
	if(!phidget_led64_is_opened(led64)) {
		ERROR("cannot toggle led of unopen PhidgetLED64.");
		return PHIDGET_RET_DEVICE_NOT_OPENED;
	}
	
	if(led_index > PHIDGETS_LED64_NUMLEDS) {
	  ERROR("there are only %i leds; cannot set status of led %i.", PHIDGETS_LED64_NUMLEDS, led_index);
		return PHIDGET_RET_DEVICE_NOT_OPENED;
	}

	if(led_level > 100) {
	        WARNING("max led_level is 100 (requested: %i); clamping level to 100.", led_level);
		led_level = 100;
	}
		  
	TRACE("setting level for led %i to %i.", led_index, led_level);

	raw_level = (unsigned char)((((int)led_level) * 6400) / 10000);
	if (raw_level == 64) {
	  raw_level = 63;
	}

	packet[0] = packet[2] = packet[4] = packet[6] = (unsigned char) led_index;
	packet[1] = packet[3] = packet[5] = packet[7] = (unsigned char) raw_level;
	
	TRACE("sending packet 0x[%02x %02x %02x %02x] to PhidgetLED64 "
		 "%s.", packet[0], packet[1], packet[2], packet[3], led64->phidget->id);

	ret = hid_set_output_report(led64->phidget->hid_iface,
		PATH, PHIDGETS_HID_PATH_DEPTH, packet, PHIDGETS_LED64_SET_PACKETLEN);
	if(ret != HID_RET_SUCCESS) {
	  WARNING("failed to send packet to PhidgetLED64 %s. (error: %i)", led64->phidget->id, ret);
		return PHIDGET_RET_HID_ERROR;
	}

	led64->led_level[led_index] = led_level;
	
	return PHIDGET_RET_SUCCESS;
}

unsigned int phidget_led64_get_level(PhidgetLED64 *const led64, unsigned int led_index) {
	
	if(!phidget_led64_is_opened(led64)) {
		ERROR("cannot get led level of unopened PhidgetLED64.");
		return PHIDGET_RET_DEVICE_NOT_OPENED;
	}
	
	if(led_index > PHIDGETS_LED64_NUMLEDS) {
	  ERROR("there are only %i leds; cannot get status of led %i.", PHIDGETS_LED64_NUMLEDS, led_index);
		return PHIDGET_RET_DEVICE_NOT_OPENED;
	}

	return led64->led_level[led_index];

}
