/** Copyright  2003 by Jean-Hugues de Raigniac <jhraigniac@workingfrog.org>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

package org.workingfrog.i18n.swing;

import java.awt.Component;
import java.beans.PropertyChangeListener;

import javax.swing.Action;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;

import org.workingfrog.i18n.util.LocaleEvent;
import org.workingfrog.i18n.util.LocaleListener;
import org.workingfrog.i18n.util.Translator;

/**
 * i18n version of javax.swing.JMenu
 *
 * @author Jean-Hugues de Raigniac
 */
public class I18NJMenu extends JMenu implements LocaleListener {

    /** bundle */
    private String bundle = "menu";

    /** i18n key */
    private String i18nKey = "menu.error";

    /**
     * Override javax.swing.JMenu.JMenu (java.lang.String).
     *
     * @param i18nKey i18n bundle key
     */
    public I18NJMenu (String i18nKey) {
        super(i18nKey);
        if (bundle != null && i18nKey != null) {
            setText(Translator.checkValue(i18nKey, this));
        }
        Translator.checkKey(i18nKey, this);
        this.i18nKey = i18nKey;
    }

    /**
     * Creates a new menu item attached to the specified
     * <code>Action</code> object and appends it to the end of this menu.
     * As of JDK 1.3, this is no longer the preferred method for adding
     * <code>Actions</code> to
     * a container. Instead it is recommended to configure a control with
     * an action using <code>setAction</code>,
     * and then add that control directly
     * to the <code>Container</code>.
     *
     * @param a the <code>Action</code> for the menu item to be added
     * @see Action
     * @return The new JMenuItem
     */
    public JMenuItem add(Action a) {
        JMenuItem mi = createActionComponent(a);
        mi.setAction(a);
        if (a instanceof LocaleListener) {
            mi.setText(Translator.checkValue(((I18NAction) a).getKey(), this));
        }
        add(mi);
        return mi;
    }

    /**
     * Factory method which creates the <code>JMenuItem</code> for
     * <code>Action</code>s added to the <code>JMenu</code>.
     * As of JDK 1.3, this is no
     * longer the preferred method. Instead it is recommended to configure
     * a control with an action using <code>setAction</code>,
     * and then adding that
     * control directly to the <code>Container</code>.
     *
     * @param a the <code>Action</code> for the menu item to be added
     * @return the new menu item
     * @see Action
     */
    protected JMenuItem createActionComponent(Action a) {
        JMenuItem mi = (a instanceof I18NAction
                        || a instanceof I18NActionWrapper)
            ? new I18NJMenuItem((a instanceof I18NAction)
                ? ((I18NAction) a).getKey()
                : ((I18NActionWrapper) a).getKey())
            : new I18NJMenuItem((String) a.getValue(Action.NAME),
                                         (Icon) a.getValue(Action.SMALL_ICON)){
            protected PropertyChangeListener
                createActionPropertyChangeListener (Action a) {
                PropertyChangeListener pcl = createActionChangeListener(this);
                if (pcl == null) {
                    pcl = super.createActionPropertyChangeListener(a);
                }
                return pcl;
            }
        };
        mi.setHorizontalTextPosition(JButton.RIGHT);
        mi.setVerticalTextPosition(JButton.CENTER);
        mi.setEnabled(a.isEnabled());
        return mi;
    }

    /**
     * Allow the use of a customized i18n resource bundle.
     *
     * @param bundle bundle binding
     */
    public void setBundle (String bundle) {
        this.bundle = bundle;

        if (i18nKey != null) {
            setText(Translator.checkValue(i18nKey, this));
        }
    }

    /**
     * Propagate the LocaleEvent to the components of this JMenu.
     *
     * @param event contains the new Locale
     */
    public void localeChanged (LocaleEvent event) {
        setText(Translator.checkValue(i18nKey, this));

        int componentNumber = getMenuComponentCount();
//System.out.println("componentNumber (JMenu) = " + componentNumber);

        for (int i = 0; i < componentNumber; i++) {
            Component component = getMenuComponent(i);

            if (component instanceof LocaleListener) {
                ((LocaleListener) component).localeChanged(event);
            } else {
                if (!(component instanceof JPopupMenu.Separator)) {
//                    Translator.log(Translator.I18N,
//                                   "JMenu, N : " + component.toString());
                }
            }
        }
    }
}
