/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This file is part of libgpiv.
   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  



-------------------------------------------------------------------------------
FILENAME:                utils.c
LIBRARY:                 libgpiv:
EXTERNAL FUNCTIONS:      gpiv_scan_parameter,
                         gpiv_scan_resourcefiles
                         gpiv_alloc_img
                         gpiv_free_img
                         gpiv_cp_img

                         gpiv_null_pivdata
			 gpiv_alloc_pivdata
			 gpiv_free_pivdata
                         gpiv_cp_pivdata
                         gpiv_0_pivdata
                         gpiv_add_dxdy_pivdata
                         gpiv_sum_dxdy_pivdata

			 gpiv_alloc_scdata
			 gpiv_free_scdata

			 gpiv_alloc_bindata
			 gpiv_free_bindata

                         gpiv_matrix, 
                         gpiv_free_matrix
			 gpiv_ucmatrix, 
                         gpiv_free_ucmatrix
			 gpiv_imatrix, 
                         gpiv_free_imatrix
                         gpiv_fftw_real_matrix
                         gpiv_free_fftw_real_matrix
                         gpiv_fftw_complex_matrix
                         gpiv_free_fftw_complex_matrix

                         gpiv_vector,
                         gpiv_free_vector, 
                         gpiv_vector_index,
                         gpiv_free_vector_index, 

                         gpiv_dvector,
                         gpiv_free_dvector, 
                         gpiv_dvector_index,
                         gpiv_free_dvector_index, 

                         gpiv_nulvector,
                         gpiv_free_nulvector, 
                         gpiv_nulvector_index,
                         gpiv_free_nulvector_index, 

                         gpiv_ulvector,
                         gpiv_free_ulvector, 
                         gpiv_ulvector_index,
                         gpiv_free_ulvector_index, 

                         gpiv_ivector,
                         gpiv_free_ivector, 
                         gpiv_ivector_index,
                         gpiv_free_ivector_index, 

                         gpiv_max, 
                         gpiv_min, 
                         gpiv_scan_iph, 
                         gpiv_fscan_iph, 
                         gpiv_scan_cph, 
                         gpiv_fscan_cph, 
                         gpiv_scan_fph, 
                         gpiv_fscan_fph, 
                         gpiv_scan_sph, 
                         gpiv_fscan_sph, 
                         gpiv_sort_3,
			 gpiv_histo
			 gpiv_cumhisto
                         gpiv_cumhisto_eqdatbin
                         gpiv_histo_gnuplot
                         gpiv_piv_gnuplot
                         gpiv_scalar_gnuplot
                         gpiv_warning, 
                         gpiv_error, 

LOCAL FUNCTIONS:         compare_float

LAST MODIFICATION DATE:  $Id: utils.c,v 1.16 2006/01/31 13:30:13 gerber Exp $
 --------------------------------------------------------------------------- */

#include <string.h>

#include <gpiv.h>

#define GPIV_END 1                 /*  Used by vector */

/*
 * Prototyping
 */
/* int */
/* compare_float(const void *a, */
/*               const void *b */
/*               ); */

/*
 * Function definitions
 */
void
gpiv_scan_parameter(const char *PAR_KEY, 
		    const char *PARFILE,
		    void * pstruct,
		    int print_par
		    )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Read parameters from local parameter file PARFILE
 *
 * PROTOTYPE LOCATATION:
 *     utils.h
 *
 * INPUTS:
 *      PAR_KEY:        Parameter key, specific for each process
 *      PARFILE:        Parameter file, specific for each process
 *      print_par:      flag for printing parameters
 *
 * OUTPUTS:
 *      pstruct:        pointer to parameter structure
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    FILE *fp_par;
    
    if ((fp_par = fopen(PARFILE,"rb")) != NULL) {
	if (strcmp(PAR_KEY, "") == 0) {
            if (print_par) printf("\n# Default %s parameters from %s: \n", 
                                  PAR_KEY, PARFILE);
            gpiv_img_read_header(fp_par, pstruct, print_par);
	} else if (strcmp(PAR_KEY, GPIV_IMAGEPROC_PAR_KEY) == 0) {
            if (print_par) printf("\n# Default %s parameters from %s: \n", 
                                  PAR_KEY, PARFILE);
            gpiv_imgproc_read_parameters(fp_par, pstruct, print_par);
	} else if (strcmp(PAR_KEY, GPIV_IMAGE_PAR_KEY) == 0) {
            if (print_par) printf("\n# Default %s parameters from %s: \n", 
                                  PAR_KEY, PARFILE);
            gpiv_img_read_parameters(fp_par, pstruct, print_par);
        } else if (strcmp(PAR_KEY, GPIV_EVAL_PAR_KEY) == 0) {
            if (print_par) printf("\n# Default %s parameters from %s: \n", 
                                  PAR_KEY, PARFILE);
	    gpiv_piv_read_parameters(fp_par, pstruct, print_par);
        } else if (strcmp(PAR_KEY, GPIV_VALID_PAR_KEY) == 0) {
            if (print_par) printf("\n# Default %s parameters from %s: \n", 
                                  PAR_KEY, PARFILE);
	    gpiv_valid_read_parameters(fp_par, pstruct, print_par);
        } else if (strcmp(PAR_KEY, GPIV_POST_PAR_KEY) == 0) {
            if (print_par) printf("\n# Default %s parameters from %s: \n", 
                                  PAR_KEY, PARFILE);
	    gpiv_post_read_parameters(fp_par, pstruct, print_par);
	}

        fclose(fp_par);
    }

}



char *
gpiv_scan_resourcefiles(const char *par_key, 
			void * pstruct,
			int print_par
			)
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Opens resource files ~/GPIV_HOME_RSC_FILE (hidden) and 
 *     /SYSTEM_RSC_DIR/GPIV_SYSTEM_RSC_FILE
 *     Reads parameters from it
 *
 * PROTOTYPE LOCATATION:
 *     utils.h
 *
 * INPUTS:
 *      par_key:        Parameter key, specific for each process
 *      print_par:      parameter to print to stdout
 *
 * OUTPUTS:
 *      pstruct:        pointer to parameter structure
 *
 * RETURNS:
 *      NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    FILE *fp_par;
    gchar *HOME_DIR = NULL;
    gchar HOME_RSC[GPIV_MAX_CHARS]; 
    gchar SYSTEM_RSC[GPIV_MAX_CHARS];
   
    HOME_DIR = g_get_home_dir ();
/*     if ((HOME_DIR=getenv("HOME")) != NULL) { */
        snprintf(HOME_RSC, GPIV_MAX_CHARS, "%s/%s", HOME_DIR, GPIV_HOME_RSC_FILE);
/*     } */
    

     if ((fp_par=fopen(HOME_RSC, "rb")) != NULL) {
	if (strcmp(par_key, GPIV_DAC_PAR_KEY) == 0) {
#ifndef DISABLE_DAC
            if (print_par) printf("\n# Default %s parameters from %s: \n", 
                                  par_key, HOME_RSC);
            gpiv_dac_read_parameters(fp_par, pstruct, print_par);
#endif 
	} else if (strcmp(par_key, GPIV_IMAGE_PAR_KEY) == 0) {
            if (print_par) printf("\n# Default %s parameters from %s: \n", 
                                  par_key, HOME_RSC);
            gpiv_img_read_parameters(fp_par, pstruct, print_par);
        } else if (strcmp(par_key, GPIV_IMAGEPROC_PAR_KEY) == 0) {
            if (print_par) printf("\n# Default %s parameters from %s: \n", 
                                  par_key, HOME_RSC);
            gpiv_imgproc_read_parameters(fp_par, pstruct, print_par);
        } else if (strcmp(par_key, GPIV_EVAL_PAR_KEY) == 0) {
            if (print_par) printf("\n# Default %s parameters from %s: \n", 
                                  par_key, HOME_RSC);
            gpiv_piv_read_parameters(fp_par, pstruct, print_par);
        } else if (strcmp(par_key, GPIV_VALID_PAR_KEY) == 0 ) {
            if (print_par) printf("\n# Default %s parameters from %s: \n", 
                                  par_key, HOME_RSC);
  	    gpiv_valid_read_parameters(fp_par, pstruct, print_par);
	} else if (strcmp(par_key, GPIV_POST_PAR_KEY) == 0) {
            if (print_par) printf("\n# Default %s parameters from %s: \n", 
                                  par_key, HOME_RSC);
  	    gpiv_post_read_parameters(fp_par, pstruct, print_par);
	} else {
            err_msg = "GPIV_SCAN_RESOURCEFILES: called with unexisting key: ";
            gpiv_warning("%s", err_msg, par_key);
            return err_msg;
	}

        fclose(fp_par);
     }

    

    snprintf(SYSTEM_RSC, GPIV_MAX_CHARS, "%s/%s", SYSTEM_RSC_DIR, 
             GPIV_SYSTEM_RSC_FILE);

    if ((fp_par=fopen(SYSTEM_RSC,"rb")) != NULL) {
	if (strcmp(par_key, GPIV_DAC_PAR_KEY) == 0) {
#ifndef DISABLE_DAC
            if (print_par) printf("\n# Default %s parameters from %s: \n", 
                                  par_key, SYSTEM_RSC);
            gpiv_dac_read_parameters(fp_par, pstruct, print_par);
#endif
	} else if (strcmp(par_key, GPIV_IMAGE_PAR_KEY) == 0) {
            if (print_par) printf("\n# Default %s parameters from %s: \n", 
                                  par_key, SYSTEM_RSC);
            gpiv_img_read_parameters(fp_par, pstruct, print_par);
	} else if (strcmp(par_key, GPIV_IMAGEPROC_PAR_KEY) == 0) {
            if (print_par) printf("\n# Default %s parameters from %s: \n", 
                                  par_key, SYSTEM_RSC);
            gpiv_imgproc_read_parameters(fp_par, pstruct, print_par);
        } else if (strcmp(par_key, GPIV_EVAL_PAR_KEY) == 0) {
            if (print_par) printf("\n# Default %s parameters from %s: \n", 
                                  par_key, SYSTEM_RSC);
	    gpiv_piv_read_parameters(fp_par, pstruct, print_par);
	} else if (strcmp(par_key, GPIV_VALID_PAR_KEY) == 0) {
            if (print_par) printf("\n# Default %s parameters from %s: \n", 
                                  par_key, SYSTEM_RSC);
  	    gpiv_valid_read_parameters(fp_par, pstruct, print_par);
	} else if (strcmp(par_key, GPIV_POST_PAR_KEY) == 0 ) {
            if (print_par) printf("\n# Default %s parameters from %s: \n", 
                                  par_key, SYSTEM_RSC);
  	    gpiv_post_read_parameters(fp_par, pstruct, print_par);
	} else {
	  err_msg = "GPIV_SCAN_RESOURCEFILES: called with unexisting key: ";
          gpiv_warning("%s%s", err_msg, par_key);
          return err_msg;
	}

        fclose(fp_par);

    } else {
	  err_msg = "GPIV_SCAN_RESOURCEFILES: Failure opening: ";
          gpiv_warning("%s%s", err_msg, SYSTEM_RSC);
          return err_msg;
    }
    
    return err_msg;
}



guint16 **
gpiv_alloc_img(GpivImagePar image_par
               )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for image
 *
 * INPUTS:
 *      image_par:      image parameters
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      img:            2-dim image data arry
 *---------------------------------------------------------------------------*/
{
    guint16 **img;
    img = gpiv_matrix_guint16(image_par.nrows, image_par.ncolumns);
    return img;
}



void 
gpiv_cp_img(guint16 **img_src, 
            guint16 **img_dest,
            GpivImagePar image_par
            )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Copies contents of img_src to img_dest
 *
 * INPUTS:
 *      image_par:      image parameters
 *      img_src:            2-dim image data arry
 *
 * OUTPUTS:
 *      img_dest:            2-dim image data arry
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    gint i, j;
    
    assert(img_src[0] != NULL);
    assert(img_dest[0] != NULL);
    
    for (i = 0; i < image_par.nrows; i++) {
        for (j = 0; j < image_par.ncolumns; j++) {
            img_dest[i][j] = img_src[i][j];
        }
    }
}



void 
gpiv_free_img(guint16 **img, 
              GpivImagePar image_par
              )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Frees memory for image
 *
 * INPUTS:
 *      image_par:      image parameters
 *      img:            2-dim image data arry
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(img[0] != NULL);
    gpiv_free_matrix_guint16(img);
}



void
gpiv_null_pivdata(GpivPivData * piv_data
                )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Sets all elements of piv_data structure to NULL
 *
 * INPUTS:
 *      piv_data:      Input PIV data structure
 *
 * OUTPUTS:
 *      piv_data:      Output PIV data structure
 *
 * RETURNS:
 *     char * to NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    piv_data->point_x = NULL;
    piv_data->point_y = NULL;
    piv_data->dx = NULL;
    piv_data->dy = NULL;
    piv_data->snr = NULL;
    piv_data->peak_no = NULL;
}



void 
gpiv_alloc_pivdata(GpivPivData * piv_data
		   )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for GpivPivData
 *
 * INPUTS:
 *      piv_data:       .nx and .ny members of GpivPivData structure
 *
 * OUTPUTS:
 *      piv_data:        point_x, point_y, dx, dy, snr, peak_no  of GpivPivData
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    int i, j;

    assert(piv_data->point_x == NULL);
    assert(piv_data->point_y == NULL);
    assert(piv_data->dx == NULL);
    assert(piv_data->dy == NULL);
    assert(piv_data->snr == NULL);
    assert(piv_data->peak_no == NULL);

    if (piv_data->ny > 0 && piv_data->nx > 0) {
        piv_data->point_x = gpiv_matrix(piv_data->ny, piv_data->nx);
        piv_data->point_y = gpiv_matrix(piv_data->ny, piv_data->nx);
        piv_data->dx = gpiv_matrix(piv_data->ny, piv_data->nx);
        piv_data->dy = gpiv_matrix(piv_data->ny, piv_data->nx);
        piv_data->snr = gpiv_matrix(piv_data->ny, piv_data->nx);
        piv_data->peak_no = gpiv_imatrix(piv_data->ny, piv_data->nx);
        
        for (i = 0; i < piv_data->ny; i++) {
            for (j = 0; j < piv_data->nx; j++) {
                piv_data->point_x[i][j] = 0.0;
                piv_data->point_y[i][j] = 0.0;
                piv_data->dx[i][j] = 0.0;
                piv_data->dy[i][j] = 0.0;
                piv_data->snr[i][j] = 0.0;
                piv_data->peak_no[i][j] = 0;
            }
        }
    } else {
        gpiv_warning("GPIV_ALLOC_PIVDATA: nx = %d ny = %d", 
                     piv_data->nx, piv_data->ny);
    }
}



void 
gpiv_free_pivdata(GpivPivData * piv_data
		  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Frees memory for GpivPivData
 *
 * INPUTS:
 *      piv_data:       PIV data structure
 *
 * OUTPUTS:
 *      piv_data:      NULL pointer to point_x, point_y, dx, dy, snr, peak_no
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(piv_data->point_x != NULL);
    assert(piv_data->point_y != NULL);
    assert(piv_data->dx != NULL);
    assert(piv_data->dy != NULL);
    assert(piv_data->snr != NULL);
    assert(piv_data->peak_no != NULL);

    gpiv_free_matrix(piv_data->point_x);
    gpiv_free_matrix(piv_data->point_y);
    gpiv_free_matrix(piv_data->dx);
    gpiv_free_matrix(piv_data->dy);
    gpiv_free_matrix(piv_data->snr);
    gpiv_free_imatrix(piv_data->peak_no);

    gpiv_null_pivdata(piv_data);
}



char *
gpiv_cp_pivdata(GpivPivData * piv_data_in,
                GpivPivData * piv_data_out
                )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Copies data from piv_data_in to piv_data_out. Both structures will have
 *      to be allocated before (with gpiv_alloc_pivdata, for example).
 *
 * INPUTS:
 *      piv_data_in:   Input PIV data structure
 *
 * OUTPUTS:
 *      piv_data_out:  Output PIV data structure
 *
 * RETURNS:
 *     char * to NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    gint i, j;

    if (piv_data_in->point_x == NULL) {
        err_msg = "GPIV_CP_PIVDATA: piv_data_in->point_x == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_in->point_y == NULL) {
        err_msg = "GPIV_CP_PIVDATA: piv_data_in->point_y == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_in->dx == NULL) {
        err_msg = "GPIV_CP_PIVDATA: piv_data_in->dx == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_in->dy == NULL) {
        err_msg = "GPIV_CP_PIVDATA: piv_data_in->dyx == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_in->snr == NULL) {
        err_msg = "GPIV_CP_PIVDATA: piv_data_in->snr == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_in->peak_no == NULL) {
        err_msg = "GPIV_CP_PIVDATA: piv_data_in->peak_no == NULL";
        gpiv_warning("%s", err_msg);
    }


    if (piv_data_out->point_x == NULL) {
        err_msg = "GPIV_CP_PIVDATA: piv_data_out->point_x == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_out->point_y == NULL) {
        err_msg = "GPIV_CP_PIVDATA: piv_data_out->point_y == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_out->dx == NULL) {
        err_msg = "GPIV_CP_PIVDATA: piv_data_out->dx == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_out->dy == NULL) {
        err_msg = "GPIV_CP_PIVDATA: piv_data_out->dyx == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_out->snr == NULL) {
        err_msg = "GPIV_CP_PIVDATA: piv_data_out->snr == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_out->peak_no == NULL) {
        err_msg = "GPIV_CP_PIVDATA: piv_data_out->peak_no == NULL";
        gpiv_warning("%s", err_msg);
    }


    if (piv_data_in->nx != piv_data_out->nx) {
        err_msg = "GPIV_CP_PIVDATA: piv_data_in->nx != piv_data_out->nx";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_in->ny != piv_data_out->ny) {
        err_msg = "GPIV_CP_PIVDATA: piv_data_in->ny != piv_data_out->ny";
        gpiv_warning("%s", err_msg);
    }

    if (err_msg != NULL) {
        return err_msg;
    }


    for (i = 0; i < piv_data_out->ny; i++) {
        for (j = 0; j < piv_data_out->nx; j++) {
            piv_data_out->point_x[i][j] = piv_data_in->point_x[i][j];
            piv_data_out->point_y[i][j] = piv_data_in->point_y[i][j];
            piv_data_out->dx[i][j] = piv_data_in->dx[i][j];
            piv_data_out->dy[i][j] = piv_data_in->dy[i][j];
            piv_data_out->snr[i][j] = piv_data_in->snr[i][j];
            piv_data_out->peak_no[i][j] = piv_data_in->peak_no[i][j];
        }
    }
    
    return NULL;
}



char *
gpiv_0_pivdata(GpivPivData * piv_data
               )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Sets estimators, snr and peak_nr of piv_data to 0 or 0.0. 
 *      The structure will have to be allocated before (with 
 *      gpiv_alloc_pivdata, for example).
 *
 * INPUTS:
 *      piv_data:   PIV data structure
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     char * to NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    gint i, j;

/*     check_if_null(piv_data, "GPIV_CP_PIVDATA: "); */
/*     char *check_if_null(GpivPivData * piv_data, char * string) */
    if (piv_data->point_x == NULL) {
        err_msg = "GPIV_CP_PIVDATA: piv_data->point_x == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data->point_y == NULL) {
        err_msg = "GPIV_CP_PIVDATA: piv_data->point_y == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data->dx == NULL) {
        err_msg = "GPIV_CP_PIVDATA: piv_data->dx == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data->dy == NULL) {
        err_msg = "GPIV_CP_PIVDATA: piv_data->dy == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data->snr == NULL) {
        err_msg = "GPIV_CP_PIVDATA: piv_data->snr == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data->peak_no == NULL) {
        err_msg = "GPIV_CP_PIVDATA: piv_data->peak_no == NULL";
        gpiv_warning("%s", err_msg);
    }


    if (err_msg != NULL) {
        return err_msg;
    }


    for (i = 0; i < piv_data->ny; i++) {
        for (j = 0; j < piv_data->nx; j++) {
/*             piv_data->point_x[i][j] = 0.0; */
/*             piv_data->point_y[i][j] = 0.0; */
            piv_data->dx[i][j] = 0.0;
            piv_data->dy[i][j] = 0.0;
            piv_data->snr[i][j] = 0.0;
            piv_data->peak_no[i][j] = 0;
        }
    }
    
    return NULL;
}



char *
gpiv_add_dxdy_pivdata(GpivPivData * piv_data_in,
                      GpivPivData * piv_data_out
                 )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Adds displacements (dx, dy), snr and peak_nr from piv_data_in to 
 *      piv_data_out. Both structures will have to be allocated before 
 *      (with gpiv_alloc_pivdata, for example).
 *
 * INPUTS:
 *      piv_data_in:   Input PIV data structure
 *
 * OUTPUTS:
 *      piv_data_out:  Output PIV data structure
 *
 * RETURNS:
 *     char * to NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    gint i, j;

    if (piv_data_in->point_x == NULL) {
        err_msg = "GPIV_ADD_DXDY_PIVDATA: piv_data_in->point_x == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_in->point_y == NULL) {
        err_msg = "GPIV_ADD_DXDY_PIVDATA: piv_data_in->point_y == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_in->dx == NULL) {
        err_msg = "GPIV_ADD_DXDY_PIVDATA: piv_data_in->dx == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_in->dy == NULL) {
        err_msg = "GPIV_ADD_DXDY_PIVDATA: piv_data_in->dyx == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_in->snr == NULL) {
        err_msg = "GPIV_ADD_DXDY_PIVDATA: piv_data_in->snr == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_in->peak_no == NULL) {
        err_msg = "GPIV_ADD_DXDY_PIVDATA: piv_data_in->peak_no == NULL";
        gpiv_warning("%s", err_msg);
    }


    if (piv_data_out->point_x == NULL) {
        err_msg = "GPIV_ADD_DXDY_PIVDATA: piv_data_out->point_x == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_out->point_y == NULL) {
        err_msg = "GPIV_ADD_DXDY_PIVDATA: piv_data_out->point_y == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_out->dx == NULL) {
        err_msg = "GPIV_ADD_DXDY_PIVDATA: piv_data_out->dx == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_out->dy == NULL) {
        err_msg = "GPIV_ADD_DXDY_PIVDATA: piv_data_out->dyx == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_out->snr == NULL) {
        err_msg = "GPIV_ADD_DXDY_PIVDATA: piv_data_out->snr == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_out->peak_no == NULL) {
        err_msg = "GPIV_ADD_DXDY_PIVDATA: piv_data_out->peak_no == NULL";
        gpiv_warning("%s", err_msg);
    }


    if (piv_data_in->nx != piv_data_out->nx) {
        err_msg = "GPIV_ADD_DXDY_PIVDATA: piv_data_in->nx != piv_data_out->nx";
        gpiv_warning("%s", err_msg);
    }
    if (piv_data_in->ny != piv_data_out->ny) {
        err_msg = "GPIV_ADD_DXDY_PIVDATA: piv_data_in->ny != piv_data_out->ny";
        gpiv_warning("%s", err_msg);
    }

    if (err_msg != NULL) {
        return err_msg;
    }


    for (i = 0; i < piv_data_out->ny; i++) {
        for (j = 0; j < piv_data_out->nx; j++) {
            piv_data_out->point_x[i][j] = piv_data_in->point_x[i][j];
            piv_data_out->point_y[i][j] = piv_data_in->point_y[i][j];
            piv_data_out->dx[i][j] += piv_data_in->dx[i][j];
            piv_data_out->dy[i][j] += piv_data_in->dy[i][j];
            piv_data_out->snr[i][j] = piv_data_in->snr[i][j];
            piv_data_out->peak_no[i][j] = piv_data_in->peak_no[i][j];
        }
    }
    
    return NULL;
}



char * 
gpiv_sum_dxdy_pivdata(GpivPivData gpd,
                      gfloat * sum
                      )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Adds all displacements in order to calculate residuals
 *      The structure will have to be allocated before (with 
 *      gpiv_alloc_pivdata, for example).
 *
 * INPUTS:
 *      piv_data:   PIV data structure
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     char * to NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    gint i, j;

    if (gpd.point_x == NULL) {
        err_msg = "SUM_DXDY_PIVDATA: gpd.point_x == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (gpd.point_y == NULL) {
        err_msg = "SUM_DXDY_PIVDATA: gpd.point_y == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (gpd.dx == NULL) {
        err_msg = "SUM_DXDY_PIVDATA: gpd.dx == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (gpd.dy == NULL) {
        err_msg = "SUM_DXDY_PIVDATA: gpd.dy == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (gpd.snr == NULL) {
        err_msg = "SUM_DXDY_PIVDATA: gpd.snr == NULL";
        gpiv_warning("%s", err_msg);
    }
    if (gpd.peak_no == NULL) {
        err_msg = "SUM_DXDY_PIVDATA: gpd.peak_no == NULL";
        gpiv_warning("%s", err_msg);
    }


    if (err_msg != NULL) {
        return err_msg;
    }


    for (i = 0; i < gpd.ny; i++) {
        for (j = 0; j < gpd.nx; j++) {
            *sum += gpd.dx[i][j];
            *sum += gpd.dy[i][j];
        }
    }
    
    return NULL;

}



void 
gpiv_alloc_scdata(GpivScalarData * scal_data
                  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for GpivScalarData
 *
 * INPUTS:
 *      scal_data:      .nx and .ny members of GpivScalarData structure
 *
 * OUTPUTS:
 *      scal_data:       point_x, point_y, scalar, flag of GpivScalarData
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    int i, j;
    
    assert(scal_data->point_x == NULL);
    assert(scal_data->point_y == NULL);
    assert(scal_data->scalar == NULL);
    assert(scal_data->flag == NULL);

    if (scal_data->ny > 0 && scal_data->nx > 0) {
        scal_data->point_x = gpiv_matrix(scal_data->ny, scal_data->nx);
        scal_data->point_y = gpiv_matrix(scal_data->ny, scal_data->nx);
        scal_data->scalar = gpiv_matrix(scal_data->ny, scal_data->nx);
        scal_data->flag = gpiv_imatrix(scal_data->ny, scal_data->nx);
        
        for (i = 0; i < scal_data->ny; i++) {
            for (j = 0; j < scal_data->nx; j++) {
                scal_data->point_x[i][j] = 0.0;
                scal_data->point_y[i][j] = 0.0;
                scal_data->scalar[i][j] = 0.0;
                scal_data->flag[i][j] = 0;
            }
        }
    } else {
        gpiv_warning("GPIV_ALLOC_SCDATA: nx = %d ny = %d",
                     scal_data->nx, scal_data->ny);
    }
}



void 
gpiv_free_scdata(GpivScalarData * scal_data
                 )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Frees memory for GpivScalarData
 *
 * INPUTS:
 *      scal_data:      scalar data structure
 *
 * OUTPUTS:
 *      scal_data:       NULL pointer to point_x, point_y, scalar, flag
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{

    assert(scal_data->point_x != NULL);
    assert(scal_data->point_y != NULL);
    assert(scal_data->scalar != NULL);
    assert(scal_data->flag != NULL);
    
    gpiv_free_matrix(scal_data->point_x);
    gpiv_free_matrix(scal_data->point_y);
    gpiv_free_matrix(scal_data->scalar);
    gpiv_free_imatrix(scal_data->flag);
    
    scal_data->point_x = NULL;
    scal_data->point_y = NULL;
    scal_data->scalar = NULL;
    scal_data->flag = NULL;
}



void
gpiv_alloc_bindata(GpivBinData * bin_data
		   )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for GpivScalarData
 *
 * INPUTS:
 *      bin_data:       nbins of GpivScalarData
 *
 * OUTPUTS:
 *      bin_data:       count, bound, centre of GpivScalarData
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{ 
    int i;

    assert(bin_data->count == NULL);
    assert(bin_data->bound == NULL);
    assert(bin_data->centre == NULL);
    
/* BUGFIX */
/*     g_warning("gpiv_alloc_bindata:: 4/4 nbins = %d", bin_data->nbins); */
    if (bin_data->nbins > 0) {
        bin_data->bound = gpiv_vector(bin_data->nbins);
        bin_data->centre = gpiv_vector(bin_data->nbins);
        bin_data->count = gpiv_ivector(bin_data->nbins);
        
        for (i = 0; i < bin_data->nbins; i++) {
            bin_data->bound[i] = 0.0;
            bin_data->centre[i] = 0.0;
            bin_data->count[i] = 0;
        }
    } else {
        gpiv_warning("GPIV_ALLOC_BINDATA: nbins = %d",
                     bin_data->nbins);
    }
    
}      



void 
gpiv_free_bindata(GpivBinData * bin_data
		  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Frees memory for GpivScalarData
 *
 * INPUTS:
 *      bin_data:       data of bins
 *
 * OUTPUTS:
 *      bin_data:       NULL pointer to count, bound, centre
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{

    assert(bin_data->count != NULL);
    assert(bin_data->bound != NULL);
    assert(bin_data->centre != NULL);
    
/*     g_warning("gpiv_free_bindata:: 0 nbins = %d", bin_data->nbins); */
    gpiv_free_ivector(bin_data->count);

    gpiv_free_vector(bin_data->centre);
    gpiv_free_vector(bin_data->bound);
    
    bin_data->bound = NULL;
    bin_data->centre = NULL;
    bin_data->count = NULL;
}



float **
gpiv_matrix_index(long nrl, 
                  long nrh, 
                  long ncl, 
                  long nch
                  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for 2-dimensional matrix of float data 
 *      with subscript range v[nrl..nrh][ncl..nch]
 *
 * INPUTS:
 *      long nrl:	lowest row index number
 *      nrh:		highest row index number
 *      ncl:		lowest column index number
 *      nch	        highest column index number
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      2-dimensional array
 *---------------------------------------------------------------------------*/
{
    long i, nrows = nrh - nrl + 1, ncolumns = nch - ncl + 1;
    float **m;

    m=(float **) malloc((size_t)((nrows + GPIV_END) * 
                           sizeof(float *)));
    if (!m) gpiv_error("GPIV_MATRIX_INDEX: allocation failure");
    m += GPIV_END;
    m -= nrl;
    
    m[nrl]=(float *) malloc((size_t)((nrows * ncolumns + GPIV_END) *
                                     sizeof(float)));
    if (!m[nrl]) gpiv_error("GPIV_MATRIX_INDEX: allocation failure");
    m[nrl] += GPIV_END;
    m[nrl] -= ncl;
    
    for (i=nrl+1; i <= nrh; i++) {
        m[i] = m[i-1] + ncolumns;
    }

    return m;
}



void 
gpiv_free_matrix_index(float **m, 
		 long nrl, 
		 long nrh, 
		 long ncl, 
		 long nch
		 )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Frees memory for 2-dimensional array of float data
 *      with subscript range v[nrl..nrh][ncl..nch]
 *
 * INPUTS:
 *      m:	        matrix
 *      nrl:	        lowest row index number
 *      long nrh:	highest row index number
 *	long ncl:	lowest column index number
 *	long nch:	highest column index number
 *
 * OUTPUTS:
 *      m:             NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(m[0] != NULL);
    g_free(m[nrl]+ncl-GPIV_END);
    g_free(m+nrl-GPIV_END);
    m = NULL;
}



float **
gpiv_matrix(long nr,
            long nc
            )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for 2-dimensional matrix of float data
 *
 * INPUTS:
 *      long nr:	row index number
 *      long nc:	column index number
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     matrix
 *---------------------------------------------------------------------------*/
{
    long i;
    float **m;

    m = (float **) g_malloc0(nr * sizeof(float *));
    m[0] = (float *) g_malloc0(nr * nc * sizeof(float));
    for(i = 1; i < nr; i++) {
        m[i] = m[0] + i * nc;
    }

    return m;
}



void 
gpiv_free_matrix(float **m
		 )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Frees memory for 2-dimensional array of float data
 *
 * INPUTS:
 *      m:	        matrix 
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      m;              NULL pointer
 *---------------------------------------------------------------------------*/
{
    assert(m[0] != NULL);
    g_free(m[0]);
    g_free(m);
    m = NULL;
}



unsigned char **
gpiv_ucmatrix(long nr, 
              long nc
              )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for 2-dimensional matrix of guint16 data
 *
 * INPUTS:
 *      nrl:	lowest row index number
 *      nrh:	highest row index number 
 *      ncl:	lowest column index number
 *      nch:	highest column index number 
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     matrix
 *---------------------------------------------------------------------------*/
{
    long i;
    unsigned char **m;

    m=(unsigned char **) g_malloc0(nr * sizeof(unsigned char*));
    m[0]=(unsigned char *) g_malloc0(nr * nc * sizeof(unsigned char));
    for(i = 1; i < nr; i++) {
        m[i] = m[0] + i * nc;
    }

    return m;
}



void 
gpiv_free_ucmatrix(unsigned char **m 
		   )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Frees memory for 2-dimensional array of guint16 data
 *
 * INPUTS:
 *      m:	        matrix 
 *	nrl:	        lowest row index number
 *	nrh:	        highest row index number
 *	ncl:	        lowest column index number
 *	ch:	        highest column index number
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      m:              NULL pointer
 *---------------------------------------------------------------------------*/
{
    assert(m[0] != NULL);
    g_free(m[0]);
    g_free(m);
    m = NULL;
}



guint8 **
gpiv_matrix_guint8(long nr, 
                   long nc
                   )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      allocate a guint8 matrix with subscript range m[0..nr][0..nc]
 *
 * INPUTS:
 *      nr:             number of rows
 *      nc:             number of columns
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      matrix
 *---------------------------------------------------------------------------*/
{
    long i;
    guint8 **m;

    m=(guint8 **) g_malloc0(nr * sizeof(guint8*));
    m[0]=(guint8 *) g_malloc0(nr * nc * sizeof(guint8));
    for(i = 1; i < nr; i++) {
        m[i] = m[0] + i * nc;
    }

    return m;
}



void 
gpiv_free_matrix_guint8(guint8 **m
                         )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      free a guint8 matrix allocated by gpiv_matrix_guint8
 *
 * INPUTS:
 *      m:	        matrix 
 *
 * OUTPUTS:
 *      m:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(m[0] != NULL);
    g_free(m[0]);
    g_free(m);
    m = NULL;
}



guint16 **
gpiv_matrix_guint16(long nr, 
                    long nc
                    )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      allocate a guint16 matrix with subscript range m[0..nr][0..nc]
 *
 * INPUTS:
 *      nr:             number of rows
 *      nc:             number of columns
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      matrix
 *---------------------------------------------------------------------------*/
{
    long i;
    guint16 **m;

    m=(guint16 **) g_malloc0(nr * sizeof(guint16*));
    m[0]=(guint16 *) g_malloc0(nr * nc * sizeof(guint16));
    for(i = 1; i < nr; i++) {
        m[i] = m[0] + i * nc;
    }

    return m;
}



void 
gpiv_free_matrix_guint16(guint16 **m
                         )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      free a guint16 matrix allocated by gpiv_matrix_guint16
 *
 * INPUTS:
 *      m:	        matrix 
 *
 * OUTPUTS:
 *      m:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(m[0] != NULL);
    g_free(m[0]);
    g_free(m);
    m = NULL;
}



int **
gpiv_imatrix_index(long nrl, 
                   long nrh, 
                   long ncl, 
                   long nch
                   )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for 2-dimensional matrix of integer data
 *      with subscript range v[nrl..nrh][ncl..nch]
 *
 * INPUTS:
 *      m:	        matrix 
 *	nrl:	        lowest row index number
 *	nrh:	        highest row index number
 *	ncl:	        lowest column index number
 *	nch:	        highest column index number
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     matrix
 *---------------------------------------------------------------------------*/
{
    long i, nrows = nrh - nrl + 1, ncolumns = nch - ncl + 1;
    int **m;

    m=(int **) g_malloc0((size_t)((nrows + GPIV_END)*
                               sizeof(int*)));
    m += GPIV_END;
    m -= nrl;
    m[nrl]=(int *) g_malloc0((size_t)((nrows * ncolumns + GPIV_END)*
                                   sizeof(int)));
    m[nrl] += GPIV_END;
    m[nrl] -= ncl;
    
    for(i = nrl + 1; i <= nrh; i++) {
        m[i] = m[i-1] + ncolumns;
    }
    
    return m;
}



void 
gpiv_free_imatrix_index(int **m, 
                        long nrl, 
                        long nrh, 
                        long ncl, 
                        long nch
                        )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Frees memory for 2-dimensional array of integer data
 *      with subscript range v[nrl..nrh][ncl..nch]
 *
 * INPUTS:
 *      m:	        matrix 
 *	nrl:	        lowest row index number
 *	nrh:	        highest row index number
 *	ncl:	        lowest column index number
 *	nch:	        highest column index number
 *
 * OUTPUTS:
 *      m:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(m[0] != NULL);
    g_free(m[nrl]+ncl - GPIV_END);
    g_free(m+nrl - GPIV_END);
    m = NULL;
}



int **
gpiv_imatrix(long nr, 
             long nc)

/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for 2-dimensional matrix of integer data
 *      unappreciated, use gpiv_imatrix_index instead
 *
 * INPUTS:
 *	nr:	        lowest row index number
 *	nc:	        lowest column index number
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     matrix
 *---------------------------------------------------------------------------*/
{
    long i;
    int **m;

    m = (int **) g_malloc0(nr * sizeof(int*));
    m[0] = (int *) g_malloc0(nr * nc * sizeof(int));
    for(i = 1; i < nr; i++) {
        m[i] = m[0] + i * nc;
    }

    return m;
}



void 
gpiv_free_imatrix(int **m
                  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Frees memory for 2-dimensional array of integer data
 *      unappreciated, use gpiv_free_imatrix_index instead
 *
 * INPUTS:
 *      m:	        matrix 
 *
 * OUTPUTS:
 *      m:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(m[0] != NULL);
    g_free(m[0]);
    g_free(m);
    m = NULL;
}



double **
gpiv_double_matrix(glong nr, 
                    glong nc 
                    )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      allocate a contiguous 2-dimensional double matrix 
 *      of nr x nc
 *
 * INPUTS:
 *      nr:             number of rows
 *      nc:             number of columns
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     matrix
 *---------------------------------------------------------------------------*/
{
    double **m;
    gint i;

    m = (double **) g_malloc0(nr * sizeof(double *));
    m[0] = (double *) g_malloc0(nr * nc * sizeof(double));
    for(i = 1; i < nr; i++) {
        m[i] = m[0] + i * nc;
    }

    return m;
}



void 
gpiv_free_double_matrix(double **m
                         )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      free a double matrix allocated by gpiv_double_matrix()
 *
 * INPUTS:
 *      m:	        matrix 
 *
 * OUTPUTS:
 *      m:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(m[0] != NULL);
    g_free(m[0]);
    g_free(m);
    m = NULL;
}


/* #undef FFTW2 */
/* #define FFTW2 */
/* #ifdef FFTW2 */
fftw_real **
gpiv_fftw_real_matrix(glong nr, 
                      glong nc 
                      )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      allocate a contiguous 2-dimensional fftw_real matrix 
 *      of nr x nc
 *
 * INPUTS:
 *      nr:             number of rows
 *      nc:             number of columns
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     matrix
 *---------------------------------------------------------------------------*/
{
    fftw_real **m;
    gint i;

    m = (fftw_real **) g_malloc0(nr * sizeof(fftw_real *));
    m[0] = (fftw_real *) g_malloc0(nr * nc * sizeof(fftw_real));
    for(i = 1; i < nr; i++) {
        m[i] = m[0] + i * nc;
    }

    return m;
}



void 
gpiv_free_fftw_real_matrix(fftw_real **m
                         )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      free a gdouble matrix allocated by gpiv_fftw_real_matrix()
 *
 * INPUTS:
 *      m:	        matrix 
 *
 * OUTPUTS:
 *      m:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(m[0] != NULL);
    g_free(m[0]);
    g_free(m);
    m = NULL;
}
/* #endif */ /* FFTW2 */


fftw_complex **
gpiv_fftw_complex_matrix(long nr, 
                         long nc 
                         )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      allocate a contiguous 2-dimensional fftw_complex matrix 
 *      of nr x nc
 *
 * INPUTS:
 *      nr:             number of rows
 *      nc:             number of columns
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     matrix
 *---------------------------------------------------------------------------*/
{
    fftw_complex **m;
    gint i;
/*
 * allocate pointers to rows
 */
    m = (fftw_complex **) fftw_malloc(nr * sizeof(fftw_complex *));
    m[0] = (fftw_complex *) fftw_malloc(nr * nc * sizeof(fftw_complex));
    for(i = 1; i < nr; i++) {
        m[i] = m[0] + i * nc;
    }
    return m;
}



void 
gpiv_free_fftw_complex_matrix(fftw_complex **m
                              )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      free a fftw_complex matrix allocated by gpiv_fftw_complex_matrix()
 *
 * INPUTS:
 *      m:	        matrix 
 *
 * OUTPUTS:
 *      m:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(m[0] != NULL);
    fftw_free(m[0]);
    fftw_free(m);
    m = NULL;
}



float *
gpiv_vector(long nl
            )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of float data
 *
 * INPUTS:
 *      nl:		vector length
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/
{
    float *v;
    
    v=(float *)g_malloc0((size_t) (nl * sizeof(float)));
    return v;
}



void 
gpiv_free_vector(float *v
		 )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of float data
 *
 * INPUTS:
 *      vector:             vector of 1-dimensional float data
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(v != NULL);
    g_free(v);
    v = NULL;
}



float *
gpiv_vector_index(long nl, 
                  long nh
                  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of float data
 *      with subscript range v[nl..nh]
 *
 * INPUTS:
 *      nl:	       lowest index number
 *      nh:	       highest index number
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/
{
    float *v;
    
    v=(float *)g_malloc0((size_t) ((nh + 1 - nl + GPIV_END) * sizeof(float)));
    return v /* - nl + GPIV_END */;
}



void 
gpiv_free_vector_index(float *v,
                       long nl,
                       long nh
                       )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of float data
 *      with subscript range v[nrl..nrh][ncl..nch]
 *
 * INPUTS:
 *      vector:        vector of 1-dimensional float data
 *      nl:	       lowest index number
 *      nh:	       highest index number
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(v != NULL);
    g_free(v /* + nl - GPIV_END */);
    v = NULL;
}



double *
gpiv_dvector(long nl 
             )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of double data
 *
 * INPUTS:
 *      nl:		vector length
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/
{
    double *v;
    
    v=(double *)g_malloc0((size_t) (nl * sizeof(double)));
    return v;
}



void
gpiv_free_dvector(double *v
                  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of double data
 *
 * INPUTS:
 *      vector:             vector of 1-dimensional float data
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(v != NULL);
    g_free(v);
    v = NULL;
}



double *
gpiv_dvector_index(long nl, 
                   long nh
                   )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of double data
 *      with subscript range v[nl..nh]
 *
 * INPUTS:
 *      nl:	       lowest index number
 *      nh:	       highest index number
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/
{
    double *v;
    
    v=(double *)g_malloc0((size_t) ((nh + 1 /* -nl+GPIV_END */) * 
                                    sizeof(double)));
    return v - nl + GPIV_END;
}



void
gpiv_free_dvector_index(double *v, 
                        long nl, 
                        long nh
                        )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of double data
 *      with subscript range v[nrl..nrh][ncl..nch]
 *
 * INPUTS:
 *      vector:        vector of 1-dimensional float data
 *      nl:	       lowest index number
 *      nh:	       highest index number
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(v != NULL);
    g_free(v + nl - GPIV_END);
    v = NULL;
}



long *
gpiv_nulvector(long nl
               )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of long data
 *
 * INPUTS:
 *      nl:		vector length
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/
{
    long *v;
    
    v=(long *)g_malloc0((size_t) (nl * sizeof(long)));
    return v /* -nl+GPIV_END */;
}



void 
gpiv_free_nulvector(long *v
                    )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of long data
 *
 * INPUTS:
 *      vector:             vector of 1-dimensional float data
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(v != NULL);
    g_free(v);
    v = NULL;
}



long *
gpiv_nulvector_index(long nl, 
                     long nh
                     )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of long data
 *      with subscript range v[nl..nh]
 *
 * INPUTS:
 *      nl:	       lowest index number
 *      nh:	       highest index number
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/
{
    long *v;
    
    v=(long *)g_malloc0((size_t) ((nh + 1 - nl + GPIV_END) * 
                                  sizeof(long)));
    return v - nl + GPIV_END;
}



void 
gpiv_free_nulvector_index(long *v, 
                          long nl, 
                          long nh)
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of long data
 *      with subscript range v[nrl..nrh][ncl..nch]
 *
 * INPUTS:
 *      vector:        vector of 1-dimensional float data
 *      nl:	       lowest index number
 *      nh:	       highest index number
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(v != NULL);
    g_free(v + nl - GPIV_END);
    v = NULL;
}



unsigned long *
gpiv_ulvector(long nl
              )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of unsigned long data
 *
 * INPUTS:
 *      nl:		vector length
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/
{
    unsigned long *v;
    
    v=(unsigned long *)g_malloc0((size_t) (nl * sizeof(long)));
    if (!v) gpiv_error("GPIV_ULVECTOR: allocation failure");
    return v;
}



void 
gpiv_free_ulvector(unsigned long *v
		   )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of unsigned long data
 *
 * INPUTS:
 *      vector:             vector of 1-dimensional float data
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(v != NULL);
    g_free(v);
    v = NULL;
}



unsigned long *
gpiv_ulvector_index(long nl, 
                    long nh
                    )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of unsigned long data
 *      with subscript range v[nl..nh]
 *
 * INPUTS:
 *      nl:	       lowest index number
 *      nh:	       highest index number
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/
{
    unsigned long *v;
    
    v=(unsigned long *)g_malloc0((size_t) ((nh + 1 - nl + GPIV_END) *
                                           sizeof(long)));
    if (!v) gpiv_error("GPIV_ULVECTOR_INDEX: allocation failure");
    return v - nl + GPIV_END;
}



void 
gpiv_free_ulvector_index(unsigned long *v, 
                         long nl, 
                         long nh
		   )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of unsigned long data
 *      with subscript range v[nrl..nrh][ncl..nch]
 *
 * INPUTS:
 *      vector:        vector of 1-dimensional float data
 *      nl:	       lowest index number
 *      nh:	       highest index number
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(v != NULL);
    g_free(v + nl - GPIV_END);
    v = NULL;
}



int *
gpiv_ivector(long nl
             )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of integer data
 *
 * INPUTS:
 *      nl:		vector length
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/
{
    int *v;

    v=(int *)g_malloc0((size_t) (nl * sizeof(int)));
    if (!v) gpiv_error("GPIV_IVECTOR: allocation failure");
    return v;
}



void 
gpiv_free_ivector(int *v
		  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of integer data
 *
 * INPUTS:
 *      vector:             vector of 1-dimensional float data
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(v != NULL);
    g_free(v);
    v = NULL;
}



int *
gpiv_ivector_index(long nl, 
                   long nh
             )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of integer data
 *      with subscript range v[nl..nh]
 *
 * INPUTS:
 *      nl:	       lowest index number
 *      nh:	       highest index number
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/
{
    int *v;

    v=(int *)g_malloc0((size_t) ((nh + 1 - nl + GPIV_END)*sizeof(int)));
    if (!v) gpiv_error("GPIV_IVECTOR_INDEX: allocation failure");
    return v -nl + GPIV_END;
}



void 
gpiv_free_ivector_index(int *v, 
                        long nl, 
                        long nh
                        )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of integer data
 *      with subscript range v[nrl..nrh][ncl..nch]
 *
 * INPUTS:
 *      vector:        vector of 1-dimensional float data
 *      nl:	       lowest index number
 *      nh:	       highest index number
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(v != NULL);
    g_free(v + nl - GPIV_END);
    v = NULL;
}



gboolean *
gpiv_gbolvector(glong nl
                )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Allocates memory for a 1-dimensional vector of gboolean data
 *
 * INPUTS:
 *      nl:		vector length
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      vector
 *---------------------------------------------------------------------------*/
{
    gboolean *v;
    
    v=(gboolean *)g_malloc0((size_t) (nl * sizeof(gboolean)));
    return v;
}



void 
gpiv_free_gbolvector(gboolean *v
                     )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      frees memory for a 1-dimensional vector of gboolean data
 *
 * INPUTS:
 *      vector:             vector of 1-dimensional boolean data
 *
 * OUTPUTS:
 *      vector:              NULL pointer
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    assert(v != NULL);
    g_free(v);
    v = NULL;
}



long
gpiv_lmax(long a, 
	  long b
	  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Calculates maximum of longs a and b
 *
 * INPUTS:
 *      a:              first variable to be tested
 *      b:              second variable to be tested
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      result of maximum
 *---------------------------------------------------------------------------*/
{
    long c;

    if (a >= b) {
        c=a;
    } else {
        c=b;
    }

    return c;
}



long
gpiv_lmin(long a, 
	  long b
	  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Calculates minimum of longs a and b
 *
 * INPUTS:
 *      a:              first variable to be tested
 *      b:              second variable to be tested
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      result of minimum
 *---------------------------------------------------------------------------*/
{
    long c;

    if (a <= b) {
        c=a;
    } else {
        c=b;
    }

    return c;
}



int 
gpiv_max(int a, 
	 int b
	 )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Calculates maximum of integers a and b
 *
 * INPUTS:
 *      a:              first variable to be tested
 *      b:              second variable to be tested
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      result of maximum
 *---------------------------------------------------------------------------*/
{
    int c;

    if (a >= b) {
        c=a;
    } else {
        c=b;
    }

    return c;
}



int 
gpiv_min(int a, 
	 int b
	 )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Calculates minimum of integers a and b
 *
 * INPUTS:
 *      a:              first variable to be tested
 *      b:              second variable to be tested
 *
 * OUTPUTS:
 *
 * RETURNS:
 *      result of minimum
 *---------------------------------------------------------------------------*/
{
    int c;

    if (a <= b) {
        c=a;
    } else {
        c=b;
    }

    return c;
}



gboolean
gpiv_scan_iph (const char *PROGNAME, 
	       char *line, 
	       char *par_name, 
	       char *key, 
	       int *parameter, 
	       int print_par, 
	       int flag_par
	       )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Function to scan line string on **integer** parameter key
 *      and value or image header value (without program key).
 *
 * PROTOTYPE LOCATATION:
 *     utils.h
 *
 * INPUTS:
 *      PROGNAME:       Program name as part of parameter key
 *      line:           line to be scanned
 *      par_name:       parameter name which is scanned in line
 *      key:            parameter key
 *      print_par:      flag to print prarameter to stdout
 *      flag_par:       flag to use program key
 *
 * INPUTS:
 *      parameter:      parameter value to be returned
 *
 * RETURNS:
 *      flag representing parameter_logic
 *      set to FALSE if parameter has been read, else TRUE
 *-------------------------------------------------------------------------- */
{
    char tmp_string[GPIV_MAX_CHARS];
    gboolean flag_par_set = FALSE;

    if (flag_par) {
        strcpy(tmp_string, PROGNAME);
        strcat(tmp_string, key);
    } else {
        strcpy(tmp_string, key);
    }
    
    if (strcmp(par_name, tmp_string) == 0) {
        sscanf(line, "%s %d", par_name, parameter);
        flag_par_set = TRUE;
        if (print_par) printf("%s %d\n", par_name, *parameter);
    }
    
    return flag_par_set;
}



gboolean
gpiv_scan_iph_nl (FILE *fp_h, 
                  const char *PROGNAME, 
                  char *line, 
                  char *par_name, 
                  char *key, 
                  int *parameter, 
                  gboolean print_par, 
                  gboolean flag_par
	       )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Function to scan line string on **integer** parameter key
 *      and value or image header value (without program key). The
 *      value to be read is on the next line of the key.
 *
 * PROTOTYPE LOCATATION:
 *     utils.h
 *
 * INPUTS:
 *      fp_h:           file pointer of header to read the data from
 *      PROGNAME:       Program name as part of parameter key
 *      line:           line to be scanned
 *      par_name:       parameter name which is scanned in line
 *      key:            parameter key
 *      print_par:      flag to print prarameter to stdout
 *      flag_par:       flag to use program key
 *
 * INPUTS:
 *      parameter:      parameter value to be returned
 *
 * RETURNS:
 *      flag representing parameter_logic
 *      set to FALSE if parameter has been read, else TRUE
 *-------------------------------------------------------------------------- */
{
    char tmp_string[GPIV_MAX_CHARS];
    gboolean flag_par_set = FALSE;

    if (flag_par) {
        strcpy(tmp_string, PROGNAME);
        strcat(tmp_string, key);
    } else {
        strcpy(tmp_string, key);
    }
    
    g_message("gpiv_scan_iph_nl:: tmp_string=%s", tmp_string);
    if (strcmp(par_name, tmp_string) == 0) {
        flag_par_set = TRUE;
        if (fgets(line, GPIV_MAX_CHARS, fp_h) != NULL) {
            g_message("gpiv_scan_iph_nl:: line = %s", line);
            sscanf(line, "%d", parameter);
            if (print_par) printf("%s %d\n", par_name, *parameter);
        } else {
            gpiv_error("gpiv_scan_iph_nl: line = NULL");
        }
    }
    
    return flag_par_set;
}



gboolean
gpiv_fscan_iph (const char *PROGNAME, 
		char *line, 
		char *par_name, 
		char *key, 
		int *parameter, 
		int print_par, 
		FILE *fp, 
		gboolean flag_par
		)
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Function to scan line string on **integer** parameter key
 *      and value or image header value (without program key). Prints result
 *      to file.
 *
 * PROTOTYPE LOCATATION:
 *      utils.h
 *
 * INPUTS:
 *      PROGNAME:       Program name as part of parameter key
 *      line:           line to be scanned
 *      par_name:       parameter name which is scanned in line
 *      key:            parameter key
 *      print_par:      flag to print prarameter to stdout
 *      fp:             file pointer to print parameter to file
 *      flag_par:       flag to use program key
 *
 * OUTPUTS:
 *      parameter:      parameter value to be returned
 *
 * RETURNS:
 *      flag representing parameter_logic
 *      set to FALSE if parameter has been read, else TRUE
 *-------------------------------------------------------------------------- */
{
    char tmp_string[GPIV_MAX_CHARS];
    gboolean flag_par_set = FALSE;
    
    if (flag_par) {
        strcpy(tmp_string, PROGNAME);
        strcat(tmp_string, key);
    } else {
        strcpy(tmp_string, key);
    }
    
    if (strcmp(par_name, tmp_string) == 0) {
        sscanf(line, "%s %d", par_name, parameter);
        flag_par_set = TRUE;
        if (print_par) printf("%s %d\n", par_name, *parameter);
        fprintf(fp, "%s %d\n", par_name, *parameter);
    }
    
    return flag_par_set;
}



gboolean 
gpiv_scan_cph (const char *PROGNAME, 
	       char *line, 
	       char *par_name, 
	       char *key, 
	       char *parameter, 
	       char print_par, 
	       int flag_par
	       )
/*---------------------------------------------------------------------------- 
 * DESCRIPTION:
 *     Function to scan line string on **character** parameter/header
 *     key and value or image header value (without program key).
 *
 * PROTOTYPE LOCATATION:
 *     utils.h
 *
 * INPUTS:
 *     PROGNAME:       Program name as part of parameter key
 *     line:           line to be scanned
 *     par_name        parameter name which is scanned in line
 *     key:            parameter key
 *     print_par:      flag to print prarameter to stdout
 *     flag_par:       flag to use program key
 *
 * OUTPUTS:
 *     parameter:      parameter value to be returned
 *
 * RETURNS:
 *      flag representing parameter_logic
 *      set to FALSE if parameter has been read, else TRUE
 *-------------------------------------------------------------------------- */
{
    char tmp_string[GPIV_MAX_CHARS];
    gboolean flag_par_set = FALSE;

    if (flag_par) {
        strcpy(tmp_string, PROGNAME);
        strcat(tmp_string, key);
    } else {
        strcpy(tmp_string, key);
    }

    if (strcmp(par_name, tmp_string) == 0) {
        sscanf(line, "%s %c", par_name, parameter);
        flag_par_set = TRUE;
        if (print_par) printf("%s %c\n", par_name, *parameter);
    }
    
    return flag_par_set;
}



gboolean
gpiv_fscan_cph (const char *PROGNAME, 
		char *line, 
		char *par_name, 
		char *key, 
		char *parameter, 
		char print_par, 
		FILE *fp, int flag_par
		)
/*---------------------------------------------------------------------------- 
 * DESCRIPTION:
 *     Function to scan line string on **character** parameter/header
 *     key and value or image header value (without program key). Prints result
 *      to file.
 *
 * PROTOTYPE LOCATATION:
 *     utils.h
 *
 * INPUTS:
 *     PROGNAME:       Program name as part of parameter key
 *     line:           line to be scanned
 *     par_name        parameter name which is scanned in line
 *     key:            parameter key
 *     print_par:      flag to print prarameter to stdout
 *     fp:             file pointer to print parameter to file
 *     flag_par:       flag to use program key
 *
 * OUTPUTS:
 *     parameter:      parameter value to be returned
 *
 * RETURNS:
 *      flag representing parameter_logic
 *      set to FALSE if parameter has been read, else TRUE
 *-------------------------------------------------------------------------- */
{
    char tmp_string[GPIV_MAX_CHARS];
    gboolean flag_par_set = FALSE;

    if (flag_par) {
        strcpy(tmp_string, PROGNAME);
        strcat(tmp_string, key);
    } else {
        strcpy(tmp_string, key);
    }
    
    if (strcmp(par_name, tmp_string) == 0) {
        sscanf(line, "%s %c", par_name, parameter);
        flag_par_set = TRUE;
        if (print_par) printf("%s %c\n", par_name, *parameter);
        fprintf(fp, "%s %c\n", par_name, *parameter);
    }
    
    return flag_par_set;
}



gboolean
gpiv_scan_fph (const char *PROGNAME, 
	       char *line, 
	       char *par_name, 
	       char *key, 
	       float *parameter, 
	       char print_par, 
	       int flag_par
	       )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Function to scan line string on **float** parameter/header
 *     key and value or image header value (without program key).
 *
 * PROTOTYPE LOCATATION:
 *     utils.h
 *
 * INPUTS:
 *     PROGNAME:       Program name as part of parameter key
 *     line:           line to be scanned
 *     par_name        parameter name which is scanned in line
 *     key:            parameter key
 *     print_par:      flag to print prarameter to stdout
 *     flag_par:       flag to use program key
 *
 * OUTPUTS:
 *     parameter:      parameter value to be returned
 *
 * RETURNS:
 *      flag representing parameter_logic
 *      set to FALSE if parameter has been read, else TRUE
 *-------------------------------------------------------------------------- */
{
    char tmp_string[GPIV_MAX_CHARS];
    gboolean flag_par_set = FALSE;
    
    if (flag_par) {
        strcpy(tmp_string, PROGNAME);
        strcat(tmp_string, key);
    } else {
        strcpy(tmp_string, key);
    }
    
    if (strcmp(par_name, tmp_string) == 0) {
        sscanf(line, "%s %f", par_name, parameter);
        flag_par_set = TRUE;
        if (print_par) printf("%s %f\n", par_name, *parameter);
    }
    
    return flag_par_set;
}



gboolean
gpiv_fscan_fph (const char *PROGNAME, 
		char *line, 
		char *par_name, 
		char *key, 
		float *parameter, 
		char print_par, FILE *fp, 
		int flag_par
		)
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Function to scan line string on **float** parameter/header
 *     key and value or image header value (without program key). Prints result
 *      to file.
 *
 * PROTOTYPE LOCATATION:
 *     utils.h
 *
 * INPUTS:
 *     PROGNAME:       Program name as part of parameter key
 *     line:           line to be scanned
 *     par_name        parameter name which is scanned in line
 *     key:            parameter key
 *     print_par:      flag to print prarameter to stdout
 *     fp:             file pointer to print parameter to file
 *     flag_par:       flag to use program key
 *
 * OUTPUTS:
 *     parameter:      parameter value to be returned
 *
 * RETURNS:
 *      flag representing parameter_logic
 *      set to FALSE if parameter has been read, else TRUE
 *-------------------------------------------------------------------------- */
{
    char tmp_string[GPIV_MAX_CHARS];
    gboolean flag_par_set = FALSE;
    
    if (flag_par) {
        strcpy(tmp_string, PROGNAME);
        strcat(tmp_string, key);
    } else {
        strcpy(tmp_string, key);
    }
    
    if (strcmp(par_name, tmp_string) == 0) {
        sscanf(line, "%s %f", par_name, parameter);
        flag_par_set = TRUE;
        if (print_par) printf("%s %f\n", par_name, *parameter);
        fprintf(fp, "%s %f\n", par_name, *parameter);
    }
    
    return flag_par_set;
}



gboolean
gpiv_scan_sph (const char *PROGNAME, 
	       char *line, 
	       char *par_name, 
	       char *key, 
	       char *parameter, 
	       char print_par, 
	       int flag_par
	       )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Function to scan line string on **string** parameter/header
 *     key and value or image header value (without program key).
 *
 * PROTOTYPE LOCATATION:
 *     utils.h
 *
 * INPUTS:
 *     PROGNAME:       Program name as part of parameter key
 *     line:           line to be scanned
 *     par_name        parameter name which is scanned in line
 *     key:            parameter key
 *     print_par:      flag to print prarameter to stdout
 *     fp:             file pointer to print parameter to file
 *     flag_par:       flag to use program key
 *
 * OUTPUTS:
 *     parameter:      parameter value to be returned
 *
 * RETURNS:
 *     flag representing parameter_logic
 *     set to FALSE if parameter has been read, else TRUE
 *---------------------------------------------------------------------------*/
{
    char tmp_string[GPIV_MAX_CHARS];
    gboolean flag_par_set = FALSE;

    size_t key_length, i = 0;
    
    if (flag_par) {
        strcpy(tmp_string, PROGNAME);
        strcat(tmp_string, key);
    } else {
        strcpy(tmp_string, key);
    }
    
    key_length = strlen(tmp_string) + 1;
    
    if (strcmp(par_name, tmp_string) == 0) {
        while (line[key_length + i] != '\0' 
               && i < GPIV_MAX_CHARS - key_length) {
            if(line[key_length + i] != '\n') {
                parameter[i] = line[key_length + i];
                i++;
            } else {
                parameter[i] = '\0';
                break;
            }
        }
        
        flag_par_set = TRUE;
        if (print_par) printf("%s %s\n", par_name, parameter);
    }

    return flag_par_set;
}



gboolean
gpiv_fscan_sph (const char *PROGNAME, 
		char *line, 
		char *par_name, 
		char *key, 
		char *parameter, 
		char print_par, FILE *fp, 
		int flag_par
		)
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Function to scan line string on **string** parameter/header
 *     key and value or image header value (without program key). Prints result
 *      to file.
 *
 * PROTOTYPE LOCATATION:
 *     utils.h
 *
 * INPUTS:
 *     PROGNAME:       Program name as part of parameter key
 *     line:           line to be scanned
 *     par_name        parameter name which is scanned in line
 *     key:            parameter key
 *     print_par:      flag to print prarameter to stdout
 *     fp:             file pointer to print parameter to file
 *     flag_par:       flag to use program key
 *
 * OUTPUTS:
 *     parameter:      parameter value to be returned
 *
 * RETURNS:
 *     flag representing parameter_logic
 *     set to FALSE if parameter has been read, else TRUE
 *---------------------------------------------------------------------------*/
{
    char tmp_string[GPIV_MAX_CHARS];
    gboolean flag_par_set = FALSE;
    int key_length, i = 0;
    
    if (flag_par) {
        strcpy(tmp_string, PROGNAME);
        strcat(tmp_string, key);
    } else {
        strcpy(tmp_string, key);
    }
    
    key_length = strlen(tmp_string) + 1;
    
    if (strcmp(par_name, tmp_string) == 0) {
        while (line[key_length + i] != '\0' 
               && i < GPIV_MAX_CHARS - key_length) {
            if(line[key_length + i] != '\n') {
                parameter[i] = line[key_length + i];
                i++;
            } else {
                parameter[i] = '\0';
                break;
            }
        }
        
        flag_par_set = TRUE;
        if (print_par) printf("%s %s\n", par_name, parameter);
        fprintf(fp, "%s %s\n", par_name, *parameter);
    }

    return flag_par_set;
}




#define SWAP(a,b) temp=(a);(a)=(b);(b)=temp;
#define M 7
#define NSTACK 50



char *
gpiv_sort_3(unsigned long n, 
	    float arr[], 
	    float arr_2[], 
	    float arr_3[]
	    )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Orders array arr AND its belonging arrays arr_2 and arr_3
 *      Also swaps accompanying arrays
 *
 * INPUTS:
 *      n:	        length of array
 *      arr[]:	        array to be sorted
 *      arr_3[]:	second belonging array to arr
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char * err_msg = NULL;
    unsigned long i,ir=n,j,k,l=1,*istack;
    int jstack=0;
    float a,b,c,temp;
    
    istack=gpiv_ulvector_index(1,NSTACK);
    for (;;) {
        if (ir-l < M) {
            for (j=l+1;j<=ir;j++) {
                a=arr[j];
                b=arr_2[j];
                c=arr_3[j];
                for (i=j-1;i>=l;i--) {
                    if (arr[i] <= a) break;
                    arr[i+1]=arr[i];
                    arr_2[i+1]=arr_2[i];
                    arr_3[i+1]=arr_3[i];
                }
                arr[i+1]=a;
                arr_2[i+1]=b;
                arr_3[i+1]=c;
            }
            if (jstack == 0) break;
            ir=istack[jstack--];
            l=istack[jstack--];
        } else {
            k=(l+ir) >> 1;
            SWAP(arr[k],arr[l+1]);
            SWAP(arr_2[k],arr_2[l+1]);
            SWAP(arr_3[k],arr_3[l+1]);
            if (arr[l] > arr[ir]) {
                SWAP(arr[l],arr[ir]);
                SWAP(arr_2[l],arr_2[ir]);
                SWAP(arr_3[l],arr_3[ir]);
            }
            if (arr[l+1] > arr[ir]) {
                SWAP(arr[l+1],arr[ir]);
                SWAP(arr_2[l+1],arr_2[ir]);
                SWAP(arr_3[l+1],arr_3[ir]);
            }
            if (arr[l] > arr[l+1]) {
                SWAP(arr[l],arr[l+1]);
                SWAP(arr_2[l],arr_2[l+1]);
                SWAP(arr_3[l],arr_3[l+1]);
            }
            i=l+1;
            j=ir;
            a=arr[l+1];
            b=arr_2[l+1];
            c=arr_3[l+1];
            for (;;) {
                do i++; while (arr[i] < a);
                do j--; while (arr[j] > a);
                if (j < i) break;
                SWAP(arr[i],arr[j]);
                SWAP(arr_2[i],arr_2[j]);
                SWAP(arr_3[i],arr_3[j]);
            }
            arr[l+1]=arr[j];
            arr_2[l+1]=arr_2[j];
            arr_3[l+1]=arr_3[j];
            arr[j]=a;
            arr_2[j]=b;
            arr_3[j]=c;
            jstack += 2;
            if (jstack > NSTACK) { 
                err_msg = "GPIV_SORT_3: NSTACK too small in sort.";
                gpiv_warning("%s", err_msg);
                return err_msg;
            }
            if (ir-i+1 >= j-l) {
                istack[jstack]=ir;
                istack[jstack-1]=i;
                ir=j-1;
            } else {
	istack[jstack]=j-1;
	istack[jstack-1]=l;
	l=i;
            }
        }
    }

    gpiv_free_ulvector_index(istack,1,NSTACK);
    return err_msg;
}

#undef M
#undef NSTACK
#undef SWAP



void 
gpiv_histo(GpivPivData * data, 
	   GpivBinData * klass
	   )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Calculates histogram from GpivPivData (NOT from ScalarData!!)
 *
 * INPUTS:
 *      data:           Input data
 *
 * OUTPUTS:
 *      klass:          Output data
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    int i, j, k;
    int nx = data->nx, ny = data->ny, **peak_no = data->peak_no;
    float **snr = data->snr;
    float delta;
    float *bound = klass->bound, *centre = klass->centre;
    int *count = klass->count, nbins = klass->nbins;
    
    assert(data->point_x != NULL);
    assert(data->point_y != NULL);
    assert(data->dx != NULL);
    assert(data->dy != NULL);
    assert(data->snr != NULL);
    assert(data->peak_no != NULL);
    
    assert(klass->count != NULL);
    assert(klass->bound != NULL);
    assert(klass->centre != NULL);
    
    klass->min = 10.0e+9, klass->max = -10.0e+9;
/*
 * find min and max value
 */
    for (i = 0; i < ny; i++) {
	for (j = 0; j < nx; j++) {
	    if (peak_no[i][j] != -1) {

		if (snr[i][j] < klass->min)
		    klass->min = snr[i][j];
		if (snr[i][j] >= klass->max)
		    klass->max = snr[i][j];

	    }
	}
    }



/*
 * Calculating boundaries of bins
 */
    delta = (klass->max - klass->min) / nbins;
    for (i = 0; i < nbins; i++) {
	centre[i] = klass->min + delta / 2.0 + (float) i *delta;
	count[i] = 0;
    }

    for (i = 0; i < nbins; i++) {
	bound[i] = klass->min + (float) i * delta;
    }



/*
 * Sorting of snr data in bins
 */
    for (i = 0; i < ny; i++) {
	for (j = 0; j < nx; j++) {
	    if (peak_no[i][j] != -1) {

		for (k = 0; k < nbins; k++) {
		    if ((snr[i][j] > bound[k])
			&& (snr[i][j] <= bound[k] + delta)) {
			count[k] = count[k] + 1;
		    }

		}
	    }
	}
    }

}



void 
gpiv_cumhisto(GpivPivData * data, 
	      GpivBinData * klass
	      )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Calculates cumulative histogram from GpivPivData (NOT from ScalarData!!)
 *
 * INPUTS:
 *      data:           Input data
 *
 * OUTPUTS:
 *      klass:          Output data
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    int i, j, k;
    int nx = data->nx, ny = data->ny, **peak_no = data->peak_no;
    float **snr = data->snr;
    float delta;
    float *bound = klass->bound, *centre = klass->centre;
    int *count = klass->count, nbins = klass->nbins;

    assert(data->point_x != NULL);
    assert(data->point_y != NULL);
    assert(data->dx != NULL);
    assert(data->dy != NULL);
    assert(data->snr != NULL);
    assert(data->peak_no != NULL);
    
    assert(klass->count != NULL);
    assert(klass->bound != NULL);
    assert(klass->centre != NULL);
  
   klass->min = 10e+9, klass->max = -10e+9;
/*
 * find min and max value
 */
    for (i = 0; i < ny; i++) {
	for (j = 0; j < nx; j++) {
	    if (peak_no[i][j] != -1) {

		if (snr[i][j] < klass->min)
		    klass->min = snr[i][j];
		if (snr[i][j] >= klass->max)
		    klass->max = snr[i][j];

	    }
	}
    }

/*     fprintf(stderr, "gpiv_cumhisto:: min=%f max=%f", klass->min, klass->max); */

/*
 * Calculating boundaries of bins
 */
    delta = (klass->max - klass->min) / nbins;
    for (i = 0; i < nbins; i++) {
	centre[i] = klass->min + delta / 2.0 + (float) i *delta;
	count[i] = 0;
    }

    for (i = 0; i < nbins; i++) {
	bound[i] = klass->min + (float) i * delta;
    }



/*
 * Sorting of snr data in bins
 */
    for (i = 0; i < ny; i++) {
	for (j = 0; j < nx; j++) {
	    if (peak_no[i][j] != -1) {

		for (k = 0; k < nbins; k++) {
		    if (snr[i][j] <= bound[k] + delta) {
			count[k] = count[k] + 1;
		    }

		}
	    }
	}
    }

}



char *
gpiv_histo_gnuplot(char *fname_out, 
		   char *title, 
		   const char *GNUPLOT_DISPLAY_COLOR,
		   const int GNUPLOT_DISPLAY_SIZE
		   )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Plots histogram on screen with gnuplot
 *
 * INPUTS:
 *      fname_out:	output filename
 *      title:	        plot title
 *      GNUPLOT_DISPLAY_COLOR:  display color of window containing graph
 *      GNUPLOT_DISPLAY_SIZE:   display size of window containing graph
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    FILE *fp_cmd;
    const gchar *tmp_dir = g_get_tmp_dir ();
    char *fname_loc= "gpiv_gnuplot.cmd";
    char *function_name= "gpiv_histo_gnuplot";
    char fname_cmd[GPIV_MAX_CHARS];
    char command[GPIV_MAX_CHARS];
    
    snprintf(fname_cmd, GPIV_MAX_CHARS, "%s/%s", tmp_dir, fname_loc);
    g_message("gpiv_histo_gnuplot:: 1 fname_cmd=%s", fname_cmd);
    if((fp_cmd = fopen(fname_cmd,"w")) == NULL) { 
        err_msg = "GPIV_HISTO_GNUPLOT: Failure opening for output";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    fprintf (fp_cmd, "set xlabel \"residu (pixels)\"");
    fprintf (fp_cmd, "\nset ylabel \"frequency\"");
    fprintf (fp_cmd, "\nplot \"%s\" title \"%s\" with boxes", 
             fname_out, title);
    
    fprintf (fp_cmd, "\npause -1 \"Hit return to exit\"");
    fprintf (fp_cmd, "\nquit");
    
    fclose (fp_cmd);
  
    
    snprintf(command, GPIV_MAX_CHARS, "gnuplot -bg %s -geometry %dx%d %s",
             GNUPLOT_DISPLAY_COLOR, GNUPLOT_DISPLAY_SIZE, 
             GNUPLOT_DISPLAY_SIZE, fname_cmd);
    
    if (system (command) != 0) {
        g_warning ("%s:%s could not exec shell command", 
                 LIBNAME, function_name);

        exit(1);
    }
    
    return err_msg;
}



char *
gpiv_piv_gnuplot(char *fname, 
                 char *title, 
                 float gnuplot_scale,
                 const char *GNUPLOT_DISPLAY_COLOR, 
                 const int GNUPLOT_DISPLAY_SIZE,
                 GpivImagePar image_par, 
                 GpivEvalPar piv_eval_par,
                 GpivPivData piv_data,
                 const char *RCSID
                 )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Plots piv data as vectors on screen with gnuplot
 *
 * INPUTS:
 *      fname:         file name containing plot
 *      title:         title of plot
 *      gnuplot_scale: vector scale
 *      GNUPLOT_DISPLAY_COLOR:  display color of window containing graph
 *      GNUPLOT_DISPLAY_SIZE:   display size of window containing graph
 *      image_par:      image parameters
 *      piv_eval_par:   piv evaluation parameters
 *      piv_data:       piv data
 *      RCSID:          program name and version that interrogated the image
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    FILE *fp_cmd;
    gchar *tmp_dir = g_get_tmp_dir ();
    char *fname_loc = "gpiv_gnuplot.cmd";
    char command[GPIV_MAX_CHARS];
    char fname_cmd[GPIV_MAX_CHARS];
    int i, j;

    snprintf(fname_cmd, GPIV_MAX_CHARS, "%s/%s", tmp_dir, fname_loc);
    
    if ((fp_cmd = fopen(fname_cmd, "w")) == NULL)
        gpiv_error ("GPIV_PIV_GNUPLOT: error: Failure opening %s for output",
                    fname_cmd);
    
    fprintf(fp_cmd, "set xlabel \"x (pixels)\"");
    fprintf(fp_cmd, "\nset ylabel \"y (pixels)\"");
    fprintf(fp_cmd, "\nset title \"Piv of %s\" ", title);
    
    for (i = 0; i < piv_data.ny; i++) {
        for (j = 0; j < piv_data.nx; j++) {
            fprintf(fp_cmd, "\nset arrow from %f, %f to %f, %f",
                    piv_data.point_x[i][j], 
                    piv_data.point_y[i][j],
                    piv_data.point_x[i][j] + piv_data.dx[i][j] * gnuplot_scale,
                    piv_data.point_y[i][j] + piv_data.dy[i][j] * gnuplot_scale);
        }
    }
    
    fprintf(fp_cmd, "\nshow arrow; set nokey");
    if (piv_eval_par.int_geo == GPIV_AOI) {
        fprintf(fp_cmd, "\nplot [%d:%d] [%d:%d] %d", 
                piv_eval_par.col_start, piv_eval_par.col_end,
                piv_eval_par.row_start, piv_eval_par.row_end,
                piv_eval_par.row_end);
    } else {
        fprintf(fp_cmd, "\nplot [%d:%d] [%d:%d] %d", 
                0, image_par.ncolumns,
                0, image_par.nrows,
                piv_eval_par.row_end);
        
    }
    fprintf(fp_cmd, "\npause -1 \"Hit return to exit\"");
    fprintf(fp_cmd, "\nquit");
    fclose(fp_cmd);
    
    snprintf(command, GPIV_MAX_CHARS, 
	     "gnuplot -bg %s -geometry %dx%d %s",
	     GNUPLOT_DISPLAY_COLOR, GNUPLOT_DISPLAY_SIZE, GNUPLOT_DISPLAY_SIZE, 
	     fname_cmd);
    
    if (system(command) != 0) {
        err_msg = "GPIV_PIV_GNUPLOT: could not exec shell command";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    return err_msg;
}



char *
gpiv_scalar_gnuplot(char * fname_out, 
		    char *title, 
		    const char *GNUPLOT_DISPLAY_COLOR, 
		    const int GNUPLOT_DISPLAY_SIZE
		    )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Graphical output with gnuplot for scalar data
 *
 * INPUTS:
 *      fname_out:     file name containing plot
 *      title:         title of plot
 *      GNUPLOT_DISPLAY_COLOR:  display color of window containing graph
 *      GNUPLOT_DISPLAY_SIZE:   display size of window containing graph
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    FILE *fp_cmd;
    gchar *tmp_dir = g_get_tmp_dir ();
    char *fname_loc = "gpiv_gnuplot.cmd";
    char *function_name= "gpiv_scalar_gnuplot";
    char fname_cmd[GPIV_MAX_CHARS];
    char command[GPIV_MAX_CHARS];

    snprintf(fname_cmd, GPIV_MAX_CHARS, "%s/%s", tmp_dir, fname_loc);
    if ((fp_cmd = fopen(fname_cmd, "w")) == NULL) {
	 err_msg = "GPIV_SCALAR_GNUPLOT: Failure opening %s for output";
         gpiv_warning("%s", err_msg);
         return err_msg;
    }

     fprintf(fp_cmd, "set xlabel \"x (pixels)\"");
     fprintf(fp_cmd, "\nset ylabel \"y (pixels)\"");
     fprintf(fp_cmd, "\nset contour; set nosurface");
     fprintf(fp_cmd, "\nset view 0, 0");
     fprintf(fp_cmd,
	     "\nsplot \"%s\" title \"%s\" with lines",
	     fname_out, title);
     fprintf(fp_cmd, "\npause -1 \"Hit return to exit\"");
     fprintf(fp_cmd, "\nquit");
     fclose(fp_cmd);

     snprintf(command, GPIV_MAX_CHARS, "gnuplot -bg %s -geometry %dx%d %s",
	      GNUPLOT_DISPLAY_COLOR, GNUPLOT_DISPLAY_SIZE, 
	      GNUPLOT_DISPLAY_SIZE, fname_cmd);

     if (system(command) != 0) {
	  err_msg = "GPIV_SCALAR_GNUPLOT: could not exec shell command";
          gpiv_warning("%s", err_msg);
          return err_msg;
     }

     return err_msg;
}




#include <stdarg.h>

void 
gpiv_warning(char *message, ...
             )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Warning message with variable argumanent list
 *
 * INPUTS:
 *      message:       warning message
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    va_list args;
    
    va_start(args, message);
    fprintf (stderr,"\n# *** message: ");
    vfprintf (stderr, message, args);
    va_end(args);
    
}



void 
gpiv_error(char *message, ...
           )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Prints error handling with variable argumanent list to stdout 
 *      and exits program
 *
 * INPUTS:
 *      message:       warning message
 *
 * OUTPUTS:
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    va_list args;
    
    va_start(args, message);
    fprintf (stderr,"\nerror: ");
    vfprintf (stderr, message, args);
    fprintf (stderr,"\n");
    va_end(args);
    
    exit(1);
}



#undef GPIV_END

