/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This file is part of libgpiv.
   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  



---------------------------------------------------------------
FILENAME:                img.c
LIBRARY:                 libgpiv:
EXTERNAL FUNCTIONS:      gpiv_img_parameters_logic
                         gpiv_img_read_header
			 gpiv_img_fread_header
                         gpiv_img_read_parameters
			 gpiv_img_fread_parameters
			 gpiv_img_check_header_read
			 gpiv_img_test_header
			 gpiv_img_test_header
			 gpiv_img_print_header
			 gpiv_img_fprint_header
			 gpiv_img_print_parameters
			 gpiv_img_fprint_parameters
			 gpiv_img_cp_parameters
			 gpiv_img_fread_hdf5_parameters
			 gpiv_img_fwrite_hdf5_parameters
                         gpiv_img_fread_davis_parameters

LAST MODIFICATION DATE:  $Id: img.c,v 1.14 2006/01/31 13:30:13 gerber Exp $
 --------------------------------------------------------------- */

#include <stdio.h>
#include <unistd.h>
#include <string.h>

#include <gpiv.h>


/*
 * Local functions
 */

static herr_t 
attr_info(hid_t loc_id, const char *name, 
          GpivImagePar * image_par
          )
/*-----------------------------------------------------------------------------
 * Operator function that actually reads the parameters.
 */
{
    hid_t attribute_id, atype;
    herr_t status;



/*
 * General image parameters
 */
    if (strcmp(name, "ncolumns") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
			 &image_par->ncolumns); 
	status = H5Aclose(attribute_id); 
	image_par->ncolumns_logic = TRUE;
    }    


    if (strcmp(name, "nrows") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
			 &image_par->nrows); 
	status = H5Aclose(attribute_id); 
	image_par->nrows_logic = TRUE;
    }


    if (strcmp(name, "x_corr") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &image_par->x_corr); 
        status = H5Aclose(attribute_id); 
        image_par->x_corr_logic = TRUE;
    }    


    if (strcmp(name, "depth") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_INT, 
                         &image_par->depth); 
        status = H5Aclose(attribute_id); 
        image_par->depth_logic = TRUE;
    }


/*
 * Variables and parameters of scale
 */
    if (strcmp(name, "s_scale") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_FLOAT, 
                         &image_par->s_scale); 
        status = H5Aclose(attribute_id); 
        image_par->s_scale_logic = TRUE;
    }


    if (strcmp(name, "t_scale") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_FLOAT, 
                         &image_par->t_scale); 
        status = H5Aclose(attribute_id); 
        image_par->t_scale_logic = TRUE;
    }


    if (strcmp(name, "z_off_x") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_FLOAT, 
                         &image_par->z_off_x); 
        status = H5Aclose(attribute_id); 
        image_par->z_off_x_logic = TRUE;
    }


    if (strcmp(name, "z_off_y") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        status = H5Aread(attribute_id, H5T_NATIVE_FLOAT, 
                         &image_par->z_off_y); 
        status = H5Aclose(attribute_id); 
        image_par->z_off_y_logic = TRUE;
    }


/*
 * Optional parameters
 */

    if (strcmp(name, "title") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        atype = H5Tcopy(H5T_C_S1);
        H5Tset_size(atype, GPIV_MAX_CHARS);
        status = H5Aread(attribute_id, atype, 
                         &image_par->title); 
        status = H5Tclose(atype);
        status = H5Aclose(attribute_id); 
        image_par->title_logic = TRUE;
    }


    if (strcmp(name, "creation_date") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        atype = H5Tcopy(H5T_C_S1);
        H5Tset_size(atype, GPIV_MAX_CHARS);
        status = H5Aread(attribute_id, atype, 
                         &image_par->creation_date); 
        status = H5Tclose(atype);
        status = H5Aclose(attribute_id); 
        image_par->creation_date_logic = TRUE;
    }


    if (strcmp(name, "location") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        atype = H5Tcopy(H5T_C_S1);
        H5Tset_size(atype, GPIV_MAX_CHARS);
        status = H5Aread(attribute_id, atype, 
                         &image_par->location); 
        status = H5Tclose(atype);
        status = H5Aclose(attribute_id); 
        image_par->location_logic = TRUE;
    }


    if (strcmp(name, "author") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        atype = H5Tcopy(H5T_C_S1);
        H5Tset_size(atype, GPIV_MAX_CHARS);
        status = H5Aread(attribute_id, atype, 
                         &image_par->author); 
        status = H5Tclose(atype);
        status = H5Aclose(attribute_id); 
        image_par->author_logic = TRUE;
    }


    if (strcmp(name, "software") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        atype = H5Tcopy(H5T_C_S1);
        H5Tset_size(atype, GPIV_MAX_CHARS);
        status = H5Aread(attribute_id, atype, 
                         &image_par->software); 
        status = H5Tclose(atype);
        status = H5Aclose(attribute_id); 
        image_par->software_logic = TRUE;
    }


    if (strcmp(name, "source") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        atype = H5Tcopy(H5T_C_S1);
        H5Tset_size(atype, GPIV_MAX_CHARS);
        status = H5Aread(attribute_id, atype, 
                         &image_par->source); 
        status = H5Tclose(atype);
        status = H5Aclose(attribute_id); 
        image_par->source_logic = TRUE;
    }


    if (strcmp(name, "usertext") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        atype = H5Tcopy(H5T_C_S1);
        H5Tset_size(atype, GPIV_MAX_CHARS);
        status = H5Aread(attribute_id, atype, 
                         &image_par->usertext); 
        status = H5Tclose(atype);
        status = H5Aclose(attribute_id); 
        image_par->usertext_logic = TRUE;
    }


    if (strcmp(name, "warning") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        atype = H5Tcopy(H5T_C_S1);
        H5Tset_size(atype, GPIV_MAX_CHARS);
        status = H5Aread(attribute_id, atype, 
                         &image_par->warning); 
        status = H5Tclose(atype);
        status = H5Aclose(attribute_id); 
        image_par->warning_logic = TRUE;
    }


    if (strcmp(name, "disclaimer") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        atype = H5Tcopy(H5T_C_S1);
        H5Tset_size(atype, GPIV_MAX_CHARS);
        status = H5Aread(attribute_id, atype, 
                         &image_par->disclaimer); 
        status = H5Tclose(atype);
        status = H5Aclose(attribute_id); 
        image_par->disclaimer_logic = TRUE;
    }


    if (strcmp(name, "comment") == 0) {
        attribute_id = H5Aopen_name(loc_id, name);
        atype = H5Tcopy(H5T_C_S1);
        H5Tset_size(atype, GPIV_MAX_CHARS);
        status = H5Aread(attribute_id, atype, 
                         &image_par->comment); 
        status = H5Tclose(atype);
        status = H5Aclose(attribute_id); 
        image_par->comment_logic = TRUE;
    }
    

    return 0;
}



/*
 * Public functions
 */

void
gpiv_img_parameters_logic(GpivImagePar * image_par,
                          gboolean flag
                          )
/*-----------------------------------------------------------------------------
 * Set flag for image_par _logic */
{
    image_par->ncolumns_logic = flag;
    image_par->nrows_logic = flag;
    image_par->depth_logic = flag;
    image_par->x_corr_logic = flag;
    image_par->s_scale_logic = flag;
    image_par->t_scale_logic = flag;
    image_par->z_off_x_logic = flag;
    image_par->z_off_y_logic = flag;
    
    image_par->title_logic = flag;
    image_par->creation_date_logic = flag;
    image_par->location_logic = flag;
    image_par->author_logic = flag;
    image_par->software_logic = flag;
    image_par->source_logic = flag;
    image_par->usertext_logic = flag;
    image_par->warning_logic = flag;
    image_par->disclaimer_logic = flag;
    image_par->comment_logic = flag;

}



void
gpiv_img_read_header(FILE *fp_h, 
		     GpivImagePar * image_par, 
		     int print_par
		     )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Reads each line of file fp_h and looks for image header parameters
 *     without IMAGE_PAR_KEY
 *
 * PROTOTYPE LOCATATION:
 *     img.h
 *
 * INPUTS:
 *     fp_h:           pointer to open file
 *     print_par:      flag to print to stdout
 *
 * OUTPUTS:
 *     image_par:      parameter structure
 *-------------------------------------------------------------------------- */
{
  char line[GPIV_MAX_CHARS], par_name[GPIV_MAX_CHARS];

    while (fgets(line, GPIV_MAX_CHARS, fp_h) != NULL) {
	if (line[0] != '#' && line[0] != '\n' && line[0] != ' '
	    && line[0] != '\t') {
	    sscanf(line, "%s", par_name);

/*
 * Compares string with keys
 */

/*
 * General image parameters
 */
	    if (image_par->ncolumns_logic == FALSE) {
		image_par->ncolumns_logic =
		    gpiv_scan_iph(GPIV_IMAGE_PAR_KEY, line, par_name, "Ncolumns:",
				  &image_par->ncolumns, print_par, 0);
	    }

	    if (image_par->nrows_logic == FALSE) {
		image_par->nrows_logic =
		    gpiv_scan_iph(GPIV_IMAGE_PAR_KEY, line, par_name, "Nrows:",
				  &image_par->nrows, print_par, 0);
	    }

	    if (image_par->x_corr_logic == FALSE) {
		image_par->x_corr_logic =
		    gpiv_scan_iph(GPIV_IMAGE_PAR_KEY, line, par_name, "X_corr:",
				  &image_par->x_corr, print_par, 0);
	    }

	    if (image_par->depth_logic == FALSE) {
		image_par->depth_logic =
		    gpiv_scan_iph(GPIV_IMAGE_PAR_KEY, line, par_name, "Depth:",
				  &image_par->depth, print_par, 0);
	    }

/*
 * Parameters used by gpiv_post_scale
 */
            if (image_par->s_scale_logic == FALSE) {
                image_par->s_scale_logic = 
                    gpiv_scan_fph(GPIV_IMAGE_PAR_KEY, line, par_name, "Sscale:",
                                  &image_par->s_scale, print_par, 0);
            }
            
            if (image_par->t_scale_logic == FALSE) {
                image_par->t_scale_logic = 
                    gpiv_scan_fph(GPIV_IMAGE_PAR_KEY, line, par_name, "Tscale:",
                                  &image_par->t_scale, print_par, 0);
            }
            
            if (image_par->z_off_x_logic == FALSE) {
                image_par->z_off_x_logic = 
                    gpiv_scan_fph(GPIV_IMAGE_PAR_KEY, line, par_name, 
                                  "Zoff_x:",
                                  &image_par->z_off_x, print_par, 0);
            }

            if (image_par->z_off_y_logic == FALSE) {
                image_par->z_off_y_logic = 
                    gpiv_scan_fph(GPIV_IMAGE_PAR_KEY, line, par_name, 
                                  "Zoff_y:",
                                  &image_par->z_off_y, print_par, 0);
            }

/*
 * Optional parameters
 */
            if (image_par->title_logic == FALSE) {
		image_par->title_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Title:",
				   image_par->title, print_par, 0);
	    }

	    if (image_par->creation_date_logic == FALSE) {
		image_par->creation_date_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "CreationDate:",
				   image_par->creation_date, print_par, 0);
	    }

	    if (image_par->location_logic == FALSE) {
		image_par->location_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Location:",
				   image_par->location, print_par, 0);
	    }

	    if (image_par->author_logic == FALSE) {
		image_par->author_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Author:",
				   image_par->author, print_par, 0);
	    }

	    if (image_par->software_logic == FALSE) {
		image_par->software_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Software:",
				   image_par->software, print_par, 0);
	    }

	    if (image_par->source_logic == FALSE) {
		image_par->source_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Source:",
				   image_par->source, print_par, 0);
	    }

	    if (image_par->usertext_logic == FALSE) {
		image_par->usertext_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Usertext:",
				   image_par->usertext, print_par, 0);
	    }

	    if (image_par->warning_logic == FALSE) {
		image_par->warning_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Warning:",
				   image_par->warning, print_par, 0);
	    }

	    if (image_par->disclaimer_logic == FALSE) {
		image_par->disclaimer_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Disclaimer:",
				   image_par->disclaimer, print_par, 0);
	    }

	    if (image_par->comment_logic == FALSE) {
		image_par->comment_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Comment:",
				   image_par->comment, print_par, 0);
	    }

	}
    }


}



void
gpiv_img_fread_header(FILE *fp_h, 
		      FILE *fp_par_out,
		      GpivImagePar * image_par, 
		      int print_par
		      )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Reads each line of the file and looks for image header parameters 
 *     without GPIV_IMAGE_PAR_KEY
 *
 * PROTOTYPE LOCATATION:
 *     img.h
 *
 * INPUTS:
 *     fp_h:           pointer to open file
 *     fp_par_out:     file pointer to print parameter to
 *     print_par:      flag to print to stdout
 *
 * OUTPUTS:
 *     image_par :     parameter structure
 *---------------------------------------------------------------------------*/
{
    char line[GPIV_MAX_CHARS], par_name[GPIV_MAX_CHARS];

    while (fgets(line, GPIV_MAX_CHARS, fp_h) != NULL) {
	if (line[0] != '#' && line[0] != '\n' && line[0] != ' '
	    && line[0] != '\t') {
	    sscanf(line, "%s", par_name);

/*
 * Compares string with Ncolumns and Nrows
 */

/*
 * General image parameters
 */
	    if (image_par->ncolumns_logic == FALSE) {
		image_par->ncolumns_logic =
		    gpiv_fscan_iph(GPIV_IMAGE_PAR_KEY, line, par_name, "Ncolumns:",
				   &image_par->ncolumns, print_par, fp_par_out, 0);
	    }
            
	    if (image_par->nrows_logic == FALSE) {
		image_par->nrows_logic =
		    gpiv_fscan_iph(GPIV_IMAGE_PAR_KEY, line, par_name, "Nrows:",
				   &image_par->nrows, print_par, fp_par_out, 0);
	    }
            
	    if (image_par->x_corr_logic == FALSE) {
		image_par->x_corr_logic =
		    gpiv_fscan_iph(GPIV_IMAGE_PAR_KEY, line, par_name, "X_corr:",
				   &image_par->x_corr, print_par, fp_par_out, 0);
                
                
	    }
            
	    if (image_par->depth_logic == FALSE) {
		image_par->depth_logic =
		    gpiv_fscan_iph(GPIV_IMAGE_PAR_KEY, line, par_name, "Depth:",
				   &image_par->depth, print_par, fp_par_out, 0);
	    }
            
/*
 * Parameters used by gpiv_post_scale
 */
            if (image_par->s_scale_logic == FALSE) {
                image_par->s_scale_logic = 
                    gpiv_fscan_fph(GPIV_IMAGE_PAR_KEY, line, par_name, "Sscale:",
                                   &image_par->s_scale, print_par, fp_par_out, 0);
            }
            
            if (image_par->t_scale_logic == FALSE) {
                image_par->t_scale_logic = 
                    gpiv_fscan_fph(GPIV_IMAGE_PAR_KEY, line, par_name, "Tscale:",
                                   &image_par->t_scale, print_par, fp_par_out, 0);
            }
            
            if (image_par->z_off_x_logic == FALSE) {
                image_par->z_off_x_logic = 
                    gpiv_fscan_fph(GPIV_IMAGE_PAR_KEY, line, par_name, 
                                   "Zoff_x:",
                                   &image_par->z_off_x, print_par, fp_par_out, 0);
            }
            
            if (image_par->z_off_y_logic == FALSE) {
                image_par->z_off_y_logic = 
                    gpiv_fscan_fph(GPIV_IMAGE_PAR_KEY, line, par_name, 
                                   "Zoff_y:",
                                   &image_par->z_off_y, print_par, fp_par_out, 0);
            }

/*
 * Optional parameters
 */
	    if (image_par->title_logic == FALSE) {
		image_par->title_logic =
		    gpiv_fscan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Title:",
				   image_par->title, print_par, fp_par_out,
				   0);
	    }

	    if (image_par->creation_date_logic == FALSE) {
		image_par->creation_date_logic =
		    gpiv_fscan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "CreationDate:",
				   image_par->creation_date, print_par, 
				   fp_par_out, 0);
	    }

	    if (image_par->location_logic == FALSE) {
		image_par->location_logic =
		    gpiv_fscan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Location:",
				   image_par->location, print_par, 
				   fp_par_out, 0);
	    }

	    if (image_par->author_logic == FALSE) {
		image_par->author_logic =
		    gpiv_fscan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Author:",
				   image_par->author, print_par, 
				   fp_par_out, 0);
	    }

	    if (image_par->software_logic == FALSE) {
		image_par->software_logic =
		    gpiv_fscan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Software:",
				   image_par->software, print_par, 
				   fp_par_out, 0);
	    }

	    if (image_par->source_logic == FALSE) {
		image_par->source_logic =
		    gpiv_fscan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Source:",
				   image_par->source, print_par, 
				   fp_par_out, 0);
	    }

	    if (image_par->usertext_logic == FALSE) {
		image_par->usertext_logic =
		    gpiv_fscan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Usertext:",
				   image_par->usertext, print_par, 
				   fp_par_out, 0);
	    }

	    if (image_par->warning_logic == FALSE) {
		image_par->warning_logic =
		    gpiv_fscan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Warning:",
				   image_par->warning, print_par, 
				   fp_par_out, 0);
	    }

	    if (image_par->disclaimer_logic == FALSE) {
		image_par->disclaimer_logic =
		    gpiv_fscan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Disclaimer:",
				   image_par->disclaimer, print_par, 
				   fp_par_out, 0);
	    }

	    if (image_par->comment_logic == FALSE) {
		image_par->comment_logic =
		    gpiv_fscan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Comment:",
				   image_par->comment, print_par, 
				   fp_par_out, 0);
	    }

	}
    }

}



char *
gpiv_img_read_pgm_header(char *fname, 
                         GpivImagePar * image_par,
                         gint *line_nr
                        )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Reads each line of the raw portable graymap (pgm) format image file
 *     and retrieves image header parameters
 *     without GPIV_IMAGE_PAR_KEY
 *
 * PROTOTYPE LOCATATION:
 *     img.h
 *
 * INPUTS:
 *     fname:           file name
 *
 * OUTPUTS:
 *     image_par :     parameter structure
 *     line_nr:        line number to start image data
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    FILE *fp;
    char *err_msg = NULL, img_type[GPIV_MAX_CHARS];
    char line[GPIV_MAX_CHARS], par_name[GPIV_MAX_CHARS];
    gint max_gray = 0, nr = 0;
    gboolean print_par = TRUE, stop_scan = FALSE;

    *line_nr = 0;
    g_message("gpiv_img_read_pgmheader:: 0 x_corr_logic = FALSE");
    image_par->x_corr_logic = FALSE;
    image_par->comment_logic = FALSE;

    if ((fp = fopen(fname, "rb")) == NULL) {
	err_msg = "GPIV_FREAD_IMAGE: Failure opening for input";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }


    fscanf(fp, "%s", img_type);
    *line_nr = *line_nr + 1;
    g_message("gpiv_img_read_pgmheader:: line_nr=%d", *line_nr);
    if (strcmp(img_type, "P5") != 0) {
        err_msg = "gpiv_fread_pgmimage: Not a raw pgm image";
        return err_msg;
    }

    while (fgets(line, GPIV_MAX_CHARS, fp) != NULL
           && !stop_scan) {
    *line_nr = *line_nr + 1;
        g_message("gpiv_img_read_pgmheader:: line_nr=%d", *line_nr);
/*         g_message("gpiv_img_read_pgmheader:: line = %s", line); */
	if (line[0] == '#' 
            ) {
            sscanf(line, "# %s", par_name);
/*
 * General image parameters
 */

            if (image_par->x_corr_logic == FALSE) {
		image_par->x_corr_logic =
		    gpiv_scan_iph(GPIV_IMAGE_PAR_KEY, line, par_name, "X_corr:",
                                  &image_par->x_corr, print_par, FALSE); 
/*
 * Parameters used by gpiv_post_scale
 */
            } 

            if (image_par->s_scale_logic == FALSE) {
                image_par->s_scale_logic = 
                    gpiv_scan_fph(GPIV_IMAGE_PAR_KEY, line, par_name, "Sscale:",
                                   &image_par->s_scale, print_par, FALSE);
            }

            if (image_par->t_scale_logic == FALSE) {
                image_par->t_scale_logic = 
                    gpiv_scan_fph(GPIV_IMAGE_PAR_KEY, line, par_name, "Tscale:",
                                   &image_par->t_scale, print_par, FALSE);
            }

            if (image_par->z_off_x_logic == FALSE) {
                image_par->z_off_x_logic = 
                    gpiv_scan_fph(GPIV_IMAGE_PAR_KEY, line, par_name, 
                                   "Zoff_x:",
                                   &image_par->z_off_x, print_par, FALSE);
            }

            if (image_par->z_off_y_logic == FALSE) {
                image_par->z_off_y_logic = 
                    gpiv_scan_fph(GPIV_IMAGE_PAR_KEY, line, par_name, 
                                   "Zoff_y:",
                                   &image_par->z_off_y, print_par, FALSE);

/*
 * Optional parameters
 */
            }

            if (image_par->title_logic == FALSE) {
		image_par->title_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Title:",
				   image_par->title, print_par, FALSE);
	    }

            if (image_par->creation_date_logic == FALSE) {
		image_par->creation_date_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "CreationDate:",
				   image_par->creation_date, print_par, FALSE);
	    }

            if (image_par->location_logic == FALSE) {
		image_par->location_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Location:",
				   image_par->location, print_par, FALSE);
	    }

            if (image_par->author_logic == FALSE) {
		image_par->author_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Author:",
                                  image_par->author, print_par, FALSE);
	    }
            
            if (image_par->software_logic == FALSE) {
		image_par->software_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Software:",
                                  image_par->software, print_par, FALSE);
	    }
            
            if (image_par->source_logic == FALSE) {
		image_par->source_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Source:",
                                  image_par->source, print_par, FALSE);
	    }
            
            if (image_par->usertext_logic == FALSE) {
		image_par->usertext_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Usertext:",
                                  image_par->usertext, print_par, FALSE);
	    }
            
            if (image_par->warning_logic == FALSE) {
		image_par->warning_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Warning:",
                                  image_par->warning, print_par, FALSE);
	    }
            
            if (image_par->disclaimer_logic == FALSE) {
		image_par->disclaimer_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Disclaimer:",
                                  image_par->disclaimer, print_par, FALSE);
	    }
            
            if (image_par->comment_logic == FALSE) {
		image_par->comment_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Comment:",
                                  image_par->comment, print_par, FALSE);
	    }
            
/*
 * Skip blank lines
 */
/*         } else if (line[0] == '\n'  */
/*                    || line[0] == ' ' */
/*                    || line[0] == '\t' */
/*                    ) { */
/*
 * Reads Ncolumns and Nrows
 */

        } else if (line[0] != '#' 
                   && line[0] != '\n' 
                   && line[0] != ' '
                   && line[0] != '\t'
                   ) {
            stop_scan = TRUE;
/*             if (image_par->ncolumns_logic == FALSE */
/*                 && image_par->nrows_logic == FALSE */
/*                 ) { */
            sscanf(line, "%d %d", &image_par->ncolumns, &nr);
            image_par->nrows = nr / 2;
            image_par->ncolumns_logic = TRUE;
            image_par->nrows_logic = TRUE;
            g_message("gpiv_img_read_pgmheader:: ncolumns=%d nrows=%d",
                      image_par->ncolumns, image_par->nrows);
/*         } */

/*             if (image_par->depth_logic == FALSE) { */
            image_par->depth_logic = TRUE;
            sscanf(line, "%d", &max_gray);
            g_message("gpiv_img_read_pgmheader:: max_gray=%d", max_gray);
        }
    }



    if (image_par->ncolumns_logic == FALSE
        || image_par->nrows_logic == FALSE) {
        gpiv_error("gpiv_img_fread_pgmheader: ncolumns or nrows not read");
    }
    if (image_par->depth_logic) {
        if (max_gray <= 255) {
            image_par->depth = 8;
        } else if (max_gray <= 65535) {
            image_par->depth = 16;
        } else {
            gpiv_error("gpiv_img_fread_pgmheader: max_gray > 65535; unvalid for pgm image");
        }
    } else {
        gpiv_error("gpiv_img_fread_pgmheader: depth not read");
    }
    g_warning("gpiv_img_fread_pgmheader:: END");

    fclose(fp);
    return err_msg;
}



void
gpiv_img_default_parameters(GpivImagePar * image_par_default,
                            gboolean force
                            )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Sets default parameter values
 *
 * INPUTS:
 *
 * OUTPUTS:
 *     piv_img_par_default:   structure of image evaluation parameters
 *
 * RETURNS:
 *---------------------------------------------------------------------------*/
{
    if (!image_par_default->depth_logic || force)
        image_par_default->depth = GPIV_IMGPAR_DEFAULT__DEPTH;
    if (!image_par_default->ncolumns_logic || force)
        image_par_default->ncolumns = GPIV_IMGPAR_DEFAULT__NCOLUMNS;
    if (!image_par_default->nrows_logic || force)
        image_par_default->nrows = GPIV_IMGPAR_DEFAULT__NROWS;
    if (!image_par_default->x_corr_logic || force)
        image_par_default->x_corr = GPIV_IMGPAR_DEFAULT__X_CORR;
    if (!image_par_default->s_scale_logic || force)
        image_par_default->s_scale = GPIV_IMGPAR_DEFAULT__S_SCALE;
    if (!image_par_default-> t_scale_logic || force)
        image_par_default-> t_scale = GPIV_IMGPAR_DEFAULT__T_SCALE;
    if (!image_par_default->z_off_x_logic || force)
        image_par_default->z_off_x = GPIV_IMGPAR_DEFAULT__Z_OFF_X;
    if (!image_par_default->z_off_y_logic || force)
        image_par_default->z_off_y = GPIV_IMGPAR_DEFAULT__Z_OFF_Y;

    if (!image_par_default->title_logic || force)
        snprintf(image_par_default->title, GPIV_MAX_CHARS, "%s", 
                 GPIV_IMGPAR_DEFAULT__TITLE);
    if (!image_par_default->creation_date_logic || force)
        snprintf(image_par_default->creation_date, GPIV_MAX_CHARS, "%s", 
                 GPIV_IMGPAR_DEFAULT__CREATION_DATE);
    if (!image_par_default->location_logic || force)
        snprintf(image_par_default->location, GPIV_MAX_CHARS, "%s", 
                 GPIV_IMGPAR_DEFAULT__LOCATION);
    if (!image_par_default->author_logic || force)
        snprintf(image_par_default->author, GPIV_MAX_CHARS, "%s", 
                 GPIV_IMGPAR_DEFAULT__AUTHOR);
    if (!image_par_default->software_logic || force)
        snprintf(image_par_default->software, GPIV_MAX_CHARS, "%s", 
                 GPIV_IMGPAR_DEFAULT__SOFTWARE);
    if (!image_par_default->source_logic || force)
        snprintf(image_par_default->source, GPIV_MAX_CHARS, "%s", 
                 GPIV_IMGPAR_DEFAULT__SOURCE);
    if (!image_par_default->usertext_logic || force)
        snprintf(image_par_default->usertext, GPIV_MAX_CHARS, "%s", 
                 GPIV_IMGPAR_DEFAULT__USERTEXT);
    if (!image_par_default->warning_logic || force)
        snprintf(image_par_default->warning, GPIV_MAX_CHARS, "%s", 
                 GPIV_IMGPAR_DEFAULT__WARNING);
    if (!image_par_default->disclaimer_logic || force)
        snprintf(image_par_default->disclaimer, GPIV_MAX_CHARS, "%s", 
                 GPIV_IMGPAR_DEFAULT__DISCLAIMER);
    if (!image_par_default->comment_logic || force)
        snprintf(image_par_default->comment, GPIV_MAX_CHARS, "%s", 
                 GPIV_IMGPAR_DEFAULT__COMMENT);

    gpiv_img_parameters_logic(image_par_default, TRUE);
}


void
gpiv_img_read_parameters(FILE *fp_h, 
			 GpivImagePar * image_par, 
			 int print_par
			 )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Reads each line of file fp_h and looks for image header parameters 
 *     including GPIV_IMAGE_PAR_KEY
 *
 * PROTOTYPE LOCATATION:
 *     img.h
 *
 * INPUTS:
 *     print_par:      flag to print to stdout
 *
 * OUTPUTS:
 *     image_par :     parameter structure
 *---------------------------------------------------------------------------*/
{
  char line[GPIV_MAX_CHARS], par_name[GPIV_MAX_CHARS];


    while (fgets(line, GPIV_MAX_CHARS, fp_h) != NULL) {
	if (line[0] != '#' && line[0] != '\n' && line[0] != ' '
	    && line[0] != '\t') {
	    sscanf(line, "%s", par_name);

/*
 * Compares string with keys
*/

/*
 * General image parameters
 */
	    if (image_par->ncolumns_logic == FALSE) {
		image_par->ncolumns_logic =
		    gpiv_scan_iph(GPIV_IMAGE_PAR_KEY, line, par_name, "Ncolumns:",
				  &image_par->ncolumns, print_par, 1);
	    }

	    if (image_par->nrows_logic == FALSE) {
		image_par->nrows_logic =
		    gpiv_scan_iph(GPIV_IMAGE_PAR_KEY, line, par_name, "Nrows:",
				  &image_par->nrows, print_par, 1);
	    }

	    if (image_par->x_corr_logic == FALSE) {
		image_par->x_corr_logic =
		    gpiv_scan_iph(GPIV_IMAGE_PAR_KEY, line, par_name, "X_corr:",
				  &image_par->x_corr, print_par, 1);
	    }

	    if (image_par->depth_logic == FALSE) {
		image_par->depth_logic =
		    gpiv_scan_iph(GPIV_IMAGE_PAR_KEY, line, par_name, "Depth:",
				  &image_par->depth, print_par, 1);
	    }

/*
 * Parameters used by gpiv_post_scale
 */
            if (image_par->s_scale_logic == FALSE) {
                image_par->s_scale_logic = 
                    gpiv_scan_fph(GPIV_IMAGE_PAR_KEY, line, par_name, "Sscale:",
                                  &image_par->s_scale, print_par, 1);
            }
            
            if (image_par->t_scale_logic == FALSE) {
                image_par->t_scale_logic = 
                    gpiv_scan_fph(GPIV_IMAGE_PAR_KEY, line, par_name, "Tscale:",
                                  &image_par->t_scale, print_par, 1);
            }
            
            if (image_par->z_off_x_logic == FALSE) {
                image_par->z_off_x_logic = 
                    gpiv_scan_fph(GPIV_IMAGE_PAR_KEY, line, par_name, 
                                  "Zoff_x:",
                                  &image_par->z_off_x, print_par, 1);
            }

            if (image_par->z_off_y_logic == FALSE) {
                image_par->z_off_y_logic = 
                    gpiv_scan_fph(GPIV_IMAGE_PAR_KEY, line, par_name, 
                                  "Zoff_y:",
                                  &image_par->z_off_y, print_par, 1);
            }

/*
 * Optional parameters
 */
	    if (image_par->title_logic == FALSE) {
		image_par->title_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Title:",
				   image_par->title, print_par, 1);
	    }

	    if (image_par->creation_date_logic == FALSE) {
		image_par->creation_date_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, 
				  "CreationDate:",
				  image_par->creation_date, print_par, 1);
	    }

	    if (image_par->location_logic == FALSE) {
		image_par->location_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Location:",
				   image_par->location, print_par, 1);
	    }

	    if (image_par->author_logic == FALSE) {
		image_par->author_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Author:",
				   image_par->author, print_par, 1);
	    }

	    if (image_par->software_logic == FALSE) {
		image_par->software_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Software:",
				   image_par->software, print_par, 1);
	    }

	    if (image_par->source_logic == FALSE) {
		image_par->source_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Source:",
				   image_par->source, print_par, 1);
	    }

	    if (image_par->usertext_logic == FALSE) {
		image_par->usertext_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Usertext:",
				   image_par->usertext, print_par, 1);
	    }

	    if (image_par->warning_logic == FALSE) {
		image_par->warning_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Warning:",
				   image_par->warning, print_par, 1);
	    }

	    if (image_par->disclaimer_logic == FALSE) {
		image_par->disclaimer_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Disclaimer:",
				   image_par->disclaimer, print_par, 1);
	    }

	    if (image_par->comment_logic == FALSE) {
		image_par->comment_logic =
		    gpiv_scan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Comment:",
				   image_par->comment, print_par, 1);
	    }

	}
    }


}



void
gpiv_img_fread_parameters(FILE *fp_h, 
			  FILE *fp_par_out,
			  GpivImagePar * image_par, 
			  int print_par
			  )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Reads each line of the file and looks for image header parameters 
 *     including GPIV_IMAGE_PAR_KEY
 *
 * PROTOTYPE LOCATATION:
 *     img.h
 *
 * INPUTS:
 *     fp_h:           file pointer to input file
 *     fp_par_out:     file pointer to print parameter to
 *     print_par:      flag to print to stdout
 *
 * OUTPUTS:
 *     image_par :     parameter structure
 *---------------------------------------------------------------------------*/
{
    char line[GPIV_MAX_CHARS], par_name[GPIV_MAX_CHARS];

    while (fgets(line, GPIV_MAX_CHARS, fp_h) != NULL) {
	if (line[0] != '#' && line[0] != '\n' && line[0] != ' '
	    && line[0] != '\t') {
	    sscanf(line, "%s", par_name);

/*
 * Compares string with Ncolumns and Nrows
 */

/*
 * General image parameters
 */
	    if (image_par->ncolumns_logic == FALSE) {
		image_par->ncolumns_logic =
		    gpiv_fscan_iph(GPIV_IMAGE_PAR_KEY, line, par_name, "Ncolumns:",
				   &image_par->ncolumns, print_par, fp_par_out, 1);
	    }

	    if (image_par->nrows_logic == FALSE) {
		image_par->nrows_logic =
		    gpiv_fscan_iph(GPIV_IMAGE_PAR_KEY, line, par_name, "Nrows:",
				   &image_par->nrows, print_par, fp_par_out, 1);
	    }

	    if (image_par->x_corr_logic == FALSE) {
		image_par->x_corr_logic =
		    gpiv_fscan_iph(GPIV_IMAGE_PAR_KEY, line, par_name, "X_corr:",
				   &image_par->x_corr, print_par, 
                                   fp_par_out, 1);


	    }

	    if (image_par->depth_logic == FALSE) {
		image_par->depth_logic =
		    gpiv_fscan_iph(GPIV_IMAGE_PAR_KEY, line, par_name, "Depth:",
				   &image_par->depth, print_par, 
                                   fp_par_out, 1);
	    }

/*
 * Parameters used by gpiv_post_scale
 */
            if (image_par->s_scale_logic == FALSE) {
                image_par->s_scale_logic = 
                    gpiv_fscan_fph(GPIV_IMAGE_PAR_KEY, line, par_name, "Sscale:",
                                  &image_par->s_scale, print_par, 
                                   fp_par_out, 1);
            }
            
            if (image_par->t_scale_logic == FALSE) {
                image_par->t_scale_logic = 
                    gpiv_fscan_fph(GPIV_IMAGE_PAR_KEY, line, par_name, "Tscale:",
                                  &image_par->t_scale, print_par, 
                                   fp_par_out, 1);
            }
            
            if (image_par->z_off_x_logic == FALSE) {
                image_par->z_off_x_logic = 
                    gpiv_fscan_fph(GPIV_IMAGE_PAR_KEY, line, par_name, 
                                  "Zoff_x:",
                                  &image_par->z_off_x, print_par, 
                                   fp_par_out, 1);
            }
 
            if (image_par->z_off_y_logic == FALSE) {
                 image_par->z_off_y_logic = 
                    gpiv_fscan_fph(GPIV_IMAGE_PAR_KEY, line, par_name, 
                                  "Zoff_y:",
                                  &image_par->z_off_y, print_par, 
                                   fp_par_out, 1);
            }

/*
 * Optional parameters
 */
	    if (image_par->title_logic == FALSE) {
		image_par->title_logic =
		    gpiv_fscan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Title:",
				   image_par->title, print_par, fp_par_out,
				   1);
	    }

	    if (image_par->creation_date_logic == FALSE) {
		image_par->creation_date_logic =
		    gpiv_fscan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "CreationDate:",
				   image_par->creation_date, print_par, 
				   fp_par_out, 1);
	    }

	    if (image_par->location_logic == FALSE) {
		image_par->location_logic =
		    gpiv_fscan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Location:",
				   image_par->location, print_par, 
				   fp_par_out, 1);
	    }

	    if (image_par->author_logic == FALSE) {
		image_par->author_logic =
		    gpiv_fscan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Author:",
				   image_par->author, print_par, 
				   fp_par_out, 1);
	    }

	    if (image_par->software_logic == FALSE) {
		image_par->software_logic =
		    gpiv_fscan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Software:",
				   image_par->software, print_par, 
				   fp_par_out, 1);
	    }

	    if (image_par->source_logic == FALSE) {
		image_par->source_logic =
		    gpiv_fscan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Source:",
				   image_par->source, print_par, 
				   fp_par_out, 1);
	    }

	    if (image_par->usertext_logic == FALSE) {
		image_par->usertext_logic =
		    gpiv_fscan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Usertext:",
				   image_par->usertext, print_par, 
				   fp_par_out, 1);
	    }

	    if (image_par->warning_logic == FALSE) {
		image_par->warning_logic =
		    gpiv_fscan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Warning:",
				   image_par->warning, print_par, 
				   fp_par_out, 1);
	    }

	    if (image_par->disclaimer_logic == FALSE) {
		image_par->disclaimer_logic =
		    gpiv_fscan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Disclaimer:",
				   image_par->disclaimer, print_par, 
				   fp_par_out, 1);
	    }

	    if (image_par->comment_logic == FALSE) {
		image_par->comment_logic =
		    gpiv_fscan_sph(GPIV_IMAGE_PAR_KEY, line, par_name, "Comment:",
				   image_par->comment, print_par, 
				   fp_par_out, 1);
	    }

	}
    }

}



char *
gpiv_img_check_header_read(GpivImagePar image_par
			   )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *   Check out if all parameters have been read
 *
 * INPUTS:
 *     image_par:      image parameters
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    if (image_par.ncolumns_logic == FALSE) {
	err_msg = "GPIV_IMG_CHECK_HEADER_READ: Ncolumns not set";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    if (image_par.nrows_logic == FALSE) {
	err_msg = "GPIV_IMG_CHECK_HEADER_READ: Nrows not set";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    if (image_par.x_corr_logic == FALSE) {
	err_msg = "GPIV_IMG_CHECK_HEADER_READ: X_corr not set";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }


/*
 * Parameters used by gpiv_post_scale
 */
   if (image_par.s_scale_logic == FALSE) {
       err_msg = "GPIV_IMG_CHECK_HEADER_READ: Sscale not set\n";
       gpiv_warning("%s", err_msg);
       return err_msg;
   }

   if (image_par.t_scale_logic == FALSE) {
	err_msg = "GPIV_IMG_CHECK_HEADER_READ: Tscale not set\n";
        gpiv_warning("%s", err_msg);
        return err_msg;
   }

   if (image_par.z_off_x_logic == FALSE) {
	err_msg = "GPIV_IMG_CHECK_HEADER_READ: Zoff_x not set\n";
        gpiv_warning("%s", err_msg);
        return err_msg;
   }

   if (image_par.z_off_y_logic == FALSE) {
	err_msg = "GPIV_IMG_CHECK_HEADER_READ: Zoff_y not set\n";
        gpiv_warning("%s", err_msg);
        return err_msg;
   }


   return err_msg;
}



char *
gpiv_img_test_header(GpivImagePar image_par
		     )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Test header information on validity
 *
 * INPUTS:
 *     image_par:      image parameters
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
 
     if (image_par.nrows < 0 || image_par.nrows > GPIV_MAX_IMG_SIZE) {
         err_msg = "GPIV_IMG_TEST_HEADER: numbers of rows larger than allowed maximum";
         gpiv_warning("%s", err_msg);
         return err_msg;
     }

     if (image_par.ncolumns < 0 || image_par.ncolumns > GPIV_MAX_IMG_SIZE) {
         err_msg = "GPIV_IMG_TEST_HEADER: numbers of columns larger than allowed maximum";
         gpiv_warning("%s", err_msg);
         return err_msg;
     }

     if (image_par.depth < 1 || image_par.depth > GPIV_MAX_IMG_DEPTH) {
         err_msg = "GPIV_IMG_TEST_HEADER: depth larger than allowed maximum";
         gpiv_warning("%s", err_msg);
         return err_msg;
     }

     return err_msg;
}


void 
gpiv_img_print_header(GpivImagePar image_par
		      )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      prints image header parameters to stdout
 *      without GPIV_IMAGE_PAR_KEY
 *
 * PROTOTYPE LOCATATION:
 *     gpiv_img.h
 *
 * INPUTS:
 *     image_par:      image parameters
 *
 * OUTPUTS:
 *---------------------------------------------------------------------------*/
{
/*
 * General image parameters
 */
    if (image_par.ncolumns_logic)
	printf("%s%s %d\n", "", "Ncolumns:", image_par.ncolumns);

    if (image_par.nrows_logic)
	printf("%s%s %d\n", "", "Nrows:", image_par.nrows);

    if (image_par.x_corr_logic)
	printf("%s%s %d\n", "", "X_corr:", image_par.x_corr);

    if (image_par.depth_logic)
	printf("%s%s %d\n", "", "Depth:", image_par.depth);

/*
 * Parameters for scale
 */
    if (image_par.s_scale_logic) 
        printf("%s%s %f\n", "", "Sscale:", image_par.s_scale);
    
    if (image_par.t_scale_logic) 
        printf("%s%s %f\n", "", "Tscale:", 
               image_par.t_scale);
    
    if (image_par.z_off_x_logic) 
        printf("%s%s %f\n", "", "Zoff_x:", 
               image_par.z_off_x);
    
    if (image_par.z_off_y_logic) 
        printf("%s%s %f\n", "", "Zoff_y:", 
               image_par.z_off_y);
  
/*
 * Optional parameters
 */
    if (image_par.title_logic)
	printf("%s%s %s\n", "", "Title:", image_par.title);

    if (image_par.creation_date_logic)
	printf("%s%s %s\n", "", "Creation_date:", 
	       image_par.creation_date);

    if (image_par.location_logic)
	printf("%s%s %s\n", "", "Location:", image_par.location);

    if (image_par.author_logic)
	printf("%s%s %s\n", "", "Author:", image_par.author);

    if (image_par.software_logic)
	printf("%s%s %s\n", "", "Software:", image_par.software);

    if (image_par.source_logic)
	printf("%s%s %s\n", "", "Source:", image_par.source);

    if (image_par.usertext_logic)
	printf("%s%s %s\n", "", "Usertext:", image_par.usertext);

    if (image_par.warning_logic)
	printf("%s%s %s\n", "", "Warning:", image_par.warning);

    if (image_par.disclaimer_logic)
	printf("%s%s %s\n", "", "Disclaimer:", image_par.disclaimer);

    if (image_par.comment_logic)
	printf("%s%s %s\n", "", "Comment:", image_par.comment);
}



void
gpiv_img_fprint_pgmheader(FILE * fp_par_out, 
                          GpivImagePar image_par
                          )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     prints image header parameters to portable graymap (pgm) image format 
 *     file
 *     without GPIV_IMAGE_PAR_KEY
 *
 * PROTOTYPE LOCATATION:
 *     img.h
 *
 * INPUTS:
 *     fp_par_out:     file pointer to output file
 *     image_par:      image parameter structure
 *
 * OUTPUTS:
 *---------------------------------------------------------------------------*/
{
    gint max_gray;
/*
 * General image parameters
 */

    if (image_par.depth <= 8) {
        max_gray = 255;
    } else {
        max_gray = 65535;
    }

    if (image_par.x_corr_logic)
        fprintf(fp_par_out, "\n# %s%s %d\n", "", "X_corr:", 
		image_par.x_corr);

    if (image_par.depth_logic)
	fprintf(fp_par_out, "# %s%s %d\n", "", "Depth:", 
		image_par.depth);

/*
 * Parameters used by gpiv_post_scale
 */
    if (image_par.s_scale_logic) 
        fprintf(fp_par_out, "# %s%s %f\n", "", "Sscale:", 
                image_par.s_scale);
    
    if (image_par.t_scale_logic) 
        fprintf(fp_par_out, "# %s%s %f\n", "", "Tscale:", 
                image_par.t_scale);
    
    if (image_par.z_off_x_logic) 
        fprintf(fp_par_out, "# %s%s %f\n", "", "Zoff_x:", 
                image_par.z_off_x);
    
    if (image_par.z_off_y_logic) 
        fprintf(fp_par_out, "# %s%s %f\n", "", "Zoff_y:", 
                image_par.z_off_y);
  

/*
 * Optional parameters
 */
    if (image_par.title_logic)
	fprintf(fp_par_out, "# %s%s %s\n", "", "Title:", 
		image_par.title);

    if (image_par.creation_date_logic)
	fprintf(fp_par_out, "# %s%s %s\n", "", "Creation_date:", 
	       image_par.creation_date);

    if (image_par.location_logic)
	fprintf(fp_par_out, "# %s%s %s\n", "", "Location:", 
		image_par.location);

    if (image_par.author_logic)
	fprintf(fp_par_out, "# %s%s %s\n", "", "Author:", 
		image_par.author);

    if (image_par.software_logic)
	fprintf(fp_par_out, "# %s%s %s\n", "", "Software:", 
		image_par.software);

    if (image_par.source_logic)
	fprintf(fp_par_out, "# %s%s %s\n", "", "Source:", 
		image_par.source);

    if (image_par.usertext_logic)
	fprintf(fp_par_out, "# %s%s %s\n", "", "Usertext:", 
		image_par.usertext);

    if (image_par.warning_logic)
	fprintf(fp_par_out, "# %s%s %s\n", "", "Warning:", 
		image_par.warning);

    if (image_par.disclaimer_logic)
	fprintf(fp_par_out, "# %s%s %s\n", "", "Disclaimer:", 
		image_par.disclaimer);

    if (image_par.comment_logic)
	fprintf(fp_par_out, "# %s%s %s\n", "", "Comment:", 
		image_par.comment);
/*
 * Required
 */
    if (image_par.ncolumns_logic && image_par.nrows_logic) {
	fprintf(fp_par_out, "# %s%s \n%d\n", "", "Ncolumns:", 
		image_par.ncolumns);
        if (image_par.x_corr) {
            fprintf(fp_par_out, "# %s%s \n%d\n",  "", "Nrows:", 
                    2 * image_par.nrows);
        } else {
            fprintf(fp_par_out, "# %s%s \n%d\n",  "", "Nrows:", 
                    image_par.nrows);
        }
        fprintf(fp_par_out, "# %s%s \n %d\n", "Max_value:", 
                max_gray);
    } else {
        gpiv_error("gpiv_img_fprint_pgmheader: ncolumns/nrows _logic = FALSE");
    }

}


void
gpiv_img_fprint_header(FILE * fp_par_out, 
		       GpivImagePar image_par
		       )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     prints image header parameters to file
 *     without GPIV_IMAGE_PAR_KEY
 *
 * PROTOTYPE LOCATATION:
 *     img.h
 *
 * INPUTS:
 *     fp_par_out:     file pointer to output file
 *     image_par:      image parameter structure
 *
 * OUTPUTS:
 *---------------------------------------------------------------------------*/
{

/*
 * General image parameters
 */

    if (image_par.ncolumns_logic)
	fprintf(fp_par_out, "%s%s %d\n", "", "Ncolumns:", 
		image_par.ncolumns);

    if (image_par.nrows_logic)
	fprintf(fp_par_out, "%s%s %d\n", "", "Nrows:", 
		image_par.nrows);

    if (image_par.x_corr_logic)
	fprintf(fp_par_out, "%s%s %d\n", "", "X_corr:", 
		image_par.x_corr);

    if (image_par.depth_logic)
	fprintf(fp_par_out, "%s%s %d\n", "", "Depth:", 
		image_par.depth);

/*
 * Parameters used by gpiv_post_scale
 */
    if (image_par.s_scale_logic) 
        fprintf(fp_par_out, "%s%s %f\n", "", "Sscale:", 
                image_par.s_scale);
    
    if (image_par.t_scale_logic) 
        fprintf(fp_par_out, "%s%s %f\n", "", "Tscale:", 
                image_par.t_scale);
    
    if (image_par.z_off_x_logic) 
        fprintf(fp_par_out, "%s%s %f\n", "", "Zoff_x:", 
                image_par.z_off_x);
    
    if (image_par.z_off_y_logic) 
        fprintf(fp_par_out, "%s%s %f\n", "", "Zoff_y:", 
                image_par.z_off_y);
  

/*
 * Optional parameters
 */
    if (image_par.title_logic)
	fprintf(fp_par_out, "%s%s %s\n", "", "Title:", 
		image_par.title);

    if (image_par.creation_date_logic)
	fprintf(fp_par_out, "%s%s %s\n", "", "Creation_date:", 
	       image_par.creation_date);

    if (image_par.location_logic)
	fprintf(fp_par_out, "%s%s %s\n", "", "Location:", 
		image_par.location);

    if (image_par.author_logic)
	fprintf(fp_par_out, "%s%s %s\n", "", "Author:", 
		image_par.author);

    if (image_par.software_logic)
	fprintf(fp_par_out, "%s%s %s\n", "", "Software:", 
		image_par.software);

    if (image_par.source_logic)
	fprintf(fp_par_out, "%s%s %s\n", "", "Source:", 
		image_par.source);

    if (image_par.usertext_logic)
	fprintf(fp_par_out, "%s%s %s\n", "", "Usertext:", 
		image_par.usertext);

    if (image_par.warning_logic)
	fprintf(fp_par_out, "%s%s %s\n", "", "Warning:", 
		image_par.warning);

    if (image_par.disclaimer_logic)
	fprintf(fp_par_out, "%s%s %s\n", "", "Disclaimer:", 
		image_par.disclaimer);

    if (image_par.comment_logic)
	fprintf(fp_par_out, "%s%s %s\n", "", "Comment:", 
		image_par.comment);

}


void 
gpiv_img_print_parameters(GpivImagePar image_par
			  )
/*----------------------------------------------------------------------------
 * DESCRIPTION:
 *     prints image header parameters to stdout
 *     including GPIV_IMAGE_PAR_KEY
 *
 * PROTOTYPE LOCATATION:
 *     img.h
 *
 * INPUTS:
 *     image_par:      image parameters
 *
 * OUTPUTS:
 *---------------------------------------------------------------------------*/
{
/*
 * General image parameters
 */
    if (image_par.ncolumns_logic)
	printf("%s%s %d\n", GPIV_IMAGE_PAR_KEY, "Ncolumns:", image_par.ncolumns);

    if (image_par.nrows_logic)
	printf("%s%s %d\n", GPIV_IMAGE_PAR_KEY, "Nrows:", image_par.nrows);

    if (image_par.x_corr_logic)
	printf("%s%s %d\n", GPIV_IMAGE_PAR_KEY, "X_corr:", image_par.x_corr);

    if (image_par.depth_logic)
	printf("%s%s %d\n", GPIV_IMAGE_PAR_KEY, "Depth:", image_par.depth);

/*
 * Parameters for scale
 */
    if (image_par.s_scale_logic) 
        printf("%s%s %f\n", GPIV_IMAGE_PAR_KEY, "Sscale:", image_par.s_scale);
    
    if (image_par.t_scale_logic) 
        printf("%s%s %f\n", GPIV_IMAGE_PAR_KEY, "Tscale:", 
               image_par.t_scale);
    
    if (image_par.z_off_x_logic) 
        printf("%s%s %f\n", GPIV_IMAGE_PAR_KEY, "Zoff_x:", 
               image_par.z_off_x);
    
    if (image_par.z_off_y_logic) 
        printf("%s%s %f\n", GPIV_IMAGE_PAR_KEY, "Zoff_y:", 
	   image_par.z_off_y);
  
/*
 * Optional parameters
 */
    if (image_par.title_logic)
	printf("%s%s %s\n", GPIV_IMAGE_PAR_KEY, "Title:", image_par.title);

    if (image_par.creation_date_logic)
	printf("%s%s %s\n", GPIV_IMAGE_PAR_KEY, "Creation_date:", 
	       image_par.creation_date);

    if (image_par.location_logic)
	printf("%s%s %s\n", GPIV_IMAGE_PAR_KEY, "Location:", image_par.location);

    if (image_par.author_logic)
	printf("%s%s %s\n", GPIV_IMAGE_PAR_KEY, "Author:", image_par.author);

    if (image_par.software_logic)
	printf("%s%s %s\n", GPIV_IMAGE_PAR_KEY, "Software:", image_par.software);

    if (image_par.source_logic)
	printf("%s%s %s\n", GPIV_IMAGE_PAR_KEY, "Source:", image_par.source);

    if (image_par.usertext_logic)
	printf("%s%s %s\n", GPIV_IMAGE_PAR_KEY, "Usertext:", image_par.usertext);

    if (image_par.warning_logic)
	printf("%s%s %s\n", GPIV_IMAGE_PAR_KEY, "Warning:", image_par.warning);

    if (image_par.disclaimer_logic)
	printf("%s%s %s\n", GPIV_IMAGE_PAR_KEY, "Disclaimer:", image_par.disclaimer);

    if (image_par.comment_logic)
	printf("%s%s %s\n", GPIV_IMAGE_PAR_KEY, "Comment:", image_par.comment);
}



void
gpiv_img_fprint_parameters(FILE * fp_par_out, 
			   GpivImagePar image_par
			   )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     prints image header parameters to file
 *     including GPIV_IMAGE_PAR_KEY
 *
 * PROTOTYPE LOCATATION:
 *     img.h
 *
 * INPUTS:
 *     fp_par_out:       file pointer to output file
 *
 * OUTPUTS:
 *---------------------------------------------------------------------------*/
{

/*
 * General image parameters
 */
    if (image_par.ncolumns_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_IMAGE_PAR_KEY, "Ncolumns:", 
		image_par.ncolumns);

    if (image_par.nrows_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_IMAGE_PAR_KEY, "Nrows:", 

		image_par.nrows);
    if (image_par.x_corr_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_IMAGE_PAR_KEY, "X_corr:", 
		image_par.x_corr);

    if (image_par.depth_logic)
	fprintf(fp_par_out, "%s%s %d\n", GPIV_IMAGE_PAR_KEY, "Depth:", 
		image_par.depth);
/*
 * Parameters used by gpiv_post_scale
 */
    if (image_par.s_scale_logic) 
        fprintf(fp_par_out ,"%s%s %f\n", GPIV_IMAGE_PAR_KEY, "Sscale:", 
                image_par.s_scale);
    
    if (image_par.t_scale_logic) 
        fprintf(fp_par_out, "%s%s %f\n", GPIV_IMAGE_PAR_KEY, "Tscale:", 
                image_par.t_scale);
    
    if (image_par.z_off_x_logic) 
        fprintf(fp_par_out, "%s%s %f\n", GPIV_IMAGE_PAR_KEY, "Zoff_x:", 
                image_par.z_off_x);
    
    if (image_par.z_off_y_logic) 
        fprintf(fp_par_out, "%s%s %f\n", GPIV_IMAGE_PAR_KEY, "Zoff_y:", 
                image_par.z_off_y);
  

/*
 * Optional parameters
 */
    if (image_par.title_logic)
	fprintf(fp_par_out, "%s%s %s\n", GPIV_IMAGE_PAR_KEY, "Title:", 
		image_par.title);

    if (image_par.creation_date_logic)
	fprintf(fp_par_out, "%s%s %s\n", GPIV_IMAGE_PAR_KEY, "Creation_date:", 
	       image_par.creation_date);

    if (image_par.location_logic)
	fprintf(fp_par_out, "%s%s %s\n", GPIV_IMAGE_PAR_KEY, "Location:", 
		image_par.location);

    if (image_par.author_logic)
	fprintf(fp_par_out, "%s%s %s\n", GPIV_IMAGE_PAR_KEY, "Author:", 
		image_par.author);

    if (image_par.software_logic)
	fprintf(fp_par_out, "%s%s %s\n", GPIV_IMAGE_PAR_KEY, "Software:", 
		image_par.software);

    if (image_par.source_logic)
	fprintf(fp_par_out, "%s%s %s\n", GPIV_IMAGE_PAR_KEY, "Source:", 
		image_par.source);

    if (image_par.usertext_logic)
	fprintf(fp_par_out, "%s%s %s\n", GPIV_IMAGE_PAR_KEY, "Usertext:", 
		image_par.usertext);

    if (image_par.warning_logic)
	fprintf(fp_par_out, "%s%s %s\n", GPIV_IMAGE_PAR_KEY, "Warning:", 
		image_par.warning);

    if (image_par.disclaimer_logic)
	fprintf(fp_par_out, "%s%s %s\n", GPIV_IMAGE_PAR_KEY, "Disclaimer:", 
		image_par.disclaimer);

    if (image_par.comment_logic)
	fprintf(fp_par_out, "%s%s %s\n", GPIV_IMAGE_PAR_KEY, "Comment:", 
		image_par.comment);

}



void
gpiv_img_cp_parameters(GpivImagePar image_par_src, 
                       GpivImagePar * image_par_dest, 
                       gboolean force,
                       gboolean print_par
                       )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Copies image parameters from src to dest
 *
 * INPUTS:
 *      piv_eval_par_src:       source piv parameters
 *      force:                  flag to force the copying, even if destination
 *                              already exists
 *      print_par:              verbose output. Not implemented yet.
 *
 * OUTPUTS:
 *      piv_eval_par_dest:     destination piv parameters
 *
 * RETURNS:
 *
 *---------------------------------------------------------------------------*/
{
    if (force 
        || (image_par_src.nrows_logic && !image_par_dest->nrows_logic)) {
        image_par_dest->nrows = image_par_src.nrows;
        image_par_dest->nrows_logic = TRUE;
        if (print_par) 
            g_message("nrows = %d", image_par_dest->nrows);
    }

    if (force 
        || (image_par_src.ncolumns_logic && !image_par_dest->ncolumns_logic)) {
        image_par_dest->ncolumns = image_par_src.ncolumns;
        image_par_dest->ncolumns_logic = TRUE;
        if (print_par) 
            g_message("ncolumns = %d", image_par_dest->ncolumns);
    }

    if(force 
       || (image_par_src.depth_logic && !image_par_dest->depth_logic)) {
        image_par_dest->depth = image_par_src.depth;
        image_par_dest->depth_logic = TRUE;
        if (print_par) 
            g_message("depth = %d", image_par_dest->depth);
    }

    if(force 
       || (image_par_src.x_corr_logic && !image_par_dest->x_corr_logic)) {
        image_par_dest->x_corr = image_par_src.x_corr;
        image_par_dest->x_corr_logic = TRUE;
        if (print_par) 
            g_message("x_corr = %d", image_par_dest->x_corr);
    }

    if (force 
        || (image_par_src.s_scale_logic && !image_par_dest->s_scale_logic)) {
        image_par_dest->s_scale = image_par_src.s_scale;
        image_par_dest->s_scale_logic = TRUE;
        if (print_par) 
            g_message("s_scale = %f", image_par_dest->s_scale);
    }

    if(force 
       || (image_par_src.t_scale_logic && !image_par_dest->t_scale_logic)) {
        image_par_dest->t_scale = image_par_src.t_scale;
        image_par_dest->t_scale_logic = TRUE;
        if (print_par) 
            g_message("t_scale = %f", image_par_dest->t_scale);
    }

    if (force 
        || (image_par_src.z_off_x_logic && !image_par_dest->z_off_x_logic)) {
        image_par_dest->z_off_x = image_par_src.z_off_x;
        image_par_dest->z_off_x_logic = TRUE;
        if (print_par) {
            g_message("z_off_x = %f", image_par_dest->z_off_x);
        }
    }

    if (force 
        || (image_par_src.z_off_y_logic && !image_par_dest->z_off_y_logic)) {
        image_par_dest->z_off_y = image_par_src.z_off_y;
        image_par_dest->z_off_y_logic = TRUE;
        if (print_par) {
            g_message("z_off_y = %f", image_par_dest->z_off_x);
        }
    }

    if (force 
        || (image_par_src.title_logic && !image_par_dest->title_logic)) {
        image_par_dest->title_logic = TRUE;
        snprintf(image_par_dest->title, GPIV_MAX_CHARS, "%s", 
                 image_par_src.title);
        if (print_par) 
            g_message("title = %S", image_par_dest->title);
    }
    
    
    if (force 
        || (image_par_src.creation_date_logic 
            && !image_par_dest->creation_date_logic)) {
        image_par_dest->creation_date_logic = TRUE;
        snprintf(image_par_dest->creation_date, GPIV_MAX_CHARS, "%s", 
                 image_par_src.creation_date);
        if (print_par) 
            g_message("creation_date = %s", image_par_dest->creation_date);
    } else {
        char time_string[GPIV_MAX_CHARS];
        struct tm time_struct;
        time_t itime;
        time(&itime);
        time_struct = *gmtime(&itime);
        strftime(time_string, GPIV_MAX_CHARS, "%a %b %d %Y %T", &time_struct);
        if (print_par)
            g_message("Setting new time = %s", time_string);
        snprintf(image_par_dest->creation_date, GPIV_MAX_CHARS, "%s", time_string);
        image_par_dest->creation_date_logic = TRUE;
    }
    
    
    if (force 
        || (image_par_src.location_logic && !image_par_dest->location_logic)) {
        image_par_dest->location_logic = TRUE;
        snprintf(image_par_dest->location, GPIV_MAX_CHARS, "%s", 
                 image_par_src.location);
        if (print_par) 
            g_message("location = %s", image_par_dest->location);
    }
    
    
    if (force 
        || (image_par_src.author_logic && !image_par_dest->author_logic)) {
        image_par_dest->author_logic = TRUE;
        snprintf(image_par_dest->author, GPIV_MAX_CHARS, "%s", 
                 image_par_src.author);
        if (print_par) 
            g_message("author = %s", image_par_dest->author);
    }

    if (force 
        || (image_par_src.software_logic && !image_par_dest->software_logic)) {
        image_par_dest->software_logic = TRUE;
        snprintf(image_par_dest->software, GPIV_MAX_CHARS, "%s", 
                 image_par_src.software);
        if (print_par) 
            g_message("software = %s", image_par_dest->software);
    }

    if (force 
        || (image_par_src.source_logic && !image_par_dest->source_logic)) {
        image_par_dest->source_logic = TRUE;
        snprintf(image_par_dest->source, GPIV_MAX_CHARS, "%s", 
                 image_par_src.source);
        if (print_par) 
            g_message("source = %s", image_par_dest->source);
    }

    if (force 
        || (image_par_src.usertext_logic && !image_par_dest->usertext_logic)) {
        image_par_dest->usertext_logic = TRUE;
        snprintf(image_par_dest->usertext, GPIV_MAX_CHARS, "%s", 
                 image_par_src.usertext);
        if (print_par) 
            g_message("usertext = %s", image_par_dest->usertext);
    }

    if (force 
        || (image_par_src.warning_logic && !image_par_dest->warning_logic)) {
        image_par_dest->warning_logic = TRUE;
        snprintf(image_par_dest->warning, GPIV_MAX_CHARS, "%s", 
                 image_par_src.warning);
        if (print_par) 
            g_message("warning = %s", image_par_dest->warning);
    }

    if (force 
        || (image_par_src.disclaimer_logic && !image_par_dest->disclaimer_logic)) {
        image_par_dest->disclaimer_logic = TRUE;
        snprintf(image_par_dest->disclaimer, GPIV_MAX_CHARS, "%s", 
                 image_par_src.disclaimer);
        if (print_par) 
            g_message("disclaimer = %s", image_par_dest->disclaimer);
    }

    if (force 
        || (image_par_src.comment_logic && !image_par_dest->comment_logic)) {
        image_par_dest->comment_logic = TRUE;
        snprintf(image_par_dest->comment, GPIV_MAX_CHARS, "%s", 
                 image_par_src.comment);
        if (print_par) 
            g_message("comment = %s", image_par_dest->comment);
    }
}




char *
gpiv_img_fread_hdf5_parameters(char * fname, 
			       GpivImagePar * image_par
			       )
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Reads image parameters from hdf5 data file
 *
 * PROTOTYPE LOCATATION:
 *     img.h
 *
 * INPUTS:
 *     fname:          complete file name
 *
 * OUTPUTS:
 *     image_par:      parameter structure
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int i;
/*
 * HDF declarations
 */
    hid_t       file_id, group_id, attribute_id, atype; 
    herr_t      status;

    if ((i = H5Fis_hdf5(fname)) == 0) {
        err_msg = "GPIV_IMG_FREAD_HDF5_PARAMETERS: not an hdf5 file";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    file_id = H5Fopen(fname, H5F_ACC_RDONLY, H5P_DEFAULT);
    group_id = H5Gopen (file_id, "IMAGE");

    H5Aiterate(group_id, NULL, attr_info, image_par);
    status = H5Gclose (group_id);
    status = H5Fclose(file_id); 
    return err_msg;
}



char *
gpiv_img_fwrite_hdf5_parameters(char *fname,
				GpivImagePar * image_par
				)
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *     Writes image parameters to an existing hdf5 data file
 *
 * PROTOTYPE LOCATATION:
 *     img.h
 *
 * INPUTS:
 *     fname:          complete file name
 *     piv_valid_par:  parameter structure
 *
 * OUTPUTS:
 *
 * RETURNS:
 *     NULL on success or *err_msg on failure
 *---------------------------------------------------------------------------*/
{
    char *err_msg = NULL;
    int i;
/*
 * HDF declarations
 */
    hid_t       file_id, dataspace_id, group_id, attribute_id,
      atype; 
    hsize_t     dims[1];
    herr_t      status;

    if ((i = H5Fis_hdf5(fname)) == 0) {
        err_msg = "GPIV_IMG_FWRITE_HDF5_PARAMETERS: not an hdf5 file";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }

    file_id = H5Fopen(fname, H5F_ACC_RDWR, H5P_DEFAULT);
    group_id = H5Gopen (file_id, "IMAGE");



    dims[0] = 1;
    dataspace_id = H5Screate_simple(1, dims, NULL);



/*
 * General image parameters
 */
    if (image_par->ncolumns_logic) {
        attribute_id = H5Acreate(group_id, "ncolumns", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &image_par->ncolumns); 
        status = H5Aclose(attribute_id); 
    }

    if (image_par->nrows_logic) {
        attribute_id = H5Acreate(group_id, "nrows", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &image_par->nrows); 
        status = H5Aclose(attribute_id); 
    }

    if (image_par->x_corr_logic) {
        attribute_id = H5Acreate(group_id, "x_corr", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &image_par->x_corr); 
        status = H5Aclose(attribute_id); 
    }

    if (image_par->depth_logic) {
        attribute_id = H5Acreate(group_id, "depth", H5T_NATIVE_INT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_INT, 
                          &image_par->depth); 
        status = H5Aclose(attribute_id); 
    }

/*
 * Variables and parameters used by gpiv_post_scale
 */
    if (image_par->s_scale_logic) {
        attribute_id = H5Acreate(group_id, "s_scale", H5T_NATIVE_FLOAT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_FLOAT, 
                          &image_par->s_scale); 
        status = H5Aclose(attribute_id); 
    }

    if (image_par->t_scale_logic) {
        attribute_id = H5Acreate(group_id, "t_scale", H5T_NATIVE_FLOAT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_FLOAT, 
                          &image_par->t_scale); 
        status = H5Aclose(attribute_id); 
    }

    if (image_par->z_off_x_logic) {
        attribute_id = H5Acreate(group_id, "z_off_x", H5T_NATIVE_FLOAT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_FLOAT, 
                          &image_par->z_off_x); 
        status = H5Aclose(attribute_id); 
    }

    if (image_par->z_off_y_logic) {
        attribute_id = H5Acreate(group_id, "z_off_y", H5T_NATIVE_FLOAT, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, H5T_NATIVE_FLOAT, 
                          &image_par->z_off_y); 
        status = H5Aclose(attribute_id); 
    }

    status = H5Sclose(dataspace_id);


/*
 * Optional parameters
 */
    if (image_par->title_logic) {
        dataspace_id = H5Screate(H5S_SCALAR);
	atype = H5Tcopy(H5T_C_S1);
	H5Tset_size(atype, GPIV_MAX_CHARS);
        attribute_id = H5Acreate(group_id, "title", atype, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, atype, 
                          &image_par->title); 
        status = H5Aclose(attribute_id); 
	status = H5Sclose(dataspace_id);
    }

    if (image_par->creation_date_logic) {
        dataspace_id = H5Screate(H5S_SCALAR);
	atype = H5Tcopy(H5T_C_S1);
	H5Tset_size(atype, GPIV_MAX_CHARS);
        attribute_id = H5Acreate(group_id, "creation_date", atype, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, atype, 
                          &image_par->creation_date); 
        status = H5Aclose(attribute_id); 
	status = H5Sclose(dataspace_id);
    }

    if (image_par->location_logic) {
        dataspace_id = H5Screate(H5S_SCALAR);
	atype = H5Tcopy(H5T_C_S1);
	H5Tset_size(atype, GPIV_MAX_CHARS);
        attribute_id = H5Acreate(group_id, "location", atype, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, atype, 
                          &image_par->location); 
        status = H5Aclose(attribute_id); 
	status = H5Sclose(dataspace_id);
    }

    if (image_par->author_logic) {
        dataspace_id = H5Screate(H5S_SCALAR);
	atype = H5Tcopy(H5T_C_S1);
	H5Tset_size(atype, GPIV_MAX_CHARS);
        attribute_id = H5Acreate(group_id, "author", atype, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, atype, 
                          &image_par->author); 
        status = H5Aclose(attribute_id); 
	status = H5Sclose(dataspace_id);
    }

    if (image_par->software_logic) {
        dataspace_id = H5Screate(H5S_SCALAR);
	atype = H5Tcopy(H5T_C_S1);
	H5Tset_size(atype, GPIV_MAX_CHARS);
        attribute_id = H5Acreate(group_id, "software", atype, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, atype, 
                          &image_par->software); 
        status = H5Aclose(attribute_id); 
	status = H5Sclose(dataspace_id);
    }

    if (image_par->source_logic) {
        dataspace_id = H5Screate(H5S_SCALAR);
	atype = H5Tcopy(H5T_C_S1);
	H5Tset_size(atype, GPIV_MAX_CHARS);
        attribute_id = H5Acreate(group_id, "source", atype, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, atype, 
                          &image_par->source); 
        status = H5Aclose(attribute_id); 
	status = H5Sclose(dataspace_id);
    }

    if (image_par->usertext_logic) {
        dataspace_id = H5Screate(H5S_SCALAR);
	atype = H5Tcopy(H5T_C_S1);
	H5Tset_size(atype, GPIV_MAX_CHARS);
        attribute_id = H5Acreate(group_id, "usertext", atype, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, atype, 
                          &image_par->usertext); 
        status = H5Aclose(attribute_id); 
	status = H5Sclose(dataspace_id);
    }

    if (image_par->warning_logic) {
        dataspace_id = H5Screate(H5S_SCALAR);
	atype = H5Tcopy(H5T_C_S1);
	H5Tset_size(atype, GPIV_MAX_CHARS);
        attribute_id = H5Acreate(group_id, "warning", atype, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, atype, 
                          &image_par->warning); 
        status = H5Aclose(attribute_id); 
	status = H5Sclose(dataspace_id);
    }

    if (image_par->disclaimer_logic) {
        dataspace_id = H5Screate(H5S_SCALAR);
	atype = H5Tcopy(H5T_C_S1);
	H5Tset_size(atype, GPIV_MAX_CHARS);
        attribute_id = H5Acreate(group_id, "disclaimer", atype, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, atype, 
                          &image_par->disclaimer); 
        status = H5Aclose(attribute_id); 
	status = H5Sclose(dataspace_id);
    }

    if (image_par->comment_logic) {
        dataspace_id = H5Screate(H5S_SCALAR);
	atype = H5Tcopy(H5T_C_S1);
	H5Tset_size(atype, GPIV_MAX_CHARS);
        attribute_id = H5Acreate(group_id, "comment", atype, 
                                 dataspace_id, H5P_DEFAULT);
        status = H5Awrite(attribute_id, atype, 
                          &image_par->comment); 
        status = H5Aclose(attribute_id); 
	status = H5Sclose(dataspace_id);
    }



    status = H5Gclose (group_id);
    status = H5Fclose(file_id); 
    return err_msg;
}



char *
gpiv_img_fread_davis_parameters(char *fname,
                                GpivImagePar *image_par
                                ) 
/* 		  enum FrameNr fn */
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Reads image specifications from Davis formatted image, with ext .IMG, 
 *      from file
 *
 * PROTOTYPE LOCATATION:
 *     img.h
 *
 * INPUTS:
 *     fname:          pointer to complete filename to be read
 *
 * RETURNS:
 *     image_par:      structure containing image dimensions and other image 
 *                     specifications
 *---------------------------------------------------------------------------*/
{
    FILE *fp;
    char *err_msg = NULL;

    image_par->nrows = 0;
    image_par->ncolumns = 0;
    image_par->depth = GPIV_DAVIS_IMG_DEPTH; 
    image_par->x_corr = 1;
    image_par->depth_logic = TRUE; 
    image_par->x_corr_logic = TRUE;

    if ((fp = fopen(fname, "rb")) == NULL) {
	err_msg = "GPIV_FREAD_DAVIS_IMAGE: Failure opening for input";
        gpiv_warning("%s", err_msg);
        return err_msg;
    }


    fseek(fp, 10, SEEK_SET);
    fread(&image_par->nrows, sizeof(guint16), 1, fp);
    fread(&image_par->ncolumns, sizeof(guint16), 1, fp);
    image_par->nrows_logic = TRUE;
    image_par->ncolumns_logic = TRUE;

    fclose(fp);
    return err_msg;
}
