/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 c-style: "K&R" -*- */

/*
   libgpiv - library for Particle Image Velocimetry

   Copyright (C) 2002, 2003, 2004 Gerber van der Graaf

   This file is part of libgpiv.

   Libgpiv is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  




---------------------------------------------------------------
FILENAME:               imgproc.h
LIBRARY:                libgpiv:
EXTERNAL FUNCTIONS:
SOURCE:                 imgproc_par.c
                        gpiv_imgproc_fscan_parameters
                        gpiv_imgproc_read_parameters 
                        gpiv_imgproc_check_parameters
                        gpiv_imgproc_test_parameters
                        gpiv_imgproc_print_parameters
                        gpiv_imgproc_fprint_parameter

SOURCE:                 imgproc.c
                        gpiv_imgproc_mktstimg
                        gpiv_imgproc_smooth
                        gpiv_imgproc_highlow
                        gpiv_imgproc_clip
                        gpiv_imgproc_getbit
                        gpiv_imgproc_fft
                        gpiv_imgproc_bandpass
                        gpiv_imgproc_correlate
                        gpiv_imgproc_convolve

LAST MODIFICATION DATE:  $Id: imgproc.h,v 1.2 2006/01/31 13:29:52 gerber Exp $
 --------------------------------------------------------------- */

#ifndef __LIBGPIV_IMGPROC_H__
#define __LIBGPIV_IMGPROC_H__


#define GPIV_IMGPROCPAR_DEFAULT__BIT 8
#define GPIV_IMGPROCPAR_DEFAULT__DIR_PREFIX ./  
#define GPIV_IMGPROCPAR_DEFAULT__FILTER 0
#define GPIV_IMGPROCPAR_DEFAULT__FIRST_DIR 0
#define GPIV_IMGPROCPAR_DEFAULT__LAST_DIR 0
#define GPIV_IMGPROCPAR_DEFAULT__FIRST_FILE 0
#define GPIV_IMGPROCPAR_DEFAULT__LAST_FILE 0
#define GPIV_IMGPROCPAR_DEFAULT__FILE_PREFIX img
#define GPIV_IMGPROCPAR_DEFAULT__OPERATOR 0
#define GPIV_IMGPROCPAR_DEFAULT__THRESHOLD 30
#define GPIV_IMGPROCPAR_DEFAULT__WINDOW 10


enum GpivImgprocOperation {
    GPIV_MK_TSTIMG = 0,
    GPIV_FI_SMOOTH = 1,
    GPIV_FI_HILO = 2,
    GPIV_FI_CLIP = 3,

    GPIV_FI_FFT = 4,
    GPIV_FI_INVFFT = 5,
    GPIV_FI_CORR = 6,
    GPIV_FI_CONV = 7,
    GPIV_FI_LOWPASS = 8,
    GPIV_FI_HIGHPASS = 9,

    GPIV_PT_GETBIT = 10 /* Pointer operation to get the N least significant bits */
} IMG_OPERATOR;

enum GpivFilterOperator {
    GPIV_EQUAL = 0,
    GPIV_SUBTRACT = 1,
    GPIV_ADD = 2,
    GPIV_MULTIPLY = 3,
    GPIV_DIVIDE = 4
} FI_OP;


/*---------------------------------------------------------------------
 Image processing parameters */
typedef struct __GpivImageProcPar GpivImageProcPar;
struct __GpivImageProcPar {
    int bit; /* BUGFIX: not in read_paramaeters */
    int dir_prefix;    
    enum GpivFilterOperator filter;
    int first_dir;
    int last_dir;
    int first_file;
    int file_prefix;
    int last_file;
    int operator;
    int window;
    int threshold;

    gboolean bit_logic; /* BUGFIX: not in read_paramaeters */
    gboolean dir_prefix_logic;     
    gboolean filter_logic;
    gboolean first_dir_logic;
    gboolean last_dir_logic;
    gboolean first_file_logic;
    gboolean last_file_logic;
    gboolean file_prefix_logic;
    gboolean operator_logic;
    gboolean window_logic;
    gboolean threshold_logic;
};



/*
 * image processing parameters
 */



void
gpiv_imgproc_read_parameters(FILE *fp_h, 
			     GpivImageProcPar * image_proc_par, 
			     gboolean print_par);

/*---------------------------------------------------------------------------- 
  Returns: NULL on success or *err_msg on failure
*/
char *
gpiv_imgproc_check_parameters_read(GpivImageProcPar image_proc_par);

/*---------------------------------------------------------------------------- 
  Returns: NULL on success or *err_msg on failure
*/
char *
gpiv_imgproc_test_parameters(GpivImageProcPar image_proc_par);


void 
gpiv_imgproc_print_parameters(GpivImageProcPar image_proc_par);


void
gpiv_imgproc_fprint_parameters(FILE * fp_par_out, 
			       GpivImageProcPar image_proc_par);



/* ----------------------------------------------------------------------------
 * Generates a test image for filtering and manipulating
 */
void
gpiv_imgproc_mktstimg(GpivImagePar image_par,
                    GpivImageProcPar image_proc_par,
                    guint16 **img_out);


/* ----------------------------------------------------------------------------
 * Smotthing filter by taking mean value of surrounding 
 * window x window pixels
 */
void
gpiv_imgproc_smooth(GpivImagePar image_par, 
                    GpivImageProcPar image_proc_par, 
                    guint16 **img_in,
                    guint16 **img_out);


/* ----------------------------------------------------------------------------
 * High-low filter to maximize contrast by stretching pixel values
 * to local max and min within window x window area
 */
void
gpiv_imgproc_highlow(GpivImagePar image_par,
                     GpivImageProcPar image_proc_par,
                     guint16 **img_in, 
                     guint16 **img_out);


/* ----------------------------------------------------------------------------
 * Sets all pixel values lower than threshold to zero
 */
void
gpiv_imgproc_clip(GpivImagePar image_par,
                  GpivImageProcPar image_proc_par,
                  guint16 **img_in, 
                  guint16 **img_out);


/* ----------------------------------------------------------------------------
 * Pointer operation to get the N least significant bits and moves them to 
 * most the significant bits
 */
void
gpiv_imgproc_getbit(GpivImagePar image_par,
                    GpivImageProcPar image_proc_par,
                    guint16 **img_in, 
                    guint16 **img_out);


/* ----------------------------------------------------------------------------
 * Fast Fourier Transformation
 */
char *
gpiv_imgproc_fft(GpivImagePar img_par, 
                 GpivImageProcPar img_proc_par,
                 guint16 **img_in, 
                 guint16 **img_out);


/*-----------------------------------------------------------------------------
 *     Lowpass filter on an image
 */
char *
gpiv_imgproc_lowpass(GpivImagePar img_par, 
                     GpivImageProcPar img_proc_par,
                     guint16 **img_in, 
                     guint16 **img_out);


/*-----------------------------------------------------------------------------
 *     Highpass filter on an image
 */
char *
gpiv_imgproc_highpass(GpivImagePar img_par, 
                      GpivImageProcPar img_proc_par,
                      guint16 **img_in, 
                      guint16 **img_out);


/*-----------------------------------------------------------------------------
 * Correlates two images
 */
void
gpiv_imgproc_correlate(GpivImagePar img_par, 
                       GpivImageProcPar img_proc_par,
                       guint16 **img1_in, 
                       guint16 **img2_in, 
                       guint16 **img_out);


/*-----------------------------------------------------------------------------
 * Convolves two images
 */
void
gpiv_imgproc_convolve(GpivImagePar img_par, 
                      GpivImageProcPar img_proc_par,
                      guint16 **img1_in, 
                      guint16 **img2_in, 
                      guint16 **img_out);


void 
gpiv_imgproc_deform(GpivPivData * piv_data,
                    GpivImagePar *image_par,
                    guint16 **img1_src,
                    guint16 **img2_src,
                    guint16 **img1_dest,
                    guint16 **img2_dest
                    );
/*-----------------------------------------------------------------------------
 * DESCRIPTION:
 *      Image shifting and deformation routine for a single exposed, double 
 *      frame PIV image pair with magnitude of PIV estimations at each pixel
 *
 * INPUTS:
 *      piv_data:       Input PIV data structure
 *      image_par:      image parameters
 *      img1_src:       first input image frame
 *      img2_src:       second input image frame
 *
 * OUTPUTS:
 *      img1_dest:      first deformed image frame
 *      img2_dest:      second deformed image frame
 *
 * RETURNS:
 *
 *---------------------------------------------------------------------------*/


#endif /* __LIBGPIV_IMGPROC_H__ */

