/**
 * =========================================
 * LibFormula : a free Java formula library
 * =========================================
 *
 * Project Info:  http://reporting.pentaho.org/libformula/
 *
 * (C) Copyright 2006-2007, by Pentaho Corporation and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * [Java is a trademark or registered trademark of Sun Microsystems, Inc.
 * in the United States and other countries.]
 *
 *
 * ------------
 * $Id: ModFunction.java 3521 2007-10-16 10:55:14Z tmorgner $
 * ------------
 * (C) Copyright 2006-2007, by Pentaho Corporation.
 */
package org.jfree.formula.function.math;

import org.jfree.formula.EvaluationException;
import org.jfree.formula.FormulaContext;
import org.jfree.formula.LibFormulaErrorValue;
import org.jfree.formula.function.Function;
import org.jfree.formula.function.ParameterCallback;
import org.jfree.formula.lvalues.TypeValuePair;
import org.jfree.formula.typing.Type;
import org.jfree.formula.typing.coretypes.NumberType;
import org.jfree.formula.util.NumberUtil;

import java.math.BigDecimal;

/**
 * This function returns the remainder when one number is divided by another number.
 *
 * @author Cedric Pronzato
 */
public class ModFunction implements Function
{
  public ModFunction()
  {
  }

  public TypeValuePair evaluate(final FormulaContext context, final ParameterCallback parameters) throws EvaluationException
  {
    final int parameterCount = parameters.getParameterCount();
    if (parameterCount != 2)
    {
      throw new EvaluationException(LibFormulaErrorValue.ERROR_ARGUMENTS_VALUE);
    }
    final Type type1 = parameters.getType(0);
    final Object value1 = parameters.getValue(0);
    final BigDecimal divided = NumberUtil.getAsBigDecimal(context.getTypeRegistry().convertToNumber(type1, value1));
    final Type type2 = parameters.getType(1);
    final Object value2 = parameters.getValue(1);
    final BigDecimal divisor = NumberUtil.getAsBigDecimal(context.getTypeRegistry().convertToNumber(type2, value2));
    if(divisor.signum() == 0)
    {
      throw new EvaluationException(LibFormulaErrorValue.ERROR_ARITHMETIC_VALUE);
    }

    final BigDecimal divide = new BigDecimal(divided.divide(divisor, 0, BigDecimal.ROUND_FLOOR).intValue());
    BigDecimal reminder = divided.subtract(divisor.multiply(divide));
    if(divide.signum() == 0)
    {
      if((divided.signum() == -1 && divisor.signum() != -1) || (divisor.signum() == -1 && divided.signum() != -1))
      {
        reminder = divided.add(divisor);
      }
    }

    return new TypeValuePair(NumberType.GENERIC_NUMBER, reminder);
  }

  public String getCanonicalName()
  {
    return "MOD";
  }

}