/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifdef HAVE_GETTIMEOFDAY
#  include <sys/times.h>
#endif
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <log.h>
#include <types.h>


/* --- Open the Linux and xBSD /dev/random devices --- */

int rand_linux_opendev(const char *name, int minor)
{
  int fd;
  struct stat sb;

  fd = open(name, O_RDONLY);
  if (fd == -1) return(0);
  if (fstat(fd, &sb)) return(0);
  if (!S_ISCHR(sb.st_mode)) return(0);

  return fd;
}

/* --- Fill a block with random bytes --- */

int rand_gather_linux(unsigned char *dest, size_t length, int level)
{
  static int fd_urandom = -1;
  static int fd_random = -1;
  size_t done;
  int fd;
  int n;
  int warn = 0;
  byte buffer[768];


  if (level >= 2)
  {
    if (fd_random == -1) fd_random = rand_linux_opendev(NAME_OF_DEV_RANDOM, 8);
    fd = fd_random;
  }
  else
  {
    if (fd_urandom == -1) fd_urandom = rand_linux_opendev(NAME_OF_DEV_URANDOM, 9);
    fd = fd_urandom;
  }
  
  done = 0;

  while (length)
  {
    fd_set rfds;
    struct timeval tv;
    int rc;

    FD_ZERO(&rfds);
    FD_SET(fd, &rfds);
    tv.tv_sec = 3;
    tv.tv_usec = 0;
    if (!(rc = select(fd + 1, &rfds, NULL, NULL, &tv)))
    {
      if (!warn) log_put_opt(LOG_DEBUG, 0, "Out of random bytes (need %d more).", length);
      warn = 1;
      continue;
    }
    else if (rc == -1)
    {
      log_put_opt(LOG_DEBUG, 0, "Random device select() error: %s", strerror(errno));
      continue;
    }

    do
    {
      int nbytes = length < sizeof(buffer) ? length : sizeof(buffer);
      n = read(fd, buffer, nbytes);
      if (n >= 0 && n > nbytes)
      {
        log_put_opt(LOG_DEBUG, 0, "Bogus read from random device (n=%d).", n );
        n = nbytes;
      }
    }
    while (n == -1 && errno == EINTR);

    if (n == -1) log_put_opt(LOG_DEBUG, 0, "Read error on random device: %s", strerror(errno));

    memcpy(dest + done, buffer, n);
    done += n;
    length -= n;
  }
  memset(buffer, 0, sizeof(buffer));

  return 0; /* success */
}
