/* Copyright (C) 1999 Hans Petter K. Jansson
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 * You can contact the library's author by sending e-mail to <hpj@styx.net>.
 */

#ifndef MT_H
#define MT_H 1

#ifdef LIBFLUX_BUILD
# include "types.h"
# include "tt.h"
#else
# include <flux/types.h>
# include <flux/tt.h>
#endif

#include <regex.h>

#define TRUE 1
#define FALSE 0

typedef TT MT;
#define MT(x) ((MT *) (x))

typedef enum
{
  MT_DATA = 0, MT_ENTITY, MT_EMPTY, MT_SPAN, MT_TYPES_NUM
} MT_TYPE;

#define MT_UNKNOWN -1

extern char *mt_types[];

#define MT_FOR_EACH(mt, c) \
  for((c) = mt_get_first_child(mt); (c); (c) = mt_get_next(c))


/* Allocation */

#define mt_size(mt) (tt_is_leaf(TT(mt)) ? 0 : tt_size(tt_get_first_child(TT(mt))))

MT *mt_new(MT_TYPE type);
#define mt_del(mt) tt_del(TT(mt))
MT *mt_dup(MT *mt);      /* Duplicate and detach node */
MT *mt_dup_all(MT *mt);  /* Duplicate and detach branch */

MT *mt_split(MT *mt, u32 pos);  /* TODO */

/* Status set */

void mt_set_ready(MT *mt, int ready);  /* TODO */

/* Status get */

int mt_has_data(MT *mt);
int mt_is_ready(MT *mt);  /* TODO */

/* Attributes set */

void mt_set_name_str(MT *mt, const char *str);

/* Attributes get */

int mt_get_type(MT *mt);

TT *mt_get_name(MT *mt);
char *mt_get_name_str(MT *mt);

#define mt_get_attrs(mt) tt_find_first_child(TT(mt), "attr", 4)
TT *mt_get_attr(MT *mt, const char *attr);
char *mt_get_attr_str(MT *mt, const char *attr);

/* Data */

void mt_data_set_str(MT *mt, const char *str);
void mt_data_set_bytes(MT *mt, byte *src, u32 start, u32 len);
void mt_data_append_bytes(MT *mt, byte *src, u32 len);
void mt_data_prepend_bytes(MT *mt, byte *src, u32 len);

#define mt_data_get(mt) tt_get_first_child(TT(mt))
char *mt_data_get_str(MT *mt);
u32 mt_data_get_bytes(MT *mt, byte *dest, u32 start, u32 len);



/* Navigation */

#define mt_is_root(mt) tt_is_root(TT(mt))
int mt_is_leaf(MT *mt);
#define mt_is_first(mt) tt_is_first(TT(mt))
#define mt_is_last(mt) tt_is_last(TT(mt))
#define mt_is_sibling(mt0, mt1) tt_is_sibling(TT(mt0), TT(mt1))

#define mt_get_root(mt) tt_get_root(TT(mt))
#define mt_get_first_sibling(mt) tt_get_first_sibling(TT(mt))
#define mt_get_last_sibling(mt) tt_get_last_sibling(TT(mt))
#define mt_get_prev(mt) tt_get_prev(TT(mt))
#define mt_get_next(mt) tt_get_next(TT(mt))
MT *mt_get_parent(MT *mt);
MT *mt_get_first_child(MT *mt);
MT *mt_get_last_child(MT *mt);
MT *mt_get_common_parent(MT *mt0, MT *mt1);

/* Linking */

void mt_add_as_first_child(MT *parent_mt, MT *mt);
void mt_add_as_last_child(MT *parent_mt, MT *mt);
void mt_add_as_first_sibling(MT *sibling_mt, MT *mt);
void mt_add_as_last_sibling(MT *sibling_mt, MT *mt);
#define mt_add_before(next_mt, mt) tt_add_before(TT(next_mt), TT(mt))
#define mt_add_after(prev_mt, mt) tt_add_after(TT(prev_mt), TT(mt))
#define mt_add(parent_mt, mt) mt_add_as_last_child(parent_mt, mt)

#define mt_swap(mt0, mt1) tt_swap(TT(mt0), TT(mt1))
#define mt_detach(mt) tt_detach(TT(mt))

#define mt_is_in_path(mt0, mt1) tt_is_in_path(TT(mt0), TT(mt1))

/* Statistics */

#define mt_depth(mt) (tt_depth(TT(mt)) >> 1)  /* Is this responsible? */
u32 mt_count_children(MT *mt);
u32 mt_count_children_all(MT *mt);
u32 mt_count_siblings(MT *mt);

/* Matching */

#ifdef FINISHED_THINKING_THIS_OVER_AND_DECIDED_TO_USE_IT

int mt_cmp(MT *mt0, MT *mt1);
int mt_casecmp(MT *mt0, MT *mt1);
int mt_memcmp(MT *mt, byte *p, u32 len);
int mt_memcasecmp(MT *mt, byte *p, u32 len);
#define mt_strcmp(mt, s) (mt_memcmp((mt), (s), strlen(s)))
#define mt_strcasecmp(mt, s) (mt_memcasecmp((mt), (s), strlen(s)))

int mt_chr(MT *mt, int c);

int mt_regcmp_precomp(MT *mt, regex_t *preg);
int mt_regcmp(MT *mt, const char *regex);
int mt_regcasecmp(MT *mt, const char *regex);

#endif

/* Searching */

/* ... */

/* XML */

MT *mt_scan_from_xml_file(FILE *in, int validate);
int mt_print_to_xml_file(MT *mt, FILE *out);


#endif /* MT_H */
