=head1 NAME

Data::Entropy::RawSource::RandomnumbersInfo - download entropy from
randomnumbers.info

=head1 SYNOPSIS

	use Data::Entropy::RawSource::RandomnumbersInfo;

	my $rawsrc = Data::Entropy::RawSource::RandomnumbersInfo->new;

	$c = $rawsrc->getc;
	# and the rest of the I/O handle interface

=head1 DESCRIPTION

This class provides an I/O handle connected to a stream of random octets
being generated by a quantum random number generator (from the company id
Quantique) connected to the randomnumbers.info server at the University
of Geneva.  This is a strong source of random bits, but is not suitable
for security applications because the bits are passed over the Internet
unencrypted.  The handle implements a substantial subset of the interface
described in L<IO::Handle>.

For use as a general entropy source, it is recommended to wrap an object
of this class using C<Data::Entropy::Source>, which provides methods to
extract entropy in more convenient forms than mere octets.

The bits generated at randomnumbers.info are, theoretically and as far as
anyone can tell, totally unbiased and uncorrelated.  However, they are
sent over the Internet in the clear, and so are subject to interception
and alteration by an adversary.  This is therefore generally unsuitable
for security applications.  Applications requiring secret entropy
should generate it locally (see L<Data::Entropy::RawSource::Local>).
Applications requiring a large amount of apparently-random data,
but not true entropy, might prefer to fake it cryptographically (see
L<Data::Entropy::RawSource::CryptCounter>).

=cut

package Data::Entropy::RawSource::RandomnumbersInfo;

use warnings;
use strict;

use Errno 1.00 qw(EIO);
use LWP 5.53_94;
use LWP::UserAgent;

our $VERSION = "0.004";

use fields qw(ua buffer bufpos error);

=head1 CONSTRUCTOR

=over

=item Data::Entropy::RawSource::RandomnumbersInfo->new

Creates and returns a handle object referring to a stream of random
octets generated by randomnumbers.info.

=cut

sub new($$) {
	my($class) = @_;
	my Data::Entropy::RawSource::RandomnumbersInfo $self =
		fields::new($class);
	$self->{ua} = LWP::UserAgent->new;
	$self->{buffer} = "";
	$self->{bufpos} = 0;
	$self->{error} = 0;
	return $self;
}

=back

=head1 METHODS

A subset of the interfaces described in L<IO::Handle> and L<IO::Seekable>
are provided.  The methods implemented are: C<clearerr>, C<close>,
C<eof>, C<error>, C<getc>, C<opened>, C<read>, C<sysread>, C<ungetc>.

C<close> does nothing.

The buffered (C<read> et al) and unbuffered (C<sysread> et al) sets
of methods are interchangeable, because no such distinction is made by
this class.

Methods to write to the file are unimplemented because the stream is
fundamentally read-only.  Methods to seek are unimplemented because the
stream is non-rewindable; C<ungetc> works, however.

=cut

sub ensure_buffer($) {
	my Data::Entropy::RawSource::RandomnumbersInfo $self = shift;
	return 1 unless $self->{bufpos} == length($self->{buffer});
	my $response = $self->{ua}->get(
		"http://www.randomnumbers.info/cgibin/wqrng.cgi?".
		"amount=256&limit=255");
	unless($response->code == 200) {
		$! = EIO;
		return 0;
	}
	my($numbers) = ($response->content =~ /((?:\d+\s+){255}\d+)/);
	unless(defined $numbers) {
		$! = EIO;
		return 0;
	}
	$self->{buffer} = "";
	$self->{buffer} .= chr($1) while $numbers =~ /(\d+)/g;
	$self->{bufpos} = 0;
	return 1;
}

sub close($) { 1 }

sub opened($) { 1 }

sub error($) {
	my Data::Entropy::RawSource::RandomnumbersInfo $self = shift;
	return $self->{error};
}

sub clearerr($) {
	my Data::Entropy::RawSource::RandomnumbersInfo $self = shift;
	$self->{error} = 0;
	return 0;
}

sub getc($) {
	my Data::Entropy::RawSource::RandomnumbersInfo $self = shift;
	unless($self->ensure_buffer) {
		$self->{error} = 1;
		return undef;
	}
	return substr($self->{buffer}, $self->{bufpos}++, 1);
}

sub ungetc($$) {
	my Data::Entropy::RawSource::RandomnumbersInfo $self = shift;
	my($cval) = @_;
	if($self->{bufpos} == 0) {
		$self->{buffer} = chr($cval).$self->{buffer};
	} else {
		$self->{bufpos}--;
	}
}

sub read($$$;$) {
	my Data::Entropy::RawSource::RandomnumbersInfo $self = shift;
	my(undef, $length, $offset) = @_;
	return undef if $length < 0;
	$_[0] = "" unless defined $_[0];
	if(!defined($offset)) {
		$offset = 0;
		$_[0] = "";
	} elsif($offset < 0) {
		return undef if $offset < -length($_[0]);
		substr $_[0], $offset, -$offset, "";
		$offset = length($_[0]);
	} elsif($offset > length($_[0])) {
		$_[0] .= "\0" x ($offset - length($_[0]));
	} else {
		substr $_[0], $offset, length($_[0]) - $offset, "";
	}
	my $original_offset = $offset;
	while($length != 0) {
		unless($self->ensure_buffer) {
			$self->{error} = 1;
			last;
		}
		my $avail = length($self->{buffer}) - $self->{bufpos};
		if($length < $avail) {
			$_[0] .= substr($self->{buffer}, $self->{bufpos},
					$length);
			$offset += $length;
			$self->{bufpos} += $length;
			last;
		}
		$_[0] .= substr($self->{buffer}, $self->{bufpos}, $avail);
		$offset += $avail;
		$length -= $avail;
		$self->{bufpos} += $avail;
	}
	my $nread = $offset - $original_offset;
	return $nread == 0 ? undef : $nread;
}

*sysread = \&read;

sub eof($) { 0 }

=head1 SEE ALSO

L<Data::Entropy::RawSource::CryptCounter>,
L<Data::Entropy::RawSource::Local>,
L<Data::Entropy::RawSource::RandomOrg>,
L<Data::Entropy::Source>,
L<http://www.randomnumbers.info>

=head1 AUTHOR

Andrew Main (Zefram) <zefram@fysh.org>

=head1 COPYRIGHT

Copyright (C) 2006, 2007 Andrew Main (Zefram) <zefram@fysh.org>

This module is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

=cut

1;
