// Copyright (c) 2002 Rob Kaper <cap@capsi.com>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License version 2.1 as published by the Free Software Foundation.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with this library; see the file COPYING.LIB.  If not, write to
// the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include <netdb.h>
#include <unistd.h>

#include "listener.h"
#include "listenport.h"
#include "socket.h"

#define	MAXLINE	1024

extern int errno;

Listener::Listener()
{
}

Listener::~Listener()
{
	while (!m_listenPorts.empty()) { delete *m_listenPorts.begin(); m_listenPorts.erase(m_listenPorts.begin()); }
	while (!m_sockets.empty()) { delete *m_sockets.begin(); m_sockets.erase(m_sockets.begin()); }
}

int Listener::addListenPort(const int port)
{
	ListenPort *listenPort = new ListenPort("0.0.0.0", port);
	m_listenPorts.push_back(listenPort);
	if ( !listenPort->isBound() )
		return -1;
	return 0;
}

void Listener::do_select()
{
	// Find whether there are any sockets to get rid of first.
	Socket *socket = 0;
	for (std::vector<Socket *>::iterator it = m_sockets.begin() ; it != m_sockets.end() && (socket = *it) ; ++it)
		if (socket->status() == Socket::Close || socket->status() == Socket::Closed)
		{
			socketHandler(socket, "");
			delSocket(socket);
			return;
		}

	// Check for events with a fairly large timeout
	FD_ZERO(&m_fdset);
	unsigned int highestFd = 0;

	ListenPort *listenPort = 0;
	for(std::vector<ListenPort *>::iterator it = m_listenPorts.begin() ; it != m_listenPorts.end() && (listenPort = *it) ; ++it)
	{
		if ( listenPort->isBound() )
		{
			FD_SET(listenPort->fd(), &m_fdset);
			if (listenPort->fd() > highestFd)
				highestFd = listenPort->fd();
		}
	}

	for (std::vector<Socket *>::iterator it = m_sockets.begin() ; it != m_sockets.end() && (socket = *it) ; ++it)
	{
		FD_SET(socket->fd(), &m_fdset);
		if (socket->fd() > highestFd)
			highestFd = socket->fd();
	}

	struct timeval tv;
	tv.tv_sec = 0;
	tv.tv_usec = 250000; // perhaps decrease with increasing amount of sockets?

	if ( !highestFd )
	{
		sleep(1);
		exit(1);
		// FIXME: try to (re)bind unbound listenports
		return;
	}

	if ((select(highestFd+1, &m_fdset, NULL, NULL, &tv)) <= 0)
		return;

	// Check for new connections
	for(std::vector<ListenPort *>::iterator it = m_listenPorts.begin() ; it != m_listenPorts.end() && (listenPort = *it) ; ++it)
		if (FD_ISSET(listenPort->fd(), &m_fdset))
		{
			newSocket(listenPort->fd()); // process listenports for new connections
		}

	// Check sockets
	for (std::vector<Socket *>::iterator it = m_sockets.begin() ; it != m_sockets.end() && (socket = *it) ; ++it)
	{
		if (socket->status() == Socket::Ok && FD_ISSET(socket->fd(), &m_fdset))
		{
			char *readBuf = new char[MAXLINE];
			int n = read(socket->fd(), readBuf, MAXLINE);
			if (n <= 0) // socket was closed
			{
				socket->setStatus(Socket::Closed);
				delete[] readBuf;
				return;
			}
			readBuf[n] = 0;
			socket->fillBuffer(readBuf);
			delete[] readBuf;

			while (socket->hasReadLine())
			{
		        std::string data = socket->readLine();

				// Call socketHandler if we have a line
				if (data.size() > 0)
				{
					socketHandler(socket, data);
					continue;
				}
			}
		}
	}
}

Socket *Listener::newSocket(unsigned int fd)
{
// TODO: reenable softbooting code
//	if (!isNew)
//	{
//		FD_SET(fd, &m_fdset);
//
//		Socket *socket = new Socket(fd);
//		socket->setStatus(Socket::Ok);
//		m_sockets.push_back(socket);
//
//		return socket;
//	}

	struct sockaddr_in clientaddr;
	struct hostent *host;

	unsigned int len = sizeof(clientaddr);
	unsigned int socketFd = accept(fd, (struct sockaddr *) &clientaddr, (socklen_t *) &len);

	Socket *socket = new Socket(socketFd);
	socket->setIpAddr((char *) inet_ntoa(clientaddr.sin_addr));
	if( (host = gethostbyaddr((char *)&clientaddr.sin_addr, sizeof(clientaddr.sin_addr), AF_INET)) != NULL)
		socket->setFqdn(host->h_name);
	m_sockets.push_back(socket);
	socketHandler(socket, "");
	return socket;
}

void Listener::delSocket(Socket *socket)
{
	FD_CLR(socket->fd(), &m_fdset);
	close(socket->fd());
		
	for (std::vector<Socket *>::iterator it = m_sockets.begin() ; it != m_sockets.end() && (*it) ; ++it)
		if (*it == socket)
		{
			delete *it;
			m_sockets.erase(it);
			return;
		}
}

Socket *Listener::findSocket(unsigned int fd)
{
	Socket *socket = 0;
	for (std::vector<Socket *>::iterator it = m_sockets.begin() ; it != m_sockets.end() && (socket = *it) ; ++it)
		if (socket->fd() == fd)
			return socket;

	return 0;
}
