/*-------------------------------------------------------------------------------

Copyright (c) 2001 CerebraSoft

Module Name:

	registry.cpp

Abstract:

	NT registry related functions

License:

   All rights reserved.

   This product includes software developed at The Center for
   Networking and Distributed Systems at The Johns Hopkins University
   for use in the Backhand project (http://www.cnds.jhu.edu/backhand).
     Creator: Theo Schlossnagle 
     Guidance: Yair Amir

   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions
   are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer. 
 
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in
     the documentation and/or other materials provided with the
     distribution.
 
  3. All advertising materials mentioning features or use of this
     software must display the following acknowledgment:
     "This product includes software developed by Cerebrasoft
     (http://www.cerebrasoft.com).
        Creator: Rob Butler"

  4. The names "Backhand Broadcaster" and "NT Backhand Broadcaster" must
     not be used to endorse or promote products derived from this
     software without prior written permission. For written permission,
     please contact www.cerebrasoft.com.
 
  5. Products derived from this software may not be called "Backhand Broadcaster"
     or "NT Backhand Broadcaster" nor may "Backhand Broadcaster" or
     "NT Backhand Broadcaster" appear in their names without prior written
     permission. For written permission, please contact www.cerebrasoft.com.
 
  6. Redistributions of any form whatsoever must retain the following
     acknowledgment:
     "This product includes software developed by Cerebrasoft
     (http://www.cerebrasoft.com).
        Creator: Rob Butler"

  THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
  PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
  ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
  NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
  STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
  OF THE POSSIBILITY OF SUCH DAMAGE.
---------------------------------------------------------------------------------*/

#include <windows.h>

/*-------------------------------------------------------------------------------

DWORD getRegDWORD(HKEY key, LPCTSTR subKey, LPCTSTR name, DWORD defVal)

	Attempts to get an DWORD from the registry.  If it cannot it returns a default value.

	HKEY key	- An open registry key handle, or one of the pre defined handles;
				HKEY_CLASSES_ROOT, HKEY_CURRENT_CONFIG, HKEY_CURRENT_USER, 
				HKEY_LOCAL_MACHINE, HKEY_USERS
	LPCTSTR subKey	- A null terminated string which is the path to sub key (path) to read
	LPCTSTR name	- A null terminated string which is the name of the key to read
	DWORD defVal	- The default value to return if the registry value cannot be found,
						or is of the wrong type.
---------------------------------------------------------------------------------*/

DWORD getRegDWORD(HKEY key, LPCTSTR subKey, LPCTSTR name, DWORD defVal){

	HKEY tmpKey = NULL;
	DWORD retVal;
	DWORD rvSize = sizeof(DWORD), type;

	if(RegOpenKeyEx(key, subKey, 0, KEY_QUERY_VALUE, &tmpKey) == ERROR_SUCCESS){
		if(RegQueryValueEx(tmpKey, name, 0, &type, (byte *) &retVal, &rvSize) == ERROR_SUCCESS && type == REG_DWORD)
			defVal = retVal;
		RegCloseKey(tmpKey);
	}
	return defVal;
}

/*-------------------------------------------------------------------------------

char *getRegString(HKEY key, LPCTSTR subKey, LPCTSTR name, char *defVal)

	Attempts to get an DWORD from the registry.  If it cannot it returns a default value.

	HKEY key	- An open registry key handle, or one of the pre defined handles;
				HKEY_CLASSES_ROOT, HKEY_CURRENT_CONFIG, HKEY_CURRENT_USER, 
				HKEY_LOCAL_MACHINE, HKEY_USERS
	LPCTSTR subKey	- A null terminated string which is the path to sub key (path) to read
	LPCTSTR name	- A null terminated string which is the name of the key to read
	char *defVal	- The default value to return if the registry value cannot be found,
						or is of the wrong type.
	bool free		- set to true if this method should free defVal
	WARNING!!!!!	- The calling method is responsible for freeing the memory used by the
					  returned string.
---------------------------------------------------------------------------------*/

char *getRegString(HKEY key, LPCTSTR subKey, LPCTSTR name, char *defVal, bool Free){

	HKEY tmpKey = NULL;
	char *retVal;
	DWORD rvSize = 0, type;

	DWORD retStatus = 0;

	if(RegOpenKeyEx(key, subKey, 0, KEY_QUERY_VALUE, &tmpKey) == ERROR_SUCCESS){
		if(RegQueryValueEx(tmpKey, name, 0, 0, NULL, &rvSize) == ERROR_SUCCESS)
			if((retVal = (char *)malloc(rvSize)) != NULL)
				if(RegQueryValueEx(tmpKey, name, 0, &type, (byte *) retVal, &rvSize) == ERROR_SUCCESS && type == REG_SZ){
					if(Free)
						free(defVal);
					defVal = retVal;
				}
		RegCloseKey(tmpKey);
	}

	return defVal;
}

/*-------------------------------------------------------------------------------
bool regSetDword(HKEY key, LPCTSTR subKey, LPCTSTR name, DWORD val)

	Attempts to set a DWORD value in the registry.  If it cannot it returns false.
	if successful returns true.

	HKEY key	- An open registry key handle, or one of the pre defined handles;
				HKEY_CLASSES_ROOT, HKEY_CURRENT_CONFIG, HKEY_CURRENT_USER, 
				HKEY_LOCAL_MACHINE, HKEY_USERS
	LPCTSTR subKey	- A null terminated string which is the path to sub key (path) to set
	LPCTSTR name	- A null terminated string which is the name of the key to set
	DWORD Val	- The value to be set
---------------------------------------------------------------------------------*/

bool regSetDword(HKEY key, LPCTSTR subKey, LPCTSTR name, DWORD val){

	HKEY tmpKey = NULL;
	if(ERROR_SUCCESS != RegCreateKeyEx(key, subKey, 0, NULL, REG_OPTION_NON_VOLATILE, KEY_SET_VALUE, NULL, &tmpKey, NULL))
		return false;

	if(ERROR_SUCCESS != RegSetValueEx(tmpKey, name, 0, REG_DWORD, (const unsigned char *) &val, sizeof(DWORD)))
		return false;

	if(ERROR_SUCCESS != RegCloseKey(tmpKey))
		return false;

	return true;
}

/*-------------------------------------------------------------------------------
bool regSetString(HKEY key, LPCTSTR subKey, LPCTSTR name, char *val, DWORD valSize)

	Attempts to set a String value in the registry.  If it cannot it returns false.
	if successful returns true.

	HKEY key	- An open registry key handle, or one of the pre defined handles;
				HKEY_CLASSES_ROOT, HKEY_CURRENT_CONFIG, HKEY_CURRENT_USER, 
				HKEY_LOCAL_MACHINE, HKEY_USERS
	LPCTSTR subKey	- A null terminated string which is the path to sub key (path) to set
	LPCTSTR name	- A null terminated string which is the name of the key to set
	char *Val	- The value to be set
	DWORD valSize	- The length of the val string.
---------------------------------------------------------------------------------*/

bool regSetString(HKEY key, LPCTSTR subKey, LPCTSTR name, char *val, DWORD valSize){

	HKEY tmpKey = NULL;
	if(ERROR_SUCCESS != RegCreateKeyEx(key, subKey, 0, NULL, REG_OPTION_NON_VOLATILE, KEY_SET_VALUE, NULL, &tmpKey, NULL))
		return false;

	if(ERROR_SUCCESS != RegSetValueEx(tmpKey, name, 0, REG_EXPAND_SZ, (const unsigned char *) val, valSize))
		return false;

	if(ERROR_SUCCESS != RegCloseKey(tmpKey))
		return false;

	return true;
}

/*-------------------------------------------------------------------------------
char * regEnumKeys(HKEY key, LPCTSTR subKey, DWORD index)

	Attempts to return a String which is a subkey of the specified subkey.
	If it cannot it returns null. if successful returns the subkey name.
	Null could be returned because the key does not exist, the subkey does not exist,
	or the required memory for the subkey name could not be allocated.

	HKEY key	- An open registry key handle, or one of the pre defined handles;
				HKEY_CLASSES_ROOT, HKEY_CURRENT_CONFIG, HKEY_CURRENT_USER, 
				HKEY_LOCAL_MACHINE, HKEY_USERS
	LPCTSTR subKey	- A null terminated string which is the path to sub key (path) to enumerate
	DWORD index	- The index of the subkey to retrieve. ( 0 for first subkey)
	WARNING!!!!!	- The calling method is responsible for freeing the memory used by the
			  returned string.
---------------------------------------------------------------------------------*/

char *regEnumKeys(HKEY key, LPCTSTR subKey, DWORD index){
	
	HKEY tmpKey = NULL;
	DWORD keySize = 0;
	char *keyName = NULL;
	long ret;

	if(RegOpenKeyEx(key, subKey, 0, KEY_ENUMERATE_SUB_KEYS, &tmpKey) == ERROR_SUCCESS){
		for(int i=1; true; i++){
			keySize = i * 50 * sizeof(char);
			keyName = (char *) realloc(keyName, keySize);
			if(keyName != NULL){
				ret = RegEnumKeyEx(tmpKey, index, keyName, &keySize, NULL, NULL, NULL, NULL);
				if(ret == ERROR_SUCCESS)
					break;
				else if(ret != ERROR_MORE_DATA){
					free(keyName);
					keyName = NULL;
					break;
				}
			}
		}
		RegCloseKey(tmpKey);
	}
	return keyName;
}

/*-------------------------------------------------------------------------------
DWORD regSubKeyCnt*KEY key, LPCTSTR subKey)

	Attempts to return the count of subkey's of the specified subkey.
	If it cannot it returns a negative number. if successful returns a positive number

	HKEY key	- An open registry key handle, or one of the pre defined handles;
				HKEY_CLASSES_ROOT, HKEY_CURRENT_CONFIG, HKEY_CURRENT_USER, 
				HKEY_LOCAL_MACHINE, HKEY_USERS
	LPCTSTR subKey	- A null terminated string which is the path to sub key (path) to enumerate
---------------------------------------------------------------------------------*/

DWORD regSubKeyCnt(HKEY key, LPCTSTR subKey){

	HKEY tmpKey = NULL;
	DWORD keyCnt = -1;
	if(RegOpenKeyEx(key, subKey, 0, KEY_QUERY_VALUE, &tmpKey) == ERROR_SUCCESS){
		RegQueryInfoKey(tmpKey, NULL, NULL, NULL, &keyCnt, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
		RegCloseKey(tmpKey);
	}
	return keyCnt;
}