/* LogJam, a GTK LiveJournal client.
 * Copyright (C) 2000,2001 Evan Martin <evan@livejournal.com>
 * vim:ts=4:sw=4:
 *
 * $Id: login.c,v 1.8 2001/11/09 16:59:07 martine Exp $
 */

#include "config.h"

#include <gtk/gtk.h>
#include <stdlib.h> /* atoi */
#include "dotconf.h"
#include "users.h"
#include "util.h"
#include "network.h"
#include "lj.h"
#include "menu.h"

#include "../pixmaps/livejournal.xpm"

typedef struct {
	ljwin *ljw;
	GtkWidget *win;
	GtkWidget *eusername, *epassword, *clogin, *cfastserver;
} login_dlg;

static void do_login(login_dlg *ldlg);

static void 
username_changed(GtkWidget *w, login_dlg *ldlg) {
	char *text;
	user *u;

	text = gtk_entry_get_text(GTK_ENTRY(w));
	u = conf_user_by_name(&conf, text);
	if (u == NULL) {
		gtk_entry_set_text(GTK_ENTRY(ldlg->epassword), "");
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ldlg->cfastserver), FALSE);
	} else {
		gtk_entry_set_text(GTK_ENTRY(ldlg->epassword), u->password);
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(ldlg->cfastserver), u->fastserver);
	}
}

void login_clicked(GtkWidget *w, login_dlg *ldlg) {
	string_replace(&conf.username, 
			gtk_editable_get_chars(GTK_EDITABLE(ldlg->eusername), 0, -1));

	string_replace(&conf.password, 
			gtk_editable_get_chars(GTK_EDITABLE(ldlg->epassword), 0, -1));

	conf.fastserver = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(ldlg->cfastserver));
	
	do_login(ldlg);
}

static gboolean
autolog_idle_cb(login_dlg *ldlg) {
	gdk_threads_enter();
	login_clicked(NULL, ldlg);
	gdk_threads_leave();
	return FALSE;
}

void set_popdown_strings(login_dlg *ldlg) {
	GList *l, *list = NULL;
	user *u;

	for (l = conf.users; l != NULL; l = l->next) {
		u = l->data;
		list = g_list_append(list, u->username);
	}
	if (list)
		gtk_combo_set_popdown_strings(GTK_COMBO(ldlg->clogin), list);
}

void edit_users(GtkWidget *w, login_dlg *ldlg) {
	user_dlg_run(ldlg->win);
	set_popdown_strings(ldlg);
}

static void
fastserver_help(GtkWidget *w, login_dlg *ldlg) {
	lj_messagebox(ldlg->win, "Fast Servers",
	"Paid users of LiveJournal.com can be redirected to our faster servers.\n"
	"If free users check this option, LiveJournal will still direct you to the free servers.\n"
	"LogJam autodetects this setting after you've logged in once.");
}

static GtkWidget*
make_login_table(login_dlg *ldlg) {
	GtkWidget *table;
	GtkWidget *hbox;
	GtkWidget *b;

	table = lj_table_new(3, 2);

	ldlg->clogin = gtk_combo_new();
	gtk_widget_set_usize(ldlg->clogin, 50, -1);
	ldlg->eusername = GTK_COMBO(ldlg->clogin)->entry;
	gtk_combo_disable_activate(GTK_COMBO(ldlg->clogin)); 
	set_popdown_strings(ldlg);
	lj_table_label(GTK_TABLE(table),   0, "User Name:");
	lj_table_content(GTK_TABLE(table), 0, ldlg->clogin);

	ldlg->epassword = gtk_entry_new(); 
	gtk_widget_set_usize(ldlg->epassword, 100, -1);
	gtk_entry_set_visibility(GTK_ENTRY(ldlg->epassword), FALSE);
	lj_table_label(GTK_TABLE(table),   1, "Password:");
	lj_table_content(GTK_TABLE(table), 1, ldlg->epassword);

	hbox = gtk_hbox_new(FALSE, 5);
	ldlg->cfastserver = gtk_check_button_new_with_label("Use fast server");
	gtk_box_pack_start(GTK_BOX(hbox), ldlg->cfastserver, TRUE, TRUE, 0);
	b = gtk_button_new_with_label(" ? ");
	gtk_signal_connect(GTK_OBJECT(b), "clicked",
			GTK_SIGNAL_FUNC(fastserver_help), ldlg);
	gtk_box_pack_start(GTK_BOX(hbox), b, FALSE, FALSE, 0);
	lj_table_fillrow(GTK_TABLE(table), 2, hbox);

	gtk_signal_connect(GTK_OBJECT(ldlg->eusername), "changed",
			GTK_SIGNAL_FUNC(username_changed), ldlg);
	if (conf.username)
		gtk_entry_set_text(GTK_ENTRY(ldlg->eusername), conf.username);
	username_changed(ldlg->eusername, ldlg);

	/* pressing enter should login */
	gtk_signal_connect(GTK_OBJECT(ldlg->eusername), "activate",
			GTK_SIGNAL_FUNC(login_clicked), ldlg);
	gtk_signal_connect(GTK_OBJECT(ldlg->epassword), "activate",
			GTK_SIGNAL_FUNC(login_clicked), ldlg);

	return table;
}

static GtkWidget*
make_pixmap(login_dlg *ldlg) {
	GtkWidget *align;
	GdkPixmap *pix;
	GdkBitmap *bit;
	GtkWidget *pixmap;

	pix = gdk_pixmap_colormap_create_from_xpm_d(ldlg->win->window, 
			NULL, &bit, NULL, livejournal_xpm);
	pixmap = gtk_pixmap_new(pix, bit);
	align = gtk_alignment_new(0.5, 0.5, 1, 1);
	gtk_container_add(GTK_CONTAINER(align), pixmap);
	return align;
}

void
login_dlg_run(ljwin *ljw, int autolog) {
	login_dlg ldlg_actual = {0}, *ldlg = &ldlg_actual;
	GtkWidget *align, *hbox, *button;

	ldlg->ljw = ljw;
	ldlg->win = lj_dialog_new(ljw->win, PROGRAMNAME " Login", -1, -1);
	gtk_signal_connect(GTK_OBJECT(ldlg->win), "destroy",
			GTK_SIGNAL_FUNC(gtk_main_quit), NULL);

	gtk_widget_realize(ldlg->win);

	hbox = gtk_hbox_new(FALSE, 5); 

	gtk_box_pack_start(GTK_BOX(hbox), make_pixmap(ldlg), FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), make_login_table(ldlg), FALSE, FALSE, 0);
	align = gtk_alignment_new(.5, .5, 1, 1);
	gtk_container_add(GTK_CONTAINER(align), hbox);

	lj_dialog_set_contents(ldlg->win, align);

	button = gtk_button_new_with_label("  Users...  ");
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			GTK_SIGNAL_FUNC(edit_users), ldlg);
	lj_dialog_add_button(ldlg->win, button);

	button = lj_dialog_add_okcancel(ldlg->win, "  Login  ",
			login_clicked, ldlg);

	gtk_widget_show(ldlg->win);

	gtk_widget_grab_focus(ldlg->eusername);

	if (autolog) {
		g_idle_add((GSourceFunc)autolog_idle_cb, ldlg);
	}

	gtk_main();
}

static void 
do_login(login_dlg *ldlg) {
	GHashTable *request, *result;
	char *value;
	user *u;

	request = net_request_new("login");

	g_hash_table_insert(request, g_strdup("clientversion"), 
			g_strdup("Gtk-LogJam/" VERSION));
	g_hash_table_insert(request, g_strdup("getmoods"), 
			g_strdup_printf("%d", mood_get_last_cached(&conf)));
	g_hash_table_insert(request, g_strdup("getmenus"), 
			g_strdup("1"));
	g_hash_table_insert(request, g_strdup("getpickws"), 
			g_strdup("1"));

	result = net_request_run(ldlg->win, "Logging in...", request);
	hash_destroy(request);

	if (!net_request_succeeded(result)) {
		conf.loginok = FALSE;
		if (result) hash_destroy(result);
		return;
	}

	conf.loginok = TRUE;

	if (conf.fullname) {
		g_free(conf.fullname);
		conf.fullname = NULL;
	}

	value = g_hash_table_lookup(result, "name");
	if (value) {
		conf.fullname = g_strdup(value);
	}

	value = g_hash_table_lookup(result, "fastserver");
	if (value && strcmp(value, "1") == 0) {
		conf.fastserver = TRUE;
	} else {
		conf.fastserver = FALSE;
	}

	u = conf_user_by_name(&conf, conf.username);
	if (u) {
		u->fastserver = conf.fastserver; /* remember the fastserver of this guy*/
	}

	/* grab moods */
	if ((value = g_hash_table_lookup(result, "mood_count")) != NULL) {
		int moodcount = atoi(value);
		int i;
		char buf[30];

		for (i = 1; i < moodcount+1; i++) { /* 1-based list */
			mood *m;
			m = g_new0(mood, 1);

			g_snprintf(buf, 30, "mood_%d_id", i);
			value = g_hash_table_lookup(result, buf);
			m->id = atoi(value);

			g_snprintf(buf, 30, "mood_%d_name", i);
			value = g_hash_table_lookup(result, buf);
			m->name = g_strdup(value);

			conf.moods = g_list_insert_sorted(conf.moods, m, 
					(GCompareFunc)mood_compare_alpha);
		}
	}

	/* grab picture keywords */
	if ((value = g_hash_table_lookup(result, "pickw_count")) != NULL) {
		int pickwcount = atoi(value);
		int i;
		char buf[30];

		for (i = 1; i < pickwcount+1; i++) { /* 1-based list */
			g_snprintf(buf, 30, "pickw_%d", i);
			value = g_hash_table_lookup(result, buf);
			conf.pickws = g_list_append(conf.pickws, g_strdup(value));
		}
	}

	/* grab usejournal, web links, etc */
	menu_update(ldlg->ljw, result);

	/* grab friend groups */
	if ((value = g_hash_table_lookup(result, "frgrp_maxnum")) != NULL) {
		int fgmax = atoi(value);
		int i;
		char buf[30];
		for (i = 1; i <= fgmax; i++) {
			g_snprintf(buf, 30, "frgrp_%d_name", i);
			value = g_hash_table_lookup(result, buf);
			if (value) {
				friendgroup *fg;
				fg = g_new0(friendgroup, 1);
				fg->id = i;
				fg->name = g_strdup(value);

				conf.friendgroups = g_list_append(conf.friendgroups, fg);
			}
		}
	}

	/* if there's a message, display it. */
	if ((value = g_hash_table_lookup(result, "message")) != NULL)
		lj_messagebox(ldlg->win, "LiveJournal Message", value);

	hash_destroy(result);
	gtk_widget_destroy(ldlg->win);
}


