#include <stdio.h>
#include <assert.h>
#include <stdarg.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <limits.h>
#include <errno.h>
#include <string.h>

#define TLMP_VERSION	"1.0"

enum FEED_TOKEN {
	T_STR,		// C++ stuff. Mostly go through without much checking
	T_MARK,		// TLMP tag
	T_SPC,		// Special character { } ; used to handle scope
	T_COMMENT,	// This is a comment
	T_EOF		// End of file
};

class FEED{
	const char *fname;
	FILE *fin;
	char buf[1000];
	char *s;
	int noline;
	bool comment_mode;
	/*~PROTOBEG~ FEED */
public:
	FEED (const char *_fname, FILE *_fin);
	void fill (void);
	char getcar (void);
	const char *getfname (void)const;
	int getnoline (void)const;
	FEED_TOKEN gettoken (char *str, char *arg1, char *arg2);
	void printerr (const char *s, ...);
	/*~PROTOEND~ FEED */
};

PUBLIC FEED::FEED(const char *_fname, FILE *_fin)
{
	fname = _fname;
	fin = _fin;
	noline = 0;
	comment_mode = false;
	fill();
}

PUBLIC void FEED::fill()
{
	s = buf;
	buf[0] = '\0';
	fgets (buf,sizeof(buf)-1,fin);
	noline++;
}

PUBLIC int FEED::getnoline() const
{
	return noline;
}
PUBLIC const char *FEED::getfname() const
{
	return fname;
}

PUBLIC void FEED::printerr (const char *s, ...)
{
	fprintf (stderr,"File %s, line %d: ",getfname(),getnoline());
	va_list list;
	va_start (list,s);
	vfprintf (stderr,s,list);
	va_end (list);
}

PUBLIC char FEED::getcar()
{
	char ret = '\0';
	if (*s == '\0') fill();
	if (*s != '\0'){
		ret = *s++;
	}
	return ret;
}

static char *tlcc_copystr(char *dst, char *s)
{
	char spccar = *s++;
	*dst++ = spccar;
	while (*s != '\0'){
		if (*s == spccar){
			*dst++ = *s++;
			break;
		}else if (*s == '\\'){
			*dst++ = *s++;
			if (*s != '\0') *dst++ = *s++;
		}else{
			*dst++ = *s++;
		}
	}
	*dst = '\0';
	return s;
}

PUBLIC FEED_TOKEN FEED::gettoken (char *str, char *arg1, char *arg2)
{
	arg1[0] = arg2[0] = str[0] = '\0';
	FEED_TOKEN ret = T_EOF;
	if (*s == '\0') fill();
	char *start = str;
	while (*s != '\0'){
		if (comment_mode){
			while (*s != '\0'
				&& !(s[0] == '*' && s[1] == '/')) *str++ = *s++;
		}else{
			while (*s != '\0'
				&& *s != '<'
				&& *s != '/'
				&& *s != '*'
				&& *s != '"'
				&& *s != '\''
				&& *s != '{'
				&& *s != '}'
				&& *s != ';') *str++ = *s++;
		}
		*str = '\0';
		if (str > start){
			ret = comment_mode ? T_COMMENT : T_STR;
			break;
		}else if (s[0] == '{' || s[0] == '}' || s[0] == ';'){
			*str++ = *s++;
			*str = '\0';
			ret = T_SPC;
			break;
		}else if (s[0] == '/' && s[1] == '/'){
			while (*s != '\0') *str++ = *s++;
			*str = '\0';
			ret = T_COMMENT;
			break;
		}else if (s[0] == '/' && s[1] == '*'){
			*str++ = '/';
			*str++ = '*';
			*str = '\0';
			s += 2;
			ret = T_COMMENT;
			comment_mode = true;
			break;
		}else if (s[0] == '*' && s[1] == '/'){
			*str++ = '*';
			*str++ = '/';
			*str = '\0';
			s += 2;
			ret = T_COMMENT;
			comment_mode = false;
			break;
		}else if (s[0] == '/' || s[0] == '*'){
			*str++ = *s++;
			*str = '\0';
			ret = T_STR;
			break;
		}else if (*s == '"' || *s == '\''){
			s = tlcc_copystr (str,s);
			ret = T_STR;
			break;
		}else if (*s == '<'){
			if (comment_mode){
				*str++ = *s++;
			}else if (isalpha(s[1])){
				char id1[1000],id2[1000],id3[1000];
				char *pt = id1;
				char *ss = s+1;
				while (isalnum(*ss) || *s == '_'){
					*pt++ = *ss++;
				}
				*pt = '\0';
				while (isspace (*ss)) ss++;
				if (*ss == '>'){
					strcpy (str,id1);
					ret = T_MARK;
					s = ss+1;
					break;
				}else if (isalpha(*ss)){
					pt = id2;
					while (isalnum(*ss) || *ss == '_'){
						*pt++ = *ss++;
					}
					*pt = '\0';
					while (isspace (*ss)) ss++;
					if (*ss == '>'){
						ret = T_MARK;
						strcpy (str,id1);
						strcpy (arg1,id2);
						s = ss+1;
						break;
					}else if (isalpha(*ss)){
						pt = id3;
						while (isalnum(*ss) || *ss == '_'){
							*pt++ = *ss++;
						}
						*pt = '\0';
						while (isspace (*ss)) ss++;
						if (*ss == '>'){
							ret = T_MARK;
							strcpy (str,id1);
							strcpy (arg1,id2);
							strcpy (arg2,id3);
							s = ss+1;
							break;
						}else{
							while (s < ss) *str++ = *s++;
						}
					}else{
						while (s < ss) *str++ = *s++;
					}
				}else{
					while (s < ss) *str++ = *s++;
				}
			}else if (s[1] == '/'){
				*str++ = '/';
				s += 2;
				while (isalpha(*s)) *str++ = *s++;
				*str = '\0';
				while (isspace (*s)) s++;
				if (*s == '>'){
					s++;
				}else{
					fprintf (stderr,"File %s: Missing closing > at line %d\n"
						,fname,noline);
				}
				ret = T_MARK;
				break;
			}else{
				*str++ = *s++;
				if (*s == '\0'){
					*str = '\0';
					ret = T_STR;
					break;
				}
			}
		}
	}
	//fprintf (stderr,"%d: %s\n",noline,start);
	return ret;
}


class BUFSTR{
	char *buf;
	int size;
	int pos;
	/*~PROTOBEG~ BUFSTR */
public:
	BUFSTR (void);
	void append (char car);
	void append (const char *s);
	void appendf (const char *ctl, ...);
	const char *getbuf (void);
	void reset (void);
	void setnoline (FEED&feed);
	~BUFSTR (void);
	/*~PROTOEND~ BUFSTR */
};

PUBLIC BUFSTR::BUFSTR()
{
	buf = (char*)malloc(1000);
	assert (buf != NULL);
	size = 1000;
	pos = 0;
	buf[0] = '\0';
}

PUBLIC BUFSTR::~BUFSTR()
{
	free (buf);
}

PUBLIC void BUFSTR::append (const char *s)
{
	int len = strlen (s);
	// fprintf (stderr,"%p %p size %d len %d pos %d\n",buf,s,size,len,pos);
	if (pos + len + 1 > size-2){
		size += 1000;
		if (len >= 500) size += len;
		buf = (char*)realloc (buf,size);
		assert (buf != NULL);
	}
	strcpy (buf+pos,s);
	pos += len;
}

PUBLIC void BUFSTR::reset ()
{
	buf[0] = '\0';
	pos = 0;
}

PUBLIC void BUFSTR::setnoline (FEED &feed)
{
	appendf ("\n#line %d \"%s\"\n",feed.getnoline(),feed.getfname());
}

PUBLIC void BUFSTR::appendf (const char *ctl, ...)
{
	va_list list;
	va_start (list,ctl);
	char tmp[1000];
	vsnprintf (tmp,sizeof(tmp)-1,ctl,list);
	va_end (list);
	append (tmp);
}

PUBLIC void BUFSTR::append (char car)
{
	char tmp[2];
	tmp[0] = car;
	tmp[1] = '\0';
	append (tmp);
}

PUBLIC const char *BUFSTR::getbuf()
{
	return buf;
}
enum COMPSTATE {
	COMPSTATE_NONE,		// Not in a
	COMPSTATE_MOD,		// Inside a <mod> </mod> block
	COMPSTATE_GLOCAL,	// Inside <glocal> block
	COMPSTATE_CALL,		// Inside <call> block, expecting <f>
	COMPSTATE_OBJ,		// Inside <obj> block, expecting <f>
	COMPSTATE_F,		// Inside <f>, expecting other <call> or <obj>
};

class MODSTATE{
public:
	int nomod;
	int nocall;
	bool in_module;
	int noglocal;
	COMPSTATE states[100];
	int nbstates;
	COMPSTATE state;
	bool some_errors;
	/*~PROTOBEG~ MODSTATE */
public:
	MODSTATE (void);
	void outofcontext (FEED&feed);
	int popstate (COMPSTATE expect, FEED&feed);
	void pushstate (COMPSTATE st, FEED&feed);
	/*~PROTOEND~ MODSTATE */
};

PUBLIC MODSTATE::MODSTATE()
{
	nomod = 0;
	nocall = 0;
	noglocal = 0;
	in_module = false;
	state = COMPSTATE_NONE;
	nbstates = 0;
	some_errors = false;
}

PUBLIC void MODSTATE::outofcontext (FEED &feed)
{
	feed.printerr ("Block is used out of context\n");
	some_errors = true;
}
	


PUBLIC void MODSTATE::pushstate (COMPSTATE st, FEED &feed)
{
	if (st == COMPSTATE_F){
		if (state != COMPSTATE_CALL && state != COMPSTATE_OBJ){
			outofcontext (feed);
		}
	}else if (st == COMPSTATE_CALL){
		if (state != COMPSTATE_MOD && state != COMPSTATE_F){
			outofcontext (feed);
		}
	}else if (st == COMPSTATE_GLOCAL){
		if (state != COMPSTATE_MOD && state != COMPSTATE_F){
			outofcontext (feed);
		}
	}
	states[nbstates++] = state;
	state = st;
}
PUBLIC int MODSTATE::popstate (COMPSTATE expect, FEED &feed)
{
	int ret = -1;
	if (nbstates == 0){
		feed.printerr ("Can't end a section, no sectio opened\n");
		some_errors = true;
	}else if (state != expect){
		feed.printerr ("Closing section missmatch\n");
	}else{
		state = states[--nbstates];
		ret = 0;
	}
	return ret;
}


static char scope_prefix[200];
static char glocal_prefix[200];

/*
	Replace all the non digit and alpha characters by a _
*/
static void tlcc_filterid (char *pt)
{
	while (*pt != '\0'){
		if (!isalpha(*pt) && !isdigit(*pt)) *pt = '_';
		pt++;
	}
}
/*
	Compute the "somewhat" unique id prefix for scope objects and glocals.
	We are using both the source file name and the current working directory
	to create a unique identifier. Almost unique. It is assumed that a source
	A in directory B is a rather unique condition.

	Ideally, we need a way to declare classes with source scope.
	Is there a way to achieve that in C++ ?
*/
static void tlcc_setscopeprefix (const char *fname)
{
	char cwd[PATH_MAX];
	if (getcwd(cwd,sizeof(cwd)-1)!=NULL){
		char *pt = strrchr(cwd,'/');
		if (pt != NULL){
			pt++;
		}else{
			pt = cwd;
		}
		snprintf(scope_prefix,sizeof(scope_prefix)-1,"__sc_%s_%s"
			,pt,fname);
		tlcc_filterid (scope_prefix);
		snprintf(glocal_prefix,sizeof(glocal_prefix)-1,"__gl_%s_%s"
			,pt,fname);
		tlcc_filterid (glocal_prefix);
	}else{
		fprintf (stderr,"Can't get the current working directory\n");
		exit (-1);
	}
}

// Put some code to insure that the module exists or that
// the header is include. If not, strange C++ errors are printed
static void tlcc_checkifdef(
	BUFSTR &insert,
	const char *module,
	const char *type)		// "class" or "module"
{
	insert.appendf ("#ifndef _TLMP_%s\n",module);
	insert.appendf ("\t#error TLMP %s %s unknown\n",type,module);
	insert.appendf ("#endif\n");
}

// One variable member of a glocal section
class GLOCAL_VAR{
	char *name;
	GLOCAL_VAR *next;
	/*~PROTOBEG~ GLOCAL_VAR */
public:
	GLOCAL_VAR (const char *_name, GLOCAL_VAR *_next);
	const char *getname (void)const;
	GLOCAL_VAR *getnext (void)const;
	~GLOCAL_VAR (void);
	/*~PROTOEND~ GLOCAL_VAR */
};

PUBLIC GLOCAL_VAR::GLOCAL_VAR(const char *_name, GLOCAL_VAR *_next)
{
	next = _next;
	name = strdup(_name);
}

PUBLIC GLOCAL_VAR::~GLOCAL_VAR()
{
	free (name);
}

PUBLIC const char *GLOCAL_VAR::getname() const
{
	return name;
}

PUBLIC GLOCAL_VAR *GLOCAL_VAR::getnext() const
{
	return next;
}

// Variables member of a glocal section and pointer
// to parent glocal sections.
class GLOCAL_VARS{
	GLOCAL_VARS *parent;
	GLOCAL_VAR *var;
	int glocal_level;
	/*~PROTOBEG~ GLOCAL_VARS */
public:
	GLOCAL_VARS (GLOCAL_VARS *_parent,
		 int _glocal_level);
private:
	void add (const char *name);
public:
	int getglevel (void)const;
	GLOCAL_VARS *getparent (void)const;
	int getscope (const char *name);
	void parse (const char *line);
	~GLOCAL_VARS (void);
	/*~PROTOEND~ GLOCAL_VARS */
};

PUBLIC GLOCAL_VARS::GLOCAL_VARS (GLOCAL_VARS *_parent, int _glocal_level)
{
	parent = _parent;
	var = NULL;
	glocal_level = _glocal_level;
}

PUBLIC GLOCAL_VARS::~GLOCAL_VARS ()
{
	while (var != NULL){
		GLOCAL_VAR *next = var->getnext();
		delete var;
		var = next;
	}
}

PUBLIC GLOCAL_VARS* GLOCAL_VARS::getparent() const
{
	return parent;
}

PUBLIC int GLOCAL_VARS::getglevel() const
{
	return glocal_level;
}

PRIVATE void GLOCAL_VARS::add (const char *name)
{
	var = new GLOCAL_VAR(name,var);
}

/*
	Find the distance of a variable from the current (closest) glocal
	block.

	Return 0 if it is in the current glocal block, 1 if it is in the one
	just before the current one and -1 if the variable is not in
	any glocal blocks.
*/
PUBLIC int GLOCAL_VARS::getscope(const char *name)
{
	int ret = -1;
	GLOCAL_VAR *p = var;
	while (p != NULL){
		if (strcmp(p->getname(),name)==0){
			ret = 0;
			break;
		}
		p = p->getnext();
	}
	if (ret == -1 && parent != NULL){
		ret = parent->getscope(name);
		if (ret != -1) ret++;
	}
	return ret;
}

static const char *tlcc_copyid (char *dst, const char *src, int maxlen)
{
	*dst = '\0';
	while (!isalpha(*src) && *src != '\0') src++;
	if (isalpha(*src)){
		int len = 0;
		while ((isalpha (*src) || isdigit(*src) || *src == '_')
			&& len < maxlen){
			len++;
			*dst++ = *src++;
		}
		*dst = '\0';
	}
	return src;
}

PUBLIC void GLOCAL_VARS::parse (const char *line)
{
	while (*line != '\0'){
		char word[1000];
		line = tlcc_copyid (word,line,sizeof(word)-1);
		// This is really minimal. We pick any valid C++ identifier
		// be it a keyword or a variable name and add it to the list
		// of known variable.
		// If the user enter glocal.const for example, we will correctly
		// find the place where const was used inside a glocal section
		// but the compiler won't be fooled.
		if (word[0] != '\0') add (word);
	}
}

/*
	Replace the sequences glocal.var in s1 with the proper
	glocal.glocal...var sequence by searching the variable var
	in the nested glocal blocks.
*/
static void tlcc_fixglocal(
	GLOCAL_VARS *vars,
	char *s1)
{
	if (vars != NULL){
		char *pt = s1;
		while ((pt = strstr(pt,"glocal."))!=NULL){
			char word[1000];
			pt += 7;	// Skil glocal.
			tlcc_copyid(word,pt,sizeof(word)-1);
			if (word[0] != '\0'){
				// We have to locate in which glocal block
				// this variable was located
				int level = vars->getscope(word);
				// fprintf (stderr,"glocal scope %s %d\n",word,level);
				if (level > 0){
					// No need to fix anything for a level 0
					// For level > 0, we must insert as many
					// .glocal as needed
					int len = strlen(pt);
					memmove (pt+level*7,pt,len+1);
					for (int i=0; i<level; i++){
						memmove (pt+i*7,"glocal.",7);
					}
					pt += level*7;
				}
			}
		}
	}
}

static void tlcc_getendparm (GLOCAL_VARS *vars, FEED &feed, BUFSTR &cur)
{
	int level = 1;
	bool comma = false;
	char txt[1000];
	char *pttxt = txt;
	while ((unsigned)(pttxt-txt)<sizeof(txt)-2){
		char carac = feed.getcar();
		if (carac == ')'){
			*pttxt++ = carac;
			level--;
			if (level == 0) break;
		}else{
			if (!isspace(carac)){
				if (!comma){
					*pttxt++ = ',';
					comma = true;
				}
			}
			*pttxt++ = carac;
		}
	}
	*pttxt = '\0';
	tlcc_fixglocal (vars,txt);
	cur.append (txt);
}
/*
	Get the (parms); sequence following a <call module> sequence
*/
static void tlcc_getparm (GLOCAL_VARS *vars, FEED &feed, BUFSTR &cur)
{
	char carac;
	while ((carac = feed.getcar()) <= ' ') cur.append (carac);
	if (carac == '('){
		tlcc_getendparm(vars,feed,cur);
	}
}

static void tlcc_delvars (
	GLOCAL_VARS *&vars,
	GLOCAL_VARS *original_vars,
	BUFSTR &mainbuf)
{
	while (vars != original_vars){
		int level1 = vars->getglevel();
		GLOCAL_VARS *parent = vars->getparent();
		delete vars;
		vars = parent;
		int level0 = 0;
		if (vars != NULL) level0 = vars->getglevel();
		char buf[100];
		sprintf (buf,"\t}\t// For nested glocals %d -> %d\n",level1,level0);
		mainbuf.append (buf);
	}
}


class SCOPE{
public:
	GLOCAL_VARS *original_vars;
	bool original_auto_glocal;
	SCOPE *prec;
	/*~PROTOBEG~ SCOPE */
public:
	SCOPE (GLOCAL_VARS *_vars,
		 SCOPE *_prec,
		 bool auto_glocal);
	SCOPE *revert (GLOCAL_VARS *&vars,
		 BUFSTR&mainbuf,
		 bool&auto_glocal);
	/*~PROTOEND~ SCOPE */
};

PUBLIC SCOPE::SCOPE (GLOCAL_VARS *_vars, SCOPE *_prec, bool auto_glocal)
{
	prec = _prec;
	original_vars = _vars;
	original_auto_glocal = auto_glocal;
}

/*
	Cleanup at the end of the SCOPE. Return the previous object
*/
PUBLIC SCOPE *SCOPE::revert(GLOCAL_VARS *&vars, BUFSTR &mainbuf, bool &auto_glocal)
{
	// fprintf (stderr,"revert %p %p\n",vars,original_vars);
	tlcc_delvars (vars,original_vars,mainbuf);
	auto_glocal = original_auto_glocal;
	return prec;
}
/*
	Generate the glocal declaration
*/
static void tlcc_startglocal(
	FEED &feed,
	MODSTATE &st,
	int &glocal_level,
	BUFSTR &insert,
	BUFSTR *&cur,
	GLOCAL_VARS *&vars,
	bool nested_glocal,		// There is already one glocal defined in this
							// block, so the new one points to it.
	const char *inmodule)	// component name (call or obj) currently process
							// or NULL
{
	st.pushstate (COMPSTATE_GLOCAL,feed);
	glocal_level = ++st.noglocal;
	insert.appendf ("struct %s_GLOCAL%d_%d{\n"
		,glocal_prefix,st.nomod,glocal_level);
	if (vars != NULL){
		int parent_glocal = vars->getglevel();
		insert.appendf ("\t%s_GLOCAL%d_%d &glocal;\n"
			,glocal_prefix,st.nomod,parent_glocal);
		if (nested_glocal){
			// We must reference the previous glocal, not the module
			// Since the previous, or its ancestor has already done this
			inmodule = NULL;
		}
		if (inmodule){
			insert.appendf ("\t_F_%s &%s;\n",inmodule,inmodule);
			insert.appendf ("\t%s_GLOCAL%d_%d(%s_GLOCAL%d_%d &gl,_F_%s *_c)\n"
					"\t\t: glocal(gl),%s(*_c)\n"
					"\t{}\n"
				,glocal_prefix
				,st.nomod,glocal_level
				,glocal_prefix
				,st.nomod,parent_glocal
				,inmodule,inmodule);
		}else{
			insert.appendf ("\t%s_GLOCAL%d_%d(%s_GLOCAL%d_%d &gl)\n"
					"\t\t: glocal(gl)\n"
					"\t{}\n"
				,glocal_prefix
				,st.nomod,glocal_level
				,glocal_prefix
				,st.nomod,parent_glocal);
		}
	}else if (inmodule != NULL){
		insert.appendf ("\t_F_%s &%s;\n",inmodule,inmodule);
		insert.appendf ("\t%s_GLOCAL%d_%d(_F_%s *_c)\n"
				"\t\t: %s(*_c)\n"
				"\t{}\n"
			,glocal_prefix
			,st.nomod,glocal_level
			,inmodule
			,inmodule);
	}
	cur = &insert;
	vars = new GLOCAL_VARS (vars,glocal_level);
	if (inmodule != NULL) vars->parse (inmodule);
}

/*
	End the glocal declaration
*/
static void tlcc_endglocal (
	FEED &feed,
	MODSTATE &st,
	BUFSTR &insert,
	BUFSTR &funcdef,
	BUFSTR &funccore,
	BUFSTR *&cur,
	GLOCAL_VARS *vars,
	const char *inmodule,
	bool nested_glocal)
{
	int glocal_level = vars->getglevel();
	st.popstate (COMPSTATE_GLOCAL,feed);
	insert.append ("};\n");
	funcdef.append ("\t{\t// Inserted to support nested glocals\n");
	if (vars->getparent() != NULL){
		if (nested_glocal){
			funcdef.appendf ("\t%s_GLOCAL%d_%d glocal(glocal);\n"
				,glocal_prefix,st.nomod,glocal_level);
		}else if (inmodule != NULL){
			funcdef.appendf ("\t%s_GLOCAL%d_%d glocal(this->glocal,this);\n"
				,glocal_prefix,st.nomod,glocal_level);
		}else{
			funcdef.appendf ("\t%s_GLOCAL%d_%d glocal(this->glocal);\n"
				,glocal_prefix,st.nomod,glocal_level);
		}
	}else if (inmodule != NULL){
		funcdef.appendf ("\t%s_GLOCAL%d_%d glocal(this);\n"
			,glocal_prefix,st.nomod,glocal_level);
	}else{
		funcdef.appendf ("\t%s_GLOCAL%d_%d glocal;\n"
			,glocal_prefix,st.nomod,glocal_level);
	}
	cur = &funccore;
	funccore.setnoline (feed);
}

static int tlcc_do (
	FEED &feed,
	BUFSTR &insert,		// Stuff which will go at the start of the C++
						// source
	BUFSTR &funcdef,	// Function definitions
	BUFSTR &mainbuf,	// Code copied and generated
	MODSTATE &st,
	FILE *fout,
	int glocal_level,
	GLOCAL_VARS *vars,
	const char *inmodule)	// component name (call or obj) currently process
							// or NULL
{
	int ret = 0;
	BUFSTR glocal;			// Definition of all glocals
	BUFSTR callobj;			// Definition and initialisation of the
							// scope object passed to module
							// It is generated right after the glocal
							// instance
	BUFSTR funccore;		// Will contain the body of a function
							// after the initialisation and the glocal
	BUFSTR callf;			// Will contains the <f > sections
							// callobj contains the definition of the
							// context object and callf contains the
							// implementation of each context object
							// member functions.
	//BUFSTR *cur = &mainbuf;
	BUFSTR *cur = &funccore;
	char module[100];
	module[0] = '\0';
	char s1[1000],s2[1000],s3[1000];
	FEED_TOKEN token;
	int nocall = 0;
	GLOCAL_VARS *original_vars = vars;
	/*
		auto_glocal will insert a glocal section before the first <call or <obj
		inside a functag. The purpose is to have access to the component
		context like this without having to declare an empty <glocal> section.

		<call comp1>();
		<f tag>
			// auto glocal section is inserted here, before the call
			<call comp2>();
			<f tag>
				glocal.comp1.some_function();
			</f>
			</call>
		</f>
		</call>

		So if no glocal section has been defined prior to the call
		one is inserted.
	*/
	bool auto_glocal = inmodule != NULL;
	SCOPE *scope = new SCOPE (vars,NULL,auto_glocal);
	while ((token=feed.gettoken(s1,s2,s3))!=T_EOF){
		// fprintf (stderr,"gettoken %d :%s: :%s:\n",token,s1,s2);
		if (token == T_STR){
			if (st.state == COMPSTATE_GLOCAL){
				assert (vars != NULL);
				vars->parse (s1);
			}else{
				tlcc_fixglocal (vars,s1);
			}
			cur->append (s1);
		}else if (token == T_SPC){
			cur->append (s1);
			if (st.state != COMPSTATE_GLOCAL){
				if (s1[0] ==  '{'){
					scope = new SCOPE (vars,scope,auto_glocal);
				}else if (s1[0] ==  '}'){
					if (scope == NULL){
						feed.printerr ("Nesting error\n");
					}else{
						SCOPE *old = scope;
						scope = scope->revert(vars,*cur,auto_glocal);
						delete old;
					}
				// Nothing special to do for ;
				// }else if (s1[0] ==  ';'){
				}
				if (cur != &funcdef){
					funcdef.append (cur->getbuf());
					cur->reset();
				}
			}
		}else if (token == T_COMMENT){
			cur->append (s1);
		}else if (strcmp(s1,"glocal")==0){
			tlcc_startglocal(feed,st,glocal_level,insert,cur
				,vars,vars!=original_vars,inmodule);
			auto_glocal = false;
		}else if (strcmp(s1,"/glocal")==0){
			tlcc_endglocal (feed,st,insert,funcdef,funccore,cur
				,vars,inmodule,vars->getparent()!=original_vars);
		}else if (strcmp (s1,"mod")==0){
			st.pushstate (COMPSTATE_MOD,feed);
			insert.append (glocal.getbuf());
			insert.append (callobj.getbuf());
			insert.append (callf.getbuf());
			mainbuf.append (funccore.getbuf());
			fputs (insert.getbuf(),fout);
			fputs (funcdef.getbuf(),fout);
			fputs (mainbuf.getbuf(),fout);
			insert.reset();
			mainbuf.reset();
			glocal.reset();
			funcdef.reset();
			funccore.reset();
			callobj.reset();
			callf.reset();

			cur = &funcdef;
			funcdef.setnoline (feed);
			st.in_module = true;
			st.nomod++;
		}else if (strcmp (s1,"/mod")==0){
			cur->setnoline (feed);
			st.popstate (COMPSTATE_MOD,feed);
			glocal_level = 0;
			st.in_module = false;
			st.noglocal = 0;
		}else if (strcmp (s1,"call")==0 ){
			if (auto_glocal){
				// We insert a glocal so the component may be referenced
				// using glocal.component syntax
				auto_glocal = false;
				tlcc_startglocal(feed,st,glocal_level,insert,cur
					,vars,false,inmodule);
				tlcc_endglocal (feed,st,insert,funcdef,funccore,cur
					,vars,inmodule,false);
			}
			st.pushstate (COMPSTATE_CALL,feed);
			st.nocall++;
			nocall = st.nocall;
			if (glocal_level > 0){
				funcdef.appendf ("\t%s%d _scopeobj%d(glocal);\n"
					,scope_prefix,nocall,nocall);
			}else{
				funcdef.appendf ("\t%s%d _scopeobj%d;\n"
					,scope_prefix,nocall,nocall);
			}
			strcpy (module,s2);
			tlcc_checkifdef(insert,module,"module");
			cur->setnoline(feed);
			cur->append (module);
			cur->appendf ("(_scopeobj%d",nocall);
			// fprintf (stderr,"call :%s: :%s:\n",mainbuf.getbuf(),cur->getbuf());
			tlcc_getparm (vars,feed,*cur);
			callobj.appendf ("class %s%d: public _F_%s{\n"
					,scope_prefix,nocall,module);
			callobj.append ("public:\n");
			if (glocal_level > 0){
				int gvar_level = vars->getglevel();
				callobj.appendf ("\t%s_GLOCAL%d_%d &glocal;\n"
					,glocal_prefix
					,st.nomod
					,gvar_level);
				callobj.appendf ("\t%s%d (%s_GLOCAL%d_%d &g)\n"
					,scope_prefix,nocall
					,glocal_prefix,st.nomod,gvar_level);
				callobj.append ("\t\t: glocal(g)\n");
			}else{
				callobj.appendf ("\t%s%d ()\n",scope_prefix,nocall);
			}
			callobj.append ("\t{\n\t}\n");
		}else if (strcmp (s1,"obj")==0 ){
			if (auto_glocal){
				// We insert a glocal so the component may be referenced
				// using glocal.component syntax
				auto_glocal = false;
				tlcc_startglocal(feed,st,glocal_level,insert,cur
					,vars,false,inmodule);
				tlcc_endglocal (feed,st,insert,funcdef,funccore,cur
					,vars,inmodule,false);
			}
			st.pushstate (COMPSTATE_OBJ,feed);
			if (s2[0] == '\0'){
				feed.printerr ("Missing class name\n");
				ret = -1;
			}else if (s3[0] == '\0'){
				feed.printerr ("Missing variable name\n");
				ret = -1;
			}
			if (!st.in_module){
				insert.append (glocal.getbuf());
				insert.append (callobj.getbuf());
				insert.append (callf.getbuf());
				mainbuf.append (funccore.getbuf());
				fputs (insert.getbuf(),fout);
				fputs (funcdef.getbuf(),fout);
				fputs (mainbuf.getbuf(),fout);
				insert.reset();
				mainbuf.reset();
				glocal.reset();
				funcdef.reset();
				funccore.reset();
				callobj.reset();
				callf.reset();
			}
			st.nocall++;
			nocall = st.nocall;
			if (glocal_level > 0){
				funcdef.appendf ("\t%s%d _scopeobj%d(glocal);\n"
					,scope_prefix,nocall,nocall);
			}else{
				funcdef.appendf ("\t%s%d _scopeobj%d;\n"
					,scope_prefix,nocall,nocall);
			}
			strcpy (module,s2);
			tlcc_checkifdef(insert,module,"class");
			cur->setnoline(feed);
			cur->appendf ("%s %s",module,s3);
			{
				cur->appendf ("(_scopeobj%d",nocall);
				char carac;
				while (isspace(carac = feed.getcar()));
				if (carac == ';'){
					cur->append (");");
				}else if (carac == '('){
					tlcc_getendparm (vars,feed,*cur);
				}else{
					feed.printerr ("Invalid obj definition\n");
					ret = -1;
				}
			}
			callobj.appendf ("class %s%d: public _F_%s{\n"
					,scope_prefix,nocall,module);
			callobj.append ("public:\n");
			if (glocal_level > 0){
				int gvar_level = vars->getglevel();
				callobj.appendf ("\t%s_GLOCAL%d_%d &glocal;\n"
					,glocal_prefix,st.nomod,gvar_level);
				callobj.appendf ("\t%s%d (%s_GLOCAL%d_%d &g)\n"
					,scope_prefix,nocall
					,glocal_prefix,st.nomod,gvar_level);
				callobj.append ("\t\t: glocal(g)\n");
			}else{
				callobj.appendf ("\t%s%d ()\n",scope_prefix,nocall);
			}
			callobj.append ("\t{\n\t}\n");
		}else if (strcmp(s1,"/call")==0){
			st.popstate (COMPSTATE_CALL,feed);
			callobj.append ("};\n");
			module[0] = '\0';
			cur->setnoline(feed);
		}else if (strcmp(s1,"/obj")==0){
			st.popstate (COMPSTATE_OBJ,feed);
			callobj.append ("};\n");
			module[0] = '\0';
			cur->setnoline(feed);
		}else if (strcmp (s1,"f")==0){
			st.pushstate (COMPSTATE_F,feed);
			// Put some code to insure that the functag exists
			// If not, strange C++ errors are printed
			if (module[0] == '\0'){
				feed.printerr ("Out of scope <f %s> construct\n",s2);
				ret = -1;
			}
			insert.appendf ("#ifndef _F_%s_%s\n",module,s2);
			insert.appendf ("\t#error Unknown logical function %s for module/class %s\n"
				,s2,module);
			insert.appendf ("#endif\n");
			callobj.appendf ("\t_F_%s_%s( );\n",module,s2,nocall);
			callf.appendf ("_F_%s_%s(%s%d::)\n",module,s2,scope_prefix,nocall);
			callf.append  ("{\n");
			callf.setnoline(feed);
			BUFSTR endcallf;
			ret |= tlcc_do (feed,insert,callf,endcallf,st,fout,glocal_level,vars,module);
			callf.append (endcallf.getbuf());
		}else if (strcmp (s1,"/f")==0){
			st.popstate (COMPSTATE_F,feed);
			funccore.append ("} // </f>\n");
			funccore.setnoline (feed);
			break;
		}else{
			fprintf (stderr,"Unknown directive %s\n",s1);
			ret = -1;
		}
	}
	insert.append (glocal.getbuf());
	insert.append (callobj.getbuf());
	insert.append (callf.getbuf());
	mainbuf.append (funccore.getbuf());
	if (scope == NULL){
		feed.printerr ("Nesting error\n");
	}else{
		SCOPE *old = scope;
		scope = scope->revert(vars,mainbuf,auto_glocal);
		delete old;
		if (scope != NULL){
			feed.printerr ("Nesting error\n");
		}
	}
	tlcc_delvars (vars,original_vars,mainbuf);
	return ret;
}

static void usage()
{
	fprintf (stderr
		,"tlmp translator version %s\n"
		 "\n"
		 "tlcc [ --name source_name ] input-file [ output-file ]\n"
		 "tlcc --name source_name  - [ output-file ]\n"
		,TLMP_VERSION);
	exit (-1);
}

static int tlcc_process (
	FILE *fin,
	FILE *fout,
	const char *source_name)
{
	int ret = 0;
	tlcc_setscopeprefix (source_name);
	FEED feed (source_name,fin);
	MODSTATE state;
	BUFSTR insert,funcdef,mainbuf;
	ret = tlcc_do (feed,insert,funcdef,mainbuf,state,fout,0,NULL,NULL);
	if (state.some_errors) ret = -1;
	fputs (insert.getbuf(),fout);
	fputs (funcdef.getbuf(),fout);
	fputs (mainbuf.getbuf(),fout);
	return ret;
}

int main (int argc, char *argv[])
{
	int ret = 0;
	bool some_errors = false;
	const char *source_name = NULL;
	int i;
	for (i=1; i<argc; i++){
		const char *arg = argv[i];
		if (strcmp(arg,"--name")==0){
			source_name = argv[i+1];
			i++;
		}else if (strcmp(arg,"--")==0){
			i++;
			break;
		}else if (strcmp(arg,"-")==0){
			break;
		}else if (arg[0] == '-'){
			usage();
			break;
		}else{
			break;
		}
	}
	if (some_errors || argc - i != 2 && argc - i != 1){
		usage();
		ret = -1;
	}else{
		FILE *fin = stdin;
		if (strcmp(argv[i],"-") != 0){
			fin = fopen (argv[i],"r");
			if (source_name == NULL) source_name = argv[i];
			if (fin == NULL){
				fprintf (stderr,"Can't open file %s (%s)\n",argv[i]
					,strerror(errno));
			}
		}else if (source_name == NULL){
			usage();
		}
		if (fin != NULL){
			FILE *fout = stdout;
			if (argc - i == 2){
				fout = fopen (argv[i+1],"w");
				if (fout == NULL){
					fprintf (stderr,"Can't open file %s (%s)\n",argv[i+1]
						,strerror(errno));
				}
			}
			if (fout != NULL){
				ret = tlcc_process (fin,fout,source_name);
				if (fout != stdout && fclose (fout) != 0) ret = -1;
			}
			if (fin != stdin) fclose (fin);
		}
	}
	return ret;
}

