#pragma implementation
#include <stdarg.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <misc.h>
#include <pwd.h>
#include <modregister.h>
#include "shellmod.h"
#include "shellmod.m"
#include "protocol.h"
#include <popen.h>
#include <dictionary.h>
#include <userconf.h>

static SSTRINGS tbdebug;

static bool debugmode = false;

/*
	Record the debug mode
*/
void protocol_setdebug (bool debug)
{
	debugmode = debug;
}

void protocol_debug(const char *ctl, ...)
{
	if (debugmode){
		va_list list;
		va_start (list,ctl);
		char line[1000];
		vsnprintf (line,sizeof(line)-1,ctl,list);
		va_end (list);
		tbdebug.add (new SSTRING (line));
	}
}


/*
	Presents the debug information (protocol stuff)
*/
void protocol_showdebug()
{
	if (debugmode){
		dialog_textbox (MSG_U(T_SHELLMODDBG,"Shellmod debugging"),tbdebug);
		tbdebug.remove_all();
	}
}


enum VAR_TYPE {
	VAR_STRING, VAR_NUM, VAR_HEXNUM, VAR_DBL,
	VAR_SEL, VAR_RADIO, VAR_ENUM};

class DIALOG_VAR: public ARRAY_OBJ{
public:
	SSTRING id;
	SSTRING str;
	int num;
	double dbl;
	VAR_TYPE type;
	char sel;
	bool do_dump;
	int nof;
	SSTRING on_val,off_val;		// Value that we must return for a
								// checkbox
	/*~PROTOBEG~ DIALOG_VAR */
public:
	DIALOG_VAR (const char *_id);
	/*~PROTOEND~ DIALOG_VAR */
};

PUBLIC DIALOG_VAR::DIALOG_VAR (const char *_id)
{
	id.setfrom(_id);
	num = 0;
	sel = 0;
	type = VAR_STRING;
	do_dump = true;
}

class DIALOG_VARS: public ARRAY{
	/*~PROTOBEG~ DIALOG_VARS */
public:
	DIALOG_VAR *getitem (int no)const;
	DIALOG_VAR *locate (const char *id)const;
	/*~PROTOEND~ DIALOG_VARS */
};

PUBLIC DIALOG_VAR *DIALOG_VARS::getitem (int no) const
{
	return (DIALOG_VAR*)ARRAY::getitem(no);
}
PUBLIC DIALOG_VAR *DIALOG_VARS::locate (const char *id) const
{
	DIALOG_VAR *ret = NULL;
	for (int i=0; i<getnb(); i++){
		DIALOG_VAR *v = getitem(i);
		if (v->id.cmp(id)==0){
			ret = v;
			break;
		}
	}
	return ret;
}
class SHELL_DIALOG: public ARRAY_OBJ{
public:
	DIALOG *dia;
	int nof;
	DIALOG_VARS vars;
	SSTRINGS tbbut;		// User defined button
	/*~PROTOBEG~ SHELL_DIALOG */
public:
	SHELL_DIALOG (void);
	void dump (POPEN&pop);
	~SHELL_DIALOG (void);
	/*~PROTOEND~ SHELL_DIALOG */
};

PUBLIC SHELL_DIALOG::SHELL_DIALOG()
{
	dia = NULL;
	nof = 0;
}

PUBLIC SHELL_DIALOG::~SHELL_DIALOG()
{
	delete dia;
}

/*
	Output the value of all fields of the dialog
*/
PUBLIC void SHELL_DIALOG::dump(POPEN &pop)
{
	protocol_debug ("%s\n",MSG_U(I_DIAVARS,"Dialog fields"));
	for (int i=0; i<vars.getnb(); i++){
		DIALOG_VAR *var = vars.getitem(i);
		if (var->do_dump){
			char tmp[20];
			const char *val = tmp;
			if (var->type == VAR_STRING){
				val = var->str.get();
			}else if (var->type == VAR_NUM){
				sprintf (tmp,"%d",var->num);
			}else if (var->type == VAR_HEXNUM){
				sprintf (tmp,"%x",var->num);
			}else if (var->type == VAR_DBL){
				sprintf (tmp,"%f",var->dbl);
			}else if (var->type == VAR_SEL || var->type == VAR_RADIO){
				strcpy (tmp,(var->sel ? var->on_val.get():var->off_val.get()));
			}
			char qval[strlen(val)*4+2+1];
			char *ptq = qval;
			*ptq++ = '\'';
			while (*val != '\0'){
				if (*val == '\''){
					*ptq++ = '\'';
					*ptq++ = '\\';
					*ptq++ = '\'';
				}
				*ptq++ = *val++;
			}
			*ptq++ = '\'';
			*ptq = '\0';
			protocol_debug ("\t%s=%s\n",var->id.get(),qval);
			pop.sendf ("%s=%s\n",var->id.get(),qval);
		}
	}
}



PUBLIC SHELL_DIALOG* SHELL_DIALOGS::getitem(int no) const
{
	return (SHELL_DIALOG*)ARRAY::getitem(no);
}


PUBLIC SHELL_INSTANCE::SHELL_INSTANCE (const char *path)
	: pop (path)
{
	if (pop.isok()){
		struct passwd *p = getpwuid(perm_getuid());
		if (p != NULL){
			sendvar ("REALUSER",p->pw_name);
		}
	}
}


/*
	Return true if the shell is still alive
*/
PUBLIC bool SHELL_INSTANCE::isalive()
{
	return pop.isok();
}


/*
	Output the value of all fields of the current dialog
*/
PRIVATE void SHELL_INSTANCE::dump()
{
	SHELL_DIALOG *sdia = dias.getitem(dias.getnb()-1);
	if (sdia != NULL) sdia->dump (pop);
}

static const char *protocol_extract (
	const char *buf,
	SSTRING &s,
	SSTRING_KEYS &longvals)
{
	buf = str_skip(buf);
	if (buf[0] == '"'){
		buf = str_extract(buf,s);
	}else{
		buf = str_extract(buf,s);
		const char *pt = s.get();
		if (pt[0] == '='){
			// This is a long variable
			SSTRING_KEY *var = longvals.getobj(pt+1);
			if (var != NULL){
				SSTRING *val = var->getobj();
				s.setfrom (*val);
				// A variable is used only once.
				longvals.remove_del (var);
			}
		}
	}
	return buf;
}

PRIVATE int SHELL_INSTANCE::parse_butopt (const char *s, SHELL_DIALOG *sdia)
{
	char *str_start,*str,*next;
	next = str = str_start = strdup(s);
	int size, ret = 0;
	*next = toupper(*next);
	while (*next != 0){
		next++;
		while (*next != ',' && *next != 0){
			*next = toupper(*next);
			next++;
		}
		size = next-str;
		if (strncmp(str,"-",size)==0){
			ret = 0;
			break;
		}else if (strncmp(str,"SAVE",size)==0){
			ret |= MENUBUT_SAVE;
		}else if (strncmp(str,"ADD",size)==0){
			ret |= MENUBUT_ADD;
		}else if (strncmp(str,"DEL",size)==0){
			ret |= MENUBUT_DEL;
		}else if (strncmp(str,"INS",size)==0){
			ret |= MENUBUT_INS;
		}else if (strncmp(str,"OK",size)==0){
			ret |= MENUBUT_OK;
		}else if (strncmp(str,"ACCEPT",size)==0){
			ret |= MENUBUT_ACCEPT;
		}else if (strncmp(str,"CANCEL",size)==0){
			ret |= MENUBUT_CANCEL;
		}else if (strncmp(str,"QUIT",size)==0){
			ret |= MENUBUT_QUIT;
		}else if (strncmp(str,"YES",size)==0){
			ret |= MENUBUT_YES;
		}else if (strncmp(str,"NO",size)==0){
			ret |= MENUBUT_NO;
		}else if (strncmp(str,"EDIT",size)==0){
			ret |= MENUBUT_EDIT;
		}else if (strncmp(str,"RESET",size)==0){
			ret |= MENUBUT_RESET;
		}else if (strncmp(str,"MORE",size)==0){
			ret |= MENUBUT_MORE;
		}else{ // It's not a default button, let's create one
			const char *s_start,*s_end,*div;
			SSTRING id,title;
			s_start = str-str_start+s; // Take unchanged string
			s_end = s_start+(next-str);
			div = s_start;
			while (*div != ':' && div != s_end) div++;
			title.setfrom(s_start,div-s_start);
			if (*div == ':'){
				id.setfrom(div+1,s_end-(div+1));
			}else{
				id.setfrom(title);
			}
			// Insert only unique buttons to avoid duplicate
			// buttons with the 'show' command
			if (sdia->tbbut.lookup(&id)==-1){
				sdia->dia->setbutinfo (MENU_USR1+sdia->tbbut.getnb(),title.get()
					,title.get());
				sdia->tbbut.add (new SSTRING(id));
			}
		}
		str = next+1;
	}
	free(str_start);
	return ret;
}

/*
	Parse commands received from the script
*/
PRIVATE int SHELL_INSTANCE::parse(
	MENUENTRIES *entries,
	COMANAGERS *managers,
	int *nof)
{
	int retcode = 0;
	bool done = false;
	FIELD_COMBO *comb = NULL;
	FIELD_ENUM *fenum = NULL;
	FIELD_LIST *list = NULL;
	int *gauge_data = NULL;
	static int tbmask[]={
		0,
		MENUBUT_USR1,
		MENUBUT_USR1|MENUBUT_USR2,
		MENUBUT_USR1|MENUBUT_USR2|MENUBUT_USR3,
		MENUBUT_USR1|MENUBUT_USR2|MENUBUT_USR3|MENUBUT_USR4
	};
	SSTRING_KEYS longvals;	// Multiline parameters
	while (!done){
		if (pop.wait(100000)<=0){
			pop.close();
			break;
		}
		char cmd[1000];
		while (pop.readout(cmd,sizeof(cmd)-1)!=-1){
			SSTRING keyword;
			const char *pt = str_extract (cmd,keyword);
			const char *keyw = keyword.get();
			SHELL_DIALOG *sdia = dias.getitem(dias.getnb()-1);
			if (strcmp(keyw,"endscope")==0){
				done = true;
				break;
			}else if (strcmp(keyw,"vreg_get")==0){
				SSTRING var,shellvar;
				pt = protocol_extract (pt,var,longvals);
				protocol_extract (pt,shellvar,longvals);
				if (!master_registry.in_session()){
					master_registry.start_session();
				}
				const char *val = master_registry.get(var.get());
				if (val == NULL){
					xconf_error (MSG_U(E_NOVREGVAR,"Variable %s does not exist")
						,var.get());
					pop.sendf ("%s=\n",shellvar.get());
				}else{
					pop.sendf ("%s=\"%s\"\n",shellvar.get(),val);
				}
			}else if (strcmp(keyw,"vreg_set")==0){
				SSTRING var,val;
				pt = protocol_extract (pt,var,longvals);
				protocol_extract (pt,val,longvals);
				if (!master_registry.in_session()){
					master_registry.start_session();
				}
				if (master_registry.set (var.get(),val.get())==-1){
					xconf_error (MSG_U(E_NOTSET,"Variable %s was not set")
						,var.get());
				}
			}else if (strcmp(keyw,"vreg_do")==0){
				if (master_registry.in_session()){
					master_registry.end_session();
				}
				pop.send ("go\n");
			}else if (strcmp(keyw,"retcode")==0){
				SSTRING s;
				protocol_extract (pt,s,longvals);
				retcode = s.getval();
			}else if (strcmp(keyw,"error")==0){
				SSTRING msg;
				protocol_extract (pt,msg,longvals);
				xconf_error ("%s",msg.get());
			}else if (strcmp(keyw,"notice")==0){
				SSTRING msg;
				protocol_extract (pt,msg,longvals);
				xconf_notice ("%s",msg.get());
			}else if (strcmp(keyw,"yesno")==0){
				SSTRING title,msg;
				pt = protocol_extract (pt,title,longvals);
				protocol_extract (pt,msg,longvals);
				if (dialog_yesno (title.get(),msg.get(),help_nil)==MENU_YES){
					pop.sendf ("CODE=yes\n");
				}else{
					pop.sendf ("CODE=no\n");
				}
				pop.sendf ("go\n");
			}else if (strcmp(keyw,"regmenu")==0){
				if (entries != NULL){
					SSTRING func,menu,title;
					pt = protocol_extract (pt,func,longvals);
					pt = protocol_extract (pt,menu,longvals);
					protocol_extract (pt,title,longvals);
					entries->add (new MENUENTRY (func.get(),menu.get()
						,title.get()));
				}else{
					xconf_error (MSG_U(E_REGISTER,"Module can't send %s command\n"
						"except in its register function"),keyw);
				}
			}else if (strcmp(keyw,"comanager")==0){
				if (managers != NULL){
					SSTRING func,dialog;
					pt = protocol_extract (pt,func,longvals);
					protocol_extract (pt,dialog,longvals);
					managers->add (new COMANAGER(func.get(),dialog.get()));
				}else{
					xconf_error (MSG_R(E_REGISTER),keyw);
				}
			}else if (strncmp(keyw,"DIALOG",6)==0){
				sdia = new SHELL_DIALOG;
				dias.add (sdia);
				DIALOG *dia = NULL;
				if (strcmp(keyw,"DIALOG")==0){
					dia = new DIALOG;
				}else if (strcmp(keyw,"DIALOG_MENU")==0){
					dia = new DIALOG_MENU;
				}else if (strcmp(keyw,"DIALOG_LISTE")==0
					|| strcmp(keyw,"DIALOG_LIST")==0){
					dia = new DIALOG_LISTE;
				}else{
					xconf_error (MSG_U(E_IVLDDIATYPE,"Invalid dialog type \"%s\"")
						,keyw);
					dia = new DIALOG;	// Create a dialog anyway so it won't crash
				}
				sdia->dia = dia;
			}else if (sdia == NULL){
				xconf_error (MSG_U(E_IVLDORDER
					,"Invalid command or out of order\n"
					 "(no dialog defined yet!)\n"
					 "Command was:\n%s")
					,cmd);
			}else if (strcmp(keyw,"settype")==0){
				SSTRING diatype;
				protocol_extract (pt,diatype,longvals);
				if (strcmp(diatype.get(),"DIATYPE_POPUP")==0){
					sdia->dia->settype (DIATYPE_POPUP);
				}
			}else if (strcmp(keyw,"remove_all")==0){
				sdia->dia->remove_all();
			}else if (strcmp(keyw,"button")==0){
				SSTRING id,title;
				pt = protocol_extract (pt,id,longvals);
				protocol_extract (pt,title,longvals);
				sdia->dia->setbutinfo (MENU_USR1+sdia->tbbut.getnb(),title.get()
					,title.get());
				sdia->tbbut.add (new SSTRING(id));
			}else if (strcmp(keyw,"setcurfield")==0){
				SSTRING id;
				protocol_extract (pt,id,longvals);
				bool found = false;
				for (int i=0; i<sdia->vars.getnb(); i++){
					DIALOG_VAR *v = sdia->vars.getitem(i);
					if (v->id.cmp(id)==0){
						sdia->nof = v->nof;
						if (nof != NULL) *nof = v->nof;
						found = true;
						break;
					}
				}
				if (!found){
					xconf_error (MSG_U(E_NOFIELD,"setcurfield command failed\n"
						"No such field \"%s\""),id.get());
				}
			}else if (strcmp(keyw,"newf_head")==0){
				SSTRING head;
				// Build a line with all paramater separated with \t
				while (1){
					SSTRING next;
					pt = protocol_extract(pt,next,longvals);
					if (next.is_empty()) break;
					if (!head.is_empty()) head.append ("\t");
					head.append (next.get());
				}
				sdia->dia->newf_head ("",head.get());
			}else if (strcmp(keyw,"comboitem")==0){
				SSTRING item,desc;
				pt = protocol_extract (pt,item,longvals);
				protocol_extract (pt,desc,longvals);
				if (comb == NULL){
					xconf_error (MSG_U(E_NOCOMBO
						,"No combo field defined.\n"
						 "Nothing to do with comboitem \"%s\"")
						,item.get());
				}else{
					comb->addopt (item.get(),desc.get());
				}
			}else if (strcmp(keyw,"listitem")==0){
				SSTRING item,desc;
				pt = protocol_extract (pt,item,longvals);
				protocol_extract (pt,desc,longvals);
				if (list == NULL){
					xconf_error (MSG_U(E_NOLIST
						,"No list field defined.\n"
						 "Nothing to do with listitem \"%s\"")
						,item.get());
				}else{
					list->addopt (item.get(),desc.get());
				}
			}else if (strcmp(keyw,"enumitem")==0){
				SSTRING item,desc;
				pt = protocol_extract (pt,item,longvals);
				protocol_extract (pt,desc,longvals);
				if (fenum == NULL){
					xconf_error (MSG_U(E_NOENUM
						,"No enum field defined.\n"
						 "Nothing to do with enumitem \"%s\"")
						,item.get());
				}else{
					fenum->addopt (item.get(),desc.get());
				}
			}else if (strcmp(keyw,"new_menuitem")==0){
				SSTRING id;
				pt = protocol_extract (pt,id,longvals);
				DIALOG_VAR *var = new DIALOG_VAR(id.get());
				var->nof = sdia->dia->getnb();
				sdia->vars.add (var);
				SSTRING title,prefix;
				pt = protocol_extract (pt,prefix,longvals);
				// Build a line with all paramater separated with \t
				while (1){
					SSTRING next;
					pt = protocol_extract(pt,next,longvals);
					if (next.is_empty()) break;
					if (!title.is_empty()) title.append ("\t");
					title.append (next.get());
				}
				sdia->dia->new_menuitem (prefix.get(),title.get());
			}else if (strcmp(keyw,"newf_title")==0){
				SSTRING pad,level,prompt,msg;
				pt = protocol_extract (pt,pad,longvals);
				pt = protocol_extract (pt,level,longvals);
				pt = protocol_extract (pt,prompt,longvals);
				protocol_extract (pt,msg,longvals);
				sdia->dia->newf_title (pad.get(),level.getval()
					,prompt.get(),msg.get());
			}else if (strncmp(keyw,"newf_",5)==0){
				SSTRING id;
				pt = protocol_extract (pt,id,longvals);
				DIALOG_VAR *var = sdia->vars.locate(id.get());
				bool reloading = true;
				if (var == NULL){
					reloading = false;
					var = new DIALOG_VAR(id.get());
					var->nof = sdia->dia->getnb();
					sdia->vars.add (var);
				}
				SSTRING title;
				pt = protocol_extract (pt,title,longvals);
				pt = protocol_extract (pt,var->str,longvals);
				const char *endkeyw = keyw + 5;
				if (reloading){
					VAR_TYPE type = var->type;
					if (type == VAR_NUM){
						var->num = var->str.getval();
					}else if (type == VAR_HEXNUM){
						sscanf (var->str.get(),"%x",&var->num);
					}else if (type == VAR_DBL){
						sscanf (var->str.get(),"%lf",&var->dbl);
					}else if (type == VAR_SEL || type == VAR_RADIO){
						var->sel = var->str.getval();
					}
					sdia->dia->reload (var->nof);
				}else if (strcmp(endkeyw,"str")==0){
					sdia->dia->newf_str (title.get(),var->str);
				}else if (strcmp(endkeyw,"textarea")==0){
					SSTRING width,height;
					pt = protocol_extract (pt,width,longvals);
					protocol_extract (pt,height,longvals);
					sdia->dia->newf_textarea (title.get(),var->str
						,width.getval(),height.getval());
				}else if (strcmp(endkeyw,"pass")==0){
					sdia->dia->newf_pass (title.get(),var->str);
				}else if (strcmp(endkeyw,"combo")==0){
					comb = sdia->dia->newf_combo (title.get(),var->str);
				}else if (strcmp(endkeyw,"list")==0){
					list = sdia->dia->newf_list (title.get(),var->str);
				}else if (strcmp(endkeyw,"enum")==0){
					var->type = VAR_NUM;
					var->num = var->str.getval();
					fenum = sdia->dia->newf_enum (title.get(),var->num);
				}else if (strcmp(endkeyw,"num")==0){
					var->type = VAR_NUM;
					var->num = var->str.getval();
					sdia->dia->newf_num (title.get(),var->num);
				}else if (strcmp(endkeyw,"hexnum")==0){
					var->type = VAR_HEXNUM;
					sscanf (var->str.get(),"%x",&var->num);
					sdia->dia->newf_hexnum (title.get(),var->num);
				}else if (strcmp(endkeyw,"dbl")==0){
					var->type = VAR_DBL;
					sscanf (var->str.get(),"%lf",&var->dbl);
					SSTRING decimal;
					protocol_extract (pt,decimal,longvals);
					sdia->dia->newf_dbl (title.get(),var->dbl
						,decimal.getval());
				}else if (strcmp(endkeyw,"chkm")==0){
					var->type = VAR_SEL;
					var->sel = var->str.getval();
					SSTRINGS titles;
					while (1){	
						SSTRING s;
						pt = protocol_extract (pt,s,longvals);
						if (s.is_empty()) break;
						titles.add (new SSTRING(s));
					}
					int n = titles.getnb();
					const char *tb[n+1];
					for (int i=0; i<n; i++){
						tb[i] = titles.getitem(i)->get();
					}
					tb[n] = NULL;
					sdia->dia->newf_chkm (title.get(),var->sel,tb);
				}else if (strcmp(endkeyw,"chk")==0){
					var->type = VAR_SEL;
					SSTRINGS parts;
					const char *bval = var->str.get();
					int n = str_splitline (bval,':',parts);
					if (n == 1 || n == 0){
						// We accept several variation to mean "selected"
						// But we will output 0 or 1 only
						var->on_val.setfrom ("1");
						var->off_val.setfrom ("0");
						if (stricmp(bval,"y")==0
							|| stricmp(bval,"yes")==0
							|| stricmp(bval,"1")==0){
							var->sel = 1;
						}else{
							var->sel = 0;
						}
					}else if (n == 3){
						// Here the user specify the value and
						// the dictionary to interpret it
						bval = parts.getitem(0)->get();
						const char *on_val = parts.getitem(1)->get();
						const char *off_val = parts.getitem(2)->get();
						var->on_val.setfrom (on_val);
						var->off_val.setfrom (off_val);
						var->sel = stricmp(bval,on_val)==0;
					}else{
						xconf_error (MSG_U(E_IVLDSYNCHK
							,"Invalid syntax for the newf_chk value\n"
							 "Enter either a 0 or a 1\n"
							 "or a triplet val:on_val:off_val")
							,bval);
					}
					SSTRING title2;
					pt = protocol_extract (pt,title2,longvals);
					sdia->dia->newf_chk (title.get(),var->sel,title2.get());
				}else if (strcmp(endkeyw,"radio")==0){
					var->type = VAR_RADIO;
					var->sel = var->str.getval();
					SSTRING instance,title2;
					pt = protocol_extract (pt,instance,longvals);
					pt = protocol_extract (pt,title2,longvals);
					// We have to pass the first instance of this radio
					for (int i=0; i<sdia->vars.getnb()-1; i++){
						DIALOG_VAR *v = sdia->vars.getitem(i);
						if (var->id.cmp(v->id)==0){
							// short circuit this instance
							var->do_dump = false;
							var = v;
							break;
						}
					}
					sdia->dia->newf_radio (title.get(),var->sel
						,instance.getval()
						,title2.get());
				}else if (strcmp(endkeyw,"gauge")==0){
					var->type = VAR_NUM;
					var->num = var->str.getval();

					if (1 || !gauge_data){
						gauge_data = &(var->num);
						var->type = VAR_NUM;
						var->num = var->str.getval();
						SSTRING range;
						protocol_extract (pt,range,longvals);
						sdia->dia->newf_gauge (title.get(),var->num,range.getval());
					}else{
						*gauge_data = var->num;
					}
				}else if (strcmp(endkeyw,"slider")==0){
					var->type = VAR_NUM;
					var->num = var->str.getval();
					SSTRING minimum,maximum;
					pt = protocol_extract (pt,minimum,longvals);
					protocol_extract (pt,maximum,longvals);
					sdia->dia->newf_slider (title.get(),var->num
						,minimum.getval(),maximum.getval());
				}else{
					xconf_error (MSG_U(E_PROTOCOL,"Invalid shellmod command (protocol): %s"),keyw);
				}
			}else if (strcmp(keyw,"edit")==0){
				SSTRING title,intro,butopt;
				int butopt_value;
				pt = protocol_extract (pt,title,longvals);
				pt = protocol_extract (pt,intro,longvals);
				pt = protocol_extract (pt,butopt,longvals);
				if (butopt.getlen() == 0){
					butopt_value = MENUBUT_CANCEL|MENUBUT_ACCEPT;
				}else{
					butopt_value = parse_butopt(butopt.get(),sdia);
				}
				MENU_STATUS code = sdia->dia->edit (title.get(),intro.get()
					,help_nil
					,sdia->nof
					,butopt_value
						|tbmask[sdia->tbbut.getnb()]);
				if (code == MENU_CANCEL || code == MENU_ESCAPE){
					pop.sendf ("CODE=cancel\n");
				}else{
					const char *cmd,*but=NULL;
					protocol_debug ("%s\n",MSG_U(I_DIARES,"Dialog result"));
					dump();
					switch (code){
						case MENU_ACCEPT: cmd = "CODE=accept\n"; break;
						case MENU_OK: cmd = "CODE=ok\n"; break;
						case MENU_RESET: cmd = "CODE=reset\n"; break;
						case MENU_QUIT: cmd = "CODE=quit\n"; break;
						case MENU_ADD: cmd = "CODE=add\n"; break;
						case MENU_INS: cmd = "CODE=ins\n"; break;
						case MENU_DEL: cmd = "CODE=del\n"; break;
						case MENU_EDIT: cmd = "CODE=edit\n"; break;
						case MENU_SAVE: cmd = "CODE=save\n"; break;
						case MENU_MORE: cmd = "CODE=more\n"; break;
						case MENU_YES: cmd = "CODE=yes\n"; break;
						case MENU_NO: cmd = "CODE=no\n"; break;
						default:
							cmd = "CODE=\"%s\"\n";
							but = sdia->tbbut.getitem(code-MENU_USR1)->get();
							break;
					}
					protocol_debug (cmd,but);
					pop.sendf (cmd,but);
					protocol_showdebug();
				}
				pop.sendf ("go\n");
			}else if (strcmp(keyw,"show")==0){
				SSTRING title,intro,butopt;
				int butopt_value;
				pt = protocol_extract (pt,title,longvals);
				pt = protocol_extract (pt,intro,longvals);
				pt = protocol_extract (pt,butopt,longvals);
				if (butopt.getlen() == 0){
					butopt_value = MENUBUT_CANCEL|MENUBUT_ACCEPT;
				}else{
					butopt_value = parse_butopt(butopt.get(),sdia);
				}
				sdia->dia->show (title.get(),intro.get()
					,help_nil
					,sdia->nof
					,butopt_value
						|tbmask[sdia->tbbut.getnb()]);
				diagui_flush();
			}else if (strcmp(keyw,"editmenu")==0){
				SSTRING title,intro,butopt;
				int butopt_value;
				pt = protocol_extract (pt,title,longvals);
				pt = protocol_extract (pt,intro,longvals);
				pt = protocol_extract (pt,butopt,longvals);
				if (butopt.getlen() == 0){
					butopt_value = 0;
				}else{
					butopt_value = parse_butopt(butopt.get(),sdia);
				}
				while(1){
					MENU_STATUS code = sdia->dia->editmenu (title.get()
						,intro.get()
						,help_nil
						,sdia->nof
						,butopt_value
							|tbmask[sdia->tbbut.getnb()]);
					if (code == MENU_ESCAPE
						|| code == MENU_QUIT
						|| code == MENU_CANCEL){
						pop.sendf ("CODE=cancel\n");
						break;
					}else if (code == MENU_OK){
						DIALOG_VAR *var = sdia->vars.getitem(sdia->nof);
						if (var != NULL){
							pop.sendf ("exec %s\n",var->id.get());
							parse(entries,managers,nof);
						}
					}else{
						const char *cmd,*but=NULL;
						protocol_debug ("%s\n",MSG_R(I_DIARES));
						dump();
						switch (code){
							case MENU_ACCEPT:
								cmd = "CODE=accept\n";
								break;
							case MENU_RESET:
								cmd = "CODE=reset\n";
								break;
							case MENU_ADD:
								cmd = "CODE=add\n";
								break;
							case MENU_INS:
								cmd = "CODE=ins\n";
								break;
							case MENU_DEL:
								cmd = "CODE=del\n";
								break;
							case MENU_EDIT:
								cmd = "CODE=edit\n";
								break;
							case MENU_SAVE:
								cmd = "CODE=save\n";
								break;
							case MENU_MORE:
								cmd = "CODE=more\n";
								break;
							case MENU_YES:
								cmd = "CODE=yes\n";
								break;
							case MENU_NO:
								cmd = "CODE=no\n";
								break;
							default:
								cmd = "CODE=\"%s\"\n";
								but = sdia->tbbut.getitem(code-MENU_USR1)->get();
								break;
						}
						protocol_debug (cmd,but);
						pop.sendf (cmd,but);
						protocol_showdebug();
						break;
					}
				}
				pop.sendf ("go\n");
			}else if (strcmp(keyw,"defval")==0){
				strip_end (cmd);
				SSTRING var;
				pt = str_extract (pt,var);
				if (isspace(pt[0])) pt++;
				SSTRING_KEY *s = longvals.getobj (var.get());
				if (s == NULL){
					longvals.add (var.get(),pt);
				}else{
					s->getobj()->appendf ("\n%s",pt);
				}
			}else if (strcmp(keyw,"end")==0){
				gauge_data = NULL;
				dias.remove_del (sdia);
			}else{
				xconf_error (MSG_R(E_PROTOCOL),keyw);
			}
		}
		while (pop.readerr(cmd,sizeof(cmd)-1)!=-1){
			fprintf (stderr,"%s",cmd);
		}
	}
	return retcode;
}

PUBLIC void SHELL_INSTANCE::sendvar (const char *var, const char *val)
{
	pop.sendf ("%s=%s\n",var,val);
}

PUBLIC int SHELL_INSTANCE::runfunc (const char *key)
{
	pop.sendf ("exec %s\n",key);
	return parse(NULL,NULL,NULL);
}

PUBLIC int SHELL_INSTANCE::collect (
	MENUENTRIES &entries,
	COMANAGERS &managers)
{
	pop.sendf ("exec register\n");
	return parse(&entries,&managers,NULL);
}

/*
	Invoke the various comanager functions in the shell
*/
PUBLIC int SHELL_INSTANCE::comanage(
	DIALOG *dia,			// dialog, passed for the setupdia call
	DICTIONARY &dict,
	const char *func_prefix,
	const char *func_suffix,
	int *nof)				// nof of the field with error, passed
							// for the validate call
{
	if (dia != NULL){
		SHELL_DIALOG *sdia = new SHELL_DIALOG;
		dias.add (sdia);
		sdia->dia = dia;
	}
	// dump the content of the dictionary
	protocol_debug(MSG_U(I_CALLING,"Comanager calling %s%s")
		,func_prefix,func_suffix);
	{
		int no=0;
		protocol_debug ("%s\n",MSG_U(I_DICTCONTENT,"Dialog information"));
		while (1){
			const char *var = dict.get_var(no);
			if (var == NULL) break;
			const char *val = dict.get_val(no);
			protocol_debug ("\t%s=\"%s\"\n",var,val);
			pop.sendf ("%s=\"%s\"\n",var,val);
			no++;
		}
	}
	// dump the dialog fields.
	dump();
	char func[100];
	snprintf (func,sizeof(func)-1,"%s_%s",func_prefix,func_suffix);
	protocol_showdebug();
	pop.sendf ("exec %s\n",func);
	return parse(NULL,NULL,nof);
}

