#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <confdb.h>
#include <fviews.h>
#include <subsys.h>
#include <configf.h>
#include "debian.h"

#define DEBIAN_DEBUG 0

static HELP_FILE help_rcs ("debian", "rcS");
static HELP_FILE help_hostname ("debian", "hostname");

static CONFIG_FILE f_rcs("/etc/default/rcS", help_rcs,
	CONFIGF_MANAGED|CONFIGF_OPTIONNAL, "root", "root", 0744);

static CONFIG_FILE f_hostname("/etc/hostname", help_nil,
	CONFIGF_MANAGED|CONFIGF_OPTIONNAL, "root", "root", 0644);

enum DEBIAN_FILES { ETC_DEFAULT_RCS, ETC_HOSTNAME, DEBIAN_dummy };

enum DEBIAN_TYPES { DEBIAN_YESNO, DEBIAN_ONOFF, DEBIAN_STR };

struct DEBIAN_CONFIG_ITEM {
	const char *key1;
	const char *key2;
	const char *var;
	DEBIAN_FILES file;
	DEBIAN_TYPES type;	
};

static DEBIAN_CONFIG_ITEM debitems[]={

	// Config items from /etc/default/rcS

	{"debian",	"tmptime",	"TMPTIME",	ETC_DEFAULT_RCS,	DEBIAN_STR},
	{"debian",	"sulogin",	"SULOGIN",	ETC_DEFAULT_RCS,	DEBIAN_YESNO},
	{"debian",	"delaylogin",	"DELAYLOGIN",	ETC_DEFAULT_RCS,	DEBIAN_YESNO},
	{"datetime",	"universal",	"UTC",		ETC_DEFAULT_RCS,	DEBIAN_YESNO},
	{"debian",	"verbose",	"VERBOSE",	ETC_DEFAULT_RCS,	DEBIAN_YESNO},
	{"debian",	"editmotd",	"EDITMOTD",	ETC_DEFAULT_RCS,	DEBIAN_YESNO},
	{"debian",	"fsckfix",	"FSCKFIX",	ETC_DEFAULT_RCS,	DEBIAN_YESNO},

	// Config items from /etc/hostname

	{"network",	"hostname",	"",		ETC_HOSTNAME,		DEBIAN_STR},
	
	// Special dummy entries (for distribution features)

	{"--distrib--",	"sysvrestart",	"",		DEBIAN_dummy,		DEBIAN_STR},
	{"--distrib--",	"privgroup",	"",		DEBIAN_dummy,		DEBIAN_STR},
	{"--distrib--",	"sysvenh",		"",		DEBIAN_dummy,		DEBIAN_STR},
	{"--distrib--",	"release",		"",		DEBIAN_dummy,		DEBIAN_STR},
};

#define DEBIAN_CONFIG_ENTRIES (sizeof(debitems)/sizeof(debitems[0]))


class CONFDB_DEBIAN: public CONFDB {
	CONFDB *orig;	// Original CONFDB object managing conf.linuxconf
	struct {
		VIEWITEMS *rcs;
		VIEWITEMS *hostname;
		bool rcs_modified;
		bool hostname_modified;
	} debcf;		// Config files managed by this module
	SSTRINGS tbstr;		// Old temporary strings returned
						// to caller of getval
	/*~PROTOBEG~ CONFDB_DEBIAN */
public:
	CONFDB_DEBIAN (CONFDB *_orig);
	void add (const char *prefix,
		 const char *key,
		 const char *val);
	int archive (SSTREAM&ss, const char *_sys);
	int extract (SSTREAM&ss, const char *_sys);
	int getall (const char *prefix,
		 const char *key,
		 SSTRINGS&lst,
		 bool copy);
	const char *getval (const char *prefix,
		 const char *key,
		 const char *defval);
private:
	const char *getextval(DEBIAN_CONFIG_ITEM *dci);
public:
	void removeall (const char *prefix,
		 const char *key);
private:
	void reset_modified (void);
public:
	int save (PRIVILEGE *priv);
	void setcursys (const char *_subsys);

#if DEBIAN_DEBUG
private:
	void debian_debug(const char *str,const char *str2) { FILE *fout = fopen ("/var/log/linuxconf_debian.log","a"); fprintf (fout,"%s> %s\n",str,str2); fclose (fout); }
	void debian_debug(const char *str,char * str2) { debian_debug(str,(const char*)str2); }
	void debian_debug(const char *str,int val) { char str2[200]; sprintf(str2,"%d",val); debian_debug(str,str2); }
#else
#define debian_debug(x,y) ;
#endif

private:
	void setextval (DEBIAN_CONFIG_ITEM *dci, const char *val);
public:
	~CONFDB_DEBIAN (void);
	/*~PROTOEND~ CONFDB_DEBIAN */
};


PRIVATE void CONFDB_DEBIAN::reset_modified()
{
	debcf.rcs_modified = false;
	debcf.hostname_modified = false;
}

PUBLIC CONFDB_DEBIAN::CONFDB_DEBIAN(CONFDB *_orig)
{
	orig = _orig;
	debcf.rcs = NULL;
	debcf.hostname = NULL;
	reset_modified();
	debian_debug("init", "1");
}

PUBLIC CONFDB_DEBIAN::~CONFDB_DEBIAN()
{
	delete orig;
	delete debcf.rcs;
	delete debcf.hostname;
	debian_debug("dstroy", "1");
}

PUBLIC void CONFDB_DEBIAN::setcursys (const char *_subsys)
{
	orig->setcursys (_subsys);
}

static DEBIAN_CONFIG_ITEM *filter_lookup (const char *key1, const char *key2)
{
	DEBIAN_CONFIG_ITEM *pt = debitems;
	for (unsigned i=0; i<DEBIAN_CONFIG_ENTRIES; i++, pt++){
		if (strcmp(pt->key1, key1)==0
			&& strcmp(pt->key2, key2)==0){
			return pt;
		}
	}
	return NULL;
}


static void saveconf (
	VIEWITEMS *&items,
	CONFIG_FILE &cf,
	PRIVILEGE *priv,
	bool modified)
{
	if (items != NULL){
		if (modified) items->write (cf,priv);
		delete items;
		items = NULL;
	}
}


PUBLIC int CONFDB_DEBIAN::save(PRIVILEGE *priv)
{
	int ret = orig->save(priv);
	if (ret != -1){
		saveconf (debcf.rcs, f_rcs, priv, debcf.rcs_modified);
		saveconf (debcf.hostname, f_hostname, priv, debcf.hostname_modified);
		reset_modified();
	}
	tbstr.remove_all();
	return ret;
}

PRIVATE const char *CONFDB_DEBIAN::getextval(DEBIAN_CONFIG_ITEM *dci)
{
	VIEWITEMS **vi = NULL;
	CONFIG_FILE *cf = NULL;

	debian_debug("GetExtVal: key1 = ", dci->key1);
	debian_debug("GetExtVal: key2 = ", dci->key2);

	if (dci->file == DEBIAN_dummy) {
		if (strcmp(dci->key2, "sysvrestart")==0) {
			return "1";
		} else if (strcmp(dci->key2, "privgroup")==0) {
			return "1";
		} else if (strcmp(dci->key2, "sysvenh")==0) {
			return "1";
		} else if (strcmp(dci->key2, "release")==0) {
			return "unknown";
		}
	} else if (dci->file == ETC_DEFAULT_RCS) {
		vi = &debcf.rcs;
		cf = &f_rcs;
	} else if (dci->file == ETC_HOSTNAME) {
		FILE_CFG *fin = f_hostname.fopen("r");
		if (fin != NULL) {
			char buf[64];
			if (fgets(buf, sizeof(buf)-1, fin) != NULL) {
				strip_end(buf);
				fclose(fin);
			        debian_debug("GetExtVal: hostname = ", buf);
				SSTRING *n = new SSTRING (buf);
				tbstr.add (n);
				return n->get();
			}
			fclose (fin);
		}
		return NULL;
	}
	if (vi != NULL) {
		if (*vi == NULL) {
			*vi = new VIEWITEMS;
			(*vi)->read (*cf);
		}
		char tmp[1000];
		const char *val = (**vi).locateval (dci->var, tmp);
		if (val != NULL) {
			if (dci->type == DEBIAN_YESNO
				|| dci->type == DEBIAN_ONOFF) {
				return stricmp(val, "yes")==0
					|| stricmp(val, "true")==0
					|| stricmp(val, "on")==0
					? "1" : "0";
			} else {
				SSTRING *n = new SSTRING (val);
				tbstr.add (n);
				return n->get();
			}
		}
	}
	return NULL;
}

PRIVATE void CONFDB_DEBIAN::setextval (
	DEBIAN_CONFIG_ITEM *dci,
	const char *val)
{
	VIEWITEMS **vi = NULL;
	CONFIG_FILE *cf = NULL;

	debian_debug("SetExtVal: key1 = ", dci->key1);
	debian_debug("SetExtVal: key2 = ", dci->key2);
	debian_debug("SetExtVal: val = ", val);

	if (dci->type == DEBIAN_YESNO) {
		val = strcmp(val,"1")==0 ? "yes" : "no";
	} else if (dci->type == DEBIAN_ONOFF) {
		val = strcmp(val,"1")==0 ? "on" : "off";
	}
	if (dci->file == ETC_DEFAULT_RCS) {
		vi = &debcf.rcs;
		cf = &f_rcs;
		debcf.rcs_modified = true;
	} else if (dci->file == ETC_HOSTNAME) {
		FILE_CFG *fout = f_hostname.fopen("w");
		if (fout != NULL) {
			fputs(val, fout);
			fclose(fout);
		}
	}
	if (cf != NULL) {
		if (*vi == NULL){
			*vi = new VIEWITEMS;
			(*vi)->read (*cf);
		}
		char newline[strlen(dci->var)+1+1+strlen(val)+1+1];
		sprintf (newline,"%s=\"%s\"", dci->var, val);
		VIEWITEM *it = (*vi)->locateassign (dci->var);
		if (it == NULL) {
			it = new VIEWITEM (newline);
			(*vi)->add (it);
		} else {
			it->line.setfrom (newline);
		}
	}
}


PUBLIC const char *CONFDB_DEBIAN::getval (
	const char *prefix,
	const char *key,
	const char *defval)
{
	DEBIAN_CONFIG_ITEM *dci = filter_lookup (prefix, key);
	if (dci != NULL) {
		return getextval(dci);
	} else {
		return orig->getval(prefix, key, defval);
	}
}


PUBLIC int CONFDB_DEBIAN::getall (
	const char *prefix,
	const char *key,
	SSTRINGS &lst,
	bool copy)	// Take a copy of the values
{
	return orig->getall (prefix, key, lst, copy);
}


PUBLIC void CONFDB_DEBIAN::removeall (const char *prefix, const char *key)
{
	orig->removeall(prefix, key);
}


PUBLIC void CONFDB_DEBIAN::add (
	const char *prefix,
	const char *key,
	const char *val)
{

	debian_debug("add: prefix = ", prefix);
	debian_debug("add: key = ", key);
	debian_debug("add: val = ", val);

	DEBIAN_CONFIG_ITEM *dci = filter_lookup (prefix, key);
	if (dci != NULL) {
		setextval(dci, val);
	} else {
		orig->add(prefix, key, val);
	}
}

PUBLIC int CONFDB_DEBIAN::archive (SSTREAM &ss, const char *_sys)
{
	return orig->archive (ss, _sys);
}


PUBLIC int CONFDB_DEBIAN::extract (SSTREAM &ss, const char *_sys)
{
	return orig->extract (ss, _sys);
}


CONFDB *filter_fctnew (CONFDB *orig)
{
	return new CONFDB_DEBIAN(orig);
}
