#include <unistd.h>
#include <stdlib.h>
#include <grp.h>
#include <sys/stat.h>
#include "../paths.h"
#include "internal.h"
#include "userconf.h"
#include "userconf.m"
#include <dialog.h>
#include <string.h>

#define ETC_GTMP "/etc/gtmp"

static USERCONF_HELP_FILE help_groups("groups");
static CONFIG_FILE f_group (ETC_GROUP,help_groups
	,CONFIGF_MANAGED|CONFIGF_TMPLOCK
	,"root","root",0644,subsys_useraccounts);

PUBLIC GROUPS::GROUPS()
{
	/* #Specification: /etc/group / strategy
		/etc/group is read "by hand" instead of using getpwent() to avoid
		getting all those NIS entries. This is done when editing local
		group account.
	*/
	nisentry = NULL;
	nis_at_end = true;
	#if 1
		FILE_CFG *fin = f_group.fopen ("r");
		if (fin != NULL){
			char *line = NULL;
			int len = 0;
			while (fgets_long (line,len,fin)!=NULL){
				strip_end (line);
				if (line[0] != '\0'){
					GROUP *ng = new GROUP(line);
					if (strcmp(ng->getname(),"+")==0){
						delete nisentry;
						nisentry = ng;
						if (getnb()==0) nis_at_end = false;
					}else{
						add (ng);
					}
				}
			}
			free (line);
			fclose (fin);
		}
	#else
		struct group *p;
		setgrent();
		while ((p=getgrent())!=NULL){
			add (new GROUP(p));
		}
		endgrent();
	#endif
	rstmodified();
}

/*
	This object will contain a copy (only pointers).
	It is used as a temporary holder for the normal object, allowing doing
	some sort.
*/
PRIVATE GROUPS::GROUPS(GROUPS *groups)
{
	nisentry = NULL;
	nis_at_end = true;
	int n = groups->getnb();
	neverdelete();
	for (int i=0; i<n; i++) add (groups->getitem(i));
}


PUBLIC GROUPS::~GROUPS()
{
	delete nisentry;
}
/*
	Get one GROUP specification of the table or NULL
*/
PUBLIC GROUP *GROUPS::getitem(int no) const
{
	return (GROUP*)ARRAY::getitem(no);
}
/*
	Return the GROUP specification from its name.
	Return NULL if not found.
*/
PUBLIC GROUP *GROUPS::getitem(const char *name)	const
{
	GROUP *ret = NULL;
	int nbg = getnb();
	for (int i=0; i<nbg; i++){
 		GROUP *grp = getitem(i);
		if (strcmp(grp->getname(),name)==0){
			ret = grp;
			break;
		}
	}
	return ret;
}
/*
	Return the gid of a group from its name.
	Return -1 if there is no group with this name.
*/
PUBLIC int GROUPS::getgid(const char *name) const
{
	GROUP *grp = getitem(name);
	return grp == NULL ? -1 : grp->getgid();
}
/*
	Write the /etc/group file with proper locking
*/
PUBLIC int GROUPS::write(PRIVILEGE *priv, CONFIG_FILE &file)
{
	int ret = -1;
	load_special();
	FILE_CFG *fout = file.fopen (priv,"w");
	if (fout != NULL){
		int nbu = getnb();
		if (nisentry != NULL && !nis_at_end) nisentry->write(fout);
		for (int i=0; i<nbu; i++){
			getitem(i)->write(fout);
		}
		if (nisentry != NULL && nis_at_end) nisentry->write(fout);
		ret = f_group.fclose(fout);
	}
	return ret;
}
/*
	Write the /etc/group file with proper locking
*/
PUBLIC int GROUPS::write(PRIVILEGE *priv)
{
	return write (priv,f_group);
}
/*
	Write the /etc/group file with proper locking
*/
PUBLIC int GROUPS::write()
{
	return write (NULL);
}
/*
	Return the name of the group to use when creating new user (default)
*/
PUBLIC const char *GROUPS::getdefault() const
{
	/* #Specification: userconf / groups / default for creation
		The default group is "users". If this group does not
		exist. then the group "group" is used. If none of those
		group exist, then no default group is proposed to
		the user.
	*/
	/* #todo: userconf / groups and users / default for creation
		It is not clear if a default setup should exist such
		as /etc/default/useradd and could be edited by userconf.

		Currently, we assumed that the default user group is 1.
	*/
	const char *ret = NULL;
	if (getitem("users")!=NULL){
		ret = "users";
	}else if (getitem("group")!=NULL){
		ret = "group";
	}
	return ret;
}

/*
	Get one GROUP specification of the table or NULL from his GID
*/
PUBLIC GROUP *GROUPS::getfromgid(int gid) const
{
	GROUP *ret = NULL;
	int nbg = getnb();
	for (int i=0; i<nbg; i++){
 		GROUP *grp = getitem(i);
		if (grp->getgid() == gid){
			ret = grp;
			break;
		}
	}
	return ret;
}

/*
	Remove one user as a member of a all groups (if there)
	This is called when a user account is deleted.

	Return != 0 if one group was modified (The user was member
	of at least one group).
*/
PUBLIC int GROUPS::delmember(const char *user)
{
	int ret = 0;
	int n = getnb();
	for (int i=0; i<n; i++){
		ret |= getitem(i)->delmember (user);
	}
	return ret;
}

/*
	Collect the alternate group for a user account
	They are simply append to alt with a space between them
*/
PUBLIC void GROUPS::getalt(const char *user, SSTRING &alt)
{
	int n = getnb();
	alt.setfrom ("");
	for (int i=0; i<n; i++){
		GROUP *grp = getitem(i);
		if (grp->is_member (user)){
			if (!alt.is_empty()) alt.append (" ");
			alt.append (grp->getname());
		}
	}
}

/*
	Register a user in several groups
	Return -1 if one group does not exist
*/
PUBLIC int GROUPS::setalt (
	const char *user,
	const char *groups,
	char delim,		// Separator in groups, either comma or a space
	bool test)		// Only check if all groups exist
{
	int ret = 0;
	SSTRINGS tb;
	int n = str_splitline (groups,delim,tb);
	for (int i=0; i<n; i++){
		const char *g = tb.getitem(i)->get();
		GROUP *grp = getitem(g);
		if (grp == NULL){
			ret = -1;
			break;
		}
	}
	if (ret == 0 && !test){
		delmember (user);
		for (int j=0; j<n; j++){
			const char *g = tb.getitem(j)->get();
			GROUP *grp = getitem(g);
			grp->addmember (user);
		}
	}
	return ret;
}

static int cmpbyid (const ARRAY_OBJ *o1, const ARRAY_OBJ *o2)
{
	GROUP *g1 = (GROUP*) o1;
	GROUP *g2 = (GROUP*) o2;
	return g1->getgid() - g2->getgid();
}
/*
	Sort the array of group by name
*/
PUBLIC void GROUPS::sortbyid()
{
	sort (cmpbyid);
}

/*
	Allocate an unused group ID.
*/
PUBLIC int GROUPS::getnew(int hint)
{
	/* #Specification: userconf / groups / group ID allocation
		When creating a new group, the group ID may be left
		blank. An unused group ID will be allocated. The first
		one unused will be allocated.

		We assume that a maximum of 65000 group may be configured.
	*/
	// We sort a temporary object to avoir changing the original
	// order of the file
	GROUPS tmp (this);
	tmp.sortbyid();
	// If over bounds, set hint to 1
	if (hint < 0 || hint > 65000) hint = 1;              

	int nbg = getnb();
	int next = hint;
	int ret = -1;
	for (int i=0; i<nbg; i++){
 		int gid = tmp.getitem(i)->getgid();
		if (gid >= hint){
			if (gid > next){
				ret = next;
				break;
			}else{
				next = gid + 1;
			}
		}
	}
	if (ret == -1) ret = next;
	if ( ret >= 65000 || ret < 1){
		fprintf (stderr,"Out of bounds Bounds: gid: %d\n", ret);
		ret = -1;
	}          
	return ret;
}

/*
	Allocate an unused group ID over 500
*/
PUBLIC int GROUPS::getnew()
{
	return getnew(500);
}

PUBLIC int GROUPS::create (
	const char *name,
	int hint)		// Try to create the group using this GID
					// Allocate another one if this fail
{
	int gid = hint != -1 && getfromgid(hint)==NULL ? hint : getnew();
	GROUP *grp = new GROUP (name,"*",gid,NULL);
	add (grp);
	return gid;
}

static int cmpbyname (const ARRAY_OBJ *o1, const ARRAY_OBJ *o2)
{
	GROUP *g1 = (GROUP*) o1;
	GROUP *g2 = (GROUP*) o2;
	return strcmp(g1->getname(),g2->getname());
}
/*
	Sort the array of group by name
*/
PUBLIC void GROUPS::sortbyname()
{
	sort (cmpbyname);
}

/*
	Sets the help list of a FIELD_COMBO with all the groups
*/
PUBLIC void GROUPS::setcombo (FIELD_COMBO *grpl)
{
	GROUPS grs (this);
	grs.sortbyname();
	for (int i=0; i<grs.getnb(); i++){
		grpl->addopt (grs.getitem(i)->getname());
	}
}

PUBLIC void GROUPS::remove_del (GROUP *grp)
{
	ARRAY::remove_del (grp);
}   

/*
	General edition (addition/deletion/correction) of /etc/group
*/

PUBLIC int GROUPS::edit()
{
	int ret = -1;
	int choice = 0;
	USERS users;
	DIALOG_LISTE dia;
	dia.newf_head ("",MSG_U(H_GROUPS,"Group\tID\tAlternate members"));
	dia.addwhat (MSG_U(I_TOADD,"Select [Add] to add a new definition"));
	while (1){
		GROUPS grs (this);
		grs.sortbyname();
		int nbg = getnb();
		for (int i=0; i<nbg; i++){
			GROUP *g = grs.getitem(i);
			char buf[100];
			unsigned len = sprintf (buf,"%d\t",g->getgid());
			int nbmem = g->tbmem.getnb();
			// We show the first 3 and put ... if there are more
			for (int m=0; m<nbmem && m < 4; m++){
				const char *u = m == 3 ? "..." : g->tbmem.getitem(m)->get();
				unsigned lenu = strlen(u);
				if (len + lenu + 2 >= sizeof(buf)) break;
				len += sprintf (buf+len,"%s ",u);
			}
			dia.set_menuitem (i,g->getname(),buf);
		}
		dia.remove_last (nbg+1);
		MENU_STATUS code = dia.editmenu (
			MSG_U(T_GROUPS,"User groups")
			,MSG_U(I_GROUPS,"You can edit, add, or delete groups")
			,help_groups
			,choice,MENUBUT_ADD);
		if (code == MENU_ESCAPE || code == MENU_QUIT){
			break;
		}else{
			GROUP *grp = grs.getitem(choice);
			if (code == MENU_OK){
				if (grp != NULL){
					int status = grp->edit(users,*this);
					if (status != -1){
						if (status == 1) remove_del (grp);
						write();
						ret = 0;
					}
				}
			}else if (perm_rootaccess(
				MSG_R(P_GROUPDB))){
				if (code == MENU_ADD){
					GROUP *ngrp = new GROUP;
					if (ngrp->edit(users,*this)==0){
						add (ngrp);
						write ();
						ret = 0;
					}else{
						delete ngrp;
					}
				}
			}
		}
	}
	return ret;
}

void groups_edit()
{
	GROUPS groups;
	groups.edit();
}

/*
	Get the ID of a group. Create it if it does not exist.
	Return -1 if it was not created.
*/
int group_getcreate (const char *name, const char *reason)
{
	GROUPS groups;
	int gid = groups.getgid (name);
	if (gid == -1){
		char buf[1000];
		sprintf (buf
			,MSG_U(I_GROUPCREATE
			 ,"The group %s does not exist on your\n"
			  "system. It is needed to %s\n"
			  "\n"
			  "Do you want to create it ? (why not)")
			,name,reason);

		if (dialog_yesno(MSG_U(Q_GROUPCREATE,"Mandatory group creation")
			,buf,help_nil) == MENU_YES){
			gid = groups.getnew();
			GROUP *grp = new GROUP (name,"*",gid,NULL);
			groups.add (grp);
			groups.write();
		}
	}
	return gid;
}

static int gid_pop=-2;
static int gid_ppp;
static int gid_uucp;
static int gid_slip;

PUBLIC void GROUPS::load_special()
{
	gid_pop = getgid (POP_GROUP);
	gid_ppp = getgid (PPP_GROUP);
	gid_uucp = getgid (UUCP_GROUP);
	gid_slip = getgid (SLIP_GROUP);
}

static void groups_locatespecial()
{
	if (gid_pop == -2){
		GROUPS grps;
		grps.load_special();
	}
}


/*
	Return the group ID of POP accounts	or -1 if this group does not exist
*/
int groups_getpop()
{
	groups_locatespecial();
	return gid_pop;
}
/*
	Return the group ID of PPP accounts	or -1 if this group does not exist
*/
int groups_getppp()
{
	groups_locatespecial();
	return gid_ppp;
}
/*
	Return the group ID of SLIP accounts or -1 if this group does not exist
*/
int groups_getslip()
{
	groups_locatespecial();
	return gid_slip;
}

/*
	Return the group ID of UUCP accounts or -1 if this group does not exist
*/
int groups_getuucp()
{
	groups_locatespecial();
	return gid_uucp;
}

/*
      Add one group, no questions asked.
      Calls getnew( ) to get an empty gid over 500.
      Added 3/98, Scimitar.
*/
int groups_add ( const char *name )
{
	GROUPS groups;
	int gid = groups.getgid(name);
	if (gid == -1){
		gid = groups.getnew();
		if (gid != -1){
			GROUP *grp = new GROUP (name, "*",gid,NULL);
			groups.add(grp);
			groups.write();
		}
	}else{
		fprintf (stderr,MSG_U(E_GEXIST,"Group %s already exists\n"),name);
		return -1;
	}
	return gid;
}

/*
	Del one group, no questions asked.
	Added 3/98, Scimitar.
*/
int groups_del ( const char *name )
{
	GROUPS groups;
	GROUP *grp = groups.getitem( name );

	if ( grp ){
		// if group found, delete it
		groups.remove_del(grp);
		groups.write();
	}else{
		fprintf (stderr,MSG_R(E_GNOTEXIST),name);
		return -1;
	}
	return 1;
}


#ifdef TEST

int main (int argc, char *argv[])
{
	groups_edit();
}

#endif


