/*
	links.cpp - A link creator/remover
	Copyright (C) 2004  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "actions.h"
#include "links.h"

#include <qcombobox.h>
#include <qfile.h>
#include <qlabel.h>

#include <kapplication.h>
#include <kdesktopfile.h>
#include <kglobalsettings.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kpushbutton.h>
#include <kstandarddirs.h>

// public

Links::Links(QWidget *parent)
	: QGrid(2, parent, "Links")
{
	setSpacing(5);
	int alignment = AlignVCenter;
	alignment |= kapp->reverseLayout() ? AlignLeft : AlignRight;

	// location
	QLabel *l_location = new QLabel(i18n("Select a location where to create the link:"), this);
	l_location->setAlignment(alignment);
	cb_location = new QComboBox(this, "QComboBox::cb_location");
	cb_location->setFocusPolicy(StrongFocus);
	cb_location->insertItem(SmallIcon("desktop"), i18n("Desktop"));
	cb_location->insertItem(SmallIcon("kmenu"), i18n("K Menu"));
	l_location->setBuddy(cb_location);

	// type
	QLabel *l_type = new QLabel(i18n("Select a type of the link:"), this);
	l_type->setAlignment(alignment);
	cb_type = new QComboBox(this, "QComboBox::cb_type");
	cb_type->setFocusPolicy(StrongFocus);
	cb_type->insertItem(kapp->miniIcon(), i18n("KShutDown"));
	cb_type->insertItem(SmallIcon("exit"), i18n("Standard Logout Dialog"));
	cb_type->insertItem(ks_actions->getIcon(Action::ShutDown), ks_actions->getName(Action::ShutDown));
	cb_type->insertItem(ks_actions->getIcon(Action::Reboot), ks_actions->getName(Action::Reboot));
	cb_type->insertItem(ks_actions->getIcon(Action::LockScreen), ks_actions->getName(Action::LockScreen));
	cb_type->insertItem(ks_actions->getIcon(Action::Logout), ks_actions->getName(Action::Logout));
	l_type->setBuddy(cb_type);

	// create link
	KPushButton *b_createLink = new KPushButton(SmallIcon("filenew"), i18n("Create Link"), this, "KPushButton::b_createLink");
	connect(b_createLink, SIGNAL(clicked()), SLOT(slotCreateLink()));

// TODO: 0.9: disable button if file does not exist
	// remove link
	KPushButton *b_removeLink = new KPushButton(SmallIcon("editdelete"), i18n("Remove Link"), this, "KPushButton::b_removeLink");
	connect(b_removeLink, SIGNAL(clicked()), SLOT(slotRemoveLink()));
}

void Links::createLink(const QString &path, const QString &command, const QString &icon, const QString &name)
{
	if (path.isNull())
		return;

// FIXME: 0.9: description/comment
	KDesktopFile *df = new KDesktopFile(path);
	df->setGroup("Desktop Entry");
	df->writeEntry("Encoding", "UTF-8");
	df->writeEntry("Exec", command);
	df->writeEntry("Icon", icon);
	if (!name.isNull())
		df->writeEntry("Name", name);
	df->writeEntry("StartupNotify", "false");
	df->writeEntry("Type", "Application");
	delete df;

	if (!QFile::exists(path))
	{
		KMessageBox::error(
			0,
			"<qt>" + i18n("Could not create file <b>%1</b>!").arg(path) + "</qt>"
		);
	}
}

void Links::removeLink(const QString &path)
{
	if (QFile::exists(path) && !QFile::remove(path))
	{
		KMessageBox::error(
			0,
			"<qt>" + i18n("Could not remove file <b>%1</b>!").arg(path) + "</qt>"
		);
	}
}

// private

QString Links::getCurrentLocationPath() const
{
	QString path;
	switch (cb_location->currentItem())
	{
		case 0:
			path = KGlobalSettings::desktopPath() + "/";
			break;
		case 1:
			path = locateLocal("apps", "") + "/";
			break;
		default:
			return QString::null;
	}
	switch (cb_type->currentItem())
	{
		case 0: return path += "kshutdown-classic.desktop";
		case 1: return path += "kshutdown-standard.desktop";
		case 2: return path += "kshutdown-shutdown.desktop";
		case 3: return path += "kshutdown-reboot.desktop";
		case 4: return path += "kshutdown-lock.desktop";
		case 5: return path += "kshutdown-logout.desktop";
		default: return QString::null;
	}
}

QString Links::getCurrentTypeCommand() const
{
	switch (cb_type->currentItem())
	{
		case 0: return "kshutdown";
		case 1: return "kshutdown --standard";
		case 2: return "kshutdown --confirm --shutdown";
		case 3: return "kshutdown --confirm --reboot";
		case 4: return "kshutdown --confirm --lock";
		case 5: return "kshutdown --confirm --logout";
		default: return QString::null;
	}
}

QString Links::getCurrentTypeIcon() const
{
	switch (cb_type->currentItem())
	{
		case 0: return "kshutdown";
		case 1: return "exit";
		// sync. with Action::getIcon
		case 2: return "exit";
		case 3: return "reload";
		case 4: return "lock";
		case 5: return "undo";
		default: return QString::null;
	}
}

// private slots

void Links::slotCreateLink()
{
	createLink(
		getCurrentLocationPath(),
		getCurrentTypeCommand(),
		getCurrentTypeIcon(),
		(cb_type->currentItem() == 1 /* evil */) ? i18n("Logout") : cb_type->currentText()
	);
}

void Links::slotRemoveLink()
{
	removeLink(getCurrentLocationPath());
}
