/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 1999-2001 by Matthias Kiefer
                            <matthias.kiefer@gmx.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

**************************************************************************** */
#include "kbabel.h"
#include "kbabeliface.h"
#include "catalog.h"
#include "kbabelsplash.h"

#include "version.h"

#include <dcopclient.h>
#include <kaboutdata.h>
#include <kcmdlineargs.h>
#include <kcursor.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kuniqueapp.h>
#include <kwin.h>

#include <qfile.h>
#include <qtimer.h>

class KBabelInterface : public KBabelIface
{
public:
    KBabelInterface();

    virtual void openURL(QCString url);
    virtual void gotoFileEntry(QCString url, QCString msgid);
};


class KBabelApp : public KUniqueApplication
{
public:
    KBabelApp();
    virtual ~KBabelApp();

    virtual int newInstance();

private:
    KBabelInterface *kbInterface;
};

KBabelApp::KBabelApp()
    : KUniqueApplication()
{
    kbInterface = new KBabelInterface;
}

KBabelApp::~KBabelApp()
{
    delete kbInterface;
}

int KBabelApp::newInstance()
{
    // see if we are starting with session management
    if (isRestored())
        RESTORE(KBabel)
    else
    {
        KCmdLineArgs *args = KCmdLineArgs::parsedArgs();
        
        QTimer timer;
        QWidget *splash=0;
        bool showSplash=args->isSet("splash");

        bool first=true;
        if(KBabel::memberList && !KBabel::memberList->isEmpty())
        {
            first=false;
        }
        
        if( showSplash && first )
        {
            timer.start(2000,true);
            setOverrideCursor(KCursor::waitCursor());
            splash = KBabelSplash::splash();
        }

        QCString msgid=args->getOption("gotomsgid");
        if(!msgid.isEmpty() && args->count() > 0)
        {
            kdDebug() << "gotomsgid" << endl;
            QString m = QString::fromLocal8Bit(msgid);
            kbInterface->gotoFileEntry(args->url(0).url().utf8(),m.utf8());
        }
        else
        {
            // no session.. just start up normally
            KBabel *widget=0;
            if(args->count() > 0)
            {
                KURL u = args->url(0);
                widget=KBabel::winForURL(u);
            }

            if(!widget)
                widget=new KBabel();
        
            while(timer.isActive()) // let the user admire the splash screen ;-)
                processEvents();
        
            widget->show();
            for (int i=0; i < args->count(); i++)
            {
               widget->open( args->url(i) , i != 0 );
            }
        }


        args->clear();

        if(splash)
        {
            delete splash;
        }
        if(showSplash)
        {
            KApplication::restoreOverrideCursor();
            /*
            KMessageBox::information(0,
                    "This is a development version of KBabel!\n"
                    "Please double check the files you edit\n"
                    "and save with this version for correctness.\n\n"
                    "Please report any bug you find to kiefer@kde.org.\n"
                    "Thanks.", "Warning");
            */
        }
    }
 
    return 0;
}

KBabelInterface::KBabelInterface()
    : DCOPObject("KBabelIFace")
{
}

void KBabelInterface::openURL(QCString url)
{
    KURL u(url.data());
    KBabel *kb = KBabel::winForURL(u);
    if(kb)
    {
        KWin::setActiveWindow(kb->topLevelWidget()->winId());
    }
    else
    {
        KMainWindow *mw = 0;
        if(KMainWindow::memberList && !KMainWindow::memberList->isEmpty())
            mw=KMainWindow::memberList->first();

        if(mw && mw->inherits("KBabel"))
        {
            kb = static_cast<KBabel*>(mw);
            KWin::setActiveWindow(kb->topLevelWidget()->winId());
            kb->open(u);
        }
        else
        {
            kb = new KBabel;
            kb->show();
            kb->open(u);
        }
    }
}

void KBabelInterface::gotoFileEntry(QCString url, QCString m)
{    
    KURL u = (QString::fromUtf8(url));
    QString msgid = QString::fromUtf8(m);
    
    KBabel *kb = KBabel::winForURL(u);

    if(kb)
    {
        KWin::setActiveWindow(kb->topLevelWidget()->winId());
    }
    else
    {
        kb = new KBabel;
        kb->show();
        kb->open(u);
    } 
    
    int index = kb->m_view->catalog()->indexForMsgid(msgid);
    if(index >= 0)
    {
        kb->m_view->gotoEntry((uint)index);
    }
}


static KCmdLineOptions options[] =
{
   {"gotomsgid <msgid>",I18N_NOOP("Go to entry with msgid <msgid>"),0},
   {"nosplash",I18N_NOOP("disable splashscreen at startup"),0},
   {"+[file]",I18N_NOOP("files to open"),0},
   {0,0,0}
};


int main(int argc, char **argv)
{
    KAboutData about("kbabel",I18N_NOOP("KBabel"),VERSION,
       I18N_NOOP("An advanced PO-file editor"),KAboutData::License_GPL,
       "(c) 1999-2001, Matthias Kiefer",0,"http://i18n.kde.org/tools/kbabel");

    about.addAuthor("Matthias Kiefer",I18N_NOOP("Author"),"kiefer@kde.org");
    about.addAuthor("Wolfram Diestel"
         ,I18N_NOOP("Wrote diff algorithm, fixed KSpell and gave a lot "
         "of useful hints."),"wolfram@steloj.de");
    about.addAuthor("Andrea Rizzi",I18N_NOOP("Wrote the dictionary plugin "
		"for searching in a database and some other code.")
            ,"rizzi@kde.org");

    about.addCredit("Claudiu Costin",I18N_NOOP("Wrote documentation and sent "
		"many bug reports and suggestions for improvements.")
         ,"claudiuc@geocities.com");
    about.addCredit("Thomas Diehl",I18N_NOOP("Gave many hints to the gui "
         "and the behaviour of KBabel and contributed the beautiful splash screen.")
            ,"thd@kde.org");
    about.addCredit("Stephan Kulow",I18N_NOOP("Helped keep KBabel up to date "
		"with the KDE API and gave alot of other help."),"coolo@kde.org");
	about.addCredit("SuSE GmbH"
					,I18N_NOOP("Sponsors development of KBabel.")
					,"suse@suse.de","http://www.suse.de");
    about.addCredit("",
            I18N_NOOP("KBabel contains code from the following projects:\n"
            "Qt by Trolltech and GNU gettext"));

    about.setTranslator(I18N_NOOP("_: NAME OF TRANSLATORS\nYour names")
            ,I18N_NOOP("_: EMAIL OF TRANSLATORS\nYour emails"));

    // Initialize command line args
    KCmdLineArgs::init(argc, argv, &about);

    // Tell which options are supported
    KCmdLineArgs::addCmdLineOptions( options );

    // Add options from other components
    KUniqueApplication::addCmdLineOptions();


    if(!KUniqueApplication::start())
    {
        return 0;
    }
    
    KBabelApp app;

    return app.exec();
}
