// ****************************************************************************
//  Project:        GUYMAGER
// ****************************************************************************
//  Programmer:     Guy Voncken
//                  Police Grand-Ducale
//                  Service de Police Judiciaire
//                  Section Nouvelles Technologies
// ****************************************************************************

#ifndef __DEVICE_H__
#define __DEVICE_H__

#include "hash.h"


#include <parted/parted.h>
#include <QMetaType>
#include <QReadWriteLock>
#include <QMutex>
#include <QTime>

#include "fifo.h"
#include "info.h"
#include "file.h"

class t_ThreadRead;
class t_ThreadHash;
class t_ThreadCompress;
class t_ThreadWrite;

#define DEVICE_DEFAULT_SECTOR_SIZE ((size_t)512)

class t_Device
{
   public:

      typedef enum
      {
         Idle,
         Acquire,
         AcquirePaused,
         Verify,
         VerifyPaused,
         Cleanup,
         Finished,
         Aborted
      } t_State;

      typedef enum
      {
         None,
         UserRequest,
         ThreadWriteFileError,
         ThreadReadFileError
      } t_AbortReason;

      static const int AutoFracLen       = -1;   // Use up to 2 digits after decimal point (min. 3 significant digits)
      static const int AutoUnitThreshold = -1;

      class t_Acquisition
      {
         public:
            QString         ImagePath;        // Always end with /
            QString         InfoPath;         // Always end with /
            QString         ImageFilename;    // Image filename, without extension
            QString         InfoFilename;     // Info filename, without extension
            t_File::Format  Format;
            bool            CalcHashes;
            bool            VerifySource;
            QString         CaseNumber;
            QString         EvidenceNumber;
            QString         Examiner;
            QString         Description;
            QString         Notes;
      };

   public:
      t_Device (const QString &SerialNumber=QString(), const PedDevice *pPedDev=NULL);
      t_Device (const QString &SerialNumber, const QString &LinuxDevice, const QString &Model,
                                             quint64 SectorSize, quint64 SectorSizePhys, quint64 Size=0);
     ~t_Device ();

      static QVariant GetSerialNumber        (t_pDevice pDevice);
      static QVariant GetLinuxDevice         (t_pDevice pDevice);
      static QVariant GetModel               (t_pDevice pDevice);
      static QVariant GetState               (t_pDevice pDevice);
      static QVariant GetSectorSize          (t_pDevice pDevice);
      static QVariant GetSectorSizePhys      (t_pDevice pDevice);
      static QVariant GetBadSectorCount      (t_pDevice pDevice);
      static QVariant GetBadSectorCountVerify(t_pDevice pDevice);
      static QVariant GetSize                (t_pDevice pDevice);
      static QVariant GetSizeHuman           (t_pDevice pDevice);
      static QVariant GetSizeHumanFrac       (t_pDevice pDevice, bool SI=true, int FracLen=AutoFracLen, int UnitThreshold=AutoUnitThreshold);
      static QVariant GetProgress            (t_pDevice pDevice);
      static QVariant GetCurrentSpeed        (t_pDevice pDevice);
      static QVariant GetAverageSpeed        (t_pDevice pDevice);
      static QVariant GetRemaining           (t_pDevice pDevice);
      static QVariant GetFifoStatus          (t_pDevice pDevice);

      inline void SetCurrentWritePos (quint64 Pos)
      {
         SemCurrentWritePos.lockForWrite ();
         CurrentWritePos = Pos;
         SemCurrentWritePos.unlock ();
      }

      inline void IncCurrentWritePos (int Inc)
      {
         SemCurrentWritePos.lockForWrite ();
         CurrentWritePos += Inc; //lint !e737 Loss of sign
         SemCurrentWritePos.unlock ();
      }

      inline quint64 GetCurrentWritePos (void)
      {
         quint64 Pos;
         SemCurrentWritePos.lockForRead ();
         Pos = CurrentWritePos;
         SemCurrentWritePos.unlock ();
         return Pos;
      }

      inline void SetCurrentVerifyPos (quint64 Pos)
      {
         SemCurrentVerifyPos.lockForWrite ();
         CurrentVerifyPos = Pos;
         SemCurrentVerifyPos.unlock ();
      }

      inline void IncCurrentVerifyPos (int Inc)
      {
         SemCurrentVerifyPos.lockForWrite ();
         CurrentVerifyPos += Inc; //lint !e737 Loss of sign
         SemCurrentVerifyPos.unlock ();
      }

      inline quint64 GetCurrentVerifyPos (void)
      {
         quint64 Pos;
         SemCurrentVerifyPos.lockForRead ();
         Pos = CurrentVerifyPos;
         SemCurrentVerifyPos.unlock ();
         return Pos;
      }

      inline void AddBadSector (quint64 Sector)
      {
         SemBadSectors.lock ();
         if (State == Acquire)
              BadSectors      .append(Sector);
         else BadSectorsVerify.append(Sector);
         SemBadSectors.unlock ();
      }

      APIRET GetBadSectors (QList<quint64> &BadSectorsCopy, bool Verify)
      {
         SemBadSectors.lock ();
         if (Verify)
              BadSectorsCopy = BadSectorsVerify;
         else BadSectorsCopy = BadSectors;
         SemBadSectors.unlock ();
         return NO_ERROR;
      }

      quint64 GetBadSectorCount (bool Verify)
      {
         quint64 Count;
         SemBadSectors.lock ();
         if (Verify)
              Count = BadSectorsVerify.count();  //lint !e732 Loss of sign
         else Count = BadSectors      .count();  //lint !e732 Loss of sign
         SemBadSectors.unlock ();
         return Count;
      }

      APIRET ClearBadSectors (void)
      {
         SemBadSectors.lock ();
         BadSectorsVerify.clear();
         BadSectors      .clear();
         SemBadSectors.unlock ();
         return NO_ERROR;
      }

      APIRET GetMessage (QString &MessageCopy)
      {
         SemMessage.lock ();
         MessageCopy = Message;
         SemMessage.unlock ();
         return NO_ERROR;
      }

      APIRET SetMessage (const QString &NewMessage)
      {
         SemMessage.lock ();
         Message = NewMessage;
         SemMessage.unlock ();
         return NO_ERROR;
      }

      bool HasHashThread         (void) const;
      bool HasCompressionThreads (void) const;
      const char *StateStr (void);

   private:
      void Initialise (void);
      void InitialiseDeviceSpecific (const QString &SerialNumber, const QString &LinuxDevice, const QString &Model,
                                    quint64 SectorSize, quint64 SectorSizePhys, quint64 Size);

   public:  QString                   SerialNumber;
            QString                   Model;
            QString                   LinuxDevice;      // for instance /dev/hda
            bool                      Local;            // it's a local device (cannot be acquired)
            quint64                   SectorSize;
            quint64                   SectorSizePhys;
            quint64                   Size;
            bool                      Removable;

            t_State                   State;
   private: QString                   Message;          // Used by the threads to communicate messages displayed in spreadsheet field "state"
   public:  bool                      AbortRequest;
            t_AbortReason             AbortReason;
            bool                      DeleteAfterAbort;

            t_Acquisition             Acquisition;
            t_Info                    Info;             // Must only be used if Aqcuisition.InfoFilename is valid!

            FILE                    *pFileSrc;
            quint64                   CurrentReadPos;   // Accessed from different threads, but never at the same time. During acquisition, it is exclusively accessed by the read threads. Using a mutex would be nicer, but could decrease performance.
   private: quint64                   CurrentVerifyPos; // Accessed concurrently, use appropriate functions
   private: quint64                   CurrentWritePos;  // Accessed concurrently, use appropriate functions
   private: QList<quint64>            BadSectors;       // Accessed concurrently, use appropriate functions
   private: QList<quint64>            BadSectorsVerify; // Accessed concurrently, use appropriate functions
   public:  bool                      FallbackMode;     // Set if an error occurs while reading a large block. If set, sectors are read individually until the next large block boundary.
            QDateTime                 StartTimestamp;
            QDateTime                 StartTimestampVerify;
            QDateTime                 StopTimestamp;

   public:  t_ThreadRead            *pThreadRead;
            t_ThreadHash            *pThreadHash;
            t_ThreadWrite           *pThreadWrite;
            QList<t_ThreadCompress *> ThreadCompressList;

   public:  t_pFifo                  pFifoRead;         // Pointers to the Fifos used by the different
            t_pFifo                  pFifoHashIn;       // threads. Some of them point to the same Fifos,
            t_pFifo                  pFifoHashOut;      // for instance pFifoRead and pFifoHashIn.
            t_pFifo                  pFifoWrite;
            t_pFifoCompressIn        pFifoCompressIn;
            t_pFifoCompressOut       pFifoCompressOut;
            int                       FifoMaxBlocks;
            unsigned int              FifoBlockSize;


            t_HashMD5Digest           MD5Digest;
            t_HashMD5Digest           MD5DigestVerify;
            t_HashSHA256Digest        SHA256Digest;
            t_HashSHA256Digest        SHA256DigestVerify;

   public:  quint64                   PrevPos;          // Some variables for
            QTime                     PrevTimestamp;    // the current speed
            double                    PrevSpeed;        // calculation.
            bool                      Checked;          // Helper variable for matching algorithm in SlotScanFinished, not used elsewhere.

   private: QReadWriteLock            SemCurrentWritePos;
   private: QReadWriteLock            SemCurrentVerifyPos;
   private: QMutex                    SemBadSectors;
   private: QMutex                    SemMessage;
};

class t_DeviceList: public QList<t_pDevice>
{
   public:
      t_DeviceList (void);
     ~t_DeviceList ();

      t_pDevice AppendNew (const QString &SerialNumber, const PedDevice *pPedDev);
      t_pDevice AppendNew (const QString &SerialNumber, const QString &LinuxDevice, const QString &Model,
                           quint64 SectorSize, quint64 SectorSizePhys, quint64 Size=0);
      APIRET MatchDevice (t_pcDevice pDevCmp, t_pDevice &pDeviceMatch);
};

typedef t_DeviceList *t_pDeviceList;

Q_DECLARE_METATYPE(t_pDeviceList); //lint !e19 Useless declaration


// ------------------------------------
//             Error codes
// ------------------------------------

enum
{
   ERROR_DEVICE_SERNR_MATCH_MODEL_MISMATCH = ERROR_BASE_DEVICE + 1,
   ERROR_DEVICE_SERNR_MATCH_LENGTH_MISMATCH,
   ERROR_DEVICE_BAD_STATE,
   ERROR_DEVICE_BAD_ABORTREASON,
   ERROR_DEVICE_NOT_CLEAN
};


#endif

