/*
 * GUI searchbar module
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <gnome.h>
#include "diff.h"
#include "gui.h"
#include "gdwin.h"
#include "basepane-widget.h"
#include "dirview.h"
#include "mergeview.h"
#include "viewmisc.h"

/* Keys for gtk_object_[get|set]_data() */
/* Two buttons share the same callback function, so I need to distinguish them. */
#define SEARCH_BUTTON_KEY	"sbk"

/* Data structure definition */
struct _GdiffSearchBar {
	GtkWidget *container;
	GtkWidget *entry;
	GtkWidget *button[MAX_NUM_COMPARE_FILES];
};

/* Private function declarations */
static void search_cb(GtkWidget *widget, gpointer data);
	

/**
 * searchbar_create:
 * Create a search-bar.
 **/
GdiffSearchBar*
searchbar_new(GDiffWindow *gdwin)
{
	GdiffSearchBar *searchbar;
	GnomeApp *app = gdwin->app;
	GtkWidget *hbox;
	GtkWidget *entry;
	GtkTooltips *tooltips;
	int n;
	const char *tooltip_str[MAX_NUM_COMPARE_FILES] = {
		N_("Search in first file"),
		N_("Search in second file"),
		N_("Search in third file"),
	};
	
	searchbar = g_new(GdiffSearchBar, 1);
	
	hbox = gtk_hbox_new(FALSE, 0);
	searchbar->container = hbox;
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 2);

	tooltips = gtk_tooltips_new();

	entry = gtk_entry_new();
	searchbar->entry = entry;
	gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);

	for (n = 0; n < MAX_NUM_COMPARE_FILES; n++) {
		GtkWidget *button;
		GtkWidget *pixmapw;

		button = gtk_button_new();
		searchbar->button[n] = button;
		gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
		gtk_object_set_data(GTK_OBJECT(button),
							SEARCH_BUTTON_KEY, GINT_TO_POINTER(n));
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
						   GTK_SIGNAL_FUNC(search_cb), gdwin);
		
		gtk_tooltips_set_tip(tooltips, button, _(tooltip_str[n]), NULL);
		pixmapw = gnome_stock_pixmap_widget(GTK_WIDGET(app), GNOME_STOCK_PIXMAP_SEARCH);
		gtk_container_add(GTK_CONTAINER(button), pixmapw);
		gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	}
		
	gtk_widget_show_all(hbox);

	gnome_app_add_docked(app, hbox, "search box", GNOME_DOCK_ITEM_BEH_NORMAL,
						 GNOME_DOCK_TOP, 1, 2, 0);

	return searchbar;
}

void
searchbar_delete(GdiffSearchBar *searchbar)
{
	g_free(searchbar);
}


void
searchbar_show(GdiffSearchBar *searchbar, gboolean to_show)
{
	if (to_show)
		gtk_widget_show(GTK_WIDGET(searchbar->container->parent));
	else
		gtk_widget_hide(GTK_WIDGET(searchbar->container->parent));
}

void
searchbar_grab_focus(GdiffSearchBar *searchbar)
{
	/* Inspired by MGEdit <http://seki.math.sci.hokudai.ac.jp:20080/gtk/mgedit/> */
	gtk_widget_grab_focus(searchbar->entry);
}

void
searchbar_show_button(GdiffSearchBar *searchbar, int num)
{
	int n;

	for (n = 0; n < num; n++)
		gtk_widget_set_sensitive(searchbar->button[n], TRUE);
	
	for (n = num; n < MAX_NUM_COMPARE_FILES; n++)
		gtk_widget_set_sensitive(searchbar->button[n], FALSE);
}

/* ---The followings are private functions--- */
static void
search_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	GtkWidget *view;
	WhichFile whichfile;
	const char *entry_string;
	
	view = gdwin_current_view(gdwin);
	if (view == NULL)
		return;

	whichfile = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(widget), SEARCH_BUTTON_KEY));
	
	entry_string = gtk_entry_get_text(GTK_ENTRY(gdwin->searchbar->entry));
	if (entry_string == NULL || entry_string[0] == '\0')
		return;

	if (GDIFF_IS_DIRVIEW(view)) {
		gdiff_dirview_search_string(GDIFF_DIRVIEW(view), entry_string, whichfile);
	} else {
		GdiffBasePane *basepane;

		
		if (GDIFF_IS_MERGEVIEW(view))/* ugly */
			basepane = view_get_pane(view, "auxpane");/* multipane-widget */
		else
			basepane = view_get_pane(view, "basepane");
		if (basepane == NULL)
			return;
		gdiff_basepane_search_string(basepane, entry_string, whichfile);
	}
}
