/*
 * GUI menubar & toolbar module
 * Almost every callback routine internally calls the related action routine
 * defined in actions.[ch].
 * Too long..., and becoming too longer... Sigh.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <gnome.h>
#include "diff.h"
#include "gui.h"
#include "menu-tool-bar.h"
#include "gdwin.h"
#include "actions.h"
#include "dirview.h"
#include "properties.h"


/* Private function declarations */
static void update_common(const Preference *pref);
static void update_dirview(GDiffWindow *gdwin, const Preference *pref);
static void menu_dirv_block(GDiffWindow *gdwin);
static void menu_dirv_unblock(GDiffWindow *gdwin);
static void update_fileview(GDiffWindow *gdwin, const Preference *pref);
static void menu_filev_block(GDiffWindow *gdwin);
static void menu_filev_unblock(GDiffWindow *gdwin);

static void file_open_cb(GtkWidget *widget, gpointer data);
static void file_close_cb(GtkWidget *widget, gpointer data);
static void mode_change_cb(GtkWidget *widget, gpointer data);
static void show_linenum_cb(GtkWidget *widget, gpointer data);
static void text_wrap_cb(GtkWidget *widget, gpointer data);
static void highlight_cb(GtkWidget *widget, gpointer data);
static void show_fill_cb(GtkWidget *widget, gpointer data);
static void go_dirview_cb(GtkWidget *widget, gpointer data);
static void move_first_cb(GtkWidget *widget, gpointer data);
static void move_last_cb(GtkWidget *widget, gpointer data);
static void move_next_cb(GtkWidget *widget, gpointer data);
static void move_prev_cb(GtkWidget *widget, gpointer data);
static void move_current_cb(GtkWidget *widget, gpointer data);
static void move_rel_next_cb(GtkWidget *widget, gpointer data);
static void move_rel_prev_cb(GtkWidget *widget, gpointer data);
static void set_preference_cb(GtkWidget *widget, gpointer data);
static void show_path_cb(GtkWidget *widget, gpointer data);
static void row_hide_func_cb(GtkWidget *widget, gpointer data);
static void row_hide_stat_cb(GtkWidget *widget, gpointer data);
static void show_toolbar_cb(GtkWidget *widget, gpointer data);
static void show_searchbar_cb(GtkWidget *widget, gpointer data);
static void show_statusbar_cb(GtkWidget *widget, gpointer data);
static void show_tabs_cb(GtkWidget *widget, gpointer data);
static void reload_cb(GtkWidget *widget, gpointer data);
static void search_cb(GtkWidget *widget, gpointer data);
static void checksum_cb(GtkWidget *widget, gpointer data);
static void insert_remove_cb(GtkWidget *w, gpointer data);
static void merge_output_cb(GtkWidget *w, gpointer data);

static void quit_cb(GtkWidget *widget, gpointer data);
static void about_cb(GtkWidget *widget, gpointer data);


/* Private variables */
/* menubar definitions */
static GnomeUIInfo file_menu[] = {
	GNOMEUIINFO_MENU_OPEN_ITEM(file_open_cb, NULL),
	GNOMEUIINFO_SEPARATOR,
	GNOMEUIINFO_MENU_CLOSE_ITEM(file_close_cb, NULL),
	GNOMEUIINFO_MENU_EXIT_ITEM(quit_cb, NULL),
	GNOMEUIINFO_END
};

static GnomeUIInfo settings_menu[] = {
	GNOMEUIINFO_MENU_PREFERENCES_ITEM(set_preference_cb, NULL),

	GNOMEUIINFO_SEPARATOR,

#define MENUITEM_SHOW_TOOLBAR	2
	{GNOME_APP_UI_TOGGLEITEM, N_("Show _Toolbar"),
	 N_("Show toolbar"),
	 show_toolbar_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
#define MENUITEM_SHOW_SEARCHBAR	3
	{GNOME_APP_UI_TOGGLEITEM, N_("Show _Searchbar"),
	 N_("Show searchbar"),
	 show_searchbar_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
#define MENUITEM_SHOW_STATUSBAR	4
	{GNOME_APP_UI_TOGGLEITEM, N_("Show St_atusbar"),
	 N_("Show statusbar"),
	 show_statusbar_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
#define MENUITEM_SHOW_TABS		5
	{GNOME_APP_UI_TOGGLEITEM, N_("Show Ta_bs"),
	 N_("Show notebook tabs"),
	 show_tabs_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
	GNOMEUIINFO_END
};

/* Keys for gtk_object_[get|set]_data() */
#define MENUITEM_HIDE_FUNC_KEY	"hfunc"
#define MENUITEM_HIDE_STAT_KEY	"hstat"

static GnomeUIInfo show_path_radio_menu[] = {
#define MENUITEM_FIRST_SHOW_PATH	0
#define MENUITEM_SHOW_PATH_FILE		0
	GNOMEUIINFO_RADIOITEM_DATA(N_("_File name"),
							   N_("Show only the file name on directory view"),
							   show_path_cb,
							   GINT_TO_POINTER(SHOW_PATH_FILE), NULL),
#define MENUITEM_SHOW_PATH_ORIG		1
	GNOMEUIINFO_RADIOITEM_DATA(N_("_Original path"),
							   N_("Show the original path on directory view"),
							   show_path_cb,
							   GINT_TO_POINTER(SHOW_PATH_ORIG), NULL),
#define MENUITEM_SHOW_PATH_REL		2
	GNOMEUIINFO_RADIOITEM_DATA(N_("_Relative path"),
							   N_("Show the relative path on directory view"),
							   show_path_cb,
							   GINT_TO_POINTER(SHOW_PATH_REL), NULL),
#define MENUITEM_LAST_SHOW_PATH		3
	GNOMEUIINFO_END
};
static GnomeUIInfo show_path_menu[] = {
	GNOMEUIINFO_RADIOLIST(show_path_radio_menu),
	GNOMEUIINFO_END
};
static GnomeUIInfo dirview_menu[] = {
	GNOMEUIINFO_SUBTREE(N_("_Show"), show_path_menu),

	GNOMEUIINFO_SEPARATOR,

#define MENUITEM_HIDE_FUNC_FIRST	2
#define MENUITEM_HIDE_FUNC_EMACS	2
	{GNOME_APP_UI_TOGGLEITEM, N_("_Hide emacs backup"),
	 N_("Hide emacs backup files on directory view"),
	 row_hide_func_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
#define MENUITEM_HIDE_FUNC_OBJ		3
	{GNOME_APP_UI_TOGGLEITEM, N_("Hide ob_ject files"),
	 N_("Hide object files on directory view"),
	 row_hide_func_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
#define MENUITEM_HIDE_FUNC_LAST		4

	GNOMEUIINFO_SEPARATOR,
	
#define MENUITEM_HIDE_STAT_FIRST	5
#define MENUITEM_HIDE_STAT_BINARY	5
	{GNOME_APP_UI_TOGGLEITEM, N_("Hide _Binary"),
	 N_("Hide binary files"),
	 row_hide_stat_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
#define MENUITEM_HIDE_STAT_ONLY		6
	{GNOME_APP_UI_TOGGLEITEM, N_("Hide _Only"),
	 N_("Hide only files"),
	 row_hide_stat_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
#define MENUITEM_HIDE_STAT_DIFF		7
	{GNOME_APP_UI_TOGGLEITEM, N_("Hide _Different"),
	 N_("Hide different files"),
	 row_hide_stat_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
#define MENUITEM_HIDE_STAT_IDENT	8
	{GNOME_APP_UI_TOGGLEITEM, N_("Hide _Identical"),
	 N_("Hide identical files"),
	 row_hide_stat_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
#define MENUITEM_HIDE_STAT_LAST		9

	GNOMEUIINFO_END
};

static GnomeUIInfo pane_radio_menu[] = {
#define MENUITEM_FIRST_VIEW			0
#define MENUITEM_ONEPANE_VIEW		0
	GNOMEUIINFO_RADIOITEM_DATA(N_("_One pane"), N_("One pane view"),
							   mode_change_cb,
							   GINT_TO_POINTER(ONEPANE_MASK_VIEW), NULL),
#define MENUITEM_MULTIPANE_VIEW		1
	GNOMEUIINFO_RADIOITEM_DATA(N_("_Multi panes"), N_("Multi panes view"),
							   mode_change_cb,
							   GINT_TO_POINTER(MULTIPANE_MASK_VIEW), NULL),
#define MENUITEM_MERGE_VIEW			2
	GNOMEUIINFO_RADIOITEM_DATA(N_("M_erge"), N_("Merge view"),
							   mode_change_cb,
							   GINT_TO_POINTER(MERGE_MASK_VIEW), NULL),
#define MENUITEM_LAST_VIEW			3
	GNOMEUIINFO_END
};
static GnomeUIInfo pane_menu[] = {
	GNOMEUIINFO_RADIOLIST(pane_radio_menu),
	GNOMEUIINFO_END
};

static GnomeUIInfo fileview_menu[] = {
	GNOMEUIINFO_SUBTREE(N_("P_ane"), pane_menu),
#define MENUITEM_LINE_SHOW		1
	{GNOME_APP_UI_TOGGLEITEM, N_("_Line number"),
	 N_("Show line numbers"),
	 show_linenum_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
#define MENUITEM_TEXT_WRAP		2
	{GNOME_APP_UI_TOGGLEITEM, N_("_Text wrap"),
	 N_("Toggle text wrap"),
	 text_wrap_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
#define MENUITEM_HIGHLIGHT		3
	{GNOME_APP_UI_TOGGLEITEM, N_("_Highlight"),
	 N_("Highlight selected lines"),
	 highlight_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
#define MENUITEM_SHOW_FILL		4
	{GNOME_APP_UI_TOGGLEITEM, N_("_Sync lines"),
	 N_("Synchronize display lines"),
	 show_fill_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 0, (GdkModifierType)0, NULL},
	GNOMEUIINFO_END
};

static GnomeUIInfo dview_action_menu[] = {
	{GNOME_APP_UI_ITEM, N_("_Reload dirs"),
	 N_("Reload the current directories"),
	 reload_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_REFRESH,
	 GDK_r, GDK_CONTROL_MASK|GDK_SHIFT_MASK, NULL},
	{GNOME_APP_UI_ITEM, N_("_Compare check sum"),
	 N_("Compare check sum of selected files"),
	 checksum_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_SPELLCHECK,
	 0, (GdkModifierType)0, NULL},
	GNOMEUIINFO_MENU_FIND_ITEM(search_cb, NULL),
	GNOMEUIINFO_END
};

static GnomeUIInfo fview_action_menu[] = {
	{GNOME_APP_UI_ITEM, N_("_Go directory view"),
	 N_("Go directory view"),
	 go_dirview_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_HOME,
	 GDK_h, GDK_CONTROL_MASK, NULL},
	{GNOME_APP_UI_ITEM, N_("_Next difference"),
	 N_("Go to next difference"),
	 move_next_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_DOWN,
	 GDK_n, GDK_CONTROL_MASK, NULL},
	{GNOME_APP_UI_ITEM, N_("_Prev difference"),
	 N_("Go to previous difference"),
	 move_prev_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_UP,
	 GDK_p, GDK_CONTROL_MASK, NULL},
	{GNOME_APP_UI_ITEM, N_("_First difference"),
	 N_("Go to the first difference"),
	 move_first_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_TOP,
	 GDK_t, GDK_CONTROL_MASK, NULL},
	{GNOME_APP_UI_ITEM, N_("_Last difference"),
	 N_("Go to the last difference"),
	 move_last_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_BOTTOM,
	 GDK_v, GDK_CONTROL_MASK, NULL},
	{GNOME_APP_UI_ITEM, N_("_Current difference"),
	 N_("Go to the current difference"),
	 move_current_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_JUMP_TO,
	 GDK_l, GDK_CONTROL_MASK, NULL},
#define MENUITEM_REL_NEXT_DIFF		6
	{GNOME_APP_UI_ITEM, N_("Relative N_ext difference"),
	 N_("Go to relative next difference"),
	 move_rel_next_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 GDK_n, GDK_SHIFT_MASK, NULL},
#define MENUITEM_REL_PREV_DIFF		7
	{GNOME_APP_UI_ITEM, N_("Relative Pre_v difference"),
	 N_("Go to relative previous difference"),
	 move_rel_prev_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_NONE, NULL,
	 GDK_p, GDK_SHIFT_MASK, NULL},
	{GNOME_APP_UI_ITEM, N_("_Reload files"),
	 N_("Reload the current files"),
	 reload_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_REFRESH,
	 GDK_r, GDK_CONTROL_MASK, NULL},
	GNOMEUIINFO_MENU_FIND_ITEM(search_cb, NULL),
	GNOMEUIINFO_END
};


/* for GnomeUIInfo merge_action_menu[] */
#include "merge-menu-d.h"

static GnomeUIInfo help_menu[] = {
	/*GNOMEUIINFO_HELP(APPNAME),*/
	GNOMEUIINFO_MENU_ABOUT_ITEM(about_cb, NULL),
	GNOMEUIINFO_END
};

static GnomeUIInfo main_menu[] = {
	GNOMEUIINFO_MENU_FILE_TREE(file_menu),
	GNOMEUIINFO_MENU_SETTINGS_TREE(settings_menu),
#define MENUITEM_DIR_VIEW		2
	GNOMEUIINFO_SUBTREE(N_("D_irectory view"), dirview_menu),
#define MENUITEM_FILE_VIEW		3
	GNOMEUIINFO_SUBTREE(N_("Fi_le view"), fileview_menu),
#define MENUITEM_DVIEW_ACTION	4
	GNOMEUIINFO_SUBTREE(N_("_Action"), dview_action_menu),
#define MENUITEM_FVIEW_ACTION	5
	GNOMEUIINFO_SUBTREE(N_("_Action"), fview_action_menu),
#define MENUITEM_MERGE_ACTION	6
	GNOMEUIINFO_SUBTREE(N_("_Merge"), merge_action_menu),
	GNOMEUIINFO_MENU_HELP_TREE(help_menu),
	GNOMEUIINFO_END
};


/* toolbar definitions */
static GnomeUIInfo fview_toolbar[] = {
	{GNOME_APP_UI_ITEM, NULL, N_("Go directory view"),
	 go_dirview_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_HOME,
	 0, 0, NULL},
	{GNOME_APP_UI_ITEM, NULL, N_("Go to next difference"),
	 move_next_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_DOWN,
	 0, 0, NULL},
	{GNOME_APP_UI_ITEM, NULL, N_("Go to previous difference"),
	 move_prev_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_UP,
	 0, 0, NULL},
	{GNOME_APP_UI_ITEM, NULL, N_("Go to the first difference"),
	 move_first_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_TOP,
	 0, 0, NULL},
	{GNOME_APP_UI_ITEM, NULL, N_("Go to the last difference"),
	 move_last_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_BOTTOM,
	 0, 0, NULL},
	{GNOME_APP_UI_ITEM, NULL, N_("Go to the current difference"),
	 move_current_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_JUMP_TO,
	 0, 0, NULL},
	{GNOME_APP_UI_ITEM, NULL, N_("Reload the current files"),
	 reload_cb, NULL, NULL,
	 GNOME_APP_PIXMAP_STOCK, GNOME_STOCK_PIXMAP_REFRESH,
	 0, 0, NULL},
	GNOMEUIINFO_END
};


/**
 * menubar_create:
 * Create menubar.
 **/
void
menubar_create(GDiffWindow *gdwin)
{
	int i;
	
	gnome_app_create_menus_with_data(gdwin->app, main_menu, gdwin);

	/* Set index to menu item, to be used in the callback function */
	for (i = MENUITEM_HIDE_FUNC_FIRST; i < MENUITEM_HIDE_FUNC_LAST; i++) {
		gtk_object_set_data(GTK_OBJECT(dirview_menu[i].widget),
							MENUITEM_HIDE_FUNC_KEY, GINT_TO_POINTER(i));
	}
	for (i = MENUITEM_HIDE_STAT_FIRST; i < MENUITEM_HIDE_STAT_LAST; i++) {
		gtk_object_set_data(GTK_OBJECT(dirview_menu[i].widget),
							MENUITEM_HIDE_STAT_KEY, GINT_TO_POINTER(i));
	}

	merge_menu_set_data();	/* Macro defined in merge-menu-d.h */
	
	menubar_update(gdwin, &g_pref, NO_VIEW);
}

/**
 * menubar_install_hints_for_statusbar:
 **/
void
menubar_install_hints_for_statusbar(GDiffWindow *gdwin)
{
	gnome_app_install_menu_hints(gdwin->app, main_menu);
}

/**
 * show_menu_item:
 **/
#define show_menu_item(w)	G_STMT_START {	\
	gtk_widget_set_sensitive(w, TRUE);		\
	gtk_widget_show(w);						\
} G_STMT_END

#define hide_menu_item(w)	G_STMT_START {	\
	gtk_widget_hide(w);						\
	gtk_widget_set_sensitive(w, FALSE);		\
} G_STMT_END

/**
 * menubar_update:
 **/
void
menubar_update(GDiffWindow *gdwin, const Preference *pref, ViewType vtype)
{
	/* top-level menus */
	if (vtype == NO_VIEW) {
		show_menu_item(main_menu[MENUITEM_DIR_VIEW].widget);
		show_menu_item(main_menu[MENUITEM_FILE_VIEW].widget);
		hide_menu_item(main_menu[MENUITEM_DVIEW_ACTION].widget);
		hide_menu_item(main_menu[MENUITEM_FVIEW_ACTION].widget);
		hide_menu_item(main_menu[MENUITEM_MERGE_ACTION].widget);
	} else if (vtype == DIR_VIEW) {
		show_menu_item(main_menu[MENUITEM_DIR_VIEW].widget);
		show_menu_item(main_menu[MENUITEM_FILE_VIEW].widget);/* show it */
		show_menu_item(main_menu[MENUITEM_DVIEW_ACTION].widget);
		hide_menu_item(main_menu[MENUITEM_FVIEW_ACTION].widget);
		hide_menu_item(main_menu[MENUITEM_MERGE_ACTION].widget);
	} else if (vtype & FILE_MASK_VIEW) {
		hide_menu_item(main_menu[MENUITEM_DIR_VIEW].widget);
		show_menu_item(main_menu[MENUITEM_FILE_VIEW].widget);
		hide_menu_item(main_menu[MENUITEM_DVIEW_ACTION].widget);
		show_menu_item(main_menu[MENUITEM_FVIEW_ACTION].widget);
		if (vtype & MERGE_MASK_VIEW) {
			show_menu_item(main_menu[MENUITEM_MERGE_ACTION].widget);
			if (vtype & COMPARE2_MASK_VIEW) {
				merge_menu_hide3();
			} if (vtype & COMPARE3_MASK_VIEW) {
				merge_menu_show3();
			}
		} else {
			hide_menu_item(main_menu[MENUITEM_MERGE_ACTION].widget);
		}
	} else {
		g_assert_not_reached();
	}

	/* During update, block signal handlers */
	menu_dirv_block(gdwin);
	menu_filev_block(gdwin);
	
	update_dirview(gdwin, pref);
	update_fileview(gdwin, pref);
	update_common(pref);
	
	menu_dirv_unblock(gdwin);
	menu_filev_unblock(gdwin);
}


/**
 * toolbar_create:
 * Create toolbar.
 * Formerly, I used gnome_app_create_toolbar_with_data(), but I stop using it,
 * because toolbar becomes hard to treat.
 **/
void
toolbar_create(GDiffWindow *gdwin)
{
	GtkWidget *toolbar;
	GnomeDockItemBehavior behavior = GNOME_DOCK_ITEM_BEH_NORMAL;/* this should be default */

	toolbar = gtk_toolbar_new(GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);
	gdwin->toolbar = toolbar;
	gnome_app_fill_toolbar_with_data(GTK_TOOLBAR(toolbar), fview_toolbar, NULL, gdwin);

	if (!gnome_preferences_get_toolbar_detachable())
		behavior |= GNOME_DOCK_ITEM_BEH_LOCKED;

	gnome_app_add_toolbar(gdwin->app, GTK_TOOLBAR(toolbar),
						  GNOME_APP_TOOLBAR_NAME, behavior,
						  GNOME_DOCK_TOP, 1, 0, 0);
}

/**
 * bars_modify:
 * This is rather a hack code. Take care of hiding toolbar etc. at startup.
 **/
void
bars_modify(GDiffWindow *gdwin)
{
	show_toolbar_cb(settings_menu[MENUITEM_SHOW_TOOLBAR].widget, gdwin);
	show_searchbar_cb(settings_menu[MENUITEM_SHOW_TOOLBAR].widget, gdwin);
	show_statusbar_cb(settings_menu[MENUITEM_SHOW_TOOLBAR].widget, gdwin);
}


/* ---The followings are private functions--- */
/**
 * update_common:
 **/
static void
update_common(const Preference *pref)
{
	/* Preferences related to window */
	gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(settings_menu[MENUITEM_SHOW_TOOLBAR].widget),
		pref->winpref.show_toolbar);
	gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(settings_menu[MENUITEM_SHOW_SEARCHBAR].widget),
		pref->winpref.show_searchbar);
	gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(settings_menu[MENUITEM_SHOW_STATUSBAR].widget),
		pref->winpref.show_statusbar);
	gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(settings_menu[MENUITEM_SHOW_TABS].widget),
		pref->winpref.show_tabs);
}

/**
 * update_dirview:
 * menu update routine for directory view case.
 **/
static void
update_dirview(GDiffWindow *gdwin, const Preference *pref)
{
	/* Various states */
	if (pref->dvpref.show_path == SHOW_PATH_FILE) {
		gtk_check_menu_item_set_active(
			GTK_CHECK_MENU_ITEM(show_path_radio_menu[MENUITEM_SHOW_PATH_FILE].widget), TRUE);
	} else if (pref->dvpref.show_path == SHOW_PATH_ORIG) {
		gtk_check_menu_item_set_active(
			GTK_CHECK_MENU_ITEM(show_path_radio_menu[MENUITEM_SHOW_PATH_ORIG].widget), TRUE);
	} else if (pref->dvpref.show_path == SHOW_PATH_REL) {
		gtk_check_menu_item_set_active(
			GTK_CHECK_MENU_ITEM(show_path_radio_menu[MENUITEM_SHOW_PATH_REL].widget), TRUE);
	} else {
		g_assert_not_reached();
	}

	gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(dirview_menu[MENUITEM_HIDE_FUNC_EMACS].widget),
		(pref->dvpref.row_hide_func_mask & ROW_HIDE_EMACS));
	gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(dirview_menu[MENUITEM_HIDE_FUNC_OBJ].widget),
		(pref->dvpref.row_hide_func_mask & ROW_HIDE_OBJ));

	gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(dirview_menu[MENUITEM_HIDE_STAT_BINARY].widget),
		(pref->dvpref.row_hide_stat_mask & BINARY_FILES));
	gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(dirview_menu[MENUITEM_HIDE_STAT_ONLY].widget),
		(pref->dvpref.row_hide_stat_mask & (ONLY_FILE1_EXISTS | ONLY_FILE2_EXISTS)));
	gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(dirview_menu[MENUITEM_HIDE_STAT_DIFF].widget),
		(pref->dvpref.row_hide_stat_mask & DIFFERENT_FILES));
	gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(dirview_menu[MENUITEM_HIDE_STAT_IDENT].widget),
		(pref->dvpref.row_hide_stat_mask & IDENTICAL_FILES));
}


/* block signal handers during update menus */
static void
menu_dirv_block(GDiffWindow *gdwin)
{
	int i;
	
	for (i = MENUITEM_FIRST_SHOW_PATH; i < MENUITEM_LAST_SHOW_PATH; i++) {
		gtk_signal_handler_block_by_func(
			GTK_OBJECT(show_path_radio_menu[i].widget),
			GTK_SIGNAL_FUNC(show_path_cb), gdwin);
	}
	for (i = MENUITEM_HIDE_FUNC_FIRST; i < MENUITEM_HIDE_FUNC_LAST; i++) {
		gtk_signal_handler_block_by_func(
			GTK_OBJECT(dirview_menu[i].widget),
			GTK_SIGNAL_FUNC(row_hide_func_cb), gdwin);
	}
	for (i = MENUITEM_HIDE_STAT_FIRST; i < MENUITEM_HIDE_STAT_LAST; i++) {
		gtk_signal_handler_block_by_func(
			GTK_OBJECT(dirview_menu[i].widget),
			GTK_SIGNAL_FUNC(row_hide_stat_cb), gdwin);
	}
}	

static void
menu_dirv_unblock(GDiffWindow *gdwin)
{
	int i;
	
	for (i = MENUITEM_FIRST_SHOW_PATH; i < MENUITEM_LAST_SHOW_PATH; i++) {
		gtk_signal_handler_unblock_by_func(
			GTK_OBJECT(show_path_radio_menu[i].widget),
			GTK_SIGNAL_FUNC(show_path_cb), gdwin);
	}
	for (i = MENUITEM_HIDE_FUNC_FIRST; i < MENUITEM_HIDE_FUNC_LAST; i++) {
		gtk_signal_handler_unblock_by_func(
			GTK_OBJECT(dirview_menu[i].widget),
			GTK_SIGNAL_FUNC(row_hide_func_cb), gdwin);
	}
	for (i = MENUITEM_HIDE_STAT_FIRST; i < MENUITEM_HIDE_STAT_LAST; i++) {
		gtk_signal_handler_unblock_by_func(
			GTK_OBJECT(dirview_menu[i].widget),
			GTK_SIGNAL_FUNC(row_hide_stat_cb), gdwin);
	}
}


/**
 * update_fileview:
 * menu update routine for file view case.
 **/
static void
update_fileview(GDiffWindow *gdwin, const Preference *pref)
{
	/* various states */
	if (pref->fvpref.view_type & ONEPANE_MASK_VIEW) {
		gtk_check_menu_item_set_active(
			GTK_CHECK_MENU_ITEM(pane_radio_menu[MENUITEM_ONEPANE_VIEW].widget), TRUE);
		gtk_widget_hide(fileview_menu[MENUITEM_SHOW_FILL].widget);
	} else if (pref->fvpref.view_type & MULTIPANE_MASK_VIEW) {
		gtk_check_menu_item_set_active(
			GTK_CHECK_MENU_ITEM(pane_radio_menu[MENUITEM_MULTIPANE_VIEW].widget),	TRUE);
		gtk_widget_show(fileview_menu[MENUITEM_SHOW_FILL].widget);
	} else if (pref->fvpref.view_type & MERGE_MASK_VIEW) {
		gtk_check_menu_item_set_active(
			GTK_CHECK_MENU_ITEM(pane_radio_menu[MENUITEM_MERGE_VIEW].widget), TRUE);
		gtk_widget_show(fileview_menu[MENUITEM_SHOW_FILL].widget);		
	} else {
		g_assert_not_reached();
	}

	gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(fileview_menu[MENUITEM_LINE_SHOW].widget),
		pref->fvpref.show_line_num);
	gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(fileview_menu[MENUITEM_TEXT_WRAP].widget),
		pref->fvpref.line_wrap);
	gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(fileview_menu[MENUITEM_HIGHLIGHT].widget),
		pref->fvpref.highlight);
	gtk_check_menu_item_set_active(
		GTK_CHECK_MENU_ITEM(fileview_menu[MENUITEM_SHOW_FILL].widget),
		pref->fvpref.show_fill);

	/* If line-wrap is on, I have to disable some actions */
	gtk_widget_set_sensitive(fview_action_menu[MENUITEM_REL_NEXT_DIFF].widget,
							 !pref->fvpref.line_wrap);
	gtk_widget_set_sensitive(fview_action_menu[MENUITEM_REL_PREV_DIFF].widget,
							 !pref->fvpref.line_wrap);
}

/* block signal handers during update menus */
static void
menu_filev_block(GDiffWindow *gdwin)
{
	int i;

	for (i = MENUITEM_FIRST_VIEW; i < MENUITEM_LAST_VIEW; i++) {
		gtk_signal_handler_block_by_func(
			GTK_OBJECT(pane_radio_menu[i].widget),
			GTK_SIGNAL_FUNC(mode_change_cb), gdwin);
	}
	gtk_signal_handler_block_by_func(
		GTK_OBJECT(fileview_menu[MENUITEM_LINE_SHOW].widget),
		GTK_SIGNAL_FUNC(show_linenum_cb), gdwin);
	gtk_signal_handler_block_by_func(
		GTK_OBJECT(fileview_menu[MENUITEM_TEXT_WRAP].widget),
		GTK_SIGNAL_FUNC(text_wrap_cb), gdwin);
	gtk_signal_handler_block_by_func(
		GTK_OBJECT(fileview_menu[MENUITEM_HIGHLIGHT].widget),
		GTK_SIGNAL_FUNC(highlight_cb), gdwin);
	gtk_signal_handler_block_by_func(
		GTK_OBJECT(fileview_menu[MENUITEM_SHOW_FILL].widget),
		GTK_SIGNAL_FUNC(show_fill_cb), gdwin);
}	

static void
menu_filev_unblock(GDiffWindow *gdwin)
{
	int i;

	for (i = MENUITEM_FIRST_VIEW; i < MENUITEM_LAST_VIEW; i++) {
		gtk_signal_handler_unblock_by_func(
			GTK_OBJECT(pane_radio_menu[i].widget),
			GTK_SIGNAL_FUNC(mode_change_cb), gdwin);
	}
	gtk_signal_handler_unblock_by_func(
		GTK_OBJECT(fileview_menu[MENUITEM_LINE_SHOW].widget),
		GTK_SIGNAL_FUNC(show_linenum_cb), gdwin);
	gtk_signal_handler_unblock_by_func(
		GTK_OBJECT(fileview_menu[MENUITEM_TEXT_WRAP].widget),
		GTK_SIGNAL_FUNC(text_wrap_cb), gdwin);
	gtk_signal_handler_unblock_by_func(
		GTK_OBJECT(fileview_menu[MENUITEM_HIGHLIGHT].widget),
		GTK_SIGNAL_FUNC(highlight_cb), gdwin);
	gtk_signal_handler_unblock_by_func(
		GTK_OBJECT(fileview_menu[MENUITEM_SHOW_FILL].widget),
		GTK_SIGNAL_FUNC(show_fill_cb), gdwin);
}	
	


/** Callback functions **/
/**
 * file_open_cb:
 **/
static void
file_open_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;

	filesel_open(gdwin);
}

/**
 * file_close_cb:
 **/
static void
file_close_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	GtkWidget *view;

	view = gdwin_current_view(gdwin);
	act_close_view(view);
}

/**
 * mode_change_cb:
 * Change the view type.
 * Update a global variable, g_pref.
 * (it's redundant, as it's updated in act_fv_mode_change()).
 **/
static void
mode_change_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	ViewType vtype;
	GtkWidget *view;

	if (!GTK_CHECK_MENU_ITEM(widget)->active)
		return;
	vtype = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(widget),
												GNOMEUIINFO_KEY_UIDATA));
	g_pref.fvpref.view_type = vtype;

	view = gdwin_current_view(gdwin);
	act_fv_mode_change(view, vtype);
}

/**
 * show_linenum_cb:
 * Toggle show(hide) the line numbers for the current view.
 * Update a global variable, "g_pref"(it's redundant, as it's updated in act_fv_show_linenum()).
 **/
static void
show_linenum_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	gboolean b_show_line_num;
	GtkWidget *view;

	b_show_line_num = GTK_CHECK_MENU_ITEM(widget)->active ? TRUE : FALSE;
	g_pref.fvpref.show_line_num = b_show_line_num;

	view = gdwin_current_view(gdwin);
	act_fv_show_linenum(view, b_show_line_num);
}

static void
text_wrap_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	gboolean b_wrap;
	GtkWidget *view;

	b_wrap = GTK_CHECK_MENU_ITEM(widget)->active ? TRUE : FALSE;
	if (g_pref.fvpref.line_wrap == b_wrap)
		return;

	view = gdwin_current_view(gdwin);
	act_fv_toggle_textwrap(view);
}

/**
 * highlight_cb:
 * Called from "Highlight selected lines" menu item.
 **/
static void
highlight_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	gboolean b_highlight;
	GtkWidget *view;

	b_highlight = GTK_CHECK_MENU_ITEM(widget)->active ? TRUE : FALSE;
	g_pref.fvpref.highlight = b_highlight;

	view = gdwin_current_view(gdwin);
	act_fv_highlight(view, b_highlight);
}

/**
 * show_fill_cb:
 * Called from "Sync line" menu item.
 **/
static void
show_fill_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	gboolean b_show_fill;
	GtkWidget *view;

	b_show_fill = GTK_CHECK_MENU_ITEM(widget)->active ? TRUE : FALSE;
	g_pref.fvpref.show_fill = b_show_fill;

	view = gdwin_current_view(gdwin);
	act_fv_show_fill(view, b_show_fill);
}

static void
go_dirview_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	GtkWidget *view;

	view = gdwin_current_view(gdwin);
	act_fv_go_dirview(view);
}

static void
move_first_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	GtkWidget *view;

	view = gdwin_current_view(gdwin);
	act_fv_move_diff(view, MOVED_FIRST);
}

static void
move_last_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	GtkWidget *view;

	view = gdwin_current_view(gdwin);
	act_fv_move_diff(view, MOVED_LAST);
}

static void
move_next_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	GtkWidget *view;

	view = gdwin_current_view(gdwin);
	act_fv_move_diff(view, MOVED_NEXT);
}

static void
move_prev_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	GtkWidget *view;

	view = gdwin_current_view(gdwin);
	act_fv_move_diff(view, MOVED_PREV);
}

static void
move_current_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	GtkWidget *view;

	view = gdwin_current_view(gdwin);
	act_fv_move_diff(view, MOVED_CURRENT);
}

static void
move_rel_next_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	GtkWidget *view;

	view = gdwin_current_view(gdwin);
	act_fv_move_diff(view, MOVED_REL_NEXT);
}

static void
move_rel_prev_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	GtkWidget *view;

	view = gdwin_current_view(gdwin);
	act_fv_move_diff(view, MOVED_REL_PREV);
}


/**
 * set_preference_cb:
 **/
static void
set_preference_cb(GtkWidget *widget, gpointer data)
{
	properties_show();
}

/**
 * show_path_cb:
 * Update a global variable, "g_pref".
 * (Actually, this is redundant, because it's updated in act_dv_show_path()).
 **/
static void
show_path_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	ShowPathDView show_path;
	GtkWidget *view;

	if (!GTK_CHECK_MENU_ITEM(widget)->active)
		return;
	show_path = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(widget),
													GNOMEUIINFO_KEY_UIDATA));
	g_pref.dvpref.show_path = show_path;
	
	view = gdwin_current_view(gdwin);
	act_dv_show_path(view, show_path);
}

/**
 * row_hide_func_cb:
 * Use the index kept in menu item widget to know what function is chosen.
 **/
static void
row_hide_func_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	GtkWidget *view;
	RowHideMask rh_mask = 0;
	gboolean b_hide;
	int index;

	index = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(widget), MENUITEM_HIDE_FUNC_KEY));

	if (index == MENUITEM_HIDE_FUNC_EMACS)
		rh_mask = ROW_HIDE_EMACS;
	else if (index == MENUITEM_HIDE_FUNC_OBJ)
		rh_mask = ROW_HIDE_OBJ;
	else {
		g_assert_not_reached();
	}
	b_hide = (gboolean)GTK_CHECK_MENU_ITEM(widget)->active;

	view = gdwin_current_view(gdwin);
	act_dv_rowhide_func(view, rh_mask, b_hide);
}

/**
 * row_hide_stat_cb:
 * Use the index kept in menu item widget to know what function is chosen.
 **/
static void
row_hide_stat_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	GtkWidget *view;
	gboolean b_hide;
	FilesStatus hide_mask = 0;
	int index;

	index = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(widget), MENUITEM_HIDE_STAT_KEY));
	if (index == MENUITEM_HIDE_STAT_BINARY)
		hide_mask = BINARY_FILES;
	else if (index == MENUITEM_HIDE_STAT_ONLY)
		hide_mask = (ONLY_FILE1_EXISTS | ONLY_FILE2_EXISTS);
	else if (index == MENUITEM_HIDE_STAT_DIFF)
		hide_mask = DIFFERENT_FILES;
	else if (index == MENUITEM_HIDE_STAT_IDENT)
		hide_mask = IDENTICAL_FILES;
	else {
		g_assert_not_reached();
	}
	b_hide = (gboolean)GTK_CHECK_MENU_ITEM(widget)->active;

	view = gdwin_current_view(gdwin);
	act_dv_rowhide_stat(view, hide_mask, b_hide);
}

/**
 * show_toolbar_cb:
 * Show or hide toolbar.
 **/
static void
show_toolbar_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	GnomeApp *app = gdwin->app;
	gboolean to_show;

	if (gdwin->toolbar == NULL)
		return;
	to_show = GTK_CHECK_MENU_ITEM(widget)->active ? TRUE : FALSE;
	if (to_show == TRUE) {
		gtk_widget_show(GTK_WIDGET(gdwin->toolbar->parent));
	} else {
		gtk_widget_hide(GTK_WIDGET(gdwin->toolbar->parent));
	}
	gtk_widget_queue_resize(GTK_WIDGET(app));

	g_pref.winpref.show_toolbar = to_show;
}

/**
 * show_searchbar_cb:
 * Show or hide searchbar.
 **/
static void
show_searchbar_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	GnomeApp *app = gdwin->app;
	gboolean to_show;

	if (gdwin->searchbar == NULL)
		return;
	to_show = GTK_CHECK_MENU_ITEM(widget)->active ? TRUE : FALSE;
	if (to_show == TRUE) {
		searchbar_show(gdwin->searchbar, TRUE);
	} else {
		searchbar_show(gdwin->searchbar, FALSE);
	}
	gtk_widget_queue_resize(GTK_WIDGET(app));

	g_pref.winpref.show_searchbar = to_show;
}

/**
 * show_statusbar_cb:
 * Show or hide statusbar.
 **/
static void
show_statusbar_cb(GtkWidget *widget, gpointer data)
{

	GDiffWindow *gdwin = data;
	GnomeApp *app = gdwin->app;
	gboolean to_show;

	if (app->statusbar == NULL)
		return;
	to_show = GTK_CHECK_MENU_ITEM(widget)->active ? TRUE : FALSE;
	if (to_show == TRUE) {
		gtk_widget_show(GTK_WIDGET(app->statusbar->parent));
	} else {
		gtk_widget_hide(GTK_WIDGET(app->statusbar->parent));
	}
	gtk_widget_queue_resize(GTK_WIDGET(app));

	g_pref.winpref.show_statusbar = to_show;
}

/**
 * show_tabs_cb:
 * Show or hide notebook tabs.
 **/
static void
show_tabs_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	GtkNotebook *notebook = gdwin->notebook;

	g_pref.winpref.show_tabs = GTK_CHECK_MENU_ITEM(widget)->active ? TRUE : FALSE;
	gtk_notebook_set_show_tabs(notebook, g_pref.winpref.show_tabs);
}

static void
reload_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;
	GtkWidget *view;

	view = gdwin_current_view(gdwin);
	act_reload(view);
}

static void
search_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;

	searchbar_grab_focus(gdwin->searchbar);
}

static void
checksum_cb(GtkWidget *widget, gpointer data)
{
	GDiffWindow *gdwin = data;

	GtkWidget *view;

	view = gdwin_current_view(gdwin);
	act_dv_checksum(view);
}

static void
insert_remove_cb(GtkWidget *w, gpointer data)
{
	GDiffWindow *gdwin = data;
	GtkWidget *view;
	int index;
	WhichFile whichfile = 0;
	gboolean to_insert;
	gboolean to_all;

	index = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(w), MENUITEM_MERGE_KEY));
	if (index & MERGE_FIRST_FILE)
		whichfile = FIRST_FILE;
	else if (index & MERGE_SECOND_FILE)
		whichfile = SECOND_FILE;
	else if (index & MERGE_THIRD_FILE)
		whichfile = THIRD_FILE;
	
	to_insert = index & MERGE_INSERT_FILE ? TRUE : FALSE;
	to_all = index & MERGE_ALL_FILE ? TRUE : FALSE;

	view = gdwin_current_view(gdwin);
	if (to_all)
		act_merge_insert_remove_all(view, whichfile, to_insert);
	else
		act_merge_insert_remove(view, whichfile, to_insert);
}

static void
merge_output_cb(GtkWidget *w, gpointer data)
{
	GDiffWindow *gdwin = data;
	GtkWidget *view;

	view = gdwin_current_view(gdwin);
	act_merge_output_file(view);
}


static void
quit_cb(GtkWidget *widget, gpointer data)
{
	gtk_main_quit();
}

static void
about_cb(GtkWidget *widget, gpointer data)
{
	static GtkWidget *about = NULL;
	static const gchar *authors[] = {
		"INOUE Seiichiro <inoue@ainet.or.jp>",
		NULL
	};

	if (about) {
		gdk_window_raise(GTK_WIDGET(about)->window);
	} else {
		about = gnome_about_new(APPNAME, VERSION,
								_("Copyright 1999, under the GNU General Public License."),
								authors,
								/* another comments */
								_("GTK+(GNOME) diff front-end."),
								NULL);
		gtk_signal_connect(GTK_OBJECT(about), "destroy",
						   GTK_SIGNAL_FUNC(gtk_widget_destroyed), &about);
		gtk_window_set_position(GTK_WINDOW(about), GTK_WIN_POS_CENTER);
		gtk_widget_show(about);
	}
	return;
}
