/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
  satlist.c: Graphical visualization module for Gnome Predict showing
             selected satellites in a list.


  Copyright (C)  2001-2003  Alexandru Csete.

  Authors:  Alexandru Csete <csete@users.sourceforge.net>

  Comments, questions and bugreports should be submitted via
  http://sourceforge.net/projects/groundstation/
  More details can be found at http://groundstation.sourceforge.net/
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the
          Free Software Foundation, Inc.,
	  59 Temple Place, Suite 330,
	  Boston, MA  02111-1307
	  USA
*/

#include <gnome.h>
#include <gconf/gconf-client.h>
#include <libgnomeui/gnome-window-icon.h>
#include "defaults.h"
#include "qth.h"
#include "fileio.h"
#include "util.h"
#include "satdata.h"
#include "satlog.h"
#include "sattabs.h"
#include "satmodules.h"
#include "iconbar.h"
#include "satlist-druid.h"
#include "satlist-info.h"
#include "satlist-single-aos.h"
#include "satlist-multi-aos.h"
#include "satlist.h"

#ifdef HAVE_CONFIG_H
#  include "../config.h"
#endif

#define COLUMNS 11

gchar *titles[COLUMNS] = {
	N_(" Satellite "),
	N_(" Lon "),
	N_(" Lat "),
	N_(" Az "),
	N_(" El "),
	N_(" Next AOS/LOS "),
	N_(" Footprint "),
	N_(" Range "),
	N_(" Alt "),
	N_(" Vel "),
	N_(" Orbit ")
};


enum satlist_mode_e {
	SATLIST_MODE_NONE = 0,  /* actually not a valid mode... */
	SATLIST_MODE_TAB,       /* widget is in the notebook */
	SATLIST_MODE_WIN,       /* widget is in a separate window */
	SATLIST_MODE_FULLSCR    /* wisget is blowing out screen! */
};


extern const gchar *modtypes[];
extern GConfClient *client; /* main.c */

/**** FIXME: SHOULD BE LIST SPECIFIC! */
GdkColor high,norm,geoh,geon,dec,noaos;      /* colors to highlight satellites */


extern GtkWidget *app;
extern qth_struc qth;    /* defined in qth.c */


/* Private function protos */
static void      satlist_rows_adjust     (GtkWidget *);
static gint      satlist_sat_prepend     (GtkWidget *, guint);
static gint      satlist_sat_prepend     (GtkWidget *, guint);
static gint      satlist_sat_append      (GtkWidget *, guint);
static gint      satlist_sat_insert      (GtkWidget *, gint, guint);
static void      satlist_sat_remove      (GtkWidget *, gint);
static void      satlist_update_visibles (GtkWidget *);
static void      satlist_clear           (GtkWidget *);
static void      satlist_read_list       (GtkWidget *, gchar *);
static gint      satlist_get_size        (GtkWidget *);
static gboolean  satlist_sat_is_visible  (GtkWidget *, gint);
static gchar    *satlist_get_satname     (GtkWidget *, gint);
static void      satlist_switch_mode_cb  (GtkWidget *, GtkWidget *);
static void      satlist_close_cb        (GtkWidget *, GtkWidget *);
static void      satlist_close_cleanup   (GtkWidget *);
static gint      satlist_timer_cb        (gpointer);
static gint      satlistwin_delete_cb    (GtkWidget *, GdkEvent *, GtkWidget *);
static void      satlistwin_close_cb     (GtkWidget *, GtkWidget *);
static gint      satlist_comp_func       (GtkCList *, gconstpointer, gconstpointer);
static void      satlist_sort            (GtkCList *, gint, gpointer);
static void      satlist_read_colors     (gpointer);
static void      satlist_prefs_changed   (GConfClient *, guint, GConfEntry *, gpointer);


void satlist_open_cb (GtkWidget *widget, gpointer data)
{
	/* This is just a gtk-callback function that calls
	   satlist_open.
	*/
	satlist_open (NULL, TRUE);
}

gint satlist_open (gchar *name, gboolean addtolist)
{
	/* This function creates the widget that shows selected
	   satellites in a list. The configuration file is expected
	   to be located at $HOME/.gpredict/name.satmap. If name
	   is NULL the druid will be invoked!
	   If addtolist is TRUE, the module will be added to the 
	   list of open modules (this should not be the case when
	   we start the program and open the modules from the
	   existing list).
	*/

	/* Widgets */
	GtkWidget *vbox,*buttonbox,*infobutton,*maosbutton,*saosbutton,*modebutton,*closebutton;
	GtkWidget *satclist,*satswin,*pixmap;
	GtkTooltips *kooltip;
	gchar *menulabel,*pixmapfile;
	guint i;
	gint timer,delay;
	guint gconf_id;

	/* Create a new list if none specified */
	if (!name)
		name = g_strdup (satlist_druid_run (NULL));

	/* user probably canceled */
	if (!name)
		return -1;

	/* Check whether module really exists */
	if (!satmodules_test_module (name, SATMOD_TYPE_LIST)) {
		/* remove from list, just in case ... */
		satmodules_remove_module (name, SATMOD_TYPE_LIST);
		return -1;
	}

	if (addtolist) {
		if (satmodules_test_open_modules (name, SATMOD_TYPE_LIST)) {
			gnome_app_error (GNOME_APP (app),
					 _("This list is already open (probably a bug)."));
			return -1;
		}
 		iconbar_remove_module (name, SATMOD_TYPE_LIST);
	 	satmodules_add_module (name, SATMOD_TYPE_LIST);
	}

	/* get some config values */
	delay = gconf_client_get_int (client, SATLIST_TIMER_PATH, NULL);
	if (!delay)
		delay = SATLIST_DEF_TIMER;


	/* create the clist widget */
	satclist = gtk_clist_new_with_titles (COLUMNS, titles);
	gtk_clist_column_titles_active (GTK_CLIST (satclist) );
	gtk_clist_set_compare_func (GTK_CLIST (satclist), satlist_comp_func );
	gtk_clist_set_sort_column (GTK_CLIST (satclist), 4 ); /* first click fix hack */
	gtk_signal_connect (GTK_OBJECT (satclist), "click-column",
			    GTK_SIGNAL_FUNC (satlist_sort), NULL);

	for (i=1; i<COLUMNS; i++)
		gtk_clist_set_column_justification (GTK_CLIST (satclist),
						    i, GTK_JUSTIFY_CENTER );

	/* load sats into list */
	satlist_read_list (satclist, name);
	satlist_read_colors (satclist);

	/* create scrolled window for satclist */
	satswin = gtk_scrolled_window_new (NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (satswin),
					GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC );
	gtk_container_add (GTK_CONTAINER (satswin), satclist);


	/* Create buttons and pack them into a button box.
	   Create custom pixmap buttons, which are much nicer
	   that the STOCK_BUTTONS/PIXMAPS available in Gtk 1.2
	   or Gnome 1.4
	*/
	buttonbox = gtk_hbutton_box_new ();
	gtk_button_box_set_spacing (GTK_BUTTON_BOX (buttonbox), 0);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (buttonbox),
				   GTK_BUTTONBOX_START );

	/* close button */
	pixmapfile = g_strconcat ( PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "icons",
				   G_DIR_SEPARATOR_S, "cancel.png", NULL);
	pixmap = gnome_pixmap_new_from_file (pixmapfile);
	closebutton = gnome_pixmap_button (pixmap, _("  Close"));
	g_free (pixmapfile);
	kooltip = gtk_tooltips_new ();
	gtk_tooltips_set_tip (kooltip, closebutton, _("Close this list"),
			      NULL);

	/* Tab/Win button */
	pixmapfile = g_strconcat ( PACKAGE_PIXMAPS_DIR, G_DIR_SEPARATOR_S, "icons",
				   G_DIR_SEPARATOR_S, "windows_small.png", NULL);
	pixmap = gnome_pixmap_new_from_file (pixmapfile);
	modebutton = gnome_pixmap_button (pixmap, _("  Mode"));
	g_free (pixmapfile);
	gtk_tooltips_set_tip (kooltip, modebutton, _("Switch betwen notebook and window mode"),
			      NULL);

	/* satellite info button */
	infobutton = gnome_pixmap_button (gnome_stock_pixmap_widget (app, GNOME_STOCK_PIXMAP_HELP),
					  _("Sat Info") );
	gtk_signal_connect (GTK_OBJECT (infobutton), "clicked",
			    GTK_SIGNAL_FUNC (satlist_info_cb), satclist);
	gtk_tooltips_set_tip (kooltip, infobutton, _("Show Keplerian elements for the selected satellite"),
			      NULL);

	/* Upcomming passes button */
	maosbutton = gnome_pixmap_button (gnome_stock_pixmap_widget (app, GNOME_STOCK_PIXMAP_TIMER),
					  "AOS/LOS" );
	gtk_signal_connect (GTK_OBJECT (maosbutton), "clicked",
			    GTK_SIGNAL_FUNC (satlist_multi_aos_cb), satclist);
	gtk_tooltips_set_tip (kooltip, maosbutton, _("Show AOS and LOS times for upcoming passes"),
			      NULL);

	/* upcomming pass in details button */
	saosbutton = gnome_pixmap_button (gnome_stock_pixmap_widget (app, GNOME_STOCK_PIXMAP_PROPERTIES),
					  _("Next AOS") );
	gtk_signal_connect (GTK_OBJECT (saosbutton), "clicked",
			    GTK_SIGNAL_FUNC (satlist_single_aos_cb), satclist);
	gtk_tooltips_set_tip (kooltip, saosbutton, _("Get detailed info about the upcoming pass"),
			      NULL);

	gtk_box_pack_start (GTK_BOX (buttonbox), GTK_WIDGET (closebutton), TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (buttonbox), GTK_WIDGET (modebutton), TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (buttonbox), GTK_WIDGET (infobutton), TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (buttonbox), GTK_WIDGET (maosbutton), TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (buttonbox), GTK_WIDGET (saosbutton), TRUE, TRUE, 0);


	/* start timer */
	timer = gtk_timeout_add (delay, satlist_timer_cb, satclist);

	/* create a vertical box */
	vbox = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (vbox), satswin, TRUE, TRUE, 0); /* extra space to swin */
	gtk_box_pack_start (GTK_BOX (vbox), buttonbox, FALSE, FALSE, 0);


	/* attach stuff to vbox */
	gtk_object_set_data (GTK_OBJECT (vbox), "name", name);
	gtk_object_set_data (GTK_OBJECT (vbox), "mode", GUINT_TO_POINTER (SATLIST_MODE_TAB));
	gtk_object_set_data (GTK_OBJECT (vbox), "timer", GINT_TO_POINTER (timer));
	gtk_object_set_data (GTK_OBJECT (vbox), "clist", satclist);

	gconf_id = gconf_client_notify_add (client, SATLIST_ROOT_PATH,
					    satlist_prefs_changed, vbox, NULL, NULL);

	gtk_object_set_data (GTK_OBJECT (vbox), "gconfid", GUINT_TO_POINTER (gconf_id));

	gtk_signal_connect (GTK_OBJECT (closebutton), "clicked",
			    GTK_SIGNAL_FUNC (satlist_close_cb), vbox);
	gtk_signal_connect (GTK_OBJECT (modebutton), "clicked",
			    GTK_SIGNAL_FUNC (satlist_switch_mode_cb), vbox);

	gtk_widget_show_all (vbox);

	menulabel = g_strconcat (name, _(" (list)"), NULL);
	sattabs_add_tab_menu (vbox, name, menulabel);
	g_free (menulabel);

	satlist_rows_adjust (satclist);

	return 0;

}



/*****************************/
/*    CList wrappers         */
/*****************************/
static void satlist_rows_adjust (GtkWidget *clist)
{
	/* Function to adjust the size of the rows in
	   sat_list to be optimal. 
	*/

	gint i;

	gtk_clist_freeze (GTK_CLIST (clist));
	for ( i=0; i<COLUMNS; i++)
//		if (gtk_clist_row_is_visible (GTK_CLIST (clist),i))
		gtk_clist_set_column_width (GTK_CLIST (clist), i,
					    gtk_clist_optimal_column_width (GTK_CLIST (clist), i));
	gtk_clist_thaw (GTK_CLIST (clist));
}


static gint satlist_sat_prepend (GtkWidget *clist, guint index)
{
	/* Function to prepend a new satellite into satclist widget.
	 */

	gchar *buf,**text;
	gint code;
	sat_t *sat;

	sat = satdata_get_sat (index);

	switch (sat->status) {
	case SAT_STATUS_DECAYED:
		buf = g_strdup_printf (_("%s;%7.2f;%7.2f;%7.2f;%7.2f;-- DECAYED --;"\
					 "%6.0f;%7.0f;%7.0f;%7.0f;%7ld"),
				       sat->name,sat->lon,sat->lat,sat->az,sat->el,sat->fp,
				       sat->range,sat->alt,sat->vel,sat->orbit);
		text = g_strsplit (buf, ";", 12);
		code = gtk_clist_prepend (GTK_CLIST (clist), text);
		gtk_clist_set_row_data (GTK_CLIST (clist), code, GUINT_TO_POINTER (index) );
		gtk_clist_set_foreground (GTK_CLIST (clist), code, &dec );
		g_free (buf);
		g_strfreev (text);
		break;
	case SAT_STATUS_GEOSTAT:
		buf = g_strdup_printf (_("%s;%7.2f;%7.2f;%7.2f;%7.2f;GEOSTATIONARY;"\
					 "%6.0f;%7.0f;%7.0f;%7.0f;%7ld"),
				       sat->name,sat->lon,sat->lat,sat->az,sat->el,sat->fp,
				       sat->range,sat->alt,sat->vel,sat->orbit);
		text = g_strsplit (buf, ";", 12);
		code = gtk_clist_prepend (GTK_CLIST (clist), text);
		gtk_clist_set_row_data (GTK_CLIST (clist), code, GUINT_TO_POINTER (index) );
		if (sat->el > 0)
			gtk_clist_set_foreground (GTK_CLIST (clist), code, &geoh);
		else
			gtk_clist_set_foreground (GTK_CLIST (clist), code, &geon);
		g_free (buf);
		g_strfreev (text);
		break;
	case SAT_STATUS_NOAOS:
		buf = g_strdup_printf (_("%s;%7.2f;%7.2f;%7.2f;%7.2f;--- NEVER ---;"\
					 "%6.0f;%7.0f;%7.0f;%7.0f;%7ld"),
				       sat->name,sat->lon,sat->lat,sat->az,sat->el,sat->fp,
				       sat->range,sat->alt,sat->vel,sat->orbit);
		text = g_strsplit (buf, ";", 12);
		code = gtk_clist_prepend (GTK_CLIST (clist), text);
		gtk_clist_set_row_data (GTK_CLIST (clist), code, GUINT_TO_POINTER (index));
		gtk_clist_set_foreground (GTK_CLIST (clist), code, &noaos);
		g_free (buf);
		g_strfreev (text);
		break;
	default:
		buf = g_strdup_printf ("%s;%7.2f;%7.2f;%7.2f;%7.2f; %s ;"\
				       "%6.0f;%7.0f;%7.0f;%7.0f;%7ld",
				       sat->name, sat->lon, sat->lat, sat->az, sat->el,
				       (sat->el > 0) ? dnum2str (sat->los) : dnum2str (sat->aos),
				       sat->fp, sat->range, sat->alt, sat->vel, sat->orbit);
		text = g_strsplit (buf, ";", 12);
		code = gtk_clist_prepend (GTK_CLIST (clist), text);
		gtk_clist_set_row_data (GTK_CLIST (clist), code, GUINT_TO_POINTER (index));
		g_free (buf);
		g_strfreev (text);
		gtk_clist_set_foreground (GTK_CLIST (clist), code, (sat->el > 0) ? &high : &norm);
	}

	return code;
}


static gint satlist_sat_append (GtkWidget *clist, guint index)
{
	/* Function to append a new satellite into the satclist widget.
	 */

	gchar *buf,**text;
	gint code;
	sat_t *sat;

	sat = satdata_get_sat (index);

	switch (sat->status) {
	case SAT_STATUS_DECAYED:
		buf = g_strdup_printf (_("%s;%7.2f;%7.2f;%7.2f;%7.2f;-- DECAYED --;"\
					 "%6.0f;%7.0f;%7.0f;%7.0f;%7ld"),
				       sat->name,sat->lon,sat->lat,sat->az,sat->el,sat->fp,
				       sat->range,sat->alt,sat->vel,sat->orbit);
		text = g_strsplit (buf, ";", 12);
		code = gtk_clist_append (GTK_CLIST (clist), text);
		gtk_clist_set_row_data (GTK_CLIST (clist), code, GUINT_TO_POINTER (index));
		gtk_clist_set_foreground (GTK_CLIST (clist), code, &dec);
		g_free (buf);
		g_strfreev (text);
		break;
	case SAT_STATUS_GEOSTAT:
		buf = g_strdup_printf (_("%s;%7.2f;%7.2f;%7.2f;%7.2f;GEOSTATIONARY;"\
					 "%6.0f;%7.0f;%7.0f;%7.0f;%7ld"),
				       sat->name,sat->lon,sat->lat,sat->az,sat->el,sat->fp,
				       sat->range,sat->alt,sat->vel,sat->orbit);
		text = g_strsplit (buf, ";", 12);
		code = gtk_clist_append (GTK_CLIST (clist), text);
		gtk_clist_set_row_data (GTK_CLIST (clist), code, GUINT_TO_POINTER (index));
		if ( sat->el>0 )
			gtk_clist_set_foreground (GTK_CLIST (clist), code, &geoh);
		else
			gtk_clist_set_foreground (GTK_CLIST (clist), code, &geon);
		g_free (buf);
		g_strfreev (text);
		break;
	case SAT_STATUS_NOAOS:
		buf = g_strdup_printf (_("%s;%7.2f;%7.2f;%7.2f;%7.2f;--- NEVER ---;"\
					 "%6.0f;%7.0f;%7.0f;%7.0f;%7ld"),
				       sat->name,sat->lon,sat->lat,sat->az,sat->el,sat->fp,
				       sat->range,sat->alt,sat->vel,sat->orbit);
		text = g_strsplit (buf, ";", 12);
		code = gtk_clist_append (GTK_CLIST (clist), text);
		gtk_clist_set_row_data (GTK_CLIST (clist), code, GUINT_TO_POINTER (index));
		gtk_clist_set_foreground (GTK_CLIST(clist), code, &noaos);
		g_free (buf);
		g_strfreev (text);
		break;
	default:
		buf = g_strdup_printf ("%s;%7.2f;%7.2f;%7.2f;%7.2f; %s ;"\
				       "%6.0f;%7.0f;%7.0f;%7.0f;%7ld",
				       sat->name,sat->lon,sat->lat,sat->az,sat->el,
				       (sat->el > 0) ? dnum2str (sat->los) : dnum2str (sat->aos),
				       sat->fp,sat->range,sat->alt,sat->vel,sat->orbit);
		text = g_strsplit (buf, ";", 12);
		code = gtk_clist_append (GTK_CLIST (clist), text);
		gtk_clist_set_row_data (GTK_CLIST (clist), code, GUINT_TO_POINTER (index));
		g_free (buf);
		g_strfreev (text);
		gtk_clist_set_foreground (GTK_CLIST (clist), code, (sat->el > 0) ? &high : &norm);
	}

	return code;
}



static gint satlist_sat_insert (GtkWidget *clist, gint row, guint index)
{
	/* Function to insert a new satellite into 
	   the satclist widget at the specified position.
	*/

	gchar *buf,**text;
	gint code;
	sat_t *sat;

	sat = satdata_get_sat (index);
	switch (sat->status) {
	case SAT_STATUS_DECAYED:
		buf = g_strdup_printf (_("%s;%7.2f;%7.2f;%7.2f;%7.2f;-- DECAYED --;"\
					 "%6.0f;%7.0f;%7.0f;%7.0f;%7ld"),
				       sat->name,sat->lon,sat->lat,sat->az,sat->el,sat->fp,
				       sat->range,sat->alt,sat->vel,sat->orbit);
		text = g_strsplit (buf, ";", 12);
		code = gtk_clist_insert (GTK_CLIST (clist), row, text);
		gtk_clist_set_row_data (GTK_CLIST (clist), code, GUINT_TO_POINTER (index));
		gtk_clist_set_foreground (GTK_CLIST (clist), code, &dec);
		g_free (buf);
		g_strfreev (text);
		break;
	case SAT_STATUS_GEOSTAT:
		buf = g_strdup_printf (_("%s;%7.2f;%7.2f;%7.2f;%7.2f;GEOSTATIONARY;"\
					 "%6.0f;%7.0f;%7.0f;%7.0f;%7ld"),
				       sat->name,sat->lon,sat->lat,sat->az,sat->el,sat->fp,
				       sat->range,sat->alt,sat->vel,sat->orbit);
		text = g_strsplit (buf, ";", 12);
		code = gtk_clist_insert (GTK_CLIST (clist), row, text);
		gtk_clist_set_row_data (GTK_CLIST (clist), code, GUINT_TO_POINTER (index));
		if ( sat->el>0 )
			gtk_clist_set_foreground (GTK_CLIST (clist), code, &geoh);
		else
			gtk_clist_set_foreground (GTK_CLIST (clist), code, &geon);
		g_free (buf);
		g_strfreev (text);
		break;
	case SAT_STATUS_NOAOS:
		buf = g_strdup_printf (_("%s;%7.2f;%7.2f;%7.2f;%7.2f;--- NEVER ---;"\
					 "%6.0f;%7.0f;%7.0f;%7.0f;%7ld"),
				       sat->name,sat->lon,sat->lat,sat->az,sat->el,sat->fp,
				       sat->range,sat->alt,sat->vel,sat->orbit);
		text = g_strsplit (buf, ";", 12);
		code = gtk_clist_insert (GTK_CLIST (clist), row, text);
		gtk_clist_set_row_data (GTK_CLIST (clist), code, GUINT_TO_POINTER (index));
		gtk_clist_set_foreground (GTK_CLIST (clist), code, &noaos);
		g_free (buf);
		g_strfreev (text);
		break;
	default:
		buf = g_strdup_printf ("%s;%7.2f;%7.2f;%7.2f;%7.2f; %s ;"\
				       "%6.0f;%7.0f;%7.0f;%7.0f;%7ld",
				       sat->name, sat->lon, sat->lat, sat->az, sat->el,
				       (sat->el > 0) ? dnum2str (sat->los) : dnum2str (sat->aos),
				       sat->fp, sat->range, sat->alt, sat->vel, sat->orbit);
		text = g_strsplit (buf, ";", 12);
		code = gtk_clist_insert (GTK_CLIST (clist), row, text);
		gtk_clist_set_row_data (GTK_CLIST (clist), code, GUINT_TO_POINTER (index));
		g_free (buf);
		g_strfreev (text);
		gtk_clist_set_foreground (GTK_CLIST (clist), code, (sat->el > 0) ? &high : &norm);
	}

	return code;
}


static void satlist_sat_remove (GtkWidget *clist, gint row)
{
	/* Function to remove a satellite from 
	   sat_l from the  given row.
	*/

	gtk_clist_remove (GTK_CLIST (clist), row);
}


static void satlist_update_visibles (GtkWidget *clist)
{
	/* This function updates the visible rows
	   int the satlist.
	*/
	guint i,cnt;
	gchar *buf,**tmp;
	sat_t *sat;

	cnt = satlist_get_size (clist);
	for ( i=0; i<cnt; i++ ) {
		if (gtk_clist_row_is_visible (GTK_CLIST (clist), i)) {
			sat = satdata_get_sat (GPOINTER_TO_UINT (gtk_clist_get_row_data (GTK_CLIST (clist), i)));
			if (!sat) {
				/* sat must have been deleted from engine? */
				satlist_sat_remove (clist, i);
				return;
			}
			switch (sat->status) {
			case SAT_STATUS_DECAYED:
				break;
			case SAT_STATUS_CAMEUP:
				gtk_clist_set_foreground (GTK_CLIST (clist), i, &high);
				gtk_clist_set_text (GTK_CLIST (clist), i, 5, dnum2str (sat->los));
				sat->status = SAT_STATUS_AHOR;
				break;
			case SAT_STATUS_WENTDOWN:
				gtk_clist_set_foreground (GTK_CLIST (clist), i, &norm);
				gtk_clist_set_text (GTK_CLIST (clist), i, 5, dnum2str (sat->aos));
				sat->status = SAT_STATUS_BHOR;
				break;
			default:
				buf = g_strdup_printf ("%7.2f;%7.2f;%7.2f;%7.2f;%6.0f;"\
						       "%7.0f;%7.0f;%7.0f;%7ld",
						       sat->lon,sat->lat,sat->az,sat->el,
						       sat->fp,sat->range,sat->alt,sat->vel,
						       sat->orbit);
				tmp = g_strsplit (buf, ";", 10);
				
				gtk_clist_set_text (GTK_CLIST (clist), i, 1, tmp[0]);
				gtk_clist_set_text (GTK_CLIST (clist), i, 2, tmp[1]);
				gtk_clist_set_text (GTK_CLIST (clist), i, 3, tmp[2]);
				gtk_clist_set_text (GTK_CLIST (clist), i, 4, tmp[3]);
				gtk_clist_set_text (GTK_CLIST (clist), i, 6, tmp[4]);
				gtk_clist_set_text (GTK_CLIST (clist), i, 7, tmp[5]);
				gtk_clist_set_text (GTK_CLIST (clist), i, 8, tmp[6]);
				gtk_clist_set_text (GTK_CLIST (clist), i, 9, tmp[7]);
				gtk_clist_set_text (GTK_CLIST (clist), i,10, tmp[8]);
				g_free (buf);
				g_strfreev (tmp);
			}
			
		}
	}
}



static void satlist_clear (GtkWidget *clist)
{
	/* Function to quick and dirty removal of ALL satellites from
	   the sat_l. Well, maybe not dirty after all...
	*/
//	rowsel = -1;
	gtk_clist_clear (GTK_CLIST (clist));
}



static void satlist_read_list (GtkWidget *clist, gchar *name)
{
	/* This function reads the list of satellites from the file
	   $(HOME)/.gpredict/name.satlist in section SATLIST_SAT_LIST_PATH
	   and append the satellites into the clist. It does NOT clear
	   the clist!
	*/

	gchar **argv,*configfile,*text;
	gint argc,i,index;

	configfile = g_strconcat ("=", g_get_home_dir (), G_DIR_SEPARATOR_S,
				  ".gpredict", G_DIR_SEPARATOR_S, name,
				  ".satlist=", SATLIST_SAT_LIST_PATH, NULL);
	gnome_config_get_vector (configfile, &argc, &argv);

	/* Empty list? */
	if (!g_strcasecmp (argv[0], "")) {
		/**********  FIXME: fix up error dialog */
		satmodules_remove_module (name, SATMOD_TYPE_LIST);
		return;
	}

	for (i=0; i<argc; i++) {
		/* Get index of satellite and add to list if the
		   satellite really exists in our database
		*/
		index = satdata_get_sat_index (0, argv[i]);
		if (index != -1) {
			satlist_sat_append (clist, index);
		}
		else {
			text = g_strdup_printf (_("%s: Satellite %s not in database!"),
						__FUNCTION__, argv[i]);
			satlog_log (SAT_LOG_WARNING, text);
			g_free (text);
		}
	}
	g_free (configfile);
	configfile = g_strdup_printf (_("SATLIST: Read %d satellites into %s"), i, name);
	satlog_log (SAT_LOG_INFO, configfile);
	g_free (configfile);
}



static gint satlist_get_size (GtkWidget *clist)
{
	/* Function to determine how many rows there are
	   in the sat_list. This is done by adding and
	   then removing a dummy row.
	*/
	gint rows;

	rows = gtk_clist_append (GTK_CLIST (clist), titles);
	gtk_clist_remove (GTK_CLIST (clist), rows);
	return rows;
}



static gboolean satlist_sat_is_visible (GtkWidget *clist, gint row)
{
	/* Is the row visible? */
	return gtk_clist_row_is_visible (GTK_CLIST (clist), row);
}



static gchar *satlist_get_satname (GtkWidget *clist, gint row)
{	
	/* The gtk_clist_get_text function needs a pointer to a
	   pointer to store the text in..
	   The function returns 1 on succes, 0 otherwise.
	*/

	/**** FIXME: THIS FUNCTION IS OBSOLETE! */
	return NULL;
/*  	if ( gtk_clist_get_text (GTK_CLIST (clist), row, 0, &tmp)) */
/*  		return tmp; */
/*  	else */
/*  		return NULL; */
}




/********************************************************************/
/*                  Button callbacks                                */
/********************************************************************/

static void
satlist_switch_mode_cb (GtkWidget *button, GtkWidget *vbox)
{
	/* This function is called when the user clicks on the
	   Tab/Window button.
	*/
	guint mode;
	gchar *name,*menulabel,*buff;
	gint page;
	GtkWidget *satwin;

	mode = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (vbox), "mode"));
	name = gtk_object_get_data (GTK_OBJECT (vbox), "name");
	switch (mode) {
	case SATLIST_MODE_TAB:
		page = sattabs_lookup_tab (vbox);
		if (page != -1) {
			/* delete from notebook */
			gtk_widget_ref (vbox);
			sattabs_remove_tab (page);
			/* update state */
			gtk_object_set_data (GTK_OBJECT (vbox), "mode",
					     GUINT_TO_POINTER (SATLIST_MODE_WIN));
			/* create window */
			satwin = gtk_window_new (GTK_WINDOW_TOPLEVEL);
			gtk_object_set_data (GTK_OBJECT (vbox), "window", satwin);
			gtk_window_set_title (GTK_WINDOW (satwin), name);
			/* try to set window size to something sensible */
			/* window icon */
			buff = g_strconcat (PACKAGE_PIXMAPS_DIR, "/icons/satlist.png", NULL);
			gnome_window_icon_set_from_file (GTK_WINDOW (satwin), buff);
			g_free (buff);
			/* connect window signals */
			gtk_signal_connect (GTK_OBJECT (satwin), "delete-event",
					    GTK_SIGNAL_FUNC (satlistwin_delete_cb), vbox);
			gtk_signal_connect (GTK_OBJECT (satwin), "destroy",
					    GTK_SIGNAL_FUNC (satlistwin_close_cb), vbox);
			gtk_container_add (GTK_CONTAINER (satwin), vbox);
			gtk_widget_show_all (satwin);
		}
		break;
	case SATLIST_MODE_WIN:
		/* get and destroy the window */
		satwin = gtk_object_get_data (GTK_OBJECT (vbox), "window");
		gtk_object_remove_data (GTK_OBJECT (vbox), "window");
		gtk_container_remove (GTK_CONTAINER (satwin), vbox);
		gtk_widget_destroy (satwin);
		/* reconfigure module */
		gtk_object_set_data (GTK_OBJECT (vbox), "mode",
				     GUINT_TO_POINTER (SATLIST_MODE_TAB));
		menulabel = g_strconcat (name, _(" (list)"), NULL);
		sattabs_add_tab_menu (vbox, name, menulabel);
		g_free (menulabel);
		break;
	case SATLIST_MODE_FULLSCR:
		break;
	default:
		satlog_log (SAT_LOG_CRITICAL, _("SATLIST: Invalid mode (BUG!)"));
	}
}



gint satlistwin_delete_cb (GtkWidget *widget, GdkEvent *event, GtkWidget *vbox)
{
	/* This function is connected to the "delete" event signal
	   of the satmap window. It returns false, so that the window
	   will be destroyed, ie. the next function will be called.
	*/
	satlist_close_cb (NULL, vbox);
	return FALSE;
}


void satlistwin_close_cb (GtkWidget *widget, GtkWidget *vbox)
{
	/****** FIXME: should clean up some memory */

	/* This function is connected to the "destroy" signal
	   of the satmap window. It calls the satmap_close
	   function, which in turn will clean up the memory
	   allocated for one module.
	*/



}


static void
satlist_close_cb (GtkWidget *button, GtkWidget *vbox)
{
	/* This function is called when the user clicks on
	   the "Close" button. It checks whether the Satlist
	   module is in a notebook tab or in a window, and
	   removes the widget.
	*/
	gint tabpos;
	GtkWidget *window;
	gchar *module;

	/* stop timer */
	gtk_timeout_remove (GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (vbox), "timer")));
	gconf_client_notify_remove (client,
				    GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (vbox), "gconfid")));

	module = g_strdup (gtk_object_get_data (GTK_OBJECT (vbox), "name"));
	/* add icon to iconbar */
	iconbar_add_module (module, SATMOD_TYPE_LIST);
	/* remove module from list */
	satmodules_remove_module (module, SATMOD_TYPE_LIST );
	g_free (module);

	tabpos = sattabs_lookup_tab (vbox);
	if (tabpos != -1) {
		sattabs_remove_tab (tabpos);
		satlist_close_cleanup (vbox);
		if (vbox)
/*			gtk_widget_destroy (vbox)*/;
	}
	else {
		window = gtk_object_get_data (GTK_OBJECT (vbox), "window");
		satlist_close_cleanup (vbox);
		if (vbox)
			gtk_widget_destroy (vbox);
		if (window)
			gtk_widget_destroy (window);
	}
}


static void
satlist_close_cleanup (GtkWidget *vbox)
{
	/* This function cleans up some memory that has been
	   allocated to a satlist. The data is attached to the
	   "vbox".
	*/

}



static gint
satlist_timer_cb (gpointer clist)
{
	/* This function does the update on the satlist.
	   It is called automatically at regular intervals.
	*/
	gtk_clist_freeze (GTK_CLIST (clist));
	satlist_update_visibles (clist);
	gtk_clist_thaw (GTK_CLIST (clist));
	return TRUE;
}


/*****************************/
/*   Row sorting functions   */
/*****************************/
static gint
satlist_comp_func (GtkCList *clist, gconstpointer s1, gconstpointer s2)
{
	/* This function is is used by Gtk+ to sort the clist
	   columns.
	   s1 and s2 are pointers to GtkClistRow structures.
	*/

	gchar *text1 = NULL;
	gchar *text2 = NULL;
	GtkCListRow *row1 = (GtkCListRow *) s1;
	GtkCListRow *row2 = (GtkCListRow *) s2;
	gdouble d1,d2;

	text1 = GTK_CELL_TEXT (row1->cell[clist->sort_column])->text;
	text2 = GTK_CELL_TEXT (row2->cell[clist->sort_column])->text;

	switch (clist->sort_column) {
	case 0: case 5: case 11:
		return g_strcasecmp (text1, text2);
		break;
	default:
		d1 = g_strtod (text1, NULL);
		d2 = g_strtod (text2, NULL);
		return (d1==d2) ? 0 : (d1<d2 ? -1 : 1);
	}
	return 0;
}


static void
satlist_sort (GtkCList *clist, gint column, gpointer data)
{
	/* Column sorting function */
	/**** WARNING: The global sat_l has been changed to clist param */

	gnome_app_flash (GNOME_APP (app), _("Sorting list"));
	if ( column == GTK_CLIST (clist)->sort_column ) {
		/* flip sorting mode (ascend/descend)  */
		gtk_clist_set_sort_type (GTK_CLIST (clist),
					 !GTK_CLIST (clist)->sort_type );
	} else {
		gtk_clist_set_sort_column (GTK_CLIST (clist), column);
	}
	gtk_clist_freeze (GTK_CLIST (clist));
	gtk_clist_sort (GTK_CLIST (clist));
	gtk_clist_thaw (GTK_CLIST (clist));
}





static void
satlist_read_colors (gpointer clist)
{
	/* This function reads the satlist colors using GConf.
	*/

	guint i,n;
	sat_t *sat;
	gchar *rgb;
	gchar **rgbv;

	/* color for satellites above horizon */
        if ( (rgb = gconf_client_get_string (client, SATLIST_HRGB_PATH, NULL)) ) {
		rgbv = g_strsplit ( rgb, ",", 3);
		high.red = (gushort) g_strtod (rgbv[0], NULL);
		high.green = (gushort)g_strtod (rgbv[1], NULL);
		high.blue = (gushort)g_strtod (rgbv[2], NULL);
		g_free (rgb);
		g_strfreev (rgbv);
	} else {
		high.red = SATLIST_DEF_HIGH_RED;
		high.green = SATLIST_DEF_HIGH_GREEN;
		high.blue = SATLIST_DEF_HIGH_BLUE;
	}
	/* color for satellites below horizon */
	if ( (rgb = gconf_client_get_string (client, SATLIST_NRGB_PATH, NULL)) ) {
		rgbv = g_strsplit (rgb, ",", 3);
		norm.red = (gushort) g_strtod (rgbv[0], NULL);
		norm.green = (gushort) g_strtod (rgbv[1], NULL);
		norm.blue = (gushort) g_strtod (rgbv[2], NULL);
		g_free (rgb);
		g_strfreev (rgbv);
	} else {
		norm.red = SATLIST_DEF_NORM_RED;
		norm.green = SATLIST_DEF_NORM_GREEN;
		norm.blue = SATLIST_DEF_NORM_BLUE;
	}
	/* Get geostationary below hor color */
	if ( (rgb = gconf_client_get_string (client, SATLIST_GNRGB_PATH, NULL)) ) {
		rgbv = g_strsplit (rgb, ",", 3);
		geon.red = (gushort) g_strtod (rgbv[0], NULL);
		geon.green = (gushort) g_strtod(rgbv[1], NULL);
		geon.blue = (gushort) g_strtod (rgbv[2], NULL);
		g_free (rgb);
		g_strfreev (rgbv);
	} else {
		geon.red = SATLIST_DEF_NORM_RED;
		geon.green = SATLIST_DEF_NORM_GREEN;
		geon.blue = SATLIST_DEF_NORM_BLUE;
	}
	/* Get geostationary above hor color */
	if ( (rgb = gconf_client_get_string (client, SATLIST_GHRGB_PATH, NULL)) ) {
		rgbv = g_strsplit (rgb, ",", 3);
		geoh.red = (gushort) g_strtod (rgbv[0], NULL);
		geoh.green = (gushort) g_strtod (rgbv[1], NULL);
		geoh.blue = (gushort) g_strtod (rgbv[2], NULL);
		g_free (rgb);
		g_strfreev (rgbv);
	} else {
		geoh.red = SATLIST_DEF_HIGH_RED;
		geoh.green = SATLIST_DEF_HIGH_GREEN;
		geoh.blue = SATLIST_DEF_HIGH_BLUE;
	}
	/* Get decayed color */
	if ( (rgb = gconf_client_get_string (client, SATLIST_DRGB_PATH, NULL)) ) {
		rgbv = g_strsplit (rgb, ",", 3);
		dec.red = (gushort) g_strtod (rgbv[0], NULL);
		dec.green = (gushort) g_strtod (rgbv[1], NULL);
		dec.blue = (gushort) g_strtod (rgbv[2], NULL);
		g_free (rgb);
		g_strfreev (rgbv);
	} else {
		dec.red = SATLIST_DEF_NORM_RED;
		dec.green = SATLIST_DEF_NORM_GREEN;
		dec.blue = SATLIST_DEF_NORM_BLUE;
	}
	/* Get no AOS color */
	if ( (rgb = gconf_client_get_string (client, SATLIST_NARGB_PATH, NULL)) ) {
		rgbv = g_strsplit (rgb, ",", 3);
		noaos.red = (gushort) g_strtod (rgbv[0], NULL);
		noaos.green = (gushort) g_strtod (rgbv[1], NULL);
		noaos.blue = (gushort) g_strtod (rgbv[2], NULL);
		g_free (rgb);
		g_strfreev (rgbv);
	} else {
		noaos.red = SATLIST_DEF_NORM_RED;
		noaos.green = SATLIST_DEF_NORM_GREEN;
		noaos.blue = SATLIST_DEF_NORM_BLUE;
	}

	/* update the colors in satlist */
	gtk_clist_freeze (GTK_CLIST (clist));
	n = satlist_get_size (clist);
	for ( i=0; i<n; i++ ) {
		sat = satdata_get_sat (GPOINTER_TO_UINT (gtk_clist_get_row_data (GTK_CLIST (clist), i)));
		switch (sat->status) {
		case SAT_STATUS_DECAYED:
			gtk_clist_set_foreground (GTK_CLIST (clist), i, &dec);
			break;
		case SAT_STATUS_GEOSTAT:
			if ( sat->el > 0 )
				gtk_clist_set_foreground (GTK_CLIST (clist), i, &geoh);
			else
				gtk_clist_set_foreground (GTK_CLIST (clist), i, &geon);
			break;
		case SAT_STATUS_NOAOS:
			gtk_clist_set_foreground (GTK_CLIST (clist), i, &noaos);
			break;
		default:
			if ( sat->el >= 0 )
				gtk_clist_set_foreground (GTK_CLIST (clist), i, &high);
			else
				gtk_clist_set_foreground (GTK_CLIST (clist), i, &norm);
			break;
		}
	}
	gtk_clist_thaw (GTK_CLIST (clist));

}


static void
satlist_prefs_changed (GConfClient *client, guint cid, GConfEntry *entry, gpointer vbox)
{
	/* This function is called automatically by GConf when the satlist
	   preferences are chenged. It is used to automatically reload
	   and update all open satellite lists.
	*/
	gint delay,timer;

	delay = gconf_client_get_int (client, SATLIST_TIMER_PATH, NULL);
	if (!delay)
		delay = SATLIST_DEF_TIMER;

	gtk_timeout_remove (GPOINTER_TO_INT (gtk_object_get_data (GTK_OBJECT (vbox), "timer")));
	satlist_read_colors (gtk_object_get_data (GTK_OBJECT (vbox), "clist"));
	timer = gtk_timeout_add (delay, satlist_timer_cb,
				 gtk_object_get_data (GTK_OBJECT (vbox), "clist"));
	gtk_object_set_data (GTK_OBJECT (vbox), "timer", GINT_TO_POINTER (timer));

}
