--|
--| Filename        : $Source: /var/cvs/gnuada/gnade/esql/scanner.ads,v $
--| Description     : Scanner for the embedded SQL translator
--| Author          : Michael Erdmann
--| Created On      : 22.12.2000
--| Last Modified By: $Author: me $
--| Last Modified On: $Date: 2001/05/26 13:30:08 $
--| Status          : $State: Exp $
--|
--| Copyright (C) 2000 Michael Erdmann
--|
--| This program is free software; you can redistribute it and/or
--| modify it under the terms of the GNU General Public License
--| as published by the Free Software Foundation; either version 2
--| of the License, or (at your option) any later version.
--|
--| This program/code is distributed in the hope that it will be useful,
--| but WITHOUT ANY WARRANTY; without even the implied warranty of
--| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--| GNU General Public License for more details.
--|
--| You should have received a copy of the GNU General Public License along
--| with this program; if not, write to the Free Software Foundation, Inc.,
--| 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
--|
--| As a special exception,  if other files  instantiate  generics from this
--| unit, or you link  this unit with other files  to produce an executable,
--| this  unit  does not  by itself cause  the resulting  executable  to  be
--| covered  by the  GNU  General  Public  License.  This exception does not
--| however invalidate  any other reasons why  the executable file  might be
--| covered by the  GNU Public License.
--|
--| Functional Description
--| ======================
--| This package provides the lexical scanner. Addtionaly for each token
--| the application may decide to copy the characters read so far into
--| the output file.
--|
--|
--| Restrictions
--| ============
--|
--| References
--| ==========
--| None
--|

--* Ada
package Scanner is

   type File_Reader is private;

   --|
   --| Token types
   --|
   type Lexical_Unit_Type is (
      Identifier_Lex,
      Numeric_Literal_Lex,
      Character_Literal_Lex,
      String_Lex,
      Delimiter_Lex,
      Comment_Lex,
      Unknown_Lex,
      End_Of_File_Lex );

   subtype Identifier_String is  String(1..48);
   Blank_Identifier : constant Identifier_String := (others=>' ');

   type Token_Type is record
         Lexicon      : Identifier_String := Blank_Identifier ;
         Lexical_Unit : Lexical_Unit_Type := Unknown_Lex;
      end record;

   Null_Token    : constant Token_Type := ( Blank_Identifier, Unknown_Lex );

   Lexical_Error : exception;
   --|
   --| Open a file reader on the given input file name. The output
   --| file is newly created. An exisiting file is overwritten.
   --|
   function Open(
      Input_File_Name  : in String;
      Output_File_Name : in String  ) return File_Reader;

   --|
   --| Close the file reader
   --|
   procedure Close(
       f : in out File_Reader );

   procedure Delete(
       f : in out File_Reader );

   --|
   --| Get the next token. The procedure returns the End_Of_File_Lex
   --| token if file end has been reached.
   --|
   procedure Get_Token(
      f     : in out File_Reader;
      Token : out Token_Type );

   --|
   --| Mark the data read till the end of the current token as
   --| accepted, which means it will be copied into the output
   --| file
   --|
   procedure Accept_Input(
      f     : in out File_reader );
   --|
   --| Ingnore the source text till the end of the current token,
   --| which means the text is not copied into the output file.
   --|
   procedure Ignore_Input(
       f : in out File_Reader );

   --|
   --| Insert a string into the accept buffer
   --|
   procedure Accept_String(
      f : in out File_Reader;
      s : in String );

   --|
   --| Write out the accepted text and the remaining text till
   --| end of the line.
   --|
   procedure Flush(
      f    : in out File_Reader );

   procedure Message(
      f    : in File_Reader;
      text : in String );

   --|
   --| Place a comment in the output source
   --|
   procedure Comment(
      f     : in out File_Reader;
      text  : in String );

   procedure Insert(
      f     : in out File_Reader;
      text  : in String );

   function Current_Line(
      F     : in File_Reader ) return Natural;

   --------------------------------------------------------------------------
private
    type File_Reader_Type;
    type File_Reader is access File_Reader_Type;

end Scanner;


