from libdesklets.controls import Control, Permission

from IYahooWeather import IYahooWeather

import re
import urllib
import types
import thread
import urlparse
import traceback

class WeatherControl(Control, IYahooWeather):
    __WEATHER_SOURCE = "http://weather.yahoo.com/search/" \
                       "weather?p=%(webcity)s,%(webcountry)s"

    __RE_MATCHES = re.compile("Weather location matches:\</font\>\</td\>\n\s*\</tr\>\</table\>\n\s*\<font face=\"arial\" size=-1\>\n\s*\<ul\>\n\s*\<li\>\n\s*\<a href=\"(?P<matchurl>[^\"]+)\"\>")

    def __init__(self):
        Control.__init__(self)
        self._attr = {}
        self.__timer = None
        self._update_weather()

    def _get_url(self):
        return self.__WEATHER_SOURCE % { "webcity" : urllib.quote(self.city),
                                         "webcountry" : urllib.quote(self.country) }

    def _update_weather(self):
        thread.start_new_thread(self.__update_weather_run, ())
        if self.__timer != None:
            self._remove_timer(self.__timer)
        self.__time = self._add_timer(self.update_interval * 1000, self.__update_weather_timer_callback)
        
    def __update_weather_timer_callback(self):
        self._update_weather()
        return False

    def __update_weather_run(self):
        try:
            fh = urllib.urlopen(self._get_url())
            data = fh.read()

            # Take first match if we have ambiguity
            if (data.find("Weather location matches") != -1):
                match_url = urlparse.urljoin(fh.geturl(), self.__RE_MATCHES.search(data).group("matchurl"))

                fh = urllib.urlopen(match_url)
                data = fh.read()

                self.__parse_weather(data)
                status = 'ambiguous'
            else:
                self.__parse_weather(data)
                status = 'ok'
        except Exception:
            print traceback.print_exc()
            status = 'fail'
            
        self._attr['status'] = status
        self._update('status')
    
        
    def __parse_weather(self, data):
        for name, regex, acc_meth, doc in self._w_props:
            v = regex.search(data).group(1)
            if isinstance(acc_meth, type):
                v = acc_meth(v)
            elif isinstance(acc_meth, types.FunctionType):
                v = acc_meth(self, v)
            self._attr[name] = v
            self._update(name)

    
    # config properties
    _props = (('city', str, _update_weather, "City", "Tokyo"),
              ('country', str, _update_weather, "Country", "Japan"),
              ('metric', bool, _update_weather, "Use metric units", True),
              ('time_format_24', bool, _update_weather, "Use 24h time format", True),
              ('update_interval', int, _update_weather, "Update interval in sec", 30 * 60),
              ('status', str, None, "Status (ok, fail, ambiguous, wait)", "wait"))
              

    def _set_attr(self, value, name, prop_type, update_func=None):
        self._attr[name] = prop_type(value)
        if update_func != None:
            update_func(self)
        self._update(name)

    # _w_props access methods
    def __get_temperature(self, value):
        if self.metric:
            return int(round((float(value) - 32) * (5 / 9.0))), "&#x0b0;C"
        else:
            return int(value), "&#x0b0;F"

    def __get_pressure(self, value):
        if self.metric:
            return float(value) * 33.8639, "hPa"
        else:
            return float(value), "in"

    def __get_pressure_change(self, value):
        return value
        chg = value.split(' ')
        if len(chg) > 1:
            return chg[3]
        return chg[1]

    def __get_wind(self, value):
        tmp = value.split(' ')
        if len(tmp) == 1:
            return "calm"
        
        val = float(tmp[1])
        uni = "mph"
        dire = tmp[0]
        if self.metric:
            val = val * 1.60934
            uni = "kph"
        return val, uni, dire

    def __get_time(self, t):
        if (self.time_format_24):
            t = t.replace(':', ' ')
            t = t.split(' ')
            m = int(t[1])
            h = int(t[0])
            if(t[2] == 'pm'):
                h = (h + 12) % 24
            return "%02d:%02d" % (h, m)
        else:
            return t

    def __get_visibility(self, value):
        tmp = value.split(' ')
        if len(tmp) == 1:
            return "unlimited"
        
        val = tmp[0]
        uni = tmp[1]
        if self.metric:
            val = int(round(float(val) * 1.60934))
            
        return val, uni

    __WEATHER_ICONS = { 1: "rain",
                        2: "rain",
                        3: "thunder",
                        4: "thunder",
                        5: "rain",
                        6: "rain",
                        7: "ice",
                        8: "rain",
                        9: "drizzle",
                        10: "rain",
                        11: "rain",
                        12: "rain",
                        13: "snow",
                        14: "snow",
                        15: "snow",
                        16: "snow",
                        17: "thunder",
                        18: "rain",
                        19: "fog",
                        20: "fog",
                        21: "fog",
                        22: "fog",
                        23: "wind",
                        24: "wind",
                        25: "ice",
                        26: "cloud",
                        27: "luna-nori", # moon
                        28: "mostly-cloudy",
                        29: "luna-ceatza", # moon
                        30: "suncloud",
                        31: "moon",
                        32: "sun",
                        33: "moon",
                        34: "sun-light-clouds",
                        35: "thunder",
                        36: "sun",
                        37: "thunder",
                        38: "thunder",
                        39: "rain",
                        40: "rain",
                        41: "snow",
                        42: "snow",
                        43: "snow",
                        44: "suncloud",
                        45: "moon-rain", # moon
                        46: "moon-snow", # moon
                        47: "moon-tstorm" #moon
                        }

    def __get_weather_icon(self, value):
        return self.__WEATHER_ICONS[int(value)]
    

    # properties and regular expressions to parse the weather data
    _w_props = (('temperature', re.compile("<b>(?P<temperature>\d+)&deg;</b>"), __get_temperature, ''),
                ('temperature_hi', re.compile("High:</font></td><td align=left width=\"50%\"><font face=Arial size=3><b>(?P<temperature_hi>\d+)&deg;</b>"), __get_temperature, ''),
                ('temperature_lo', re.compile("High:</font></td><td align=left width=\"50%\"><font face=Arial size=3><b>(?P<temperature_lo>\d+)&deg;</b>"), __get_temperature, ''),
                ('temperature_feel', re.compile("<b>Feels Like:</b></font></td>\n<td><font face=Arial size=2>\n(?P<temperature_feel>\d+)"), __get_temperature, ''),
                ('pressure', re.compile("<b>Barometer:</b></font></td>\n<td><font face=Arial size=2>\n(?P<pressure>[\d\.]+) \w+ \nand (?P<pressure_change>.+)\n"), __get_pressure, ''),
                ('pressure_change', re.compile("<b>Barometer:</b></font></td>\n<td><font face=Arial size=2>\n[\d\.]+ \w+ \nand (?P<pressure_change>.+)\n"), __get_pressure_change, ''),
                ('dewpoint', re.compile("<b>Dewpoint:</b></font></td>\n<td><font face=Arial \nsize=2> (?P<dewpoint>\d+)"), __get_temperature, ''),
                ('wind', re.compile("<b>Wind:</b></font></td>\n<td><font face=Arial size=2>\n(?P<wind>[\w\s]+)"), __get_wind, ''),
                ('humidity', re.compile("<b>Humidity:</b></font></td>\n<td><font face=Arial size=2>\n(?P<humidity>\d+)\%"), int, 'in %'),
                ('sunrise', re.compile("<b>Sunrise:</b></font></td>\n<td><font face=Arial size=2>\n(?P<sunrise>.+)</font></td>"), __get_time, ''),
                ('sunset', re.compile("<b>Sunset:</b></font></td>\n<td><font face=Arial size=2>\n(?P<sunset>.+)</font></td>"), __get_time, ''),
                ('visibility', re.compile("<b>Visibility:</b></font></td>\n<td><font face=Arial size=2>\n(?P<visibility>\S+)"), __get_visibility, ''),
                ('sky', re.compile("<br><font face=Arial size=2>(?P<sky>[^\<]+)\</font\>\</td\>"), str, ''),
                ('icon', re.compile("\<img src=\"http://us.i1.yimg.com/us.yimg.com/i/us/we/52/(?P<icon>\d+).gif\""), __get_weather_icon, ''))
            

for name, prop_type, update_func, doc, default in WeatherControl._props:
    if getattr(WeatherControl, name) != Permission.READ:
        set_func = lambda self, value, name=name, prop_type=prop_type, update_func=update_func: self._set_attr(value, name, prop_type, update_func)
    else:
        set_func = None
    setattr(WeatherControl, name, property(fget = lambda self, name=name, default=default: self._attr.get(name, default),
                                           fset = set_func,
                                           doc=doc))

    del set_func

for name, regex, acc_meth, doc in WeatherControl._w_props:
    setattr(WeatherControl, name, property(fget = lambda self, name=name: self._attr.get(name),
                                           doc=doc))

def get_class():
    return WeatherControl
