/* GAdmin-Rsync - An easy to use GTK+ frontend for the rsync backup client and server.
 * Copyright (C) 2007 Magnus Loef <magnus-swe@telia.com> 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
*/



#include "../config.h"
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <gtk/gtk.h>
#include "gettext.h"
#include "widgets.h"
#include "allocate.h"
#include "show_info.h"
#include "make_settings_checkbuttons.h"
#include "make_settings_entries.h"
#include "make_settings_spinbuttons.h"
#include "make_settings_combos.h"
#include "create_rsync_tab.h"
#include "add_backup_menu.h"
#include "save_backup_settings.h"
#include "key_handling.h"

int global_backup_local		= 1;
int global_backup_local_remote 	= 0;
int global_backup_remote_local	= 0;

GtkTreeIter iter_filesel;

extern gchar *global_key_path;



/* Backup local checkbutton callback */
void backup_local_checkbutton_toggled(GtkButton *button, struct w *widgets)
{
    if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widgets->backup_chkbtn_local)) )
    {
        global_backup_local        = 1;
	global_backup_local_remote = 0;
	global_backup_remote_local = 0;
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(widgets->backup_chkbtn_local_remote), FALSE);
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(widgets->backup_chkbtn_remote_local), FALSE);
    }
}

/* Backup local to remote checkbutton callback */
void backup_local_remote_checkbutton_toggled(GtkButton *button, struct w *widgets)
{
    if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widgets->backup_chkbtn_local_remote)) )
    {    
        global_backup_local  	   = 0;
        global_backup_local_remote = 1;
        global_backup_remote_local = 0;
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(widgets->backup_chkbtn_local), FALSE);
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(widgets->backup_chkbtn_remote_local), FALSE);
    }
}

/* Backup remote to local checkbutton callback */
void backup_remote_local_checkbutton_toggled(GtkButton *button, struct w *widgets)
{
    if( gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widgets->backup_chkbtn_remote_local)) )
    {    
        global_backup_local  	   = 0;
        global_backup_local_remote = 0;
        global_backup_remote_local = 1;
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(widgets->backup_chkbtn_local), FALSE);
	gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(widgets->backup_chkbtn_local_remote), FALSE);
    }
}

/* Scroll to the last row in the treeview */
void scroll_to_last_row(GtkTreeView *treeview)
{
    /* Scroll to the last row */
    GtkTreeModel *model;
    GtkTreePath *path;
    GtkTreeSelection *selection;

    model = gtk_tree_view_get_model(GTK_TREE_VIEW(treeview));
    path = gtk_tree_model_get_path(GTK_TREE_MODEL(model), &iter_filesel);

    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
    gtk_tree_selection_select_path(GTK_TREE_SELECTION(selection), path);
    gtk_tree_view_set_cursor(GTK_TREE_VIEW(treeview), path, 0, TRUE);
    gtk_tree_path_free(path);
}

/* Source file selection ok clicked */
void src_filesel_ok(struct w *widgets)
{
    /* Adds source directory to the treeview */
    gchar *utf8 = NULL;
    G_CONST_RETURN gchar *path = gtk_file_selection_get_filename(GTK_FILE_SELECTION(widgets->src_filesel));
    utf8 = g_locale_to_utf8(path, strlen(path), NULL, NULL, NULL);

    /* Insert the backup source in the treeview row */
    gtk_list_store_append(GTK_LIST_STORE(widgets->backup_store), &iter_filesel);
    gtk_list_store_set(widgets->backup_store, &iter_filesel, 1, utf8, -1);
    if( utf8!=NULL )
      g_free(utf8);

    /* Check the backup checkbutton in the treeview */
    gtk_list_store_set(widgets->backup_store, &iter_filesel, 6, TRUE, -1);

    gtk_widget_destroy(widgets->src_filesel);

    /* If destination is local, show dest file selector */
    if( global_backup_local )
	show_dst_filesel(widgets);
    else
    if( global_backup_local_remote )
      show_server_menu("dest", widgets);
    else
      printf("This is 2 odd.\n");
}


/* Destination file selection ok clicked */
void dst_filesel_ok(struct w *widgets)
{
    /* Adds destination directory to the treeview */
    gchar *utf8 = NULL;

    G_CONST_RETURN gchar *path = gtk_file_selection_get_filename(GTK_FILE_SELECTION(widgets->dst_filesel));
    utf8 = g_locale_to_utf8(path, strlen(path), NULL, NULL, NULL);

    /* Using a global treeiter */
    gtk_list_store_set(widgets->backup_store, &iter_filesel, 3, utf8, -1);
    /* Check the rsync checkbutton in the treeview */
    gtk_list_store_set(widgets->backup_store, &iter_filesel, 6, TRUE, -1);

    if( utf8!=NULL )
      g_free(utf8);

    gtk_widget_destroy(widgets->dst_filesel);

    /* Save the backup settings */
    save_backup_settings(widgets);
}


/* Show source file or server selection */
void show_src_filesel(struct w *widgets)
{
    /* Source is remote */
    if( global_backup_remote_local )
    {
	/* Show source server selection */
	show_server_menu("source", widgets);

	if( GTK_IS_WINDOW(widgets->backup_menu_window) )
          gtk_widget_destroy(widgets->backup_menu_window);
          
        return;
    }
    
    /* Show the source file selection */
    widgets->src_filesel = gtk_file_selection_new("Select source directory or file.");
    gtk_widget_show(widgets->src_filesel);

    /* Connect the ok button to the file selection ok function */
    g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(widgets->src_filesel)->ok_button), "clicked",
                    	     G_CALLBACK(src_filesel_ok), widgets);
                        	          
    /* Cancel button destroys the fileselection widget */
    g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(widgets->src_filesel)->cancel_button),
                        "clicked", G_CALLBACK(src_filesel_cancel), widgets);

    if( GTK_IS_WINDOW(widgets->backup_menu_window) )
      gtk_widget_destroy(widgets->backup_menu_window);
}


/* Show destination file selection */
void show_dst_filesel(struct w *widgets)
{
    G_CONST_RETURN gchar *host, *path, *port, *user, *pass, *keytype, *keylen;
    gchar *info, *key_name_priv, *key_path_priv;

    /* Destination is remote */
    if( global_backup_local_remote )
    {
	/* Show destination server selection */
	show_server_menu("dest", widgets);

	if( GTK_IS_WINDOW(widgets->backup_menu_window) )
          gtk_widget_destroy(widgets->backup_menu_window);
          
        return;
    }
    
    /* Save source server selection */
    if( global_backup_remote_local )
    {
	/* Insert source server data into the treeview and upload keys */
	host    = gtk_entry_get_text(GTK_ENTRY(widgets->remote_server_entry));
	path    = gtk_entry_get_text(GTK_ENTRY(widgets->remote_path_entry));
	port    = gtk_entry_get_text(GTK_ENTRY(widgets->remote_port_entry));
	user    = gtk_entry_get_text(GTK_ENTRY(widgets->remote_user_entry));
	pass    = gtk_entry_get_text(GTK_ENTRY(widgets->remote_password_entry));
	keytype = gtk_combo_box_get_active_text(GTK_COMBO_BOX(widgets->key_type_combo));
	keylen  = gtk_entry_get_text(GTK_ENTRY(widgets->key_length_spinbutton));

	if( is_empty((gchar *)host)  ||  is_empty((gchar *)path)
	||  is_empty((gchar *)port)  ||  is_empty((gchar *)user)
	||  is_empty((gchar *)pass)  ||  is_empty((gchar *)keytype)
	||  is_empty((gchar *)keylen) )
	{
	    info = g_strdup_printf(_("All fields must be filled in.\n"));
	    show_info(info);
	    if( info!=NULL )
	      g_free(info);
	    return;    
	}

	key_name_priv = mk_key_name((char *)user, (char *)host, "priv");
	key_path_priv = g_strdup_printf("%s/%s", global_key_path, key_name_priv);
	g_free(key_name_priv);

	/* Insert the backup source in the treeview row */
	gtk_list_store_append(GTK_LIST_STORE(widgets->backup_store), &iter_filesel);

	gtk_list_store_set(widgets->backup_store, &iter_filesel, 0, host, -1);
	gtk_list_store_set(widgets->backup_store, &iter_filesel, 1, path, -1);
	gtk_list_store_set(widgets->backup_store, &iter_filesel, 8, user, -1);
	gtk_list_store_set(widgets->backup_store, &iter_filesel, 9, port, -1);
	gtk_list_store_set(widgets->backup_store, &iter_filesel, 10, key_path_priv, -1);

	/* Check the backup checkbutton in the treeview */
	gtk_list_store_set(widgets->backup_store, &iter_filesel, 6, TRUE, -1);

	g_free(key_path_priv);

	/* Generate and setup keys if there are no keys for this host */
	setup_keys((char *)user, (char *)pass, (char *)host, (char *)port, (char *)keytype, (char *)keylen, widgets);

	/* Check the backup checkbutton in the treeview */
	gtk_list_store_set(widgets->backup_store, &iter_filesel, 6, TRUE, -1);

	if( GTK_IS_WINDOW(widgets->server_menu_window) )
          gtk_widget_destroy(widgets->server_menu_window);
	/* Dont return */    
    }

    /* Show the destination file selection */
    widgets->dst_filesel = gtk_file_selection_new("Select destination directory or file.");
    gtk_widget_show(widgets->dst_filesel);

    /* Connect the ok button to the file selection ok function */
    g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(widgets->dst_filesel)->ok_button), "clicked",
                    	     G_CALLBACK(dst_filesel_ok), widgets);
                        	          
    /* Cancel button destroys the fileselection widget */
    g_signal_connect_swapped(G_OBJECT(GTK_FILE_SELECTION(widgets->dst_filesel)->cancel_button),
                        "clicked", G_CALLBACK(dst_filesel_cancel), widgets);
}


/* Destination server selection forward button is clicked */
void dest_server_forward_clicked(struct w *widgets)
{
    /* Insert destination server data into the treeview and upload keys */
    G_CONST_RETURN gchar *host, *path, *port, *user, *pass, *keytype, *keylen;
    gchar *info, *key_name_priv, *key_path_priv;

    host    = gtk_entry_get_text(GTK_ENTRY(widgets->remote_server_entry));
    path    = gtk_entry_get_text(GTK_ENTRY(widgets->remote_path_entry));
    port    = gtk_entry_get_text(GTK_ENTRY(widgets->remote_port_entry));
    user    = gtk_entry_get_text(GTK_ENTRY(widgets->remote_user_entry));
    pass    = gtk_entry_get_text(GTK_ENTRY(widgets->remote_password_entry));
    keytype = gtk_combo_box_get_active_text(GTK_COMBO_BOX(widgets->key_type_combo));
    keylen  = gtk_entry_get_text(GTK_ENTRY(widgets->key_length_spinbutton));

    if( is_empty((gchar *)host)  ||  is_empty((gchar *)path)
    ||  is_empty((gchar *)port)  ||  is_empty((gchar *)user)
    ||  is_empty((gchar *)pass)  ||  is_empty((gchar *)keytype)
    ||  is_empty((gchar *)keylen) )
    {
	info = g_strdup_printf(_("All fields must be filled in.\n"));
	show_info(info);
	if( info!=NULL )
	  g_free(info);
	return;    
    }

    key_name_priv = mk_key_name((char *)user, (char *)host, "priv");
    key_path_priv = g_strdup_printf("%s/%s", global_key_path, key_name_priv);
    g_free(key_name_priv);

    gtk_list_store_set(widgets->backup_store, &iter_filesel, 2, host, -1);
    gtk_list_store_set(widgets->backup_store, &iter_filesel, 3, path, -1);
    gtk_list_store_set(widgets->backup_store, &iter_filesel, 8, user, -1);
    gtk_list_store_set(widgets->backup_store, &iter_filesel, 9, port, -1);
    gtk_list_store_set(widgets->backup_store, &iter_filesel, 10, key_path_priv, -1);

    /* Check the backup checkbutton in the treeview */
    gtk_list_store_set(widgets->backup_store, &iter_filesel, 6, TRUE, -1);

    g_free(key_path_priv);

    /* Generate and setup keys if there are no keys for this host */
    setup_keys((char *)user, (char *)pass, (char *)host, (char *)port, (char *)keytype, (char *)keylen, widgets);

    /* Save the backup settings */
    save_backup_settings(widgets);

    gtk_widget_destroy(widgets->server_menu_window);
}


/* Source file selection canceled */
void src_filesel_cancel(struct w *widgets)
{
    gtk_widget_destroy(widgets->src_filesel);
}

/* Destination file selection canceled */
void dst_filesel_cancel(struct w *widgets)
{
    /* Scroll to the last row in the treeview and delete it */
    scroll_to_last_row(GTK_TREE_VIEW(widgets->backup_treeview));
    del_treeview_row(widgets);

    gtk_widget_destroy(widgets->dst_filesel);
}


/* The server menu selection window */
void show_server_menu(char type[128], struct w *widgets)
{
    GtkWidget *frame1;
    GtkWidget *table1;
    GtkWidget *menu_vbox;
    GtkTooltips *tooltips;
    gchar *utf8=NULL;
    gchar *info, *text;
    int a=0, b=1;

    widgets->server_menu_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_position(GTK_WINDOW (widgets->server_menu_window), GTK_WIN_POS_CENTER);
    gtk_widget_set_size_request(widgets->server_menu_window, 480, -1);

    /* Set window information */
    if( strstr(type, "source") )
	info = g_strdup_printf(_("GAdmin-Rsync %s source server selection"), VERSION);
    else
	info = g_strdup_printf(_("GAdmin-Rsync %s destination server selection"), VERSION);
    
    gtk_window_set_title(GTK_WINDOW(widgets->server_menu_window), info);
    g_free(info);


    menu_vbox = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(widgets->server_menu_window), menu_vbox);

    tooltips = gtk_tooltips_new();

    /* 1 tables with some settings and 2 columns */
    table1 = gtk_table_new(2, 2, FALSE);

    if( strstr(type, "source") )
	frame1 = gtk_frame_new(_("Supply source server information:"));
    else
	frame1 = gtk_frame_new(_("Supply destination server information:"));

    gtk_box_pack_start(GTK_BOX(menu_vbox), frame1, TRUE, TRUE, 1);
    gtk_container_add(GTK_CONTAINER(frame1), table1);


    widgets->remote_server_entry   = make_entry_with_label(GTK_TABLE(table1), _(" Remote server: "), 0,1,a,b, 300);
    a++; b++;
    widgets->remote_path_entry     = make_entry_with_label(GTK_TABLE(table1), _(" Remote path: "), 0,1,a,b, 300);
    a++; b++;
    widgets->remote_port_entry     = make_entry_with_label(GTK_TABLE(table1), _(" Remote port: "), 0,1,a,b, 300);
    a++; b++;
    widgets->remote_user_entry     = make_entry_with_label(GTK_TABLE(table1), _(" Remote user: "), 0,1,a,b, 300);
    a++; b++;
    widgets->remote_password_entry = make_entry_with_label(GTK_TABLE(table1), _(" Remote password: "), 0,1,a,b, 300);
    a++; b++;

    /* Use asterisks for the password field */
    gtk_entry_set_visibility(GTK_ENTRY(widgets->remote_password_entry), FALSE);

    widgets->key_type_combo = make_combo_with_label(GTK_TABLE(table1), _(" Key type: "), 0,1,a,b, 300);
    a++; b++;
    widgets->key_length_spinbutton = make_spinbutton_with_label(GTK_TABLE(table1), _(" Key length: "), 0,1,a,b, 300);
    a++; b++;


    text = g_strdup_printf("22");
    utf8 = g_locale_to_utf8(text, strlen(text), NULL, NULL, NULL);
    gtk_entry_set_text(GTK_ENTRY(widgets->remote_port_entry), utf8);
    if( text!=NULL )
      g_free(text);
    if( utf8!=NULL )
      g_free(utf8);

    text = g_strdup_printf("%s", g_get_user_name());
    utf8 = g_locale_to_utf8(text, strlen(text), NULL, NULL, NULL);
    gtk_entry_set_text(GTK_ENTRY(widgets->remote_user_entry), utf8);
    if( text!=NULL )
      g_free(text);
    if( utf8!=NULL )
      g_free(utf8);

    gtk_spin_button_set_value(GTK_SPIN_BUTTON(widgets->key_length_spinbutton), 1024);

    /* Set combo choises */
    text = g_strdup_printf("dsa");
    utf8 = g_locale_to_utf8(text, strlen(text), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->key_type_combo), utf8);
    gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->key_type_combo), 0);
    if( text!=NULL )
      g_free(text);
    if( utf8!=NULL )
      g_free(utf8);

    text = g_strdup_printf("rsa");
    utf8 = g_locale_to_utf8(text, strlen(text), NULL, NULL, NULL);
    gtk_combo_box_append_text(GTK_COMBO_BOX(widgets->key_type_combo), utf8);
    if( text!=NULL )
      g_free(text);
    if( utf8!=NULL )
      g_free(utf8);


//    FIXME: Add a progress bar after the key type row on the same line.


    /* Buttons, "Cancel", "Forward"*/
    GtkWidget *hbutton_box = gtk_hbutton_box_new();
    gtk_button_box_set_layout(GTK_BUTTON_BOX(hbutton_box), GTK_BUTTONBOX_SPREAD);

    GtkWidget *cancel_button, *forward_button;
    cancel_button = gtk_button_new_from_stock(GTK_STOCK_QUIT);
    forward_button = gtk_button_new_from_stock(GTK_STOCK_GO_FORWARD);

    gtk_box_pack_start(GTK_BOX(hbutton_box), cancel_button, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(hbutton_box), forward_button, FALSE, FALSE, 0);
    gtk_container_add(GTK_CONTAINER(menu_vbox), hbutton_box);


    /* Window exit signal */
    g_signal_connect(GTK_WINDOW(widgets->server_menu_window), "delete_event",
                     G_CALLBACK(gtk_widget_destroy), NULL);



    /* Quit / Cancel button */
    g_signal_connect_swapped((gpointer)cancel_button, "clicked",
                    	    G_CALLBACK(gtk_widget_destroy),
                            GTK_OBJECT(widgets->server_menu_window));

    /* Forward button signals */
    
    /* We are asked to show a source server selection */
    if( strstr(type, "source") )
    {
        /* Pressing the forward button shows a destination selection */
	g_signal_connect_swapped((gpointer)forward_button, "clicked",
                    		G_CALLBACK(show_dst_filesel), widgets);
    }
    else /* We are asked to show a destination server selection */
    {
	g_signal_connect_swapped((gpointer)forward_button, "clicked",
                    		G_CALLBACK(dest_server_forward_clicked), widgets);
    }
    

    gtk_widget_show_all(widgets->server_menu_window);
}



/* The Main backup menu selection window */
void show_backup_menu(struct w *widgets)
{
    GtkWidget *frame1;
    GtkWidget *table1;
    GtkWidget *menu_vbox;
    GtkTooltips *tooltips;
    gchar *utf8=NULL;
    gchar *info;
    int a=0, b=1;

    widgets->backup_menu_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_position(GTK_WINDOW (widgets->backup_menu_window), GTK_WIN_POS_CENTER);
    gtk_widget_set_size_request(widgets->backup_menu_window, 450, -1);

    /* Set window information */
    info = g_strdup_printf(_("GAdmin-Rsync %s backup selection"), VERSION);
    gtk_window_set_title(GTK_WINDOW(widgets->backup_menu_window), info);
    g_free(info);

    g_signal_connect(GTK_WINDOW(widgets->backup_menu_window), "delete_event",
                     G_CALLBACK(gtk_widget_destroy), NULL);

    menu_vbox = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(widgets->backup_menu_window), menu_vbox);

    tooltips = gtk_tooltips_new();

    /* 1 tables with some settings and 2 columns */
    table1 = gtk_table_new(2, 2, FALSE);

    frame1 = gtk_frame_new("Backup type selection:");
    gtk_box_pack_start(GTK_BOX(menu_vbox), frame1, TRUE, TRUE, 1);
    gtk_container_add(GTK_CONTAINER(frame1), table1);

    /* Backup local checkbutton */
    widgets->backup_chkbtn_local = make_checkbutton_with_label(GTK_TABLE(table1), _("Local backup:"), 0,1,a,b);
    a++; b++;
    widgets->backup_chkbtn_local_remote = make_checkbutton_with_label(GTK_TABLE(table1), _("Local to remote backup:"), 0,1,a,b);
    a++; b++;
    widgets->backup_chkbtn_remote_local = make_checkbutton_with_label(GTK_TABLE(table1), _("Remote to local backup:"), 0,1,a,b);
    a++; b++;
	      
    /* Buttons, "Cancel", "Forward"*/
    GtkWidget *hbutton_box = gtk_hbutton_box_new();
    gtk_button_box_set_layout(GTK_BUTTON_BOX(hbutton_box), GTK_BUTTONBOX_SPREAD);

    GtkWidget *cancel_button, *forward_button;
    cancel_button = gtk_button_new_from_stock(GTK_STOCK_QUIT);
    forward_button = gtk_button_new_from_stock(GTK_STOCK_GO_FORWARD);

    gtk_box_pack_start(GTK_BOX(hbutton_box), cancel_button, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(hbutton_box), forward_button, FALSE, FALSE, 0);
    gtk_container_add(GTK_CONTAINER(menu_vbox), hbutton_box);

    /* Set default backup states.. same as in gadmin-rsync.c (globals origin) */
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(widgets->backup_chkbtn_local), TRUE);
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(widgets->backup_chkbtn_local_remote), FALSE);
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(widgets->backup_chkbtn_remote_local), FALSE);

    /* We need to reset global_backup* options so that it matches the default settings */
    global_backup_local   	= 1;
    global_backup_local_remote  = 0;
    global_backup_remote_local  = 0;

    /* Signal hookups for the checkbuttons, all uses the same function to set globals */
    g_signal_connect((gpointer)widgets->backup_chkbtn_local, "toggled",
                	    G_CALLBACK(backup_local_checkbutton_toggled), widgets);

    g_signal_connect((gpointer)widgets->backup_chkbtn_local_remote, "toggled",
                	    G_CALLBACK(backup_local_remote_checkbutton_toggled), widgets);

    g_signal_connect((gpointer)widgets->backup_chkbtn_remote_local, "toggled",
                	    G_CALLBACK(backup_remote_local_checkbutton_toggled), widgets);

    /* Quit button */
    g_signal_connect_swapped((gpointer)cancel_button, "clicked",
                    	    G_CALLBACK(gtk_widget_destroy),
                            GTK_OBJECT(widgets->backup_menu_window));
    /* Forward button */
    g_signal_connect_swapped((gpointer)forward_button, "clicked",
                    	    G_CALLBACK(show_src_filesel), widgets);

    gtk_widget_show_all(widgets->backup_menu_window);

    if( utf8 != NULL )
      g_free(utf8);
}
