/***************************************

    This is part of frox: A simple transparent FTP proxy
    Copyright (C) 2000 James Hollingshead

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

  sstr.c

***************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <ctype.h>

#include "sstr.h"
#include "sstr_private.h"

void (*on_error) (void)=NULL;

void sstr_setopts(void (*func) (void), int flags)
{
	on_error = func;
}

/* ------------------------------------------------------------- **
**  mlen=maximum no. of chars to hold. mlen=0 gives a dynamically
**  growing string.
**
**  NB. We malloc one extra byte to give space for a NUL termination,
**  but we do not use one internally. However this allows us to NUL
**  terminate our internal buffer before calling eg. atoi() on it, or
**  returning a pointer to it.
**  ------------------------------------------------------------- */
sstr *sstr_init(int mlen)
{
	sstr *ret;

	ret = malloc(sizeof(sstr));
	if (ret == NULL) {
		if (on_error)
			on_error();
		return (NULL);
	}

	if (mlen>0) {
		ret->maxlen = mlen;
		ret->growable = 0;
	} else {
		ret->growable = 1;
		ret->maxlen = 50;
	}
	ret->buf = malloc(ret->maxlen + 1);
	if (ret->buf == NULL) {
		free(ret);
		if (on_error)
			on_error();
		return (NULL);
	}

	ret->len = 0;
	return ret;
}

void sstr_free(sstr * p)
{
	free(p->buf);
	free(p);
}

void sstr_empty(sstr * p)
{
	p->len = 0;
	shrink(p);
}

int sstr_len(const sstr * p)
{
	return (p->len);
}

int sstr_cpy2(sstr * dest, const char *src)
{
	if(!src) return(-1);
	return (sstr_ncpy2(dest, src, strlen(src)));
}

int sstr_ncpy2(sstr * dest, const char *src, int len)
{
	if (src == NULL || len == 0) {
		dest->len = 0;
		return (0);
	}

	if (len < 0) return (-1);
	if (len > dest->maxlen && !grow(dest, len))
		return (-1);

	memcpy(dest->buf, src, len);
	dest->len = len;

	shrink(dest);
	return (0);
}

int sstr_ncat2(sstr * dest, const char *src, int len)
{
	if (src == NULL || len == 0) return (0);

	if (len < 0) return (-1);
	if (dest->len + len > dest->maxlen && !grow(dest, dest->len + len))
		return (-1);

	memcpy(dest->buf + dest->len, src, len);
	dest->len += len;
	return (0);
}

int sstr_cat(sstr * dest, const sstr * src)
{
	if (src == NULL) return (0);

	return (sstr_ncat2(dest, src->buf, src->len));
}

int sstr_cpy(sstr * dest, const sstr * src)
{
	if (src == NULL) return (0);

	return (sstr_ncpy2(dest, src->buf, src->len));
}

const char *sstr_buf(const sstr * p)
{
	p->buf[p->len] = 0;
	return p->buf;
}

sstr *sstr_dup(const sstr *buf)
{
	sstr *ret;

	ret = sstr_init(0);
	sstr_cpy(ret, buf);
	return (ret);
}

sstr *sstr_dup2(const char *buf)
{
	sstr *ret;

	ret = sstr_init(0);
	sstr_cpy2(ret, buf);
	return (ret);
}

int sstr_casecmp2(const sstr * s1, const char *s2)
{
	if(!s2) return(-1);
	s1->buf[s1->len] = 0;
	return (strcasecmp(s1->buf, s2));
}

int sstr_ncasecmp2(const sstr * s1, const char *s2, int len)
{
	if(!s2) return(-1);
	s1->buf[s1->len] = 0;
	return (strncasecmp(s1->buf, s2, len));
}

int sstr_cmp(const sstr * s1, const sstr *s2)
{
	s1->buf[s1->len] = 0;
	s2->buf[s2->len] = 0;
	return (strcmp(s1->buf, s2->buf));
}

int sstr_cmp2(const sstr * s1, const char *s2)
{
	if(!s2) return(-1);
	s1->buf[s1->len] = 0;
	return (strcmp(s1->buf, s2));
}

int sstr_atoi(const sstr * p)
{
	p->buf[p->len] = 0;
	return (atoi(p->buf));
}

int sstr_chr(const sstr * p, int c)
{
	int i;
	if (c > 127)
		c -= 256;
	for (i = 0; i < p->len; i++)
		if (p->buf[i] == c)
			return (i);

	return (-1);
}

int sstr_pbrk2(const sstr * p, const char *accept)
{
	int i;
	for (i = 0; i < p->len; i++)
		if (strchr(accept, p->buf[i]))
			return (i);

	return (-1);
}

void sstr_strip(sstr * p, const char *strip)
{
	int i;
	for (i = 0; i < p->len && strchr(strip, p->buf[i]); i++);
	sstr_split(p, NULL, 0, i);
}

int sstr_escapebs(sstr * p)
{
	int i;

	for (i = 0; i < p->len; i++) {
		if (p->buf[i] == '/') {
			if (p->len + 2 > p->maxlen && !grow(p, p->len + 2))
				return (-1);
			memmove(p->buf + i + 2, p->buf + i, p->len - i - 1);
			p->buf[i++] = '%';
			p->buf[i++] = '2';
			p->buf[i] = 'f';
			p->len += 2;
		}
	}
	return (0);
}

int sstr_token(sstr * in, sstr * tok, const char *delim, int flags)
{
	int sep, quote, i;
	if (in->len == 0)
		return (-1);

	sstr_strip(in, delim);

	if ((flags & SSTR_QTOK) && (*in->buf == '"' || *in->buf == '\'')) {
		quote = *in->buf;
		for (i = 1; i < in->len && in->buf[i] != quote; i++);
		if (i == in->len)
			return (-1);
		sstr_split(in, tok, 1, i - 1);
		sstr_split(in, NULL, 0, 2);	/* Remove quotes */
	} else {
		if ((i = strcspn(in->buf, delim)) >= in->len)
			return (-1);
		sstr_split(in, tok, 0, i);
	}
	sep = *in->buf;
	sstr_strip(in, delim);

	return (sep);
}

int sstr_getchar(const sstr * p, int i)
{
	if (i >= p->len || i < 0)
		return (-1);
	return (p->buf[i]);
}

int sstr_setchar(const sstr * p, int i, int c)
{
	if (i >= p->len || i < 0)
		return (-1);
	p->buf[i] = c;
	return (0);
}

int sstr_split(sstr * in, sstr * out, int start, int cnt)
{
	if (start + cnt > in->len || start < 0 || cnt < 0)
		return (-1);
	if (!cnt) {
		if (out)
			sstr_empty(out);
		return (0);
	}

	if (out)
		sstr_ncpy2(out, in->buf + start, cnt);

	if (in->len > start + cnt)
		memmove(in->buf + start, in->buf + start + cnt,
			in->len - start - cnt);
	in->len -= cnt;
	shrink(in);
	return (0);
}

int sstr_makeprintable(sstr * p, int c)
{
	int i, j = 0;

	for (i = 0; i < p->len; i++) {
		if (!isprint(p->buf[i])) {
			p->buf[i] = c;
			j++;
		}
	}
	return (j);
}

int grow(sstr * p, int size)
{
	char *tmp;

	if (!p->growable)
		return (0);
	tmp = realloc(p->buf, size + 50);
	if (tmp == NULL) {
		if (on_error)
			on_error();
		else
			return (0);
	}
	p->buf = tmp;
	p->maxlen = size + 50;
	return (1);
}

int shrink(sstr * p)
{
	char *tmp;
	int diff;

	if (!p->growable)
		return (0);
	diff = p->maxlen - p->len;
	if (diff > 50) {
		tmp = realloc(p->buf, p->len + 25);
		if (tmp == NULL) {
			if (on_error)
				on_error();
			else
				return (0);
		}
		p->buf = tmp;
		p->maxlen = p->len + 25;
	}
	return (1);
}
