/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/


/*****************************************************************
*
*  File: painter.c 
*
*  Contents:  Routines to accumulate and sort facets for display
*                 back to front.  Not device specific, but calls
*                 device specific routines  for actual display.
*                 Also does transformations on edges.
*                 painter_end() uses Newell-Newell-Sancha algorithm
*                 to depth sort and display facets.
*/

#include "include.h"

static int count;      /* number of facets */
static int maxcount;  /* number allocated */
struct tsort *trilist; /* list for depth sorting triangles */

/* results of comparing depth of facets */
#define  DISJOINT     1
#define  FIRST_BACK  2
#define  SECOND_BACK 4
#define  ASPLITTINGB     8
#define  BSPLITTINGA     16
#define  NOTKNOWN     32
#define  COPLANAR  64

REAL tableau[7][3];  /* simplex tableau */
void pivot ARGS((int ,int ));
int newell_split ARGS((struct tsort *,struct tsort *,struct tsort *,struct tsort *));
int edgeedgecompare ARGS((struct tsort *,struct tsort *));
int edgefacetcompare ARGS((struct tsort *,struct tsort *));
int facetfacetcompare ARGS((struct tsort *,struct tsort *));
int edgedegencompare ARGS((struct tsort *,struct tsort *));
int degenfacetcompare ARGS((struct tsort *,struct tsort *));
int degendegencompare ARGS((struct tsort *,struct tsort *));
int plane_test ARGS((struct tsort *,struct tsort *));

static int ttcompare(t1,t2)  /* depth comparison for sort */
struct tsort **t1,**t2;
{
  /* First, compare back z */
  if ( t1[0]->mins[2] > t2[0]->mins[2] ) return 1;
  if ( t1[0]->mins[2] < t2[0]->mins[2] ) return -1;
  /* Break ties with front z */
  if ( t1[0]->maxs[2] > t2[0]->maxs[2] ) return 1;
  if ( t1[0]->maxs[2] < t2[0]->maxs[2] ) return -1;
  /* Finally, break ties with id to get consistent ordering,
     independent of how many elements are being sorted */
  if ( t1[0]->f_id > t2[0]->f_id ) return 1;
  if ( t1[0]->f_id < t2[0]->f_id ) return -1;
  return 0;

}

/*****************************************************************
*
*  Function: find_bbox()
*
*  Purpose: find 3D bounding box for facet.
*
*/

void find_bbox(t)
struct tsort *t;
{ int n;

  for ( n = 0 ; n < SDIM ; n++ )
     { if ( t->x[0][n] < t->x[1][n] )
          { if  ( t->x[2][n] < t->x[0][n] )
                { t->maxs[n] = t->x[1][n];
                  t->mins[n] = t->x[2][n];
                }
             else if ( t->x[1][n] < t->x[2][n] )
                { t->maxs[n] = t->x[2][n];
                  t->mins[n] = t->x[0][n];
                }
             else
                { t->maxs[n] = t->x[1][n];
                  t->mins[n] = t->x[0][n];
                }
            }
         else
          { if  ( t->x[2][n] < t->x[1][n] )
                { t->maxs[n] = t->x[0][n];
                  t->mins[n] = t->x[2][n];
                }
             else if ( t->x[0][n] < t->x[2][n] )
                { t->maxs[n] = t->x[2][n];
                  t->mins[n] = t->x[1][n];
                }
             else
                { t->maxs[n] = t->x[0][n];
                  t->mins[n] = t->x[1][n];
                }
          }
     }
}

/* For setting quadtree code and checking if in bounding box. */
/* Coded for 32 bit ints. */
#define OUTOFBOX 0
#define INTHEBOX 1
int setquadcode(t)
struct tsort *t;
{ unsigned int q = 0;  /* the quadcode */
  unsigned int bit = 1;  /* for shifting to quad bit position */
  int n;
  REAL midx = 0.0, midy = 0.0, delta = 0.75;

  if ( t->maxs[0] < -1.5 || t->mins[0] > 1.5 || t->maxs[1] < -1.5 ||
          t->mins[1] > 1.5 ) return OUTOFBOX;

  for ( n = 0  ; n < 8 ; n++, delta /= 2 )
  {
    if ( t->maxs[0] <= midx ) 
    { q |= bit; midx -= delta; }
    else if ( t->mins[0] >= midx )
    { q |= bit<<1; midx += delta; }
    else break;
    bit <<= 2;
    if ( t->maxs[1] <= midy ) 
    { q |= bit; midy -= delta; }
    else if ( t->mins[1] >= midy )
    { q |= bit<<1; midy += delta; }
    else break;
    bit <<= 2;
  }

  t->quadcode = q;
  return INTHEBOX;
}

void painter_start()
{
  long allocsize;

  /* allocate space for depth sort list */
  if ( web.representation == STRING )
    {  maxcount = web.skel[EDGE].count + 5;
        if ( web.torus_flag ) maxcount *= 2;
    }
  else
    { if ( web.torus_flag )
        if ( torus_display_mode == TORUS_CLIPPED_MODE )
          maxcount = 5*web.skel[FACET].count+ bare_edge_count + 5;
        else maxcount = 2*web.skel[FACET].count+ bare_edge_count + 5;
     else
        maxcount = web.skel[FACET].count + bare_edge_count + 5;
    }
  if ( transforms_flag ) maxcount *= (1 + transform_count);
  if ( web.dimension > 2 )
     maxcount *= web.dimension+1; /* each simplex face becomes facet */
  allocsize = (long)maxcount*sizeof(struct tsort);
  if ( allocsize >= MAXALLOC  )
     maxcount = MAXALLOC/sizeof(struct tsort);
  trilist = (struct tsort *)temp_calloc(maxcount,sizeof(struct tsort));
  count = 0;
}


void painter_edge(gdata,e_id)
struct graphdata *gdata;
edge_id e_id; 
{
  struct tsort *t;
  int i,j,n;
  REAL a[MAXCOORD+1],b[MAXCOORD+1];
  REAL dx,dy,dz,mag,len,width;

  if ( gdata->color == CLEAR ) return;
  if ( count >= maxcount-2 )
    { trilist = (struct tsort *)temp_realloc((char*)trilist,
         (maxcount+200)*sizeof(struct tsort),maxcount*sizeof(struct tsort));
      maxcount += 200;
    }

  t = trilist + count;
  for ( j =  SDIM ; j < HOMDIM-1 ; j++ ) a[j] = 0.0; /* filler */
  for ( i = 0 ; i < 2 ; i++ )
     {
        for ( j = 0 ; (j < SDIM) && (j < HOMDIM-1) ; j++ ) 
            a[j] = gdata[i].x[j];
        a[HOMDIM-1] = 1.0;
        matvec_mul(view,a,b,HOMDIM,HOMDIM);  /* transform */
        if ( SDIM <= 2 )
          for ( j = 0 ; j < 3 ; j++ ) t->x[i][j] = (float)b[j];
        else  for ( j = 0 ; j < 3 ; j++ ) t->x[i][j] = (float)b[(j+1)%3];
        t->x[i][2] += (float).0001;  /* bias edges in front of facets */
     }

  /* width of edge, in descending order of thickness */
  if ( gdata->etype & BARE_EDGE ) width = edgewidths[0];
  else if ( gdata->etype & FIXED_EDGE ) width = edgewidths[1];
  else if ( gdata->etype & CONSTRAINT_EDGE ) width = edgewidths[2];
  else if ( gdata->etype & BOUNDARY_EDGE ) width = edgewidths[3];
  else if ( gdata->etype & SINGLE_EDGE ) width = edgewidths[4];
  else if ( gdata->etype & TRIPLE_EDGE ) width = edgewidths[5];
  else width = edgewidths[6]; /* regular grid interior edge */
  t->width = (float)width;

  t->f_id = e_id;
  t->color = t->ecolor[0] =  gdata->color;
  t->etype[0] = gdata->etype;
  t->flag = EDGE;
  /* find extents */
  for ( n = 0 ; n < SDIM ; n++ )
     { if ( t->x[0][n] < t->x[1][n] )
          { t->maxs[n] = t->x[1][n];
             t->mins[n] = t->x[0][n];
            }
         else
          { t->maxs[n] = t->x[0][n];
             t->mins[n] = t->x[1][n];
          }
     }
  /* adjust extents for thickness */
  dx = fabs(t->x[1][0] - t->x[0][0]);
  dy = fabs(t->x[1][1] - t->x[0][1]);
  len = sqrt(dx*dx+dy*dy);
  if ( len > 0.0 )
  { t->maxs[0] += (float)(dy/len*width/2);
    t->mins[0] -= (float)(dy/len*width/2);
    t->maxs[1] += (float)(dx/len*width/2);
    t->mins[1] -= (float)(dx/len*width/2);
  }

  /* normal vector, closest to z axis */
  dx = t->x[1][0] - t->x[0][0];
  dy = t->x[1][1] - t->x[0][1];
  dz = t->x[1][2] - t->x[0][2];
  t->normal[0] = (float)(dx*dz);
  t->normal[1] = (float)(dy*dz);
  t->normal[2] = -(float)(dx*dx+dy*dy);
  mag = sqrt(dotf(t->normal,t->normal,3));
  if ( mag != 0.0 )
      for ( i = 0 ; i < 3; i++ ) t->normal[i] /= (float)mag;

  if ( setquadcode(t) == OUTOFBOX ) return;
  count++;
}


void painter_facet(gdata,f_id)
struct graphdata *gdata;
facet_id f_id;
{
  int i,j;
  REAL a[MAXCOORD+1],b[FACET_VERTS][MAXCOORD+1];
  struct tsort *t;
  REAL normal[MAXCOORD],mag;

  if ( gdata[0].color == UNSHOWN )
  { /* just do edges */
    struct graphdata ggdata[2];
    for ( i = 0 ; i < FACET_EDGES ; i++ )
    { if ( gdata[i].etype == INVISIBLE_EDGE ) continue;
      ggdata[0] = gdata[i]; 
      ggdata[0].color = gdata[i].ecolor;
      ggdata[1] = gdata[i==2 ? 0 : i+1];
      painter_edge(ggdata,NULLID);
    }
    return;
  }

  if ( count >= maxcount )
    { trilist = (struct tsort *)temp_realloc((char*)trilist,
         (maxcount+200)*sizeof(struct tsort),maxcount*sizeof(struct tsort));
      maxcount += 200;
    }
  t = trilist + count;
  t->flag = FACET;
  t->f_id = f_id;
  t->color = gdata[0].backcolor;  /* not sure why, but works */
  for ( i = 0 ; i < FACET_EDGES ; i++ ) 
     { t->ecolor[i] = gdata[i].ecolor;
        t->etype[i] = gdata[i].etype;
        t->v_id[i] = gdata[i].v_id;
     }

  /* accumulate list of triangles to display */

  for ( j = SDIM ; j < HOMDIM-1 ; j++ ) a[j] = 0.0;
  for ( i = 0 ; i < FACET_VERTS ; i++ )
     {
        for ( j = 0 ; (j < SDIM) && (j < HOMDIM-1) ; j++ ) 
          a[j] = gdata[i].x[j];
        a[HOMDIM-1] = 1.0;
        matvec_mul(view,a,b[i],HOMDIM,HOMDIM);  /* transform */
        t->x[i][0] = (float)b[i][1];
        t->x[i][1] = (float)b[i][2];
        t->x[i][2] = (float)b[i][0];
     } 
  vnormal(b[0],b[1],b[2],normal); 
  mag = sqrt(SDIM_dot(normal,normal));
  if ( mag > 0.0 )
  { t->normal[0] = (float)(normal[1]/mag); 
     t->normal[1] = (float)(normal[2]/mag); 
     t->normal[2] = (float)(normal[0]/mag);
     if ( fabs(t->normal[2]) < 1e-6 ) t->normal[2] = 0.0; 
  } else
  { t->normal[0] = 0.0f; 
     t->normal[1] = 0.0f; 
     t->normal[2] = 1.0f;
  }
  if ( t->normal[2] > (float)0.0 ) /* frontward normal */
  { int c;
     vertex_id tv;
     for ( i = 0 ; i < SDIM ; i++ )
        { float temp = (float)t->x[1][i];
          t->x[1][i] = t->x[2][i];
          t->x[2][i] = temp;
          t->normal[i] = -t->normal[i];
        }
        c = t->ecolor[0];
        t->ecolor[0] = t->ecolor[2];
        t->ecolor[2] = c;
        c = t->etype[0];
        t->etype[0] = t->etype[2];
        t->etype[2] = c;
        t->color = gdata[0].color;
        tv = t->v_id[1];
        t->v_id[1] = t->v_id[2];
        t->v_id[2] = tv;
        t->flag |= FLIPPED_FACET;
  }
  else
  { 
     if ( backcull_flag && (gdata[0].color == gdata[0].backcolor) ) return;
  }

  /* find extents */
  find_bbox(t);
  if ( setquadcode(t) == OUTOFBOX ) return;

  count++;
}
 /* stats for analyzing performance */
 int in_back_calls;
 int box_overlaps;
 int facetfacet;
 int facetedge;
 int edgeedge;
 int crossings;
 int done;
 int loopbailouts;

 int tcursor; /* cursor position in topological sort */
 struct tsort **tlist;
/**************************************************************************
*
* function: topsortnode()
*
* purpose: Handle one node in topological sort of facets, displaying if
*          all sons done and recursing to parents.
*/
void topsortnode(k)
int k;
{ struct tsort *tk = tlist[k];
  struct tlink *link;
       { done++;
         if ( (tk->flag & 0xF) == FACET )
            (*display_facet)(tk);
         else (*display_edge)(tk);
         tk->flag = 0; 
         for ( link = tk->linkhead ; link != NULL ; link = link->next )
         { tlist[link->node]->sons--;
           if ( (link->node < tcursor) && (tlist[link->node]->sons == 0) )
               topsortnode(link->node);         
         }
       }
}
int loopnumber;
int loopfind(k)
#define INPROGRESS 1
#define VISITED    2
int k;  /* index in tlist */
{ struct tsort *tk = tlist[k];
  struct tlink *link;
  int ret;
  char *cmp;

  tk->from = INPROGRESS; /* to prevent 0 */
  for ( link = tk->linkhead ; link != NULL ; link = link->next )
  { struct tsort *t = tlist[link->node];
    if ( t->from == VISITED ) continue;
    if ( t->from == INPROGRESS )
    { /* found loop */
      if ( id_type(t->f_id) == EDGE )
      { printf("Loop edge  %5d",ordinal(t->f_id)+1); 
        set_edge_color(t->f_id,loopnumber);
      }
      else
      { printf("Loop facet %5d",ordinal(t->f_id)+1); 
      }

      if ( k < link->node )
      {
      switch(in_back(tk,t))
      { case DISJOINT: cmp="DISJOINT";break;
        case ASPLITTINGB: cmp="ASPITTINGB";break;
        case BSPLITTINGA: cmp="BSPITTINGA";break;
        case FIRST_BACK: cmp="FIRST_BACK";break;
        case SECOND_BACK: cmp="SECOND_BACK";break;
        case COPLANAR: cmp="COPLANAR";break;
      }
      printf ("      %d %d %s\n",ordinal(tk->f_id)+1,ordinal(t->f_id)+1,cmp);
      }
      else
      {
      switch(in_back(t,tk))
      { case DISJOINT: cmp="DISJOINT";break;
        case ASPLITTINGB: cmp="ASPITTINGB";break;
        case BSPLITTINGA: cmp="BSPITTINGA";break;
        case FIRST_BACK: cmp="FIRST_BACK";break;
        case SECOND_BACK: cmp="SECOND_BACK";break;
        case COPLANAR: cmp="COPLANAR";break;
      }
      printf ("      %d %d %s\n",ordinal(t->f_id)+1,ordinal(tk->f_id)+1,cmp);
      }

      return 1;
    }
    ret = loopfind(link->node);
    if ( ret )
    { 
      if ( id_type(t->f_id) == EDGE )
      { printf("Loop edge  %5d",ordinal(t->f_id)+1); 
      }
      else
      { printf("Loop facet %5d",ordinal(t->f_id)+1); 
      }
      switch(in_back(tk,t))
      { case DISJOINT: cmp="DISJOINT";break;
        case ASPLITTINGB: cmp="ASPITTINGB";break;
        case BSPLITTINGA: cmp="BSPITTINGA";break;
        case FIRST_BACK: cmp="FIRST_BACK";break;
        case SECOND_BACK: cmp="SECOND_BACK";break;
        case COPLANAR: cmp="COPLANAR";break;
      }
      printf ("      %d %d %s\n",ordinal(tk->f_id)+1,ordinal(t->f_id)+1,cmp);
      return ret;
    }
  }
  tk->from = VISITED;
  return 0; /* loop not found */
}
/**************************************************************************
*
* function: painter_end() NEW
*
* purpose: sort and display facets and edges from trilist.
*   Uses topological sort
*/

void new_painter_end()
{
  int j,k;
 

  struct tsort **ll,*tt;
#define TEXTRA 100
  struct tsort textra[TEXTRA]; /* for triangle fragments */
  int linkmax, linkcount;
  struct tlink *linkbase;

done = 0;

  in_back_calls = box_overlaps = facetfacet = facetedge = edgeedge = crossings = 0;
  loopbailouts = 0;
  tlist=NULL;
  if ( count > maxcount ) count = maxcount;    /* in case there was excess */

  /* find bounding box */
  if ( need_bounding_box )
    { struct tsort *t;
      bbox_minx = bbox_miny = 1e20;
      bbox_maxx = bbox_maxy = -1e20;
      for ( k = 0, t = trilist ; k < count ; k++,t++ )
        { if ( t->mins[0] < bbox_minx ) bbox_minx = (REAL)t->mins[0];
          if ( t->mins[1] < bbox_miny ) bbox_miny = (REAL)t->mins[1];
          if ( t->maxs[0] > bbox_maxx ) bbox_maxx = (REAL)t->maxs[0];
          if ( t->maxs[1] > bbox_maxy ) bbox_maxy = (REAL)t->maxs[1];
        }
    }

  (*init_graphics)();

  if ( SDIM == 2 )  /* don't bother with depth */
     { for ( k = 0 ; k < count ; k++ )
          if ( (trilist[k].flag & 0xF) == FACET )
              (*display_facet)(trilist + k);
          else (*display_edge)(trilist + k);
        goto end_exit;
     } 

  /* now sort on min z, moving pointers instead of structures */
  /* leaving room at front of list for extra fragments */
     tlist = (struct tsort **)temp_calloc(count+TEXTRA,sizeof(struct tsort *));
     for ( k = 0, ll=tlist+TEXTRA, tt=trilist ; k < count ; k++ ) *(ll++) = tt++;
     qsort((char *)(tlist+TEXTRA),count,sizeof(struct tsort *),FCAST ttcompare); 
     for ( k = 0 ; k < TEXTRA ; k++ ) 
        { tlist[k] = textra+k; tlist[k]->flag = 0; }

  /* Get obscuration relations for topological sort */

  linkmax = 10*count;
  linkbase = (struct tlink *)temp_calloc(linkmax,sizeof(struct tlink));
  linkcount = 0;
  for ( k = TEXTRA ; k < count+TEXTRA ; k++ )
     { struct tsort *tk = tlist[k];
        struct tsort *tj;


        if ( breakflag ) break;
        if ( !tk->flag ) { continue; }

        /* k is current back facet */
        for ( j = k+1 ; j < count+TEXTRA ; j++ )
          { int retval;
             tj = tlist[j];
             if ( breakflag ) break;
             if ( !tj->flag ) continue;
             if ( tk->maxs[2] < tj->mins[2] )
             { 
             break ;
             }

             retval = in_back(tk,tj);
             if ( retval & (COPLANAR|DISJOINT) ) continue;

             if ( !(retval & (FIRST_BACK|SECOND_BACK)) ) 
             {
               crossings++;
               retval = SECOND_BACK; 
             }

             /* add link */
             if ( retval & FIRST_BACK )
             { struct tlink *newlink;
                           tj->sons++;
               newlink = linkbase + linkcount++;
               newlink->next = tk->linkhead;
               newlink->node = j;
               tk->linkhead = newlink;
             }
             else
             { struct tlink *newlink;
                           tk->sons++;
               newlink = linkbase + linkcount++;
               newlink->next = tj->linkhead;
               newlink->node = k;
               tj->linkhead = newlink;
             }

           }
    }
    
         /* Topological sort and display */
    for ( k = TEXTRA ; k < count+TEXTRA ; k++ )
     { tcursor = k;
  
       if ( tlist[k]->sons == 0 )
         topsortnode(tcursor);
     }

     loopnumber = 0;
     while ( done < count )
     {  int flag;
 
        loopbailouts++;
        for ( k = TEXTRA,flag=1 ; k < count+TEXTRA ; k++ )                
        { tcursor = k;
          if ( flag && tlist[k]->flag ) 
            { 
              /* find loop */
              tlist[k]->from = -1;  /* start */
              loopnumber++;
              loopfind(k);  /* start recursion */

              tlist[k]->sons = 0;  flag = 0; 
            }
          if ( (tlist[k]->sons == 0) && (tlist[k]->flag) )
             topsortnode(tcursor);
        }
     }
 
end_exit:

  if ( verbose_flag )
  {
         printf("in_back_calls: %d\n",in_back_calls);
         printf("box_overlaps:  %d\n",box_overlaps);
         printf("facetfacet:    %d\n",facetfacet);
         printf("facetedge:     %d\n",facetedge);
         printf("edgeedge:      %d\n",edgeedge);
         printf("crossings:     %d\n",crossings);
         printf("loop bailouts: %d\n",loopbailouts);
  }

  if ( tlist ) temp_free((char *)tlist);
  temp_free((char *)trilist); trilist = NULL;
  (*finish_graphics)();
}

/**************************************************************************
*
* function: old_painter_end()
*
* purpose: sort and display facets and edges from trilist.
*/

void painter_end()
{
  int j,k;
  int maxmark;
  int loopcount; /* for emergency loop bailout */
  struct tsort **tlist=NULL,**ll,*tt;
#define TEXTRA 100
  struct tsort textra[TEXTRA]; /* for triangle fragments */

  in_back_calls = box_overlaps = facetfacet = facetedge = edgeedge = crossings = 0;
  loopbailouts = 0;

  if ( count > maxcount ) count = maxcount;    /* in case there was excess */

  /* find bounding box */
  if ( need_bounding_box )
    { struct tsort *t;
      bbox_minx = bbox_miny = 1e20;
      bbox_maxx = bbox_maxy = -1e20;
      for ( k = 0, t = trilist ; k < count ; k++,t++ )
        { if ( t->mins[0] < bbox_minx ) bbox_minx = (REAL)t->mins[0];
          if ( t->mins[1] < bbox_miny ) bbox_miny = (REAL)t->mins[1];
          if ( t->maxs[0] > bbox_maxx ) bbox_maxx = (REAL)t->maxs[0];
          if ( t->maxs[1] > bbox_maxy ) bbox_maxy = (REAL)t->maxs[1];
        }
    }

  (*init_graphics)();

  if ( SDIM == 2 )  /* don't bother with depth */
     { for ( k = 0 ; k < count ; k++ )
          if ( (trilist[k].flag & 0xF) == FACET )
              (*display_facet)(trilist + k);
          else (*display_edge)(trilist + k);
        goto end_exit;
     } 

  /* now sort on min z, moving pointers instead of structures */
  /* leaving room at front of list for extra fragments */
     tlist = (struct tsort **)temp_calloc(count+TEXTRA,sizeof(struct tsort *));
     for ( k = 0, ll=tlist+TEXTRA, tt=trilist ; k < count ; k++ ) *(ll++) = tt++;
     qsort((char *)(tlist+TEXTRA),count,sizeof(struct tsort *),FCAST ttcompare); 
     for ( k = 0 ; k < TEXTRA ; k++ ) 
        { tlist[k] = textra+k; tlist[k]->flag = 0; }

  /* display */
  maxmark = 0;
  loopcount = 0;
  for ( k = TEXTRA ; k < count+TEXTRA ;  )
     { struct tsort *tk = tlist[k];
        struct tsort *tj;

        if ( breakflag ) break;
        if ( !tk->flag ) { k++; continue; }

        /* k is current back facet */
        for ( j = k+1 ; j < count+TEXTRA ; j++ )
          { int retval;
             tj = tlist[j];
             if ( breakflag ) break;
             if ( !tj->flag ) continue;
             if ( (j > maxmark) && (tk->maxs[2] <= tj->mins[2]) ) 
                break; 

             retval = in_back(tk,tj);  
             if ( retval & (FIRST_BACK|COPLANAR|DISJOINT) ) continue;

             /* test for possible looping, and if found, split tk */
             if ( (tj->flag & WAS_BACKMOST) )
             { int ret;

                crossings++;
                 if ( ++loopcount > 100 ) 
                 { loopbailouts++; break; }

                /* need to split */
               
                if ( k < 2 ) break; /* not enough room */
                if( retval & BSPLITTINGA )
                {
                  ret = newell_split(tk,tj,tlist[k-1],tlist[k-2]);
                  if ( ret )
                  {
                    k -= ret;
                    goto repeat_tests;  /* might not have split */
                  }
                }
                else if ( retval & ASPLITTINGB )
                {
                  /* try splitting the other way */
                  tlist[k] = tj; tlist[j] = tk;
                  ret = newell_split(tj,tk,tlist[k-1],tlist[k-2]);
                  if ( ret )
                  { k -= ret;
                    goto repeat_tests;
                  }
                }
                
             }

             tk->flag |= WAS_BACKMOST;
             tlist[k] = tj;  /* swap tj and tk */
             tlist[j] = tk;
             if ( maxmark < j ) maxmark = j;
             goto repeat_tests;
          }

        if ( (tk->flag & 0xF) == FACET )
            (*display_facet)(tk);
        else (*display_edge)(tk);
        loopcount = 0;
        tk->flag = 0; 
repeat_tests:
        continue;
     }
 
end_exit:

    if ( verbose_flag )
    {
         printf("in_back_calls: %d\n",in_back_calls);
         printf("box_overlaps:  %d\n",box_overlaps);
         printf("facetfacet:    %d\n",facetfacet);
         printf("facetedge:     %d\n",facetedge);
         printf("edgeedge:      %d\n",edgeedge);
         printf("crossings:     %d\n",crossings);
         printf("loop bailouts: %d\n",loopbailouts);
    }

  if ( tlist ) temp_free((char *)tlist);
  temp_free((char *)trilist); trilist = NULL;
  (*finish_graphics)();
}

/**************************************************************************
*
* Function: edgefacetdepthcompare()
*
* Purpose: depth sort compare of an edge and a facet.
*
* Return value: FIRST_BACK, SECOND_BACK, BSPLITTINGA, or DISJOINT.
*/
int edgefacetdepthcompare(ta,tb)
struct tsort *ta; /* the edge */
struct tsort *tb; /* the facet */
{ int k;
  REAL s1[3],s2[3],e1[3],e2[3],a,b,c,gamma,lambda,z;
  REAL d1[3],d2[3],maxlambda,minlambda;
;
 
       for ( k = 0 ; k < 3; k++ )
       { s1[k] = tb->x[1][k]-tb->x[0][k];
         s2[k] = tb->x[2][k]-tb->x[0][k];
         e1[k] = ta->x[0][k]-tb->x[0][k];
         e2[k] = ta->x[1][k]-tb->x[0][k];
       }
       /* Find max, min lambda of seg in triagle projection. */
       d1[0] = s1[0]*e1[1] - s1[1]*e1[0];
       d1[2] = e1[0]*s2[1] - e1[1]*s2[0];
       d1[1] = s1[0]*s2[1] - s1[1]*s2[0] - d1[0] - d1[2];
       d2[0] = s1[0]*e2[1] - s1[1]*e2[0];
       d2[2] = e2[0]*s2[1] - e2[1]*s2[0];
       d2[1] = s1[0]*s2[1] - s1[1]*s2[0] - d2[0] - d2[2];
       maxlambda = 1; minlambda = 0;
       for ( k = 0 ; k < 3 ; k++ ) 
       { REAL diff = d2[k] - d1[k];
         if ( diff > 0.0 )
         { lambda = d1[k]/diff;
           if ( lambda > minlambda ) minlambda = lambda;
         }
         else if ( diff < 0.0 )
         { lambda = d1[k]/diff;
           if ( lambda < maxlambda ) maxlambda = lambda;
         }
         else if ( d1[k] < 0.0 ) return DISJOINT;
       }
       if ( minlambda > maxlambda ) return DISJOINT;
       if ( minlambda < maxlambda )
       { /* have actual finite segment */
         /* find barycentric points of crossing */
         a = triple_prod(s1,s2,e1); 
         b = triple_prod(s1,s2,e2); 
         lambda = a/(a-b);
         if ( lambda <= minlambda || lambda >= maxlambda )
         { /* whole segment on same side */
           if ( (2-minlambda-maxlambda)*a + (minlambda+maxlambda)*b >= 0.0 )
             return SECOND_BACK;
           else return FIRST_BACK;
         }
         else return BSPLITTINGA;
       }
       /* Now down to degenerate case of point crossing */
       { lambda = minlambda;
         z = lambda*e1[2] + (1-lambda)*e2[2];
         /* see if crosses interior */
         a = e1[0]*e2[1] - e1[1]*e2[0];
         b = (e1[0]-s1[0])*(e2[1]-s1[1]) - (e1[1]-s1[1])*(e2[0]-s1[0]);
         c = (e1[0]-s2[0])*(e2[1]-s2[1]) - (e1[1]-s2[1])*(e2[0]-s2[0]);
         if ( (a < 0.0 || b < 0.0 || c < 0.0) && (a>0.0 || b>0.0 || c>0.0) )
         { /* have crossing, so see if any facet edges in front */
           if ( a-b != 0.0 )
           { gamma = a/(a-b);
             if ( gamma <= 1.0 && gamma*s1[2] > z ) return FIRST_BACK;
           }
           if ( a-c != 0.0 )
           { gamma = a/(a-c);
             if ( gamma <= 1.0 && gamma*s2[2] > z ) return FIRST_BACK;
           }
           if ( b-c != 0.0 )
           { gamma = b/(b-c);
             if ( gamma <= 1.0 && gamma*s1[2]+(1-gamma)*s2[2] > z ) 
               return FIRST_BACK;
           }
           return SECOND_BACK;
         }
         else return DISJOINT;
       }
       return DISJOINT;
}

/*********************************************************************
*
* function: in_back()
*
* purpose: see if one facet or edge obscures another.
*
* returns DISJOINT, FIRST_BACK, SECOND_BACK, ASPLITTINGB, BSPLITTINGA, 
*   or COPLANAR (possibly bitwise OR)
*/
int in_back(ta,tb)
struct tsort *ta,*tb;
{
  int n;
  
  int retval;
   
  if ( verbose_flag )
  {
  in_back_calls++;
  if ( (ta->flag & 0xF) == FACET  )
  {   if ( (tb->flag & 0xF) == FACET  )          facetfacet++;
      else facetedge++;
  }
   else  {   if ( (tb->flag & 0xF) == FACET  )          facetedge++;
      else edgeedge++;
  }
  }

  /* quick test with quadcodes */
  if ( ((ta->quadcode & tb->quadcode) != ta->quadcode) &&
  ((ta->quadcode & tb->quadcode) != tb->quadcode) )
             return DISJOINT;

  /* test x and y extent overlap */
  for ( n = 0 ; n < 2 ; n++ )
     if ( (tb->maxs[n] <= ta->mins[n]) || (tb->mins[n] >= ta->maxs[n]) )
          return DISJOINT;  
  if ( ta->maxs[2] <= tb->mins[2] ) return FIRST_BACK;

  box_overlaps++;   /* for verbose stats */

  retval = plane_test(ta,tb);
  if ( retval & (FIRST_BACK|COPLANAR|DISJOINT) ) return retval;

  /* now the nitty gritty check to see if they overlap */
#ifdef ZZZ
  if ( (ta->flag & 0xF) == FACET  )
  { if ( (tb->flag & 0xF) == FACET  ) 
      return facetfacetcompare(ta,tb);
     else 
     {    retval = edgefacetcompare(tb,ta);
        if ( retval & (FIRST_BACK|SECOND_BACK) )
           return retval ^ (FIRST_BACK|SECOND_BACK);
        else return retval;
    }
  }
  else
  { if ( (tb->flag & 0xF) == FACET  ) 
      return edgefacetcompare(ta,tb);
    else
    { retval = edgeedgecompare(tb,ta);
      if ( retval & (FIRST_BACK|SECOND_BACK) )
           return retval ^ (FIRST_BACK|SECOND_BACK);
      else return retval;
    }

  }
#endif

  if ( separating_line(ta,tb) == DISJOINT ) return DISJOINT;  

  return retval;
}  


/*************************************************************************
*
* function: newell_split()
*
* purpose: split one triangle by plane of another.
*
* return: number of new elements generated.
*/

int newell_split(ta,tb,tc,td)
struct tsort *ta;  /* splittee and fragment return */
struct tsort *tb;  /* splitter */
struct tsort *tc;  /* fragment return */
struct tsort *td;  /* fragment return */
{ 
  int i;
  REAL d0,d1,d2,db;
  int retval;

  if ( (tb->flag & 0xF) == EDGE )
  { struct tsort *t;
     if ( (ta->flag & 0xF) == EDGE ) return 0;
     t = ta; ta = tb; tb = t; /* swap so edge first */
  }
  if ( (ta->flag & 0xF) == EDGE ) /* tb assumed to be facet */
  {
     db = dotf(tb->normal,tb->x[0],SDIM);
     d0 = dotf(tb->normal,ta->x[0],SDIM); 
     d1 = dotf(tb->normal,ta->x[1],SDIM); 

     /* fill in fragment info */
     *tc = *ta;
     for ( i = 0 ; i < SDIM ; i++ )
        ta->x[1][i] = tc->x[0][i] =
          (float)(((db-d0)*ta->x[1][i] + (d1-db)*ta->x[0][i])/(d1-d0));
     for ( i = 0 ; i < SDIM ; i++ )
     { ta->mins[i] = (ta->x[0][i]<ta->x[1][i]) ? ta->x[0][i] : ta->x[1][i];
        tc->mins[i] = (tc->x[0][i]<tc->x[1][i]) ? tc->x[0][i] : tc->x[1][i];
        ta->maxs[i] = (ta->x[0][i]>ta->x[1][i]) ? ta->x[0][i] : ta->x[1][i];
        tc->maxs[i] = (tc->x[0][i]>tc->x[1][i]) ? tc->x[0][i] : tc->x[1][i];
     }
     return 1;
  }

  /* figure out which vertices of ta on same side, and get as 0,1 */
  db = dotf(tb->normal,tb->x[0],SDIM);
  d0 = dotf(tb->normal,ta->x[0],SDIM); 
  d1 = dotf(tb->normal,ta->x[1],SDIM); 
  d2 = dotf(tb->normal,ta->x[2],SDIM); 

if ( (d0<db) && (d1<db) && (d2<db) ) /* should never happen */
{ struct tsort t;  
  t = *ta; *ta = *tb; *tb = t; return 0; }
if ( (d0>db) && (d1>db) && (d2>db) )
{ return 0; }

  retval = 0;
  if ( db == d0 ) /* split thru vertex 0 of ta */
  { *tc = *ta;

     /* fill in fragment info */
     for ( i = 0 ; i < SDIM ; i++ )
     { ta->x[2][i] = tc->x[1][i] = 
          (float)(((db-d1)*ta->x[2][i] + (d2-db)*ta->x[1][i])/(d2-d1));
     }
     /* internal edges invisible */
     ta->etype[2] = tc->etype[0] = SPLITTING_EDGE;
     retval = 1;
  }
  if ( db == d1 ) /* split thru vertex 1 of ta */
  { *tc = *ta;

     /* fill in fragment info */
     for ( i = 0 ; i < SDIM ; i++ )
     { ta->x[2][i] = tc->x[0][i] = 
          (float)(((db-d0)*ta->x[2][i] + (d2-db)*ta->x[0][i])/(d2-d0));
     }
     /* internal edges invisible */
     ta->etype[1] = tc->etype[0] = SPLITTING_EDGE;
     retval = 1;
  }
  if ( db == d2 ) /* split thru vertex 2 of ta */
  { *tc = *ta;

     /* fill in fragment info */
     for ( i = 0 ; i < SDIM ; i++ )
     { ta->x[1][i] = tc->x[0][i] = 
          (float)(((db-d0)*ta->x[1][i] + (d1-db)*ta->x[0][i])/(d1-d0));
     }
     /* internal edges invisible */
     ta->etype[1] = tc->etype[2] = SPLITTING_EDGE;
     retval = 1;
  }

  if ( retval == 1 )
  {
     /* set mins and maxs */
     for ( i = 0 ; i < SDIM ; i++ )
     { int j;
        ta->mins[i] = tc->mins[i] = (float)1e30;
        ta->maxs[i] = tc->maxs[i] = (float)(-1e30);
        for ( j = 0 ; j < FACET_VERTS ; j++ )
        { if ( ta->x[j][i] < ta->mins[i] ) ta->mins[i] = ta->x[j][i];
          if ( tc->x[j][i] < tc->mins[i] ) tc->mins[i] = tc->x[j][i];
          if ( ta->x[j][i] > ta->maxs[i] ) ta->maxs[i] = ta->x[j][i];
          if ( tc->x[j][i] > tc->maxs[i] ) tc->maxs[i] = tc->x[j][i];
        }
     }
     return 1;
  }

 
  if ( (d0-db)*(d2-db) > 0.0 )
  { int c = ta->ecolor[1]; 
     short e = ta->etype[1];
     REAL d = d1;
     for ( i = 0 ; i < SDIM ; i++ )
     { float temp = ta->x[1][i]; 
        ta->x[1][i] = ta->x[0][i]; ta->x[0][i] = ta->x[2][i];
        ta->x[2][i] = temp;
     }
     ta->ecolor[1] = ta->ecolor[0]; ta->ecolor[0] = ta->ecolor[2];
     ta->ecolor[2] = c;
     ta->etype[1] = ta->etype[0]; ta->etype[0] = ta->etype[2];
     ta->etype[2] = e;
     d1 = d0; d0 = d2; d2 = d;
  }
  else if ( (d1-db)*(d2-db) > 0.0 )
  { int c = ta->ecolor[1]; 
     short e = ta->etype[1];
     REAL d = d1;
     for ( i = 0 ; i < SDIM ; i++ )
     { float temp = ta->x[1][i]; 
        ta->x[1][i] = ta->x[2][i]; ta->x[2][i] = ta->x[0][i];
        ta->x[0][i] = temp;
     }
     ta->ecolor[1] = ta->ecolor[2]; ta->ecolor[2] = ta->ecolor[0];
     ta->ecolor[0] = c;
     ta->etype[1] = ta->etype[2]; ta->etype[2] = ta->etype[0];
     ta->etype[0] = e;
     d1 = d2; d2 = d0; d0 = d;
  }

  *tc = *ta; *td = *ta; /* copy all info to fragments */

  /* fill in fragment info */
  for ( i = 0 ; i < SDIM ; i++ )
  { ta->x[2][i] = tc->x[0][i] = td->x[0][i] =
        (float)(((db-d0)*td->x[2][i] + (d2-db)*ta->x[0][i])/(d2-d0));
     tc->x[2][i] = td->x[1][i] =
        (float)(((db-d1)*td->x[2][i] + (d2-db)*ta->x[1][i])/(d2-d1));
  }
  /* internal edges invisible */
  ta->etype[1] = tc->etype[0] = tc->etype[2] = td->etype[0] = SPLITTING_EDGE;

  /* set mins and maxs */
  for ( i = 0 ; i < SDIM ; i++ )
  { int j;
     ta->mins[i] = tc->mins[i] = td->mins[i] = (float)1e30;
     ta->maxs[i] = tc->maxs[i] = td->maxs[i] = (float)(-1e30);
     for ( j = 0 ; j < 3 ; j++ )
     { if ( ta->x[j][i] < ta->mins[i] ) ta->mins[i] = ta->x[j][i];
        if ( tc->x[j][i] < tc->mins[i] ) tc->mins[i] = tc->x[j][i];
        if ( td->x[j][i] < td->mins[i] ) td->mins[i] = td->x[j][i];
        if ( ta->x[j][i] > ta->maxs[i] ) ta->maxs[i] = ta->x[j][i];
        if ( tc->x[j][i] > tc->maxs[i] ) tc->maxs[i] = tc->x[j][i];
        if ( td->x[j][i] > td->maxs[i] ) td->maxs[i] = td->x[j][i];
     }
  }

  return 2;
}

/*********************************************************************
*
* function: separating_plane()
*
* purpose: See if two facets have a separating plane in 3D.
*             Meant to be called when known the facets overlap in 2D.
*
* returns  FIRST_BACK, SECOND_BACK, or ASPLITTINGB | BSPLITTINGA
*/
int separating_plane(ta,tb,depth)
struct tsort *ta,*tb;
int depth; /* to limit recursion to depth 2 */
{ int i,j;
  int na,nb; /* vertices on respective elements */
  int nna; /* number of vertex pairs to check on ta */
  float *a[3],*b[3];
  int retval;
  REAL da,db,d;
  int n,k,bnear=0,beq=0,bfar=0,anear=0,afar=0,aeq=0;

  /* see where tb is with respect to ta plane */
  if ( (ta->flag & 0xF) == FACET  )
  { da = dotf(ta->normal,ta->x[0],SDIM);
     n = ((tb->flag & 0xF) == FACET) ? 3 : 2;
     for ( k = 0 ; k < n ; k++ )
     {
        d = dotf(ta->normal,tb->x[k],SDIM); 
        if ( d < da - 0.0001 ) { bnear++; continue; }
        if ( d < da + 0.0001 ) { beq++; continue; }
        bfar++;
     }
     if ( beq == n ) 
                 return ((tb->flag&0xF)==EDGE)?FIRST_BACK:DISJOINT; /* both in same plane */
     if ( bfar == 0 ) return FIRST_BACK;
  }

  /* see where ta is with respect to tb plane */
  if ( (tb->flag & 0xF) == FACET )
  { db = dotf(tb->normal,tb->x[0],SDIM);
     n = ((ta->flag & 0xF) == FACET) ? 3 : 2;
     for ( k = 0 ; k < n ; k++ )
     {
        d = dotf(tb->normal,ta->x[k],SDIM); 
        if ( d < db - 0.0001 ) { anear++; continue; }
        if ( d < db + 0.0001 ) { aeq++; continue; }
        afar++;
     }
     if ( aeq == n ) 
                 return ((ta->flag&0xF)==EDGE)?SECOND_BACK:DISJOINT; /* both in same plane */
     if ( anear == 0 ) return FIRST_BACK;
  }
  
  
  na = (ta->flag & 0xF) == FACET ? 3 : 2;
  nna = (ta->flag & 0xF) == FACET ? 3 : 1;
  nb = (tb->flag & 0xF) == FACET ? 3 : 2;

  for ( i = 0 ; i < nna ; i++ )
  for ( j = 0 ; j < nb ; j++ )
  { REAL c[3],d; /* coefficients for plane */
    REAL da[3],db[3],minarea,s[3],length,dar[3],dbr[3];
    int ii,jj,jjj;

    for ( ii = 0 ; ii < na ; ii++ )
       a[ii] = ta->x[(i+ii)%na];
    for ( jj = 0 ; jj < nb ; jj++ )
       b[jj] = tb->x[(j+jj)%nb];
         for ( ii = 0 ; ii < 3 ; ii++ ) s[ii] = a[1][ii]-a[0][ii];
         length = sqrt(s[0]*s[0]+s[1]*s[1]+s[2]*s[2]);
         minarea = 1e-4*length;
         c[0] = s[1]*(b[0][2]-a[0][2]) - s[2]*(b[0][1]-a[0][1]);
          c[1] = s[2]*(b[0][0]-a[0][0]) - s[0]*(b[0][2]-a[0][2]);
         c[2] = s[0]*(b[0][1]-a[0][1]) - s[1]*(b[0][0]-a[0][0]);

     if ( c[0]*c[0] + c[1]*c[1] + c[2]*c[2] <= minarea*minarea ) 
         continue; /* degenerate */

     d = c[0]*a[0][0] + c[1]*a[0][1] + c[2]*a[0][2];

     for ( ii = 2 ; ii < na ; ii++ )
     { dar[ii] = c[0]*a[ii][0] + c[1]*a[ii][1] +c[2]*a[ii][2] - d;
       da[ii] = ( dar[ii] < -minarea ? -1.0 : ( dar[ii] > minarea ? 1.0 : 0.0));
     }
     for ( jj = 1 ; jj < nb ; jj++ )
     { dbr[jj] = c[0]*b[jj][0] + c[1]*b[jj][1] +c[2]*b[jj][2] - d;
       db[jj] = ( dbr[jj] < -minarea ? -1.0 : ( dbr[jj] > minarea ? 1.0 : 0.0));
     }

     /* test opposite sidedness */
     for ( jj = 1 ; jj < nb ; jj++ )
      for ( jjj = jj+1 ; jjj < nb ; jjj++ )
        if ( db[jj]*db[jjj] < 0.0 ) goto keeptrying;
     for ( ii = 2 ; ii < na ; ii++ )
      for ( jj = 1 ; jj < nb ; jj++ )
        if ( da[ii]*db[jj] > 0.0 ) goto keeptrying;

     /* have separating plane */
     { REAL asum,bsum;
       /* decide which is in front */
       for ( ii = 2, asum = 0.0 ; ii < na ; ii++ ) asum += c[2]*da[ii];
       for ( jj = 1, bsum = 0.0 ; jj < nb ; jj++ ) bsum += c[2]*db[jj];
       if ( asum > 0.0 || bsum < 0.0 ) return SECOND_BACK;
       else   return FIRST_BACK;
     }
keeptrying: ;
  }

  /* might have case of separating plane having two vertices on tb */
  if ( depth == 2 ) 
      return ASPLITTINGB|BSPLITTINGA;
  retval = separating_plane(tb,ta,2);
  if ( retval == FIRST_BACK ) 
      return SECOND_BACK;
  if ( retval == SECOND_BACK ) 
      return FIRST_BACK;
  return retval;

}

/*********************************************************************
*
* function: separating_line()
*
* purpose: See if two elements have a separating line in 2D.
*             Includes small tolerance.
*
* returns  DISJOINT or result of separating_plane()
*/
int separating_line(ta,tb)
struct tsort *ta,*tb;
{ int i;
  int same = 0;
  int na,nb; /* vertices on respective elements */
  int nna,nnb; /* number of lines to try */
  int apos,aneg,bpos,bneg;
  float *a[3],*b[3];
  REAL width; /* thickness of separating line */ 

  /* get edge first, if any */
  if ( ((ta->flag & 0xF) == FACET) && ((tb->flag & 0xF) == EDGE ) )
  { struct tsort *tmp = ta; ta = tb; tb = tmp; }

  /* want to prevent overlap of facet with thick edge; not going
     to worry about edge-edge overlap, since that too weird. */
  if ( ((ta->flag & 0xF) == EDGE) && ((tb->flag & 0xF) == FACET ) )
  {
    width = ta->width/2;  /* actually need half-width */
  }
  else width = -1e-5;  /* allow slight overlap for numerical purposes */

  na = (ta->flag & 0xF) == FACET ? 3 : 2;
  nb = (tb->flag & 0xF) == FACET ? 3 : 2;
  nna = (ta->flag & 0xF) == FACET ? 3 : 1;
  nnb = (tb->flag & 0xF) == FACET ? 3 : 1;

  /* Try using edges of ta */
  for ( i = 0 ; i < nna ; i++ )
  { REAL cx,cy,d; /* coefficients for line */
    REAL dar[3],dbr[3],minarea;
    int ii,jj;

    for ( ii = 0 ; ii < na ; ii++ )
       a[ii] = ta->x[(i+ii)%na];

    for ( jj = 0 ; jj < nb ; jj++ )
       b[jj] = tb->x[jj];

     cx = a[1][1] - a[0][1]; cy = a[0][0] - a[1][0]; 
     d = cx*a[0][0] + cy*a[0][1];

     minarea = width*sqrt(cx*cx + cy*cy);

     if ( fabs(minarea) < 1e-20 ) 
      { same++;    continue; /* same point */ }

     apos = aneg = bpos = bneg = 0;
     for ( ii = 2 ; ii < na ; ii++ )
     { dar[ii] = cx*a[ii][0] + cy*a[ii][1] - d;
       if ( dar[ii] > minarea ) apos++;
       if ( dar[ii] < -minarea ) aneg++;
     }
     for ( jj = 0 ; jj < nb ; jj++ )
     { dbr[jj] = cx*b[jj][0] + cy*b[jj][1] - d;
       if ( dbr[jj] > minarea ) bpos++;
       if ( dbr[jj] < -minarea ) bneg++;
     }

     /* test opposite sidedness */
     if ( apos == (na-2) && bneg == nb ) return DISJOINT;
     if ( aneg == (na-2) && bpos == nb ) return DISJOINT;
  }

  /* Try using edges of tb */
  for ( i = 0 ; i < nnb ; i++ )
  { REAL cx,cy,d; /* coefficients for line */
    REAL dar[3],dbr[3],minarea;
    int ii,jj;

    for ( ii = 0 ; ii < nb ; ii++ )
       a[ii] = tb->x[(i+ii)%nb];

    for ( jj = 0 ; jj < na ; jj++ )
       b[jj] = ta->x[jj];

     cx = a[1][1] - a[0][1]; cy = a[0][0] - a[1][0]; 
     d = cx*a[0][0] + cy*a[0][1];

     width = -1e-5;
     minarea = width*sqrt(cx*cx + cy*cy);

     if ( fabs(minarea) < 1e-20 ) 
      { same++;    continue; /* same point */ }

     apos = aneg = bpos = bneg = 0;
     for ( ii = 2 ; ii < nb ; ii++ )
     { dar[ii] = cx*a[ii][0] + cy*a[ii][1] - d;
       if ( dar[ii] > minarea ) apos++;
       if ( dar[ii] < -minarea ) aneg++;
     }
     for ( jj = 0 ; jj < na ; jj++ )
     { dbr[jj] = cx*b[jj][0] + cy*b[jj][1] - d;
       if ( dbr[jj] > minarea ) bpos++;
       if ( dbr[jj] < -minarea ) bneg++;
     }

     /* test opposite sidedness */
     if ( apos == (nb-2) && bneg == na ) return DISJOINT;
     if ( aneg == (nb-2) && bpos == na ) return DISJOINT;
  }

  return NOTKNOWN;
 }

/***************************************************************************
*
* function: edgeedgecompare()
*
* purpose: Depth compare two edges, including all degenerate cases.
*/
int edgeedgecompare(ta,tb)
struct tsort *ta,*tb; /* the edges */
{ int atype = 0, btype = 0; /* for possibly pointlike */
#define POINTLIKE 37
  REAL det;
  REAL alambda0,alambda1,minalambda,maxalambda;
  REAL blambda0,blambda1,minblambda,maxblambda;
  REAL a,b,az,bz,zdiff1,zdiff2,diffx,diffy,zdiff;

          det = (ta->x[0][0]-ta->x[1][0])*(tb->x[1][1]-tb->x[0][1])
             -  (ta->x[0][1]-ta->x[1][1])*(tb->x[1][0]-tb->x[0][0]);
          if ( det == 0.0 )
          { REAL det1,det2;
            /* parallel, including if either pointlike */
            /* see if collinear */
            det1 = (ta->x[1][0]-ta->x[0][0])*(tb->x[1][1]-ta->x[0][1])
             -  (ta->x[1][1]-ta->x[0][1])*(tb->x[1][0]-ta->x[0][0]);
            det2 = (tb->x[0][0]-ta->x[0][0])*(tb->x[1][1]-ta->x[0][1])
             -  (tb->x[0][1]-ta->x[0][1])*(tb->x[1][0]-ta->x[0][0]);
            if ( det1 != 0.0 || det2 != 0.0 ) return DISJOINT; /* parallel */

            /* now collinear; see if overlap */
            diffx = ta->x[1][0] - ta->x[0][0];
            diffy = ta->x[1][1] - ta->x[0][1];
            if ( fabs(diffx) > fabs(diffy) )
            { alambda0 = (tb->x[0][0] - ta->x[0][0])/diffx;
              alambda1 = (tb->x[1][0] - ta->x[0][0])/diffx;
            }
            else if ( diffy == 0.0 )
            { /* ta is pointlike */
              alambda0 = 0.0; alambda1 = 1.0; atype = POINTLIKE;
            }
            else
            { alambda0 = (tb->x[0][1] - ta->x[0][1])/diffy;
              alambda1 = (tb->x[1][1] - ta->x[0][1])/diffy;
            }
        minalambda = alambda0 < alambda1 ? alambda0 : alambda1;
        if ( minalambda < 0.0 ) minalambda = 0.0;

        maxalambda = alambda0 > alambda1 ? alambda0 : alambda1;
        if ( maxalambda > 1.0 ) maxalambda = 1.0;

            if ( minalambda > 1.0 || maxalambda < 0.0 ) return DISJOINT;

            diffx = tb->x[1][0] - tb->x[0][0];
            diffy = tb->x[1][1] - tb->x[0][1];
            if ( fabs(diffx) > fabs(diffy) )
            { blambda0 = (ta->x[0][0] - tb->x[0][0])/diffx;
              blambda1 = (ta->x[1][0] - tb->x[0][0])/diffx;
            }
            else if ( diffy == 0.0 )
            { /* tb is pointlike */
              blambda0 = 0.0; blambda1 = 1.0;  btype = POINTLIKE;
            }
            else
            { blambda0 = (ta->x[0][1] - tb->x[0][1])/diffy;
              blambda1 = (ta->x[1][1] - tb->x[0][1])/diffy;
            }

        minblambda = blambda0 < blambda1 ? blambda0 : blambda1;
        if ( minblambda < 0.0 ) minblambda = 0.0;

        maxblambda = blambda0 > blambda1 ? blambda0 : blambda1;
        if ( maxblambda > 1.0 ) maxblambda = 1.0;

            if ( minblambda > 1.0 || maxblambda < 0.0 ) return DISJOINT;
            if ( atype == POINTLIKE && btype == POINTLIKE )
            { if ( ta->x[0][0] != tb->x[0][0] ) return DISJOINT;
              if ( ta->x[0][1] != tb->x[0][1] ) return DISJOINT;
              az = (ta->x[0][2]>ta->x[1][2])? ta->x[0][2]:ta->x[1][2];
              bz = (tb->x[0][2]>tb->x[1][2])? tb->x[0][2]:tb->x[1][2];
              if ( az > bz ) return SECOND_BACK;
              if ( az < bz ) return FIRST_BACK;
              return DISJOINT;
            }
            /* compare z values at ends of intersection segment */
            if ( (ta->x[1][0]-ta->x[0][0])*(tb->x[1][0]-tb->x[0][0])
               + (ta->x[1][1]-ta->x[0][1])*(tb->x[1][1]-tb->x[0][1]) < 0.0 )
            { /* antiparallel */
              zdiff1 = (1-minalambda)*ta->x[0][2]+minalambda*ta->x[1][2]
                     - ((1-maxblambda)*tb->x[0][2]+maxblambda*tb->x[1][2]);
              zdiff2 = (1-maxalambda)*ta->x[0][2]+maxalambda*ta->x[1][2]
                     - ((1-minblambda)*tb->x[0][2]+minblambda*tb->x[1][2]);
              if ( zdiff1*zdiff2 < 0.0 ) 
                  return ASPLITTINGB|BSPLITTINGA;
              if ( zdiff1+zdiff2 > 0.0 ) return SECOND_BACK;
              if ( zdiff1+zdiff2 < 0.0 ) return FIRST_BACK;
              return DISJOINT;
            }
            else 
            { /* parallel */
              zdiff1 = (1-minalambda)*ta->x[0][2]+minalambda*ta->x[1][2]
                     - ((1-minblambda)*tb->x[0][2]+minblambda*tb->x[1][2]);
              zdiff2 = (1-maxalambda)*ta->x[0][2]+maxalambda*ta->x[1][2]
                     - ((1-maxblambda)*tb->x[0][2]+maxblambda*tb->x[1][2]);
              if ( zdiff1*zdiff2 < 0.0 ) 
                  return ASPLITTINGB|BSPLITTINGA;
              if ( zdiff1+zdiff2 > 0.0 ) return SECOND_BACK;
              if ( zdiff1+zdiff2 < 0.0 ) return FIRST_BACK;
              return DISJOINT;
            }
          } /* done parallel */

       /* solve for intersection barycentric coordinates */
          a = ((tb->x[1][1]-tb->x[0][1])*(tb->x[1][0]-ta->x[1][0])
             -  (tb->x[1][0]-tb->x[0][0])*(tb->x[1][1]-ta->x[1][1]))/det;
          if ( (a < 0.0) || (a > 1.0) ) return DISJOINT;
          b = ((ta->x[1][1]-ta->x[0][1])*(tb->x[1][0]-ta->x[1][0])
             -  (ta->x[1][0]-ta->x[0][0])*(tb->x[1][1]-ta->x[1][1]))/det;
          if ( (b < 0.0) || (b > 1.0)) return DISJOINT;
          zdiff = a*ta->x[0][2]+(1-a)*ta->x[1][2]
                    - b*tb->x[0][2] + (1-b)*tb->x[1][2];
          if ( zdiff > 0.0 ) return SECOND_BACK;
          if ( zdiff < 0.0 ) return FIRST_BACK;
          return DISJOINT;
}

/***************************************************************************
*
* function: edgefacetcompare()      old
*
* purpose: Depth compare two edge and facet, including all degenerate cases.
*/
int old_edgefacetcompare(ta,tb)
struct tsort *ta; /* the edge */
struct tsort *tb; /* the facet */
{ int k;
  REAL s1[3],s2[3],e1[3],e2[3],a,b,lambda,gamma,length;
  REAL d1[3],d2[3],t[3],maxlambda,minlambda,zdiff;

  if ( tb->normal[2] == 0.0 ) return edgedegencompare(ta,tb);

  /* now have nondegenerate facet */
 
       for ( k = 0 ; k < 3; k++ )
       { s1[k] = tb->x[1][k]-tb->x[0][k];
         s2[k] = tb->x[2][k]-tb->x[0][k];
         e1[k] = ta->x[0][k]-tb->x[0][k];
         e2[k] = ta->x[1][k]-tb->x[0][k];
       }
       /* Find max, min lambda of seg in triangle projection. */
       /* dets of triangle sides with edge endpoints */
       d1[0] = s1[0]*e1[1] - s1[1]*e1[0];
       d1[2] = e1[0]*s2[1] - e1[1]*s2[0];
       d1[1] = s1[0]*s2[1] - s1[1]*s2[0] - d1[0] - d1[2];
       d2[0] = s1[0]*e2[1] - s1[1]*e2[0];
       d2[2] = e2[0]*s2[1] - e2[1]*s2[0];
       d2[1] = s1[0]*s2[1] - s1[1]*s2[0] - d2[0] - d2[2];
       /* dets of edge with triangle vertices */
       t[0] = e2[0]*e1[1] - e2[1]*e1[0];
       t[2] = t[0] + d1[2] - d2[2];
       t[1] = d1[1] - d2[1] + t[2];
       maxlambda = -1e30; minlambda = 1e30;
       for ( k = 0 ; k < 3 ; k++ ) 
       { REAL diff,difft;
  
         /* triangle side parameter */
         difft = t[k] - t[(k+1)%3];
         if ( difft != 0.0 )
         { gamma = t[k]/difft;
           if ( gamma < 0.0 || gamma > 1.0 ) continue;
         }

         /* edge side parameter */
         diff = d1[k] - d2[k];
         if ( diff != 0.0 )
         { lambda = d1[k]/diff;
           if ( lambda < minlambda ) minlambda = lambda;
           if ( lambda > maxlambda ) maxlambda = lambda;
         }
         else if ( d1[k] < 0.0 ) return DISJOINT;
       }
       if ( maxlambda > 1.0 ) maxlambda = 1.0;
       if ( minlambda < 0.0 ) minlambda = 0.0;
       if ( minlambda > maxlambda ) return DISJOINT;
       a = triple_prod(s1,s2,e1); 
       b = triple_prod(s1,s2,e2); 
       if ( minlambda < maxlambda )
       { /* have actual finite segment */
         /* find barycentric points of crossing */
         if ( a == b ) /* edge parallel to facet */
         { if ( a < 0.0 ) return SECOND_BACK;
           if ( a > 0.0 ) return FIRST_BACK;
           return SECOND_BACK;  /* give edge priority */
         }
         lambda = a/(a-b);
         length = (e1[0]-e2[0])*(e1[0]-e2[0])+(e1[1]-e2[1])*(e1[1]-e2[1]);
         length = sqrt(length);
         if ( (lambda - minlambda)*length < .0001 || 
               (maxlambda-lambda)*length < .0001 )
         { /* whole segment on same side */
           if ( (2-minlambda-maxlambda)*a + (minlambda+maxlambda)*b <= 0.0 )
             return SECOND_BACK;
           else return FIRST_BACK;
         }
         else return BSPLITTINGA;
       }
       /* Now down to degenerate case of point crossing, at vertex */
       /* Give presumption to DISJOINT, unless one way in front */
       zdiff = (1-minlambda)*a + minlambda*b;
       if ( zdiff < -.001 ) return FIRST_BACK;
       if ( zdiff > .001 ) return SECOND_BACK;
       return DISJOINT;  
     
}

/***************************************************************************
*
* function: facetfacetcompare()
*
* purpose: Depth compare two facets, including all degenerate cases.
*/
int facetfacetcompare(ta,tb)
struct tsort *ta,*tb;
{
  if ( ta->normal[2] == 0.0 && tb->normal[2] == 0.0 ) 
    return degendegencompare(ta,tb);
  if ( ta->normal[2] == 0.0 ) 
    return degenfacetcompare(ta,tb);
  if ( tb->normal[2] == 0.0 ) 
  { int retval = degenfacetcompare(tb,ta);
    if ( retval == SECOND_BACK ) return FIRST_BACK;
    if ( retval == FIRST_BACK ) return SECOND_BACK;
    return retval;
  }

  /* now have two nondegenerate facets */

  return separating_line(ta,tb);
}

/***************************************************************************
*
* function: edgedegencompare()
*
* purpose: Depth compare edge and degenerate facet.
*/
int edgedegencompare(ta,tb)
struct tsort *ta; /* edge */
struct tsort *tb; /* degenerate facet */
{ 
return DISJOINT;  
#ifdef ZZZ
  struct tsort t; /* for constructing edge temp structure */
  int a,b,c;  /* compares of edge and edge of degenerate */
  int i;

  a = edgeedgecompare(ta,tb);
  if ( a == CROSSING ) 
      return CROSSING;
  t = *tb;
  for ( i = 0 ; i < 3 ; i++ ) t.x[1][i] = tb->x[2][i];
  b = edgeedgecompare(ta,&t);
  if ( b == CROSSING ) 
      return CROSSING;
  for ( i = 0 ; i < 3 ; i++ ) t.x[0][i] = tb->x[1][i];
  c = edgeedgecompare(ta,&t);
  if ( c == CROSSING ) 
      return CROSSING;

  if ( (a==DISJOINT) && (b==DISJOINT) && (c==DISJOINT) ) return DISJOINT;
  if ( ((a==SECOND_BACK)||(a==DISJOINT)) &&
       ((b==SECOND_BACK)||(b==DISJOINT)) &&
       ((c==SECOND_BACK)||(c==DISJOINT))) return SECOND_BACK; 
  return FIRST_BACK;
#endif
}
/***************************************************************************
*
* function: degenfacetcompare()
*
* purpose: Depth compare facet and degenerate facet.
*/
int degenfacetcompare(ta,tb)
struct tsort *ta; /* the degenerate facet */
struct tsort *tb;
{ 
 return DISJOINT; 
#ifdef ZZZ
  struct tsort t; /* for constructing edge temp structure */
  int a,b,c;  /* compares of edge of degenerate and facet */
  int i;
  a = edgefacetcompare(ta,tb);
  if ( a == CROSSING ) return CROSSING;
  t = *ta;
  for ( i = 0 ; i < 3 ; i++ ) t.x[1][i] = ta->x[2][i];
  b = edgefacetcompare(&t,tb);
  if ( b == CROSSING ) return CROSSING;
  for ( i = 0 ; i < 3 ; i++ ) t.x[0][i] = ta->x[1][i];
  c = edgefacetcompare(&t,tb);
  if ( c == CROSSING ) return CROSSING;

  if ( (a==DISJOINT) && (b==DISJOINT) && (c==DISJOINT) ) return DISJOINT;
  if ( ((a==SECOND_BACK)||(a==DISJOINT)) || 
       ((b==SECOND_BACK)||(b==DISJOINT)) || 
       ((c==SECOND_BACK)||(c==DISJOINT))) return SECOND_BACK; 
  return FIRST_BACK;
#endif
}
/***************************************************************************
*
* function: degendegencompare()
*
* purpose: Depth compare two degenerate facets.
*/
int degendegencompare(ta,tb)
struct tsort *ta,*tb;     /* the degenerate facets */
{ 
return DISJOINT;  
#ifdef ZZZ
  struct tsort t; /* for constructing edge temp structure */
  int a,b,c;  /* compares of edge of degenerate and degenerate */
  int i;
  a = edgedegencompare(ta,tb);
  if ( a == CROSSING ) return CROSSING;
  t = *ta;
  for ( i = 0 ; i < 3 ; i++ ) t.x[1][i] = ta->x[2][i];
  b = edgedegencompare(&t,tb);
  if ( b == CROSSING ) return CROSSING;
  for ( i = 0 ; i < 3 ; i++ ) t.x[0][i] = ta->x[1][i];
  c = edgedegencompare(&t,tb);
  if ( c == CROSSING ) return CROSSING;

  if ( (a==DISJOINT) && (b==DISJOINT) && (c==DISJOINT) ) return DISJOINT;
  if ( ((a==SECOND_BACK)||(a==DISJOINT)) &&
       ((b==SECOND_BACK)||(b==DISJOINT)) &&
       ((c==SECOND_BACK)||(c==DISJOINT))) return SECOND_BACK; 
  return FIRST_BACK;
#endif
}

/***************************************************************************
*
* function: edgefacetcompare()
*
* purpose: Depth compare two edge and facet, including all degenerate cases.
*          Does Simplex algorithm on facet triangle and thick edge
*          rectangle.
*/
REAL tab[4+3+2][2+1];  /* the tableau */
int tabrows = 9;

int edgefacetcompare(ta,tb)
struct tsort *ta; /* the edge */
struct tsort *tb; /* the facet */
{ int i,j,k;
  REAL width,dx,dy,length;
  REAL a,b,det,minratio,maxp;
  int pivj,pivi;
  int marker[7],label[2];
  int lastcol,cornercount,efront;

  /* width of edge, in descending order of thickness */
  if ( ta->etype[0] & BARE_EDGE ) width = edgewidths[0];
  else if ( ta->etype[0] & FIXED_EDGE ) width = edgewidths[1];
  else if ( ta->etype[0] & CONSTRAINT_EDGE ) width = edgewidths[2];
  else if ( ta->etype[0] & BOUNDARY_EDGE ) width = edgewidths[3];
  else if ( ta->etype[0] & SINGLE_EDGE ) width = edgewidths[4];
  else if ( ta->etype[0] & TRIPLE_EDGE ) width = edgewidths[5];
  else width = edgewidths[6]; /* regular grid interior edge */

  /* fill in tableau */
  /* edge outline */
  dx = ta->x[1][0] - ta->x[0][0];
  dy = ta->x[1][1] - ta->x[0][1];
  length = sqrt(dx*dx+dy*dy);
  tab[0][0] = dy;
  tab[0][1] = -dx;
  tab[0][2] = dy*ta->x[0][0] - dx*ta->x[0][1] + width/2*length;
  tab[1][0] = -dy;
  tab[1][1] = dx;
  tab[1][2] = -dy*ta->x[0][0] + dx*ta->x[0][1] + width/2*length;
  tab[2][0] = -dx;
  tab[2][1] = -dy;
  tab[2][2] = -dx*ta->x[0][0] - dy*ta->x[0][1]; 
  tab[3][0] = dx;
  tab[3][1] = dy;
  tab[3][2] = dx*ta->x[1][0] + dy*ta->x[1][1]; 
  /* edge z value */
  a = (ta->x[1][2]-ta->x[0][2])/(length*length);
  b = ta->x[0][2] - a*(ta->x[0][0]*dx + ta->x[0][1]*dy);
  tab[7][0] = -a*dx;
  tab[7][1] = -a*dy;
  tab[7][2] = b;
  /* facet outline, recalling vertices in clockwise order */
  dx = tb->x[1][0] - tb->x[0][0];
  dy = tb->x[1][1] - tb->x[0][1];
  tab[4][0] = -dy;
  tab[4][1] = dx;
  tab[4][2] = -dy*tb->x[0][0] + dx*tb->x[0][1];
  dx = tb->x[2][0] - tb->x[1][0];
  dy = tb->x[2][1] - tb->x[1][1];
  tab[5][0] = -dy;
  tab[5][1] = dx;
  tab[5][2] = -dy*tb->x[1][0] + dx*tb->x[1][1];
  dx = tb->x[0][0] - tb->x[2][0];
  dy = tb->x[0][1] - tb->x[2][1];
  tab[6][0] = -dy;
  tab[6][1] = dx;
  tab[6][2] = -dy*tb->x[2][0] + dx*tb->x[2][1];
  /* facet z value */
  det = tb->x[1][0]*tb->x[2][1] - tb->x[1][1]*tb->x[2][0]
      + tb->x[2][0]*tb->x[0][1] - tb->x[2][1]*tb->x[0][0]
      + tb->x[0][0]*tb->x[1][1] - tb->x[0][1]*tb->x[1][0];
  tab[8][0] = -
       (tb->x[1][2]*tb->x[2][1] - tb->x[1][1]*tb->x[2][2]
      + tb->x[2][2]*tb->x[0][1] - tb->x[2][1]*tb->x[0][2]
      + tb->x[0][2]*tb->x[1][1] - tb->x[0][1]*tb->x[1][2])/det;
  tab[8][1] = -
       (tb->x[1][0]*tb->x[2][2] - tb->x[1][2]*tb->x[2][0]
      + tb->x[2][0]*tb->x[0][2] - tb->x[2][2]*tb->x[0][0]
      + tb->x[0][0]*tb->x[1][2] - tb->x[0][2]*tb->x[1][0])/det;
  tab[8][2] = 
       (tb->x[1][0]*tb->x[2][1]*tb->x[0][2] - tb->x[1][1]*tb->x[2][0]*tb->x[0][2]
      + tb->x[2][0]*tb->x[0][1]*tb->x[1][2] - tb->x[2][1]*tb->x[0][0]*tb->x[1][2]
      + tb->x[0][0]*tb->x[1][1]*tb->x[2][2] - tb->x[0][1]*tb->x[1][0]*tb->x[2][2])
      /det;

  /* Phase 0: get rid of x and y */
  for ( i = 0 ; i < 2 ; i++ )
  { /* pick maximum magnitude pivot in each column */
    for ( j = 0, maxp = 0.0, pivj = -1 ; j < 7-i ; j++ )
       if ( fabs(tab[j][i]) > maxp ) { maxp = fabs(tab[j][i]); pivj = j; }
    /* pivot */
    pivot(pivj,i);
    /* swap to known row, x to row 6, y to row 5 (0 index, recall) */
    if ( pivj != 6-i )
     for ( k = 0 ; k < 3 ; k++ ) 
     { REAL tmp; tmp = tab[pivj][k]; tab[pivj][k]=tab[6-i][k];
       tab[6-i][k] = tmp;
     }
  }


  /* Phase 1: find feasible region (only top 5 rows in play for pivot) */
  /* Want to get rhs (col 2) all nonnegative */
  for(;;)
  { /* pick neg rhs */
    for ( j = 0, pivj = -1 ; j < 5 ; j++ )
    { if ( tab[j][2] < 0.0 ) { pivj = j; break; } }
    if ( pivj == -1 ) break;  /* done Phase 1 */
    /* pick pivot column */
    for ( i = 0, pivi = -1 ; i < 2 ; i++ )
    { if ( tab[pivj][i] < 0.0 ) { pivi = i; break; } }
    if ( pivi == -1 ) 
        return DISJOINT; /* no feasible region */
    /* get pivot in pivot column */
    minratio = tab[pivj][2]/tab[pivj][pivi];
    for ( j = 0 ; j < 5 ; j++ )
    { if ( tab[j][2] > 0 && tab[j][pivi] > 0.0 &&
           tab[j][2]/tab[j][pivi] < minratio )
      { pivj = j; minratio = tab[j][2]/tab[j][pivi]; }
    }
    /* do pivot */
    pivot(pivj,pivi);
  }

  /* Phase 2: tour feasible region */
  lastcol = 0;
  for ( i = 0 ; i < 5 ; i++ ) marker[i] = i;
  label[0] = 5; label[1] = 6;
  for(cornercount=0, efront=0;;)
  { /* record z info */
    int tmp;
    cornercount++;
    if ( tab[7][2] > tab[8][2] ) efront++;
    /* do column not done last */
    pivi = lastcol; lastcol = 1-lastcol;
    /* find minimum ratio in column */
    for ( j = 0, minratio = 1e30 ; j < 5 ; j++ )
      if ( tab[j][pivi] > 0.0 && tab[j][2]/tab[j][pivi] < minratio )
      { pivj = j; minratio = tab[j][2]/tab[j][pivi]; }
    pivot(pivj,pivi);
    tmp=label[pivi]; label[pivi] = marker[pivj]; marker[pivj] = tmp; 
    if ( label[0]==5 && label[1] == 6 ) break;
  }
  /* analyze results */
  if ( efront == 0 ) return FIRST_BACK;
  if ( efront == cornercount ) return SECOND_BACK;
  return ASPLITTINGB|BSPLITTINGA;

}

void pivot(i,j)
int i,j; /* pivot row and column */
{
  int n,m;
  REAL p = tab[i][j];

  for ( n = 0 ; n < tabrows  ; n++ )
    { if ( n == i ) continue;
      for ( m = 0 ; m < 3 ; m++ )
         { if ( m == j ) continue;
            tab[n][m] -= tab[n][j]*tab[i][m]/p;
         }
      tab[n][j] /= -p;
    }
  tab[i][j] = 1.0;
  for ( m = 0 ; m < 3 ; m++ )
     tab[i][m] /= p;
}

/*********************************************************************
*
* function: plane_test()
*
* purpose: See if one facet or edge is in front or back of element plane.
*          Suitable for Newell-Newell-Sancha algorithm.
*
* returns DISJOINT, FIRST_BACK, SECOND_BACK, ASPLITTINGB, BSPLITTINGA, or COPLANAR 
* Returns FIRST_BACK if guaranteed first does not obscure any of second.
*  Possibly bitwise OR of properties.
*/
int plane_test(ta,tb)
struct tsort *ta,*tb;
{
  REAL da=0.0,db=0.0;
  int k,n;
  int afar=0,aeq=0,anear=0;  /* count of ta vertices relative to tb plane */
  int bfar=0,beq=0,bnear=0;  /* count of tb vertices relative to ta plane */
  REAL d;
  int retval = NOTKNOWN;

  /* see where tb is with respect to ta plane */
    da = dotf(ta->normal,ta->x[0],SDIM);
     n = ((tb->flag & 0xF) == FACET) ? 3 : 2;
     for ( k = 0 ; k < n ; k++ )
     {
        d = dotf(ta->normal,tb->x[k],SDIM); 
        if ( d < da - 0.0001 ) { bnear++; continue; }
        if ( d < da + 0.0001 ) { beq++; continue; }
        bfar++;
     }
     if ( beq == n ) return COPLANAR; /* both in same plane */
     if ( bfar == 0 ) return FIRST_BACK; 
     if ( bnear > 0 ) 
     { if ( (ta->flag & 0xF) == FACET ) retval = ASPLITTINGB; }
     else retval = SECOND_BACK;
  
  /* see where ta is with respect to tb plane */

    db = dotf(tb->normal,tb->x[0],SDIM);
    n = ((ta->flag & 0xF) == FACET) ? 3 : 2;
    for ( k = 0 ; k < n ; k++ )
     {
        d = dotf(tb->normal,ta->x[k],SDIM); 
        if ( d < db - 0.0001 ) { anear++; continue; }
        if ( d < db + 0.0001 ) { aeq++; continue; }
        afar++;
     }
    if ( aeq == n ) return COPLANAR; /* both in same plane */
    if ( anear == 0 ) return FIRST_BACK;
    if ( afar > 0 )  
    { if ( (tb->flag & 0xf) == FACET ) retval |= BSPLITTINGA;   }
    else retval |= SECOND_BACK;
  
  return retval;

}
