/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <glib.h>
#include <ncurses.h>
#include <panel.h>
#include <frontend.h>

#include "common.h"
#include "window.h"
#include "menu.h"
#include "clist.h"
#include "dialog.h"
#include "selwin.h"
#include "views.h"
#include "enum.h"
#include "task.h"
#include "plugin.h"
#include "readable.h"
#include "fsutils.h"
#include "logging.h"

/**
 *	volume_has_fsim - checks to see if volume has associated FSIM
 *	@volume: volume's object handle
 *
 *	This routine checks to see if the volume has an associated
 *	FSIM and returns TRUE if it does.
 */
gboolean volume_has_fsim(object_handle_t volume)
{
	int rc;
	gboolean result = FALSE;
	handle_object_info_t *object;

	rc = evms_get_info(volume, &object);
	if (rc != 0) {
		log_error("%s: evms_get_info() returned error %d.\n", __FUNCTION__, rc);
	} else {
		result = object->info.volume.file_system_manager != 0;
		evms_free(object);
	}
	return result;
}

/**
 *	can_volume_be_formatted - check to see if any FSIM can mkfs the volume
 *	@volume: volume's object handle
 *
 *	This routine checks to see if any FSIM can mkfs it
 *	and if so returns TRUE.
 */
gboolean can_volume_be_formatted(object_handle_t volume)
{
	int rc;
	gboolean result = FALSE;
	handle_array_t  *plugins;

	rc = evms_get_plugin_list(EVMS_FILESYSTEM_INTERFACE_MODULE, 0, &plugins);
	if (rc != 0) {
		log_error("%s: evms_get_plugin_list() returned error code %d.\n", __FUNCTION__, rc);
	} else {
		gint i;

		for (i = 0; i < plugins->count && result == FALSE; i++) {
			result = evms_can_mkfs(volume, plugins->handle[i]) == 0;
		}
		evms_free(plugins);
	}
	return result;
}

/**
 *	filter_fsim_for_volume - filter FSIM that can do a mkfs on a give volume
 *	@fsim: the FSIM handle
 *	@user_data: contains the volume handle
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given plugin can mkfs on a given volume.
 */
int filter_fsim_for_volume(engine_handle_t fsim, void *user_data)
{
	return evms_can_mkfs(GPOINTER_TO_UINT(user_data), fsim);
}

/**
 *	filter_fsckable_volume - filter volume that can be fsck'd
 *	@handle: the volume handle
 *	@user_data: not used
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given volume can be fsck'd (and I mean that in a nice way).
 */
int filter_fsckable_volume(engine_handle_t handle, void *user_data)
{
	return evms_can_fsck(handle);
}

/**
 *	filter_unmkfsable_volumes - filter volumes that can be unmkfs'd
 *	@handle: the volume handle
 *	@user_data: not used
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given volume can be unmkfs'd.
 */
int filter_unmkfsable_volume(engine_handle_t handle, void *user_data)
{
	return evms_can_unmkfs(handle);
}

/**
 *	context_fsck_menuitem_activated - fsck initiated from context menu
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to fsck a filesystem on the volume selected
 *	from the context popup menu.
 */
int context_fsck_menuitem_activated(struct menu_item *item)
{
	struct dialog_window *dialog;

	dialog = create_task_dialog(NULL, GPOINTER_TO_UINT(item->user_data),
				EVMS_Task_fsck,
				_("Check/Repair File System"),
				_("Check"),
				NULL);
	if (dialog != NULL) {
		process_dialog_list_events(dialog->list);
		delete_dialog_list(dialog->list->list);
	}
	return 0;
}

/**
 *	on_delete_filesys_task_dialog - callback invoked when dialog is to be deleted
 *	@dialog: the plugin selection dialog
 *
 *	This routine is invoked when the filesystem task dialog is deleted. We
 *	take care of freeing any dynamically allocated entries in the hash table
 *	before freeing the hash table associated with the dialog.
 */
void on_delete_filesys_task_dialog(struct dialog_window *dialog)
{
	GHashTable *hash_table = dialog->user_data;

	if (hash_table != NULL) {
		g_free(g_hash_table_lookup(hash_table, "title"));
		g_hash_table_destroy(hash_table);
	}
}

/**
 *	show_filesys_task_dialog - present dialog with list of volumes to process a fsck task
 *	@title: the dialog window title
 *	@action: the task action code (EVMS_Task_fsck)
 *
 *	This routine displays volumes to fsck and handles the processing
 *	necessary for the follow-on task dialogs.
 */
void show_filesys_task_dialog(char *title, task_action_t action)
{
	struct selwin *selwin;
	GHashTable *hash_table;
	struct dialog_list dialogs;
	struct dialog_window *dialog;

	/*
	 * Allocate a hash table for storing multiple pieces of data needed by the
	 * activation callback. Save off the task action and title for starters.
	 */
	hash_table = g_hash_table_new(g_str_hash, g_str_equal);
	g_hash_table_insert(hash_table, "title", g_strdup(title));
	g_hash_table_insert(hash_table, "action", GUINT_TO_POINTER(action));

	selwin = create_selection_window(title,
					NULL, NULL,
					NULL,
					(menuitem_activate_cb)on_init_task_menuitem_activated,
					NULL,
					(menuitem_activate_cb)NULL,
					hash_table);

	dialog = (struct dialog_window *)selwin;

	print_clist_column_title(selwin->clist, 0, " ");
	print_clist_column_title(selwin->clist, 1, _("Volume"));
	print_clist_column_title(selwin->clist, 2, _("Size"));

	clist_populate(selwin->clist, enum_volumes,
			filter_fsckable_volume,
			format_standard_item, NULL, NULL);

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	set_menu_item_visibility(dialog->prev_button, FALSE);
	set_dialog_delete_cb(dialog, (dialog_delete_cb)on_delete_filesys_task_dialog);

	init_dialog_list(&dialogs);
	append_dialog_to_list(dialog, &dialogs);
	process_dialog_list_events(&dialogs);
	delete_dialog_list(dialogs.list);
}

/**
 *	actions_fsck_menuitem_activated - fsck initiated from Actions menu
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the File System->Check/Repair Actions pulldown
 *	menu item.
 */
int actions_fsck_menuitem_activated(struct menu_item *item)
{
	show_filesys_task_dialog(_("Check/Repair File System"), EVMS_Task_fsck);
	return 0;
}

/**
 *	create_mkfs_task_dialog - create task, set selected object and create options dialog
 *	@volume: the volume handle
 *	@plugin: the FSIM handle
 *
 *	This routine creates the EVMS_Task_mkfs task, sets the given volume as the selected
 *	object and creates the option dialog to configure the operation.
 */
struct dialog_window *create_mkfs_task_dialog(object_handle_t volume, object_handle_t plugin)
{
	int rc;
	task_handle_t task;
	struct dialog_window *dialog = NULL;

	rc = evms_create_task(plugin, EVMS_Task_mkfs, &task);
	if (rc == 0) {
		rc = set_selected_object(task, volume);
		if (rc == 0) {
			GHashTable *hash_table;

			hash_table = g_hash_table_new(g_str_hash, g_str_equal);
			g_hash_table_insert(hash_table, "task", GUINT_TO_POINTER(task));
			g_hash_table_insert(hash_table, "verb", g_strdup(_("Make File System")));

			dialog = create_task_options_dialog(task, _("Make File System"),
					_("Make"), NULL, hash_table);
			set_dialog_delete_cb(dialog, (dialog_delete_cb)on_delete_task_dialog);
		} else {
			evms_destroy_task(task);
			show_message_dialog(_("Set Selected Object Error"),
					_("Received an error communicating volume to FSIM plugin: %s"),
					evms_strerror(rc));
			log_error("%s: evms_set_selected_objects() returned error code %d.\n", __FUNCTION__, rc);
		}
	} else {
		show_message_dialog(_("Error starting mkfs task"),
				_("Received the following error starting Make File System task: %s"),
				evms_strerror(rc));
		log_error("%s: evms_create_task() returned error code %d.\n", __FUNCTION__, rc);
	}
	return dialog;
}

/**
 *	fsim_for_volume_button_activated - callback invoke when the FSIM is selected for the volume
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Next button is activated after selecting
 *	a FSIM to mkfs a volume. We then create a task now that we know the
 *	volume and the FSIM.
 */
int fsim_for_volume_button_activated(struct menu_item *item)
{
	GHashTable *hash_table;
	struct selwin *selwin = item->user_data;
	object_handle_t saved_plugin, curr_plugin;
	struct dialog_window *dialog = item->user_data;

	hash_table = dialog->user_data;
	curr_plugin = get_selected_handle(selwin->clist);
	saved_plugin = GPOINTER_TO_UINT(g_hash_table_lookup(hash_table, "plugin"));
	
	if (curr_plugin != saved_plugin) {
		GList *next_dialog;
		object_handle_t volume;
		struct dialog_window *new_dialog;
		/*
		 * Since the selected feature has changed we need to delete any
		 * dialogs that we may have built for the mkfs task.
		 */
		next_dialog = get_next_dialog(dialog);
		if (next_dialog != NULL)
			delete_dialog_list(next_dialog);

		volume = GPOINTER_TO_UINT(g_hash_table_lookup(hash_table, "volume"));
		new_dialog = create_mkfs_task_dialog(volume, curr_plugin);

		if (new_dialog != NULL) {
			g_hash_table_insert(hash_table, "plugin", GUINT_TO_POINTER(curr_plugin));
			append_dialog_to_list(new_dialog, dialog->list);
			dialog->list->current = get_next_dialog(dialog);
		}
	} else {
		dialog->list->current = get_next_dialog(dialog);
	}

	return 0;
}

/**
 *	delete_fsim_dialog_cb - callback invoked when FSIM selection dialog is being deleted
 *	@dialog: the FSIM plugin selection dialog
 *
 *	This routine is invoked when the FSIM selection dialog is being deleted. We take
 *	this opportunity to destroy the hash table associated with the dialog window.
 */
void delete_fsim_dialog_cb(struct dialog_window *dialog)
{
	g_hash_table_destroy(dialog->user_data);
}

/**
 *	show_fsims_for_volume - create a dialog with a list of FSIMs that can mkfs the given volume
 *	@handle: the handle of the volume we want to create a filesystem on
 *
 *	This routine creates a dialog window with a list of FSIM that can create a filesystem
 *	on the given volume.
 */
void show_fsims_for_volume(object_handle_t handle)
{
	struct selwin *selwin;
	GHashTable *hash_table;
	struct dialog_list dialogs;
	struct dialog_window *dialog;

	hash_table = g_hash_table_new(g_str_hash, g_str_equal);
	g_hash_table_insert(hash_table, "volume", GUINT_TO_POINTER(handle));

	selwin = create_selection_window(_("Make File System - Plugin Selection"),
					NULL, NULL, NULL,
					(menuitem_activate_cb)fsim_for_volume_button_activated,
					NULL, NULL, hash_table);

	dialog = (struct dialog_window *)selwin;
	set_clist_column_count(selwin->clist, 2);
	set_clist_column_info(selwin->clist, 0, calc_clist_column_width(selwin->clist, 0.05),
				0,
				CLIST_TEXT_JUSTIFY_CENTER);
	set_clist_column_info(selwin->clist, 1, calc_clist_column_width(selwin->clist, 0.95),
				get_clist_column_end(selwin->clist, 0),
				CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(selwin->clist, 0, "");
	print_clist_column_title(selwin->clist, 1, _("FSIM"));

	clist_populate(selwin->clist, enum_fsim_plugins, filter_fsim_for_volume,
			format_standard_item, NULL, GUINT_TO_POINTER(handle));

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	set_dialog_delete_cb(dialog, (dialog_delete_cb)delete_fsim_dialog_cb);
	set_menu_item_visibility(dialog->prev_button, FALSE);

	init_dialog_list(&dialogs);
	append_dialog_to_list(dialog, &dialogs);
	process_dialog_list_events(&dialogs);
	delete_dialog_list(dialogs.list);
}

/**
 *	context_mkfs_menuitem_activated - mkfs initiated from context menu
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to make a filesystem on the volume selected
 *	and is initiated from the context popup menu.
 */
int context_mkfs_menuitem_activated(struct menu_item *item)
{
	show_fsims_for_volume(GPOINTER_TO_UINT(item->user_data));
	return 0;
}

/**
 *	actions_mkfs_menuitem_activated - mkfs initiated from Actions menu
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the File System->Make Actions pulldown
 *	menu item.
 */
int actions_mkfs_menuitem_activated(struct menu_item *item)
{
	process_plugin_task(_("Make File System"), EVMS_FILESYSTEM_INTERFACE_MODULE,
				FALSE, EVMS_Task_mkfs);
	return 0;
}

/**
 *	unmkfs_button_activated - callback to remove a file system
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Remove button is activated to remove the
 *	file system from a volume.
 */
int unmkfs_button_activated(struct menu_item *item)
{
	int rc;
	struct selwin *selwin = item->user_data;

	rc = evms_unmkfs(get_selected_handle(selwin->clist));
	update_status(_("Remove"), rc);

	if (rc == 0) {
		((struct dialog_window *)selwin)->status = DLG_STATUS_CLOSING;
		refresh_views();
	}
	return 0;
}

/**
 *	format_remove_segment_manager_item - return column strings for a row in remove segment manager clist
 *	@handle: the volume handle
 *	@not_used: extra info that we don't pay attention to
 *	@text: the array in which to place the row's column text
 *
 *	This routine is called to produce the row/column text for a volume placed
 *	in the remove file system clist. We return the name of the object and the name
 *	of the plugin currently assigned.
 */
int format_unmkfs_item(object_handle_t handle, void *not_used, GPtrArray *text)
{
	int rc;
	handle_object_info_t *object;

	rc = evms_get_info(handle, &object);
	if (rc == 0) {
		g_ptr_array_add(text, g_strdup(" "));
		g_ptr_array_add(text, g_strdup(object->info.volume.name));
		g_ptr_array_add(text, get_plugin_name(object->info.volume.file_system_manager));
		evms_free(object);
	}
	return rc;
}

/**
 *	show_unmkfs_dialog - display dialog to allow user to remove a filesystem
 *	@handle: non-zero when invoked from context popup menu
 *
 *	This routine displays a selection window to allow the user to remove a filesystem
 *	from one of the volumes listed.
 **/
int show_unmkfs_dialog(object_handle_t handle)
{
	struct selwin *selwin;
	struct dialog_window *dialog;

	selwin = create_selection_window(_("Remove File System"),
					NULL, NULL,
					_("_Remove"),
					(menuitem_activate_cb)unmkfs_button_activated,
					NULL,
					(menuitem_activate_cb)NULL,
					NULL);

	dialog = (struct dialog_window *)selwin;

	set_clist_column_info(selwin->clist, 2, calc_clist_column_width(selwin->clist, 0.45),
				get_clist_column_end(selwin->clist, 1),
				CLIST_TEXT_JUSTIFY_LEFT);

	print_clist_column_title(selwin->clist, 0, " ");
	print_clist_column_title(selwin->clist, 1, _("Volume"));
	print_clist_column_title(selwin->clist, 2, _("FSIM"));

	if (handle == 0)
		clist_populate(selwin->clist, enum_volumes,
				filter_unmkfsable_volume,
				format_unmkfs_item, NULL, NULL);
	else
		clist_populate_handle(selwin->clist, handle,
				format_unmkfs_item, NULL, NULL);

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	set_menu_item_visibility(dialog->prev_button, FALSE);

	process_modal_dialog(dialog);
	return 0;
}

/**
 *	context_unmkfs_menuitem_activated - unmkfs initiated from context menu
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to remove a filesystem on the volume selected
 *	and is initiated from the context popup menu.
 */
int context_unmkfs_menuitem_activated(struct menu_item *item)
{
	show_unmkfs_dialog(GPOINTER_TO_UINT(item->user_data));
	return 0;
}

/**
 *	actions_unmkfs_menuitem_activated - unmkfs initiated from Actions menu
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the File System->Remove Actions pulldown
 *	menu item.
 */
int actions_unmkfs_menuitem_activated(struct menu_item *item)
{
	show_unmkfs_dialog(0);
	return 0;
}
