/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: add.c
 */

#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "add.h"
#include "widget.h"
#include "task.h"
#include "logging.h"

/*
 *
 *   void add_feature_to_volume (GtkButton *, object_handle_t)
 *
 *   Description:
 *      This routine initiates the task of setting a
 *      add a feature to the given volume and if so returns TRUE.
 *
 *   Entry:
 *      button - the button that initiated the action
 *      volume - volume's object handle
 *
 *   Exit:
 *      See Description.
 *
 */
void add_feature_to_volume(GtkButton * button, object_handle_t volume)
{
	GList *window_list;
	GtkCList *clist;
	GtkWidget *next_window;
	GtkWidget *selection_window;
	plugin_handle_t feature;
	plugin_handle_t prev_feature;

	selection_window = gtk_widget_get_toplevel(GTK_WIDGET(button));

	clist = GTK_CLIST(lookup_widget(GTK_WIDGET(button), "selection_window_clist"));
	feature = GPOINTER_TO_UINT(get_single_select_current_row_data(clist));
	window_list = get_window_list(selection_window);
	next_window = gtk_object_get_data(GTK_OBJECT(selection_window), "next_window_id");
	prev_feature = GPOINTER_TO_UINT(gtk_object_get_data(GTK_OBJECT(selection_window),
							    "previous_feature"));

	if (window_list == NULL)
		window_list = g_list_append(window_list, selection_window);

	if (feature != prev_feature) {
		gint rc;
		task_handle_t handle;

		rc = evms_create_task(feature, EVMS_Task_Add_Feature, &handle);

		if (rc != SUCCESS) {
			log_error("%s: evms_create_task() returned error code %d.\n", __FUNCTION__,
				  rc);

			display_results_window(rc, NULL, _("Unable to create the task context."),
					       NULL, FALSE, selection_window);
		} else {
			/*
			 * Shortcut the task process. Since the process starts with selecting the volume
			 * first (either through action menu or context popup menu), we can simply set
			 * the volume as the selected object and continue on to configure options.
			 */

			gchar mem[sizeof(handle_array_t) + sizeof(object_handle_t)];
			handle_array_t * ha = (handle_array_t *) mem;

			ha->count = 1;
			ha->handle[0] = volume;

			rc = evms_set_selected_objects(handle, ha, NULL, NULL);

			if (rc != SUCCESS) {
				log_error
				    ("%s: evms_set_selected_objects() returned error code %d.\n",
				     __FUNCTION__, rc);

				display_results_window(rc, NULL,
						       _
						       ("Feature plug-in did not accept the volume."),
						       NULL, FALSE, selection_window);
			} else {
				/*
				 * If we already created the next window for options,
				 * make sure we destroy the window since our selection
				 * has apperently changed and so the panel needs to be
				 * recreated.
				 */

				if (next_window != NULL)
					destroy_window_list(g_list_find(window_list, next_window));

				next_window =
				    create_standard_task_window(_
								("Add Feature to Volume Configuration"),
								_("Add"), NULL,
								on_button_clicked_display_prev_window,
								on_button_clicked_destroy_window_list,
								handle);

				window_list = g_list_append(window_list, next_window);

				set_window_list(next_window, window_list);
				set_window_list(selection_window, window_list);

				bind_options_callout_to_toplevel_widget(next_window,
									options_window_callout);

				gtk_object_set_data(GTK_OBJECT(selection_window), "next_window_id",
						    next_window);
				gtk_object_set_data(GTK_OBJECT(selection_window),
						    "previous_feature", GUINT_TO_POINTER(feature));

				/*
				 * Connect the destroy signal to the next window. If the window is
				 * destroyed, so is the task context.
				 */

				gtk_signal_connect(GTK_OBJECT(next_window), "destroy",
						   GTK_SIGNAL_FUNC(on_task_window_destroy),
						   GUINT_TO_POINTER(handle));

				gtk_widget_show(next_window);
				gtk_widget_hide(selection_window);
			}
		}
	} else {
		gtk_widget_show(next_window);
		gtk_widget_hide(selection_window);
	}
}

/*
 *
 *   gboolean can_volume_accept_new_feature (object_handle_t)
 *
 *   Description:
 *      This routine checks to see if any feature plugin can
 *      add a feature to the given volume and if so returns TRUE.
 *
 *   Entry:
 *      volume - volume's object handle
 *
 *   Exit:
 *      See Description.
 *
 */
gboolean can_volume_accept_new_feature(object_handle_t volume)
{
	gint rc;
	gboolean result = FALSE;
	handle_array_t *plugins;

	rc = evms_get_plugin_list(EVMS_FEATURE, 0, &plugins);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_plugin_list() returned error code %d.\n", __FUNCTION__, rc);
	} else {
		gint i;

		for (i = 0; i < plugins->count && result == FALSE; i++) {
			result = evms_can_add_feature_to_volume(volume, plugins->handle[i]) == 0;
		}

		evms_free(plugins);
	}

	return result;
}
