/*
 *   (C) Copyright IBM Corp. 2001, 2005
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: raid0_mgr.c
 *
 * Description: This file contains all of the required engine-plugin APIs
 *              for the Raid0 MD region manager.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include <malloc.h>

#include "md.h"
#include "raid0_mgr.h"

#define my_plugin_record raid0_plugin

static list_anchor_t raid0_expand_shrink_list = NULL;
static list_anchor_t raid0_delay_kill_sector_list = NULL;

static int raid0_compare_func(void *thing1, void *thing2, void *user_data)
{
	storage_object_t *obj1 = (storage_object_t *)thing1;
	storage_object_t *obj2 = (storage_object_t *)thing2;
	md_volume_t *vol = (md_volume_t *)user_data;
	sector_count_t size1;
	sector_count_t size2;

	size1 = md_object_usable_size(obj1, &vol->sb_ver, vol->chunksize);
	size2 = md_object_usable_size(obj2, &vol->sb_ver, vol->chunksize);
	
	if (size1 < size2) return -1;
	if (size1 > size2) return 1;
	return 0;
}

static void raid0_free_private_data(md_volume_t * volume)
{
	raid0_conf_t * conf = (raid0_conf_t *)volume->private_data;
	int i;
	struct strip_zone *zone;

	LOG_ENTRY();

	if (!conf) {
		LOG_WARNING("Nothing to free!!!.\n");
		LOG_EXIT_VOID();
		return;
	}

	if (conf->strip_zone) {
		for (i=0; i<conf->nr_strip_zones; i++) {
			zone = conf->strip_zone + i;
			if (zone->dev) {
				EngFncs->engine_free(zone->dev);
				zone->dev = NULL;
			}
		}
		EngFncs->engine_free(conf->strip_zone);
		conf->strip_zone = NULL;
	}
	if (conf->hash_table) {
		EngFncs->engine_free(conf->hash_table);
		conf->hash_table = NULL;
	}
	EngFncs->engine_free(volume->private_data);
	volume->private_data = NULL;
	LOG_EXIT_VOID();
}

static int create_strip_zones (md_volume_t * vol)
{
	int rc = 0;
	int i, c, cur;
	u_int64_t current_offset;
	u_int64_t curr_zone_offset;
	u_int64_t size;
	u_int64_t zone0_size;
	raid0_conf_t * conf = mdvol_to_conf(vol);
	storage_object_t * child_object;
	storage_object_t * child_object1;
	storage_object_t * child_object2;
	md_member_t *member;
	md_member_t *smallest;
	md_member_t *member1;
	md_member_t *member2;
	list_element_t iter;
	list_element_t iter1;
	list_element_t iter2;
	struct strip_zone *zone;

	LOG_ENTRY();

	// if this volume is corrupt, can't build the stripes correctly, so just return
	if (vol->flags & MD_CORRUPT) {
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (vol->raid_disks == 0) {
		LOG_MD_BUG();
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	conf->chunksize = vol->chunksize;

	LOG_DEBUG("Intializing private data for RAID0 region %s, chunksize(%d sectors).\n",
		  vol->name, conf->chunksize);

	LIST_FOR_EACH(vol->members, iter1, member1) {
		LOG_DEBUG("  %s: dev_number(%d), data_offset(%"PRIu64") data_size(%"PRIu64").\n",
			  member1->obj->name, member1->dev_number,
			  member1->data_offset, member1->data_size);
	}

	/*
	 * The number of 'same size groups'
	 */
	conf->nr_strip_zones = 0;
	LIST_FOR_EACH(vol->members, iter1, member1) {
		child_object1 = member1->obj;
		LOG_DEBUG("Looking at %s\n", child_object1->name);
		c = 0;
		LIST_FOR_EACH(vol->members, iter2, member2) {
			child_object2 = member2->obj;
			LOG_DEBUG("Comparing %s with %s\n", child_object1->name, child_object2->name);
			if (child_object2 == child_object1) {
				LOG_DEBUG("  END\n");
				break;
			}
			if (member1->data_size == member2->data_size) {
				/*
				 * Not unique, dont count it as a new
				 * groupvolume->super_block->chunk_size >> EVMS_VSECTOR_SIZE_SHIFT
				 */
				LOG_DEBUG("  EQUAL\n");
				c = 1;
				break;
			}
			LOG_DEBUG("  NOT EQUAL\n");
		}
		if (!c) {
			LOG_DEBUG("  ==> UNIQUE\n");
			conf->nr_strip_zones++;
			LOG_DEBUG("  %d zones\n", conf->nr_strip_zones);
		}
	}
	LOG_DEBUG("FINAL %d zones\n", conf->nr_strip_zones);

	conf->strip_zone = EngFncs->engine_alloc(conf->nr_strip_zones * sizeof(struct strip_zone));
	if (!conf->strip_zone) {
		rc = ENOMEM;
		LOG_CRITICAL("Error %d allocating memory for strip zone structures.\n", rc);
		goto out;
	}

	for (i=0; i<conf->nr_strip_zones; i++) {
		zone = conf->strip_zone + i;
		zone->dev = EngFncs->engine_alloc(sizeof(md_member_t *) * vol->raid_disks);
		if (!zone->dev) {
			rc = ENOMEM;
			LOG_CRITICAL("Error %d allocating memory device list.\n", rc);
			goto out;
		}
	}

	conf->smallest_zone = NULL;
	current_offset = 0;
	curr_zone_offset = 0;

	for (i = 0; i < conf->nr_strip_zones; i++) {
		zone = conf->strip_zone + i;

		LOG_DEBUG("Zone %d\n", i);
		zone->dev_offset = current_offset;
		smallest = NULL;
		c = 0;

		LIST_FOR_EACH(vol->members, iter, member) {
			child_object = member->obj;
			LOG_DEBUG("  checking %s ...\n", child_object->name);
			if (member->data_size > current_offset) {
				LOG_DEBUG("  contained as device %d\n", c);
				zone->dev[c] = member;
				c++;
				if (!smallest || (member->data_size < smallest->data_size)) {
					smallest = member;
					LOG_DEBUG("  (%"PRIu64") is smallest!.\n", smallest->data_size);
				}
			} else {
				LOG_DEBUG("  nope.\n");
			}
		}

		zone->nb_dev = c;
		zone->size = (smallest->data_size - current_offset) * c;
		LOG_DEBUG("zone->nb_dev: %d, size: %"PRIu64"\n", zone->nb_dev, zone->size);

		if (!conf->smallest_zone || (zone->size < conf->smallest_zone->size))
			conf->smallest_zone = zone;

		zone->zone_offset = curr_zone_offset;
		curr_zone_offset += zone->size;

		current_offset = smallest->data_size;
		LOG_DEBUG("current zone offset: %"PRIu64"\n", current_offset);
	}

	conf->nr_zones = (md_volume_calc_size(vol) + conf->smallest_zone->size - 1) / conf->smallest_zone->size;
	LOG_DEBUG("Number of zones is %d.\n", conf->nr_zones);

	/* Set up the hash tables. */
	cur = 0;

	conf->hash_table = EngFncs->engine_alloc(sizeof (struct raid0_hash) * conf->nr_zones);
	if (!conf->hash_table) {
		rc = ENOMEM;
		LOG_CRITICAL("Error %d allocating memory for zone hash table.\n", rc);
		goto out;
	}
	size = conf->strip_zone[cur].size;

	i = 0;
	while (cur < conf->nr_strip_zones) {
		conf->hash_table[i].zone0 = conf->strip_zone + cur;

		/*
		 * If we completely fill the slot
		 */
		if (size >= conf->smallest_zone->size) {
			conf->hash_table[i++].zone1 = NULL;
			size -= conf->smallest_zone->size;

			/*
			 * If there is nothing left in the strip zone,
			 * move to the next srip zone.  Else, the
			 * next iteration of the loop will hit the
			 * code below where zone1 is filled in for this
			 * hash entry.
			 */
			if (!size) {
				if (++cur == conf->nr_strip_zones)
					continue;
				size = conf->strip_zone[cur].size;
			}
			continue;
		}
		if (++cur == conf->nr_strip_zones) {
			/*
			 * Last dev, set unit1 as NULL
			 */
			conf->hash_table[i].zone1=NULL;
			continue;
		}

		/*
		 * Here we use a 2nd dev to fill the slot
		 */
		zone0_size = size;
		size = conf->strip_zone[cur].size;
		conf->hash_table[i++].zone1 = conf->strip_zone + cur;
		size -= conf->smallest_zone->size - zone0_size;
	}

out:
	if (rc) {
		raid0_free_private_data(vol);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid0_setup_evms_plugin
 *
 *  This function gets called shortly after the plugin is loaded by the
 *  Engine. It performs all tasks that are necessary before the initial
 *  discovery pass.
 */
static int raid0_setup_evms_plugin(engine_functions_t * functions) {
	int rc = 0;

	/* Parameter check */
	if (!functions) {
		return EINVAL;
	}

	EngFncs = functions;

	my_plugin = raid0_plugin;
	LOG_ENTRY();
	rc = md_register_name_space();

	if (rc != 0) {
		LOG_SERIOUS("Failed to register the MD name space.\n");
	}

	if (raid0_expand_shrink_list == NULL) {
		raid0_expand_shrink_list = EngFncs->allocate_list();
	}

	if (raid0_delay_kill_sector_list == NULL) {
		raid0_delay_kill_sector_list = EngFncs->allocate_list();
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/****** Region Checking Functions ******/


/* All of the following md_can_* functions return 0 if they are able to
 * perform the specified action, or non-zero if they cannot.
 */


/* Function: raid0_can_delete
 *
 * Can we remove the specified MD logical volume?  Yes.
 */
static int raid0_can_delete( storage_object_t * region ) {

	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}


/* Function: raid0_can_expand
 *
 */
static int raid0_can_expand(
	storage_object_t * region,
	u_int64_t expand_limit,
	list_anchor_t expansion_points )
{
	int rc = 0;
	md_volume_t *vol = (md_volume_t *)region->private_data;
	expand_object_info_t * expand_object;
	list_anchor_t acceptable_objects = NULL;
	list_element_t iter, li=NULL;
	storage_object_t *obj;
	sector_count_t expand_size = 0;
	sector_count_t size;
	int disk_count;
	logical_volume_t *evms_volume;
	md_super_info_t info;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	/* Don't allow expanding if the region is corrupt */
	if (region->flags & SOFLAG_CORRUPT) {
		LOG_EXIT_INT(EINVAL);
		return(EINVAL);
	}

	/* Don't allow expanding if volume is mounted */
	if (EngFncs->is_offline(region, &evms_volume) == FALSE) {
		LOG_EXIT_INT(EINVAL);
		return(EINVAL);
	}

	/* Don't allow expanding if changes are pending */
	if (region->flags & SOFLAG_DIRTY) {
		LOG_EXIT_INT(EBUSY);
		return(EBUSY);
	}

	md_volume_get_super_info(vol, &info);

	/* This region can expand if nr_disks <= max_disks */
	disk_count = info.nr_disks;
	if ( disk_count > MAX_DISKS(vol)) {
		/* TODO: We should allow the last child to expand */
		rc = EINVAL;
		LOG_EXIT_INT(rc);
		return (rc);
	}

	/* Get all available objects */
	rc = EngFncs->get_object_list(DISK | SEGMENT | REGION,
				DATA_TYPE,
				NULL,
				region->disk_group,
				VALID_INPUT_OBJECT | NO_DISK_GROUP,
				&acceptable_objects);

	if (rc) {
		LOG_WARNING("Error getting available object list.\n");
		LOG_EXIT_INT(rc);
		return rc;
	}
	
	if (!acceptable_objects || !EngFncs->list_count(acceptable_objects)) {
		goto out;
	}
	
	/* Remove all parents of this MD region from acceptable list */
	remove_parent_regions_from_list(acceptable_objects, region);
	
	/*
	 * Sort the list.
	 * Since the compare function needs chunk size information, we need to pass
	 * the pointer to MD volume.
	 */
	rc = EngFncs->sort_list(acceptable_objects, raid0_compare_func, (void*)vol);
	if (rc) {
		goto out;
	}

	LIST_FOR_EACH(acceptable_objects, iter, obj) {
		if ( disk_count == MAX_DISKS(vol)) {
			break;
		}

		if ( obj != region) {
			size = md_object_usable_size(obj, &vol->sb_ver, vol->chunksize);
			if ((expand_size + size) > expand_limit) {
				break;
			}
			expand_size += size;
			disk_count++;
		}
	}

	if (expand_size) {

		expand_object = (expand_object_info_t *) EngFncs->engine_alloc( sizeof(expand_object_info_t) );
		if (expand_object) {
			expand_object->object = region;
			expand_object->max_expand_size = expand_size;

			li = EngFncs->insert_thing(expansion_points,
						   expand_object,
						   INSERT_AFTER,
						   NULL);

			if (!li) {
				EngFncs->engine_free( expand_object );
				rc = ENOMEM;
			}
		} else {
			rc = ENOMEM;
		}
	} else {
		/* TODO: We should allow the last child to expand */
		rc = EINVAL;
	}

out:
	if (acceptable_objects) {
		EngFncs->destroy_list(acceptable_objects);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int raid0_can_shrink(
	storage_object_t * region,
	u_int64_t shrink_limit,
	list_anchor_t shrink_points )
{
	int rc = 0;
	md_volume_t *vol;
	storage_object_t *obj;
	shrink_object_info_t * shrink_object;
	u_int64_t shrink_size = 0;
	list_anchor_t my_list = NULL;
	list_element_t li = NULL;
	logical_volume_t *evms_volume;
	md_member_t *member;
	list_element_t iter;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	/* Don't allow shrinking if the region is corrupt */
	if (region->flags & SOFLAG_CORRUPT) {
		LOG_EXIT_INT(EINVAL);
		return(EINVAL);
	}
	
	/* Don't allow shrinking if volume is mounted */
	if (EngFncs->is_offline(region, &evms_volume) == FALSE) {
		LOG_EXIT_INT(EINVAL);
		return(EINVAL);
	}

	/* Don't allow shrinking if changes are pending */
	if (region->flags & SOFLAG_DIRTY) {
		LOG_EXIT_INT(EBUSY);
		return(EBUSY);
	}

	vol = (md_volume_t *)region->private_data;
	
	/*
	 * Create a list and insert all child objects into the list
	 */
	my_list = EngFncs->allocate_list();
	if (!my_list) {
		rc = ENOMEM;
		goto out;
	}

	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj) {
			li = EngFncs->insert_thing(my_list, member, INSERT_AFTER, NULL);
			if (!li) {
				rc = ENOMEM;
				goto out;
			}
		}
	}
	
	/*
	 * Sort the list.
	 * Since the compare function needs chunk size information, we need to pass
	 * the pointer to MD volume.
	 */
	rc = EngFncs->sort_list(my_list, raid0_compare_func, (void*)vol);
	if (rc) {
		goto out;
	}
	
	/* The first child object is the smallest. */
	member = EngFncs->first_thing(my_list, NULL);

	/*
	 * If the shrink limit is less than the smallest child object,
	 * we will not shrink this region.  However, the return code is 0.
	 */
	if (shrink_limit < member->data_size) {
		rc = 0;
		goto out;
	}

	/*
	 * Loop through the list, increase the shrink_size until we hit the limit.
	 */
	LIST_FOR_EACH(my_list, li, obj) {
		if ((shrink_size + member->data_size) > shrink_limit)
			break;
		shrink_size += member->data_size;
	}

	/*
	 * Allocate a shrink_info
	 * Set shrink_info's object to this region
	 * Set maximum shrink size
	 * Append to the shrink points list
	 */
	shrink_object = (shrink_object_info_t *) EngFncs->engine_alloc( sizeof(shrink_object_info_t) );
	if (shrink_object) {
		shrink_object->object          = region;
		shrink_object->max_shrink_size = shrink_size;

		li = EngFncs->insert_thing(shrink_points, shrink_object, INSERT_AFTER, NULL);

		if (!li) {
			EngFncs->engine_free( shrink_object );
			rc = ENOMEM;
		}
	} else {
		rc = ENOMEM;
	}

out:
	if (my_list) {
		EngFncs->destroy_list(my_list);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int raid0_can_replace_child(storage_object_t *region,
				   storage_object_t *child,
				   storage_object_t *new_child)
{
	int rc;
	my_plugin = raid0_plugin;
	LOG_ENTRY();
	rc = md_can_replace_child(region, child, new_child);
	LOG_EXIT_INT(rc);
	return rc;
}

static int raid0_create_region(md_volume_t *vol, list_anchor_t output_list, boolean final_call)
{
	int rc = 0;
	storage_object_t * region = NULL;
	md_member_t *saved_member = NULL;
	md_saved_info_t *saved_info = NULL;
	md_member_t *member;
	int i;
	mdu_array_info_t info;
	int length;

	LOG_ENTRY();

	if (!vol->sb) {
		LOG_MD_BUG();
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}


	if ((vol->nr_disks < vol->raid_disks) & !final_call) {
		LOG_DETAILS("Region %s is missing members, delaying discovery.\n",
			    vol->name);
		LOG_EXIT_INT(0);
		return 0;
	}

	rc = EngFncs->allocate_region(vol->name, &region);
	if (rc) {
		LOG_WARNING("Region %s is already created, try new name.\n", vol->name);
		rc = md_volume_get_alternative_name(vol, 255);
		if (!rc) {
			LOG_WARNING("Trying new region name: %s...\n", vol->name);
			rc = EngFncs->allocate_region(vol->name, &region);
			if (!rc) {
				LOG_WARNING("OK. got it.\n");
			} else {
				LOG_ERROR("Give up.\n");
			}
		}
	}

	region->data_type = DATA_TYPE;
	region->plugin = raid0_plugin;
	region->private_data = (void *)vol;
	vol->flags |= MD_DISCOVERED;
	vol->region = region;

	md_analyze_volume(vol);

	for (i=0; i<vol->raid_disks; i++) {
		member = md_volume_find_member(vol, i);
		if (member) {
			md_append_region_to_object(region, member->obj);
		} else {
			length = sprintf(message_buffer, _("  The disk indexed %d is missing.\n"), i);
			md_queue_corrupt_message(vol->personality, message_buffer, length);
			vol->flags |= MD_CORRUPT;
		}
	}
	
	region->size = md_volume_calc_size(vol);

	/* If the MD array is corrupt, don't allocate private data. */
	if (vol->flags & MD_CORRUPT) {
		goto out;
	}

	vol->private_data = EngFncs->engine_alloc(sizeof (raid0_conf_t));
	if (vol->private_data) {
		rc = create_strip_zones(vol);
	} else {
		LOG_CRITICAL("Error %d allocating memory for raid 0 configuration structure.\n", rc);
		vol->flags |= MD_CORRUPT;
		goto out;
	}

	/*
	 * Check EXPAND/SHRINK flag in the "saved" superblock.
	 * If EXPAND_IN_PROGRESS, we will need to unwind the expansion.
	 * If SHRINK_IN_PROGRESS, we will need to resume the shrinking.
	 */
	if (md_check_for_expand_shrink_in_progress(vol, &saved_member)) {
		saved_info = saved_member->saved_info;
		if (saved_info->sector_mark > 0) {
			if (saved_info->flags & MD_SAVED_INFO_EXPAND_IN_PROGRESS) {
				rc = raid0_unwind_expansion(region);
				if (!rc) {
					MESSAGE(_("The process to expand region %s was interrupted.  "
						  "The orginal configuration will be restored."),
						region->name);
				}
			} else
			if (saved_info->flags & MD_SAVED_INFO_SHRINK_IN_PROGRESS) {
				rc = raid0_resume_shrinking(region);
				if (!rc) {
					MESSAGE(_("The process to shrink region %s was interrupted.  "
						  "The process will be resumed."),
						region->name);
				}
			} else {
				LOG_MD_BUG();
			}
		} else {
			LOG_WARNING("%s: The sector mark is 0.\n", region->name);
		}
	}

	/*
	 * Query device-mapper for the status of this MD object.
	 * if this MD object is active, it's already activated as
	 * an DM device.  Otherwise, check with the MD kernel driver.
	 */
	rc = EngFncs->dm_update_status(region);
	if (!rc && (region->flags & SOFLAG_ACTIVE)) {
		LOG_DEBUG("Region %s is an active DM device (%d:%d)\n",
			region->name, region->dev_major, region->dev_minor);
	} else {
		rc = 0;
		region->dev_major = MD_MAJOR;
		region->dev_minor = vol->md_minor;
		md_get_kernel_info(region, &info);
	}


out:
	if (vol->flags & MD_CORRUPT) {
		region->flags |= SOFLAG_CORRUPT;
	}

	if (region) {
		md_add_object_to_list(region, output_list);
	}
	
	LOG_EXIT_INT(rc);
	return rc;
}

int raid0_discover_regions( list_anchor_t output_list, int *count, boolean final_call )
{
	int rc = 0;
	md_volume_t * volume = volume_list_head;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	while (volume != NULL) {
		if ((!(volume->flags & MD_DISCOVERED)) && (volume->personality == RAID0)) {
		       	rc = raid0_create_region(volume, output_list, final_call);
		       	if (volume->flags & MD_DISCOVERED) {
		       		*count = *count + 1;
		       	}
		}
		volume = volume->next;
	}
	if (final_call) {
		md_display_corrupt_messages(RAID0);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid0_discover
 *
 *  Examine all disk segments and find MD PVs. Assemble volume groups
 *  and export all MD logical volumes as EVMS regions.
 *
 *  All newly created regions must be added to the output list, and all
 *  segments from the input list must either be claimed or moved to the
 *  output list.
 */
static int raid0_discover( list_anchor_t input_list,
			   list_anchor_t output_list,
			   boolean final_call ) {
	int count = 0;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	// Parameter check
	if (!input_list || !output_list) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (final_call) {
		md_discover_final_call(input_list, output_list, &count);
	} else {
		md_discover_volumes(input_list, output_list);
		LOG_DETAILS("PV discovery complete.\n");

		// LV discovery and exporting
		raid0_discover_regions(output_list, &count, final_call);
		LOG_DETAILS("RAID0 volume discovery complete.\n");
	}

	LOG_EXIT_INT(count);
	return count;
}



/****** Region Functions ******/


static int raid0_get_create_options( option_array_t *options, u_int32_t *chunksize, md_sb_ver_t *sb_ver)
{
	int i;
	int rc = 0;
	boolean ver1_superblock = FALSE;

	LOG_ENTRY();

	for (i = 0; i < options->count; i++) {

		if (options->option[i].is_number_based) {

			switch (options->option[i].number) {
			case RAID0_CREATE_OPT_SB1_INDEX:
				ver1_superblock = options->option[i].value.b;
				break;
			
			case RAID0_CREATE_OPT_CHUNK_SIZE_INDEX:
				// Option is in 1K byte blocks
				*chunksize = 2 * options->option[i].value.ui32;
				break;

			default:
				break;
			}

		} else {

			if (strcmp(options->option[i].name, RAID0_CREATE_OPT_CHUNK_SIZE_NAME) == 0) {
				// Option is in 1K byte blocks
				*chunksize = 2 * options->option[i].value.ui32;
			} else if (strcmp(options->option[i].name, RAID0_CREATE_OPT_SB1_NAME) == 0){
				ver1_superblock =  options->option[i].value.b;
			}
		}
	}

	if (ver1_superblock == TRUE) {
		sb_ver->major_version = MD_SB_VER_1;
		sb_ver->minor_version = 0;
		sb_ver->patchlevel = 0;
	} else {
		sb_ver->major_version = MD_SB_VER_0;
		sb_ver->minor_version = 90;
		sb_ver->patchlevel = 0;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int raid0_create_new_region(md_volume_t * vol, list_anchor_t output_list)
{
	int rc=0;
	md_member_t *member;
	list_element_t iter;
	storage_object_t *region = NULL;

	LOG_ENTRY();

	rc = EngFncs->allocate_region(vol->name, &region);
	if (rc) {
		LOG_ERROR("Region %s is already created (rc=%d).\n",
			  vol->name, rc);
	}

	LOG_DEBUG("Creating new region %s: nr_disks=%d, raid_disks=%d, spares=%d, actives=%d, working=%d\n",
		  vol->name, vol->nr_disks, vol->raid_disks, vol->spare_disks, vol->active_disks, vol->working_disks);
	
	if (!rc) {
		vol->private_data = EngFncs->engine_alloc(sizeof (raid0_conf_t));
		if (!vol->private_data) {
			rc = ENOMEM;
			goto error_out;
		}
		rc = create_strip_zones(vol);
		if (!rc) {
			LIST_FOR_EACH(vol->members, iter, member) {
				md_append_region_to_object(region, member->obj);
			}
			region->size = md_volume_calc_size(vol);
			region->data_type = DATA_TYPE;
			region->plugin = raid0_plugin;
			region->private_data = (void *)vol;
			region->dev_major = MD_MAJOR;
			region->dev_minor = vol->md_minor;
			vol->region = region;
			region->flags |= SOFLAG_DIRTY;
			md_add_object_to_list(region, output_list);
		}
	}

error_out:
	if (rc) {
		if (region){
			EngFncs->free_region(region);
		}
		if (vol->private_data) {
			EngFncs->engine_free(vol->private_data);
			vol->private_data = NULL;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid0_create
 *
 *  Create a new MD RAID0 volume using the specified objects and options.
 */
static int raid0_create( list_anchor_t    objects,
			 option_array_t * options,
			 list_anchor_t    new_region_list )
{
	md_volume_t * volume = NULL;
	storage_object_t * object;
	unsigned long size = -1;
	list_element_t iter1, iter2;
	md_sb_ver_t sb_ver = {MD_SB_VER_0, 90, 0};
	md_member_t *member;
	u_int32_t chunksize;
	int rc = 0;

	my_plugin = raid1_plugin;
	LOG_ENTRY();

	// Parameter check
	if (!objects || !options || !new_region_list) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	// Must have at least 1 disk
	if (EngFncs->list_count(objects) < 1) {
		LOG_CRITICAL("Must have at least 1 object.\n");
		rc = EINVAL;
		goto out;
	}

	if (!(volume = md_allocate_volume())) {
		rc = ENOMEM;
		goto out;
	}

	rc = md_volume_get_available_name(volume, 256);
	if (rc) {
		goto error_free;
	}

	raid0_get_create_options(options, &chunksize, &sb_ver);

	LIST_FOR_EACH(objects, iter1, object) {
		size = min(size, md_object_usable_size(object, &sb_ver, chunksize));
	}

	rc = md_init_sb(volume, &sb_ver, MD_LEVEL_RAID0, 0, size, chunksize);
	if (rc) {
		goto error_free;
	}
	// Add raid members
	LIST_FOR_EACH_SAFE(objects, iter1, iter2, object) {
		member = md_allocate_member(object);
		if (member) {
			// This will add the member and update the MD superblock.
			member->data_size = md_object_usable_size(object, &sb_ver, chunksize);
			member->flags |= (MD_MEMBER_NEW | MD_MEMBER_DISK_ACTIVE | MD_MEMBER_DISK_SYNC);
			rc = md_volume_add_new_member(volume, member);
			if (rc) {
				md_free_member(member);
				goto error_free;
			}
		} else {
			rc = ENOMEM;
		}
		if (rc) {
			goto error_free;
		}
		EngFncs->delete_element(iter1);
	}

	rc = raid0_create_new_region(volume, new_region_list);
	if (!rc) {
		volume->flags |= MD_DIRTY;
	}
error_free:
	if (rc && volume) {
		md_free_volume(volume);
	}
out:
	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: w_delete
 *
 * Worker function for raid0_delete and raid0_discard
 *
 */
static int w_delete(storage_object_t *region, list_anchor_t children, boolean tear_down)
{
	md_volume_t * volume;
	int rc = 0;

	LOG_ENTRY();

	// Check that this region can be removed.
	if ((rc = raid0_can_delete(region))) {
		LOG_EXIT_INT(rc);
		return rc;
	}
	volume = region->private_data;

	// Remove the parent/child associations with the PVs
	md_clear_child_list(region, children);

	//Delete raid0 private_data
	raid0_free_private_data(volume);

	// Delete the volume.
	md_delete_volume(volume, tear_down);
	EngFncs->free_region(region);

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid0_delete
 *
 *  Remove the specified region.
 */
static int raid0_delete(storage_object_t * region, list_anchor_t children)
{
	int           rc;

	my_plugin = raid0_plugin;
	LOG_ENTRY();
	rc = w_delete(region, children, TRUE);
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Function: raid0_discard
 *
 * This function is similar to delete.  Just call delete to free all
 * data structures related to the regions.
 */
static int raid0_discard(list_anchor_t regions)
{
	storage_object_t * region;
	list_element_t le;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	LIST_FOR_EACH(regions, le, region) {
		w_delete(region, NULL, FALSE);
	}

	LOG_EXIT_INT(0);
	return 0;
}

static int add_active_disk(md_volume_t *vol, storage_object_t *new_disk)
{
	md_member_t *member = NULL;
	md_super_info_t info;
	int rc=0;

	LOG_ENTRY();

	md_volume_get_super_info(vol, &info);
	if (vol->nr_disks != info.nr_disks) {
		LOG_MD_BUG();
		rc = EINVAL;
		goto out;
	}
                                                                               
	member = md_allocate_member(new_disk);
	if (!member) {
		rc = ENOMEM;
		goto out;
	}

	rc = md_volume_find_empty_slot(vol, &member->dev_number);
	if (rc) {
		goto out;
	}
	
	member->data_size = md_object_usable_size(new_disk, &vol->sb_ver, vol->chunksize);
	member->flags |= (MD_MEMBER_NEW | MD_MEMBER_DISK_ACTIVE | MD_MEMBER_DISK_SYNC);
	rc = md_volume_add_new_member(vol, member);
	if (rc) {
		goto out;
	}

	md_append_region_to_object(vol->region, new_disk);

out:
	if (rc && member) {
		md_free_member(member);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

static int remove_active_disk(md_volume_t *vol, storage_object_t *child)
{
	int rc = 0;
	md_member_t *member;
	list_element_t iter;
	boolean found = FALSE;

	LOG_ENTRY();

	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj == child) {
			found = TRUE;
			break;
		}
	}
	
	if (found == FALSE) {
		LOG_MD_BUG();
		rc = EINVAL;
		goto out;
	}

	rc = md_volume_remove_member(member, TRUE);
	if (!rc) {
		md_free_member(member);
		/*
		 * Just in case of error, hold on to the child for now.
		 * Don't unbind the child from the MD region.
		 */
	}
out:
	LOG_EXIT_INT(rc);
	return (rc);
	
}

/* Function: raid0_expand
 */
static int raid0_expand( storage_object_t    * region,
                         storage_object_t    * expand_object,
                         list_anchor_t         input_objects,
                         option_array_t      * options )
{
	int rc = 0;
	list_element_t iter;
	list_element_t li = NULL;
	storage_object_t *obj;
	md_volume_t *org_vol = (md_volume_t *)region->private_data;
	md_volume_t *new_vol = NULL;
	logical_volume_t *evms_volume;
	u_int64_t add_size = 0;
	md_member_t *member;

	my_plugin = raid0_plugin;
	
	LOG_ENTRY();

	/* Don't allow expanding if volume is mounted */
	if (EngFncs->is_offline(region, &evms_volume) == FALSE) {
		LOG_WARNING("Hmm... %s is mounted.\n", evms_volume->name);
		LOG_EXIT_INT(EINVAL);
		return(EINVAL);
	}

	/* Ask the engine if it's ok to add these objects. */
	LIST_FOR_EACH(input_objects, iter, obj) {
		add_size += obj->size;
	}
	rc = EngFncs->can_expand_by(region, &add_size);
	if (rc) {
		LOG_ERROR("Expand of region %s rejectd by the engine.\n",
			  region->name);
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* Make a copy of the orignal MD volume */
	new_vol = md_clone_volume(org_vol);
	if (!new_vol) {
		rc = ENOMEM;
		goto out;
	}

	li = EngFncs->insert_thing(raid0_expand_shrink_list, org_vol, INSERT_AFTER, NULL);
	if (!li) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	LIST_FOR_EACH(input_objects, iter, obj) {
		rc = add_active_disk(new_vol, obj);
		if (rc) {
			goto out;
		}
	}

	/* Build a new raid0_conf */
	new_vol->private_data = EngFncs->engine_alloc(sizeof (raid0_conf_t));
	if (!new_vol->private_data) {
		rc = ENOMEM;
		goto out;
	}
	rc = create_strip_zones(new_vol);
	if (!rc) {
		/* recalculate size */
		new_vol->flags |= MD_NEEDS_UPDATE_SIZE;
		region->private_data = new_vol;
		region->size = md_volume_calc_size(new_vol);

		region->flags |= SOFLAG_DIRTY;
		if (region->flags & SOFLAG_ACTIVE)
			region->flags |= (SOFLAG_NEEDS_DEACTIVATE | SOFLAG_NEEDS_ACTIVATE);
		new_vol->region_mgr_flags |= MD_RAID0_EXPAND_PENDING;
	}

out:
	if (rc) {

		if (new_vol) {
			/* Error, unwinding... */
	
			/* For each of the added members
			 *    (ie. it's not in the orginal volume)
			 * - remove the object from the region
			 * - delete the member from the volume member list
			 * - free the member
			 */
	
			LIST_FOR_EACH(new_vol->members, iter, member) {
				if (!md_volume_find_object(org_vol, member->obj)) {
					md_remove_region_from_object(region, member->obj);
				}
			}

			if (new_vol->private_data) {
				raid0_free_private_data(new_vol);
			}

			md_free_volume(new_vol);
		}

		region->private_data = org_vol;
		region->size = md_volume_calc_size(org_vol);
		if (li) {
			EngFncs->delete_element(li);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Function: raid0_unwind_expansion
 *	This function should be called during RAID0 discovery time to set up the "original"
 * md region.
 *	
 */
int raid0_unwind_expansion( storage_object_t *region)
{
	md_volume_t *new_vol = NULL;
	md_volume_t *volume = (md_volume_t *)region->private_data;
	md_member_t *member;
	md_member_t *saved_member = NULL;
	md_saved_info_t *info;
	list_element_t li = NULL;
	list_element_t iter;
	int i, rc = 0;
	list_anchor_t remove_list = NULL;
	storage_object_t *obj;

	LOG_ENTRY();
	
	/* Make a copy of the orignal MD volume */
	new_vol = md_clone_volume(volume);
	if (!new_vol) {
		rc = ENOMEM;
		goto out;
	}

	li = EngFncs->insert_thing(raid0_expand_shrink_list, volume, INSERT_AFTER, NULL);

	/*
	 * Find out which child object's saved area
	 * has the interrupted expansion information.
	 */
	if (md_check_for_expand_shrink_in_progress(volume, &saved_member)) {
		info = saved_member->saved_info;
	} else {
		LOG_MD_BUG();
		rc = EINVAL;
		goto out;
	}

	/*
	 * Check the saved expand info for the new objects that were added to the region.
	 * Put these objects in a list to remove.
	 */
	remove_list = EngFncs->allocate_list();
	if (!remove_list) {
		rc = ENOMEM;
		goto out;
	}

	for (i=0; !rc && i<info->expand_shrink_cnt; i++) {
		int idx = info->expand_shrink_devs[i];
		member = md_volume_find_member(volume, idx);
		if (member) {
			li = EngFncs->insert_thing(remove_list, member->obj, INSERT_AFTER, NULL);
			if (!li) {
				rc = ENOMEM;
			}
		} else {
			LOG_MD_BUG();
			rc = EINVAL;
		}
	}
	
	if (rc) {
		goto out;
	}
	
	if (EngFncs->list_count(remove_list) == 0) {
		LOG_MD_BUG();
		rc = EINVAL;
		goto out;
	}
	
	/* Remove all objects in the list from the region */
	LIST_FOR_EACH(remove_list, iter, obj) {
		rc = remove_active_disk(new_vol, obj);
		if (rc) {
			goto out;
		}
	}

	/* Build a new raid0_conf */
	new_vol->private_data = EngFncs->engine_alloc(sizeof (raid0_conf_t));
	if (!new_vol->private_data) {
		rc = ENOMEM;
		goto out;
	}
	
	rc = create_strip_zones(new_vol);
	if (!rc) {
		/* recalculate size */
		new_vol->flags |= MD_NEEDS_UPDATE_SIZE;
		region->private_data = new_vol;
		region->size = md_volume_calc_size(new_vol);
		region->flags |= SOFLAG_DIRTY;
		if (region->flags & SOFLAG_ACTIVE)
			region->flags |= (SOFLAG_NEEDS_DEACTIVATE | SOFLAG_NEEDS_ACTIVATE);
		new_vol->region_mgr_flags |= MD_RAID0_UNWIND_EXPANSION_PENDING;
	}

	if (!rc) {
		LOG_DEBUG("%s: expanded size: %"PRIu64", original size: %"PRIu64".\n",
			  region->name, md_volume_calc_size(volume), region->size);
	}

out:
	if (rc) {
		if (new_vol) {
			if (new_vol->private_data) {
				raid0_free_private_data(new_vol);
			}
			md_free_volume(new_vol);
		}
		if (li) {
			EngFncs->delete_element(li);
		}
		region->private_data = volume;
		region->size = md_volume_calc_size(volume);
		volume->flags |= MD_CORRUPT;
		region->flags |= SOFLAG_CORRUPT;		
	}

	if (remove_list) {
		EngFncs->destroy_list(remove_list);
	}

	LOG_EXIT_INT(rc);
	return rc;
}



/* Function: raid0_shrink
 */
static int raid0_shrink( storage_object_t    * region,
                         storage_object_t    * shrink_object,
                         list_anchor_t         input_objects,
                         option_array_t      * options )
{
	int rc = 0;
	list_element_t iter;
	list_element_t li = NULL;
	storage_object_t *obj;
	md_volume_t *org_vol = (md_volume_t *)region->private_data;
	md_volume_t *new_vol = NULL;
	logical_volume_t *evms_volume;
	u_int64_t shrink_size = 0;
	md_member_t *member;

	my_plugin = raid0_plugin;
	
	LOG_ENTRY();

	/* Don't allow shrinking if volume is mounted */
	if (EngFncs->is_offline(region, &evms_volume) == FALSE) {
		LOG_WARNING("Hmm... %s is mounted.\n", evms_volume->name);
		LOG_EXIT_INT(EINVAL);
		return(EINVAL);
	}

	/* Ask the engine if it's ok to remove these objects. */
	LIST_FOR_EACH(input_objects, iter, obj) {
		member = md_volume_find_object(org_vol, obj);
		if (!member) {
			LOG_ERROR("object %s is not found in region %s.\n",
				  obj->name, org_vol->name);
			LOG_EXIT_INT(EINVAL);
			return(EINVAL);
		}
		shrink_size += member->data_size;
	}
	rc = EngFncs->can_shrink_by(region, &shrink_size);
	if (rc) {
		LOG_ERROR("Shrink of region %s rejected by the engine.\n",
			  region->name);
		LOG_EXIT_INT(rc);
		return rc;
	}

	/* Make a copy of the orignal MD volume */
	new_vol = md_clone_volume(org_vol);
	if (!new_vol) {
		rc = ENOMEM;
		goto out;
	}

	li = EngFncs->insert_thing(raid0_expand_shrink_list, org_vol, INSERT_AFTER, NULL);
	if (!li) {
		rc = ENOMEM;
		goto out;
	}

	LIST_FOR_EACH(input_objects, iter, obj) {
		rc = remove_active_disk(new_vol, obj);
		if (rc) {
			goto out;
		}
	}

	/* Build a new raid0_conf */
	new_vol->private_data = EngFncs->engine_alloc(sizeof (raid0_conf_t));
	if (!new_vol->private_data) {
		rc = ENOMEM;
		goto out;
	}
	rc = create_strip_zones(new_vol);
	if (!rc) {
		/* recalculate size */
		new_vol->flags |= MD_NEEDS_UPDATE_SIZE;
		region->private_data = new_vol;
		region->size = md_volume_calc_size(new_vol);

		region->flags |= SOFLAG_DIRTY;
		if (region->flags & SOFLAG_ACTIVE)
			region->flags |= (SOFLAG_NEEDS_DEACTIVATE | SOFLAG_NEEDS_ACTIVATE);
		new_vol->region_mgr_flags |= MD_RAID0_SHRINK_PENDING;
	}
out:
	if (rc) {
		if (new_vol) {
			if (new_vol->private_data) {
				raid0_free_private_data(new_vol);
			}
			md_free_volume(new_vol);
		}
		region->private_data = org_vol;
		region->size = md_volume_calc_size(org_vol);
		if (li) {
			EngFncs->delete_element(li);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Function: raid0_resume_shrinking
 *	This function should be called during RAID0 discovery time to set up for
 * continuation of the interruped shrinking.
 */
int raid0_resume_shrinking( storage_object_t *region)
{
	md_volume_t *new_vol = NULL;
	md_volume_t *volume = (md_volume_t *)region->private_data;
	md_member_t *saved_member = NULL;
	md_saved_info_t *info = NULL;
	list_element_t li = NULL;
	list_element_t iter;
	int i, rc = 0;
	list_anchor_t remove_list = NULL;
	storage_object_t *obj;
	md_member_t *member;

	LOG_ENTRY();
	
	/* Make a copy of the orignal MD volume */
	new_vol = md_clone_volume(volume);
	if (!new_vol) {
		rc = ENOMEM;
		goto out;
	}
	
	li = EngFncs->insert_thing(raid0_expand_shrink_list, volume, INSERT_AFTER, NULL);
	if (!li) {
		rc = ENOMEM;
		goto out;
	}

	/*
	 * Find out which child object's saved area
	 * has the interrupted SHRINK information.
	 */
	if (md_check_for_expand_shrink_in_progress(volume, &saved_member)) {
		info = saved_member->saved_info;
	} else {
		LOG_MD_BUG();
		rc = EINVAL;
		goto out;
	}

	/*
	 * Check the saved shrink info for the objects that were removed
	 * from the region.  Put these objects in a list to remove.
	 */
	remove_list = EngFncs->allocate_list();
	for (i=0; !rc && i<info->expand_shrink_cnt; i++) {
		int idx = info->expand_shrink_devs[i];
		member = md_volume_find_member(volume, idx);
		if (member) {
			li = EngFncs->insert_thing(remove_list, member->obj, INSERT_AFTER, NULL);
			if (!li) {
				rc = ENOMEM;
			}
		} else {
			LOG_MD_BUG();
			rc = EINVAL;
		}
	}
	
	if (rc) {
		goto out;
	}
	
	if (EngFncs->list_count(remove_list) == 0) {
		LOG_MD_BUG();
		rc = EINVAL;
		goto out;
	}
	
	/* Remove all objects in the list from the region */
	LIST_FOR_EACH(remove_list, iter, obj) {
		rc = remove_active_disk(new_vol, obj);
		if (rc) {
			goto out;
		}
	}

	/* Build a new raid0_conf */
	new_vol->private_data = EngFncs->engine_alloc(sizeof (raid0_conf_t));
	if (!new_vol->private_data) {
		rc = ENOMEM;
		goto out;
	}

	rc = create_strip_zones(new_vol);
	if (!rc) {
		/* recalculate size */
		new_vol->flags |= MD_NEEDS_UPDATE_SIZE;
		region->private_data = new_vol;
		region->size = md_volume_calc_size(new_vol);
		region->flags |= SOFLAG_DIRTY;
		if (region->flags & SOFLAG_ACTIVE)
			region->flags |= (SOFLAG_NEEDS_DEACTIVATE | SOFLAG_NEEDS_ACTIVATE);
		new_vol->region_mgr_flags |= MD_RAID0_RESUME_SHRINKING_PENDING;
	}

	if (!rc) {
		LOG_DEBUG("%s: shrunk size: %"PRIu64", original size: %"PRIu64".\n",
			  region->name, region->size, md_volume_calc_size(volume));
	}

out:
	if (rc) {
		if (new_vol) {
			if (new_vol->private_data) {
				raid0_free_private_data(new_vol);
			}
			md_free_volume(new_vol);
		}
		
		if (li) {
			EngFncs->delete_element(li);
		}

		region->private_data = volume;
		region->size = md_volume_calc_size(volume);

		volume->flags |= MD_CORRUPT;
		region->flags |= SOFLAG_CORRUPT;		
	}


	if (remove_list) {
		EngFncs->destroy_list(remove_list);
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int get_child_run( md_volume_t       * volume,
			  lsn_t               lsn,
			  sector_count_t      count,
			  md_member_t      ** child,
			  lsn_t             * child_lsn,
			  sector_count_t    * child_count) {

	unsigned int sect_in_chunk;
	unsigned int chunksize_bits;
	raid0_conf_t *conf = mdvol_to_conf(volume);
	struct raid0_hash * hash;
	struct strip_zone * zone;
	unsigned long chunk;

	LOG_ENTRY();

	chunksize_bits = calc_log2(conf->chunksize);

	/* Sanity checks */
	if (!conf->hash_table || !conf->smallest_zone) {
		LOG_WARNING("Uninitialized raid0 configuration for %s\n",
			    volume->name);
		LOG_EXIT_INT(EIO);
		return EIO;
	}

	hash = conf->hash_table + (lsn / conf->smallest_zone->size);
	if (!hash) {
		LOG_WARNING("hash == NULL for lsn %"PRIu64"\n", lsn);
		LOG_EXIT_INT(EIO);
		return EIO;
	}

	if (!hash->zone0) {
		LOG_WARNING("hash->zone0 == NULL for lsn %"PRIu64"\n", lsn);
		LOG_EXIT_INT(EIO);
		return EIO;
	}

	if (lsn >= (hash->zone0->size + hash->zone0->zone_offset)) {
		if (!hash->zone1) {
			LOG_WARNING("hash->zone1 == NULL for lsn %"PRIu64"\n", lsn);
			LOG_EXIT_INT(EIO);
			return EIO;
		}
		zone = hash->zone1;
	} else {
		zone = hash->zone0;
	}

	sect_in_chunk = lsn & (conf->chunksize - 1);
	chunk = (lsn - zone->zone_offset) / (zone->nb_dev << chunksize_bits);
	*child = zone->dev[(lsn >> chunksize_bits) % zone->nb_dev];
	*child_lsn = ((chunk << chunksize_bits) + zone->dev_offset) + sect_in_chunk;
	*child_count = min(count, conf->chunksize - sect_in_chunk);

	LOG_EXIT_INT(0);
	return 0;
}

/* Function: raid0_volume_rw
 *
 *  Perform a logical-to-physical remapping, and send the read/write
 *  down to the next plugin.
 */
static int raid0_volume_rw(
	md_volume_t * volume,
	lsn_t lsn,
	sector_count_t count,
	void * buffer,
	int rw )
{
	int rc=0;
	md_member_t *member;
	lsn_t child_lsn;
	sector_count_t child_count;
	
	//LOG_ENTRY();
	while ((count != 0) && (rc == 0)) {
		rc = get_child_run(volume, lsn, count, &member, &child_lsn, &child_count);
		if (rc == 0) {
			if (rw == 1) {
				rc = WRITE(member->obj, child_lsn + member->data_offset, child_count, buffer);
			} else {
				rc = READ(member->obj, child_lsn + member->data_offset, child_count, buffer);
			}
			lsn += child_count;
			count -= child_count;
			buffer += child_count << EVMS_VSECTOR_SIZE_SHIFT;
		}
	}
	//LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Function: raid0_copy_data
 *
 *	Copy data from source volume to target volume.
 *
 * PARAMETERS:
 *	src : source volume
 *	target : target volume
 *	sectors (IN/OUT) :
 *		On input, number of sectors to copy
 *		On ouput, the number of sectors successfully copied
 *	forward : copy direction
 *		TRUE, forward copy start fron LSN 0
 *		FALSE, backward copy
 *	show_progress : TRUE, show progress indicator
 *	message : message to be displayed to user
 *
 * RETURN:
 *	ENOMEM - not enough memory to perform I/O
 *	other - from READ/WRITE function
 */
static int raid0_copy_data(
	md_volume_t *src,
	md_volume_t *target,
	sector_count_t *sectors,
	boolean forward,
	boolean show_progress,
	char *message)
{
	int rc=0;
	lsn_t lsn;
	sector_count_t transfer_sects;
	char *buf = NULL;
	progress_t progress;
	u_int32_t buf_size;
	md_member_t *saved_member = NULL;
	md_saved_info_t *info;
	raid0_conf_t * conf = mdvol_to_conf(src);

	LOG_ENTRY();

	LOG_DEFAULT("Region: %s, nr_disks (src:%d, target:%d),"
		    " sectors: %"PRIu64", copy direction: %s\n",
		    src->name, src->nr_disks, target->nr_disks,
		    *sectors, forward ? "FORWARD" : "BACKWARD");

	memset(&progress, 0, sizeof(progress_t));
	progress.total_count = *sectors;

	buf_size = conf->chunksize << EVMS_VSECTOR_SIZE_SHIFT;

	while (buf == NULL) {
		/* Allocate a buffer that is aligned on a 4KB boundary. */
		buf = memalign(4096, buf_size);
		if ((buf==NULL) && (buf_size > 4096)) {
			buf_size >>= 1;
		}
	}
	
	if (buf == NULL) {
		/* memalign() failed.  Don't give up, try malloc() */
		buf_size = conf->chunksize >> EVMS_VSECTOR_SIZE_SHIFT;
		while (buf == NULL) {
			buf = malloc(buf_size);
			if ((buf==NULL) && (buf_size > 4096)) {
				buf_size >>= 1;
			}
		}
	}

	if (buf == NULL) {
		*sectors = 0;
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	transfer_sects = buf_size >> EVMS_VSECTOR_SIZE_SHIFT;

	if (show_progress == TRUE) {
		progress.title = message;
		progress.description = "Transferring data, please wait...";
		progress.type = DISPLAY_PERCENT;
		EngFncs->progress(&progress);
	}

	/*
	 * Find out which child object's saved area has the
	 * EXPAND_IN_PROGRESS / SHRINK_IN_PROGRESS set, use that child object
	 * to keep track of copy progress.
	 */
	if (md_check_for_expand_shrink_in_progress(target, &saved_member)) {
		info = saved_member->saved_info;
	} else {
		if (md_check_for_expand_shrink_in_progress(src, &saved_member)) {
			info = saved_member->saved_info;
		} else {
			LOG_ERROR("Can't keep track of copy progress.\n");
			LOG_EXIT_INT(EINVAL);
			return EINVAL;
		}
	}
	
	if (forward) {
		lsn = 0;
	} else {
		lsn = progress.total_count;
	}

	while (progress.count < progress.total_count) {

		/* VERY IMPORTANT:
		 *  If buf_size > chunk_size, MUST enable the following code
		 */

		//if ((progress.count + transfer_sects) > progress.total_count) {
		//	transfer_sects = total_sects - progress.count;
		//}
		
		if (forward == TRUE) {
			lsn = progress.count;
		} else {
			lsn -= transfer_sects;
		}

		rc = raid0_volume_rw(src, lsn, transfer_sects, buf, 0);
		if (!rc) {
			rc = raid0_volume_rw(target, lsn, transfer_sects, buf, 1);
		}

		if (rc)
			break;

		progress.count += transfer_sects;

		/* update progress indicator */
		if (show_progress == TRUE) {
			EngFncs->progress(&progress);
		}

		if (forward)
			info->sector_mark = progress.count;
		else
			info->sector_mark = lsn;
		rc = md_write_saved_info(saved_member);
	}

	*sectors = progress.count;
	
	if (progress.count > progress.total_count) {
		LOG_WARNING("count=(%"PRIu64") is greater than total_count(%"PRIu64").\n",
			    progress.count, progress.total_count);
	}

	if ((show_progress == TRUE) && (progress.count < progress.total_count)) {
		/* close the progress indicator by setting count = total_count */
		progress.count = progress.total_count;
		EngFncs->progress(&progress);
	}

	free(buf);

	LOG_DEFAULT("Last LSN=%"PRIu64", used %"PRIu64"-sector blocks.\n", lsn, transfer_sects);

	LOG_EXIT_INT(rc);
	return rc;
}

static int raid0_do_kill_sector(storage_object_t *region)
{
	md_volume_t *volume;
	list_element_t iter1, iter2;
	raid0_delay_kill_sector_t *killsect;
	void *buffer;
	int current_buffer_size = 4096;
	int buffer_size_needed = 0;
	int rc=0;

	LOG_ENTRY();

	buffer = EngFncs->engine_alloc(current_buffer_size);
	if (!buffer) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	volume = (md_volume_t *)region->private_data;
	LIST_FOR_EACH_SAFE(raid0_delay_kill_sector_list, iter1, iter2, killsect) {
		if (killsect->region == region) {
			/* Grow our zero filled buffer if needed. */
			buffer_size_needed = killsect->count * EVMS_VSECTOR_SIZE;
			if (current_buffer_size < buffer_size_needed) {
				buffer = EngFncs->engine_realloc(buffer, buffer_size_needed);
				if (buffer != NULL) {
					current_buffer_size = buffer_size_needed;

				} else {
					LOG_CRITICAL("Error allocating memory for a zero filled"
						     " buffer for killing sectors.\n");
					rc = ENOMEM;
				}
			}

			/* Zap the sectors. */
			if (rc == 0) {
				rc = raid0_volume_rw(volume, killsect->lsn,
						     killsect->count, buffer, 1);
				if (rc == 0) {
					/*
					 * The sectors were killed.
					 * Remove the ksr from the list
					 * and free it.
					 */
					EngFncs->delete_element(iter1);
					EngFncs->engine_free(killsect);

				}
			}
		}
	}

	EngFncs->engine_free(buffer);

	LOG_EXIT_INT(rc);
	return rc;
}

static int raid0_do_expand(storage_object_t *region)
{
	int rc = 0;
	int rc2;
	list_element_t iter;
	md_volume_t *volume;
	md_volume_t *org_vol = NULL;
	boolean found = FALSE;
	sector_count_t sectors;
	md_saved_info_t *info = NULL;
	md_member_t *saved_member;
	md_member_t *member;

	LOG_ENTRY();

	volume = (md_volume_t *)region->private_data;

	LIST_FOR_EACH(raid0_expand_shrink_list, iter, org_vol) {
		if (org_vol->region == region) {
			found = TRUE;
			break;
		}
	}
	if (found == FALSE) {
		LOG_CRITICAL("Internal Error, could not find original volume to expand region %s.\n",
			     region->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	/* Create saved_info struct on the first disk to keep track of the expansion progress */
	info = EngFncs->engine_alloc(MD_SAVED_INFO_BYTES);
	if (!info) {
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}
	saved_member = EngFncs->first_thing(volume->members, NULL);
	saved_member->saved_info = info;

	/* Set EXPAND flag on the first disk */
	info->flags |= MD_SAVED_INFO_EXPAND_IN_PROGRESS;
	info->sector_mark = 0;
		
	/* Keep track of newly added objects */
	LIST_FOR_EACH(volume->members, iter, member) {
		if (!md_volume_find_object(org_vol, member->obj)) {
			info->expand_shrink_devs[info->expand_shrink_cnt] = member->dev_number;
			info->expand_shrink_cnt++;
		}
	}

	rc = md_write_saved_info(saved_member);
	if (rc) {
		LOG_CRITICAL("Failed to write info on expand progress for regions %s.\n",
			     region->name);
		goto error;
	}

	sectors = md_volume_calc_size(org_vol);
	
	sprintf(message_buffer, "Expanding RAID0 region %s...", region->name);
	LOG_DEBUG("%s (sectors=%"PRIu64")\n",message_buffer, sectors);
	
	rc = raid0_copy_data(org_vol, volume, &sectors, TRUE, TRUE, message_buffer);
	
	LOG_DEBUG("raid0_copy_data returned rc=%d, (sectors=%"PRIu64")\n", rc, sectors);
	
	/* Clear EXPAND flag  */
	info->flags &= ~MD_SAVED_INFO_EXPAND_IN_PROGRESS;
	rc2 = md_write_saved_info(saved_member);
	if (rc2) {
		LOG_CRITICAL("Failed to update info on expand progress for regions %s.\n",
			     region->name);
	}

	switch (rc) {
	case 0:
		rc = raid0_do_kill_sector(region);
		raid0_free_private_data(org_vol);
		EngFncs->remove_thing(raid0_expand_shrink_list, org_vol);
		md_free_volume(org_vol);
		volume->region_mgr_flags &= ~MD_RAID0_EXPAND_PENDING;
		goto out;
		break;
	default:
		/* Error, unwinding... */
		if (sectors) {
			sprintf(message_buffer, "RAID0 region %s failed to expand, restoring data...", region->name);
			rc2 = raid0_copy_data(volume, org_vol, &sectors, FALSE, TRUE, message_buffer);
			if (rc2) {
				LOG_CRITICAL("Error restoring data after expand failure.\n");
			}
		}

		break;
	}

error:
	if (rc && org_vol) {

		/* For each of the added members
		 *    (ie. it's not in the orginal volume)
		 * - remove the object from the region
		 * - delete the member from the volume member list
		 * - free the member
		 */

		LIST_FOR_EACH(volume->members, iter, member) {
			if (!md_volume_find_object(org_vol, member->obj)) {
				md_remove_region_from_object(region, member->obj);
			}
		}

		raid0_free_private_data(volume);
		md_free_volume(volume);
		region->size = md_volume_calc_size(org_vol);
		region->private_data = org_vol;
		EngFncs->remove_thing(raid0_expand_shrink_list, org_vol);
		org_vol->region_mgr_flags &= ~MD_RAID0_EXPAND_PENDING;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

static int raid0_do_unwind_expansion(storage_object_t *region)
{
	md_volume_t *volume;
	md_volume_t *exp_vol;
	list_element_t iter;
	sector_count_t sectors = 0;
	md_saved_info_t *info = NULL;
	md_member_t *saved_member;
	md_member_t *member;
	int rc = 0;
	boolean found = FALSE;

	LOG_ENTRY();

	volume = (md_volume_t *)region->private_data;

	LIST_FOR_EACH(raid0_expand_shrink_list, iter, exp_vol) {
		if (exp_vol->region == region) {
			found = TRUE;
			break;
		}
	}
	if (found == FALSE) {
		LOG_CRITICAL("Internal Error, Could not find original volume "
			     "to unwind the interrupred expansion of region %s.\n",
			     region->name);
		rc = EINVAL;
		goto error_free;
	}
	
	/* Get EXPAND info : sector_mark */
	if (md_check_for_expand_shrink_in_progress(volume, &saved_member)) {
		info = saved_member->saved_info;
		sectors = info->sector_mark;
	} else {
		LOG_CRITICAL("%s: Internal error: No expand info.\n", region->name);
		rc = EINVAL;
		goto error_free;
	}
	
	if (!rc && sectors) {
		sprintf(message_buffer, "RAID0 region %s failed to expand, restoring data...", region->name);
		rc = raid0_copy_data(exp_vol, volume, &sectors, FALSE, TRUE, message_buffer);
		if (rc) {
			LOG_CRITICAL("Error restoring data after expand failure.\n");
			goto error_free;
		}
	}
	
	/*
	 * Great!  Unwinding of incomplete expansion succeeded.
	 * - free the raid0 private data
	 * - free "expanded" volume which was built during discovery
	 */
	rc = md_zero_saved_info(saved_member, TRUE);
	if (rc) {
		goto error_free;
	}

	LIST_FOR_EACH(exp_vol->members, iter, member) {
		if (!md_volume_find_object(volume, member->obj)) {
			LOG_DEFAULT("Delete MD superblock on %s.\n", member->obj->name);
				/* zero out the superblock & saved superblock */
			md_zero_superblock(member, TRUE);
			md_remove_region_from_object(region, member->obj);
		}
	}

	EngFncs->remove_thing(raid0_expand_shrink_list, exp_vol);
	if (exp_vol->private_data)
		raid0_free_private_data(exp_vol);
	md_free_volume(exp_vol);
	volume->region_mgr_flags &= ~MD_RAID0_UNWIND_EXPANSION_PENDING;

	LOG_EXIT_INT(0);
	return 0;

error_free:
	
	EngFncs->remove_thing(raid0_expand_shrink_list, exp_vol);
	if (exp_vol->private_data)
		raid0_free_private_data(exp_vol);
	md_free_volume(exp_vol);

	volume->region_mgr_flags &= ~MD_RAID0_UNWIND_EXPANSION_PENDING;
	region->flags |= SOFLAG_CORRUPT;
	volume->flags |= MD_CORRUPT;
	
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * raid0_do_shrink
 *
 * This function is used to shrink or resume shrinking of MD region.
 */
static int raid0_do_shrink(storage_object_t *region)
{
	int rc = 0;
	int rc2;
	list_element_t iter;
	md_volume_t *volume;
	md_volume_t *org_vol;
	boolean found = FALSE;
	sector_count_t sectors;
	md_saved_info_t *info = NULL;
	md_member_t *saved_member;
	md_member_t *member;

	LOG_ENTRY();
	

	volume = (md_volume_t *)region->private_data;

	LIST_FOR_EACH(raid0_expand_shrink_list, iter, org_vol) {
		if (org_vol->region == region) {
			found = TRUE;
			break;
		}
	}
	if (found == FALSE) {
		LOG_CRITICAL("Internal Error, could not find original volume to shrink region %s.\n",
			     region->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	if (volume->region_mgr_flags & MD_RAID0_RESUME_SHRINKING_PENDING) {
		/* Find the disk that has the saved shrink info */
		if (md_check_for_expand_shrink_in_progress(volume, &saved_member)) {
			info = saved_member->saved_info;
		} else {
			LOG_MD_BUG();
			LOG_EXIT_INT(EINVAL);
			return EINVAL;
		}
	} else {
		/* Create saved_info struct on the first disk to keep track of the shrink progress */
		info = EngFncs->engine_alloc(MD_SAVED_INFO_BYTES);
		if (!info) {
			LOG_EXIT_INT(ENOMEM);
			return ENOMEM;
		}
		saved_member = EngFncs->first_thing(volume->members, NULL);
		saved_member->saved_info = info;
	}

	if (info->flags & MD_SAVED_INFO_SHRINK_IN_PROGRESS) {
		/* resume shrinking */
		sectors = info->sector_mark;
	} else {
		sectors = md_volume_calc_size(volume);
	}
	
	info->flags |= MD_SAVED_INFO_SHRINK_IN_PROGRESS;
	info->sector_mark = sectors;

	/*
	 * The original volume structure contains the orignal set of child objects.
	 * The new (shrunk) volume structure contains the new set of child objects.
	 * Set up the bit map to keep track of removed objects.
	 */
	info->expand_shrink_cnt = 0;
	LIST_FOR_EACH(org_vol->members, iter, member) {
		if (!md_volume_find_object(volume, member->obj)) {
			info->expand_shrink_devs[info->expand_shrink_cnt] = member->dev_number;
			info->expand_shrink_cnt++;
		}
	}
	
	rc = md_write_saved_info(saved_member);
	if (rc) {
		LOG_EXIT_INT(rc);
		return rc;
	}
	
	
	sprintf(message_buffer, "Shrinking RAID0 region %s...", region->name);
	LOG_DEBUG("%s (sectors=%"PRIu64")\n",message_buffer, sectors);
	
	rc = raid0_copy_data(org_vol, volume, &sectors, FALSE, TRUE, message_buffer);
	
	LOG_DEBUG("raid0_copy_data returned rc=%d, (sectors=%"PRIu64")\n", rc, sectors);
	
	/* Clear SHRINK flag  */
	info->flags &= ~MD_SAVED_INFO_SHRINK_IN_PROGRESS;
	rc2 = md_write_saved_info(saved_member);
	if (rc2) {
		LOG_EXIT_INT(rc2);
		return rc2;
	}

	switch (rc) {
	case 0:
		/* Delete superblocks */
		LIST_FOR_EACH(org_vol->members, iter, member) {
			if (!md_volume_find_object(volume, member->obj)) {
				md_remove_region_from_object(org_vol->region, member->obj);
				md_zero_superblock(member, TRUE);
				md_zero_saved_info(member, TRUE);
			}
		}
		EngFncs->remove_thing(raid0_expand_shrink_list, org_vol);
		raid0_free_private_data(org_vol);
		md_free_volume(org_vol);
		volume->region_mgr_flags &= ~MD_RAID0_SHRINK_PENDING;
		break;
	default:
		/* Error, unwinding... */
		if (sectors) {
			sprintf(message_buffer,
				"RAID0 region %s failed to shrink, restoring data...",
				region->name);
			rc2 = raid0_copy_data(volume, org_vol, &sectors, FALSE, TRUE, message_buffer);
			if (rc2) {
				LOG_CRITICAL("Error restoring data after shrink failure.\n");
			}
		}

		raid0_free_private_data(volume);
		md_free_volume(volume);
		region->size = md_volume_calc_size(org_vol);
		region->private_data = org_vol;
		EngFncs->remove_thing(raid0_expand_shrink_list, org_vol);
		org_vol->region_mgr_flags &= ~MD_RAID0_SHRINK_PENDING;
		break;
	}


	LOG_EXIT_INT(rc);
	return rc;
}

static int raid0_do_resume_shrinking(storage_object_t *region)
{
	int rc = 0;
	md_volume_t *volume;

	LOG_ENTRY();

	volume = (md_volume_t *)region->private_data;

	rc = raid0_do_shrink(region);
	if (rc) {
		volume = (md_volume_t *)region->private_data;
		volume->flags |= MD_CORRUPT;
		region->flags |= SOFLAG_CORRUPT;
	}
	
	volume->region_mgr_flags &= ~MD_RAID0_RESUME_SHRINKING_PENDING;

	LOG_EXIT_INT(rc);
	return rc;
}

static int raid0_replace_child(storage_object_t *region,
			       storage_object_t *child,
			       storage_object_t *new_child)
{
	int rc;
	my_plugin = raid0_plugin;
	LOG_ENTRY();
	rc = md_replace_child(region, child, new_child);
	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid0_add_sectors_to_kill_list
 *
 *  The kill_sectors list contains a list of sectors that need to be zeroed
 *  during the next commit. This function is very similar to read/write.
 */
static int raid0_add_sectors_to_kill_list( storage_object_t * region,
					   lsn_t              lsn,
					   sector_count_t     count ) {

	md_volume_t     *volume = (md_volume_t *)region->private_data;
	int             rc = 0;
	md_member_t *   member;
	lsn_t           child_lsn;
	sector_count_t  child_count;
	raid0_delay_kill_sector_t *killsect;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	if (volume->flags & MD_CORRUPT) {
		LOG_ERROR("MD Object %s is corrupt, data is suspect.\n",volume->name);
		LOG_EXIT_INT(EIO);
		return EIO;
	}

	if (volume->region_mgr_flags & MD_RAID0_EXPAND_PENDING) {
		killsect = EngFncs->engine_alloc(sizeof(raid0_delay_kill_sector_t));
		if (killsect) {
			killsect->region = region;
			killsect->lsn = lsn;
			killsect->count = count;
			EngFncs->insert_thing(raid0_delay_kill_sector_list,
					      killsect, INSERT_AFTER, NULL);
			LOG_EXIT_INT(0);
			return 0;
		}
		/* Note: If memory allocation fails, fall thru the next while loop */
	}

	while ((count != 0) && (rc == 0)) {
		rc = get_child_run(volume, lsn, count, &member, &child_lsn, &child_count);
		if (rc == 0) {
			rc = KILL_SECTORS(member->obj, child_lsn + member->data_offset, child_count);
			count -= child_count;
		}
	}
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * FUNCTION: raid0_commit_expand
 *
 * Steps to expand raid0 region:
 *	- update superblocks
 *	- expand the region (raid0_do_expand)
 *	- if expansion failed, rewrite superblocks
 */
static int raid0_commit_expand(storage_object_t *region)
{
	md_volume_t * volume = (md_volume_t *)region->private_data;
	logical_volume_t *evms_volume;
	md_volume_t * org_vol;
	boolean found = FALSE;
	list_element_t iter;
	md_member_t *member;
	int rc = 0;
	int rc2 = 0;

	LOG_ENTRY();
	
	/* Don't allow expansion if volume is mounted */
	if (EngFncs->is_offline(region, &evms_volume) == FALSE) {
		LOG_WARNING("Hmm... %s is mounted.\n", evms_volume->name);
		rc = EINVAL;
		goto out;
	}
	
	LIST_FOR_EACH(raid0_expand_shrink_list, iter, org_vol) {
		if (org_vol->region == region) {
			found = TRUE;
			break;
		}
	}
	if (found == FALSE) {
		LOG_CRITICAL("Internal Error,"
			     " could not find original volume to expand region %s.\n",
			     region->name);
		rc = EINVAL;
		goto out;
	}

	/* Assume success, new disks are now active and in sync */
	//raid_disk = volume->raid_disks;
	LIST_FOR_EACH(volume->members, iter, member) {
		if (!md_volume_find_object(org_vol, member->obj)) {
			//member->raid_disk = raid_disk;
			//raid_disk++;
			member->flags = (MD_MEMBER_DISK_ACTIVE | MD_MEMBER_DISK_SYNC);
			//volume->sb_func->set_device_state(member);
		}
	}

	volume->flags |= MD_DIRTY;
	volume->commit_flag |= MD_COMMIT_DONT_CHECK_ACTIVE;	
	rc = md_write_sbs_to_disk(volume);
	if (!rc) {
		rc = raid0_do_expand(region);
		if (rc) {
			/* If expansion failed, write back old superblocks */
			volume = (md_volume_t *)region->private_data;
			volume->flags |= MD_DIRTY;
			volume->commit_flag |= MD_COMMIT_DONT_CHECK_ACTIVE;
			rc2 = md_write_sbs_to_disk(volume);
		}
	}
out:
	region->flags &= ~SOFLAG_DIRTY;
	LOG_EXIT_INT(rc | rc2);
	return (rc | rc2);
}

/*
 * FUNCTION: raid0_commit_shrink
 *
 * Steps to shrink raid0 region:
 *	- shrink the region (raid0_do_shrink)
 *	- update superblocks
 */
static int raid0_commit_shrink(storage_object_t *region)
{
	md_volume_t * volume = (md_volume_t *)region->private_data;
	logical_volume_t *evms_volume;
	int rc = 0;
	
	LOG_ENTRY();
	
	/* Don't allow shrinking if volume is mounted */
	if (EngFncs->is_offline(region, &evms_volume) == FALSE) {
		region->flags &= ~SOFLAG_DIRTY;
		LOG_WARNING("Hmm... %s is mounted.\n", evms_volume->name);
		LOG_EXIT_INT(EINVAL);
		return(EINVAL);
	}
	
	rc = raid0_do_shrink(region);
	if (!rc) {
		/* update superblocks */
		volume->flags |= MD_DIRTY;
		volume->commit_flag |= MD_COMMIT_DONT_CHECK_ACTIVE;
		rc = md_write_sbs_to_disk(volume);
	}
	region->flags &= ~SOFLAG_DIRTY;
	LOG_EXIT_INT(rc);
	return (rc);
}

static int raid0_commit_unwind_expansion(storage_object_t *region)
{
	md_volume_t * volume = (md_volume_t *)region->private_data;
	int rc = 0;

	LOG_ENTRY();
	rc = raid0_do_unwind_expansion(region);
	if (!rc) {
		volume->flags |= MD_DIRTY;
		volume->commit_flag |= MD_COMMIT_DONT_CHECK_ACTIVE;
		rc = md_write_sbs_to_disk(volume);
	}

	region->flags &= ~SOFLAG_DIRTY;
	LOG_EXIT_INT(rc);
	return rc;
}

static int raid0_commit_resume_shrinking(storage_object_t *region)
{
	md_volume_t * volume = (md_volume_t *)region->private_data;
	int rc = 0;

	LOG_ENTRY();
	rc = raid0_do_resume_shrinking(region);
	if (!rc) {
		volume->flags |= MD_DIRTY;
		volume->commit_flag |= MD_COMMIT_DONT_CHECK_ACTIVE;
		rc = md_write_sbs_to_disk(volume);
	}

	region->flags &= ~SOFLAG_DIRTY;
	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid0_commit_changes
 *
 *  Newly created regions may have the ZERO flag set. In this case, clear
 *  the first 1k of the LV.
 *
 *  All other commit operations are done in commit_container_changes.
 */
static int raid0_commit_changes( storage_object_t * region,
				 uint               phase ) {

	md_volume_t * volume;
	int         rc = 0;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	if (!region ||  !(volume = (md_volume_t *)region->private_data)) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	// Make sure this region belongs to MD, and is dirty
	if (region->plugin != raid0_plugin) {
		LOG_ERROR("Region %s does not belong to MD.\n", region->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}
	if (!(region->flags & SOFLAG_DIRTY)) {
		LOG_WARNING("Region %s is not dirty - not committing.\n", region->name);
		LOG_EXIT_INT(0);
		return 0;
	}

	switch (phase) {
	case FIRST_METADATA_WRITE:
		if (volume->region_mgr_flags & MD_RAID0_EXPAND_PENDING) {
			rc = raid0_commit_expand(region);
			break;
		}
		if (volume->region_mgr_flags & MD_RAID0_UNWIND_EXPANSION_PENDING) {
			rc = raid0_commit_unwind_expansion(region);
			break;
		}
		if (volume->region_mgr_flags & MD_RAID0_SHRINK_PENDING) {
			/* Wait until SECOND_METADATA_WRITE */
			break;
		}
		if (volume->region_mgr_flags & MD_RAID0_RESUME_SHRINKING_PENDING) {
			rc = raid0_commit_resume_shrinking(region);
			break;
		}

		/* For error cases, region's private data (volume) could have been changed.
		 * Set volume pointer again
		 */

		volume = (md_volume_t *) region->private_data;

		volume->flags |= MD_DIRTY;
		volume->commit_flag |= MD_COMMIT_DONT_CHECK_ACTIVE;
		rc = md_write_sbs_to_disk(volume);   // write super blocks
		region->flags &= ~SOFLAG_DIRTY;	     // mark clean
		break;
	case SECOND_METADATA_WRITE:
		if (volume->region_mgr_flags & MD_RAID0_SHRINK_PENDING) {
			rc = raid0_commit_shrink(region);
			break;
		}

	default :
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int raid0_can_activate_region(storage_object_t * region)
{
	md_volume_t   * volume = (md_volume_t *)region->private_data;
	my_plugin = raid0_plugin;
	LOG_ENTRY();

	if (volume->flags & MD_CORRUPT) {
		LOG_WARNING("MD region %s is corrupt.\n", volume->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	LOG_EXIT_INT(0);
	return 0;
}

static int raid0_activate_region(storage_object_t * region)
{
	int rc = 0;
	md_volume_t *vol = (md_volume_t *)region->private_data;
	dm_target_t *targets = NULL, *target=NULL;
	dm_target_stripe_t *stripe = NULL;
	int i, j;
	raid0_conf_t *conf = mdvol_to_conf(vol);
	
	my_plugin = raid0_plugin;
	LOG_ENTRY();
	
	if (vol->flags & MD_CORRUPT) {
		LOG_WARNING("%s is corrupt.\n", vol->name);
		LOG_EXIT_INT(ENODEV);
		return ENODEV;
	}

	for (i=0; (!rc) && (i<conf->nr_strip_zones); i++) {
		struct strip_zone *zone = conf->strip_zone + i;
		
		LOG_DEBUG("%s(zone[%d]): zone_offset=%"PRIu64", dev_offset=%"PRIu64", size=%"PRIu64" nb_dev=%d\n",
			region->name, i, zone->zone_offset, zone->dev_offset, zone->size, zone->nb_dev);
		
		target = EngFncs->dm_allocate_target(
					DM_TARGET_STRIPE,
					zone->zone_offset,
					zone->size,
					zone->nb_dev, 0);
		if (target) {
			stripe = target->data.stripe;
			stripe->num_stripes = zone->nb_dev;
			stripe->chunk_size = conf->chunksize;
			for (j=0; (!rc) && j<stripe->num_stripes; j++) {
				lsn_t lsn;
				md_member_t *member;
				lsn_t child_lsn;
				sector_count_t child_count;
				
				lsn = zone->zone_offset + (j*stripe->chunk_size);
				
				rc = get_child_run(vol, lsn, 1, &member, &child_lsn, &child_count);
				if (!rc) {
					
					LOG_DEBUG("%s(zone[%d]): for lsn=%"PRIu64", %s was selected and I/O offset=%"PRIu64"\n",
						region->name, i, lsn, member->obj->name, child_lsn);
				
					stripe->devices[j].major = member->obj->dev_major;
					stripe->devices[j].minor = member->obj->dev_minor;
					stripe->devices[j].start = zone->dev_offset + member->data_offset;
				} else {
					LOG_ERROR("Could not find out which child to setup stripe target!\n");
				}
			}
		
		} else {
			rc = ENOMEM;
			break;
		}
		EngFncs->dm_add_target(target, &targets);
	}

	if (!rc) {
		rc = EngFncs->dm_activate(region, targets);
		if (!rc) {
			region->flags &= ~SOFLAG_NEEDS_ACTIVATE;
			LOG_DEBUG("Region %s has been activated, DM device(%d, %d)\n",
				  region->name, region->dev_major, region->dev_minor);
		}
	}

	if (targets) EngFncs->dm_deallocate_targets(targets); 	

	LOG_EXIT_INT(rc);
	return rc;

}

static int raid0_can_deactivate_region(storage_object_t * region)
{
	my_plugin = raid0_plugin;
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}

static int raid0_deactivate_region(storage_object_t * region)
{
	int rc;
	mdu_array_info_t info;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	rc = md_ioctl_get_array_info(region, &info);
	if (rc == 0) {
		/*
		 * This MD array was started by another tool.
		 * Stop the array via ioctl to the kernel MD driver.
		 */
		 rc = md_deactivate_region(region);
	} else {
		rc = EngFncs->dm_deactivate(region);
		if (!rc) {
			region->flags &= ~SOFLAG_NEEDS_DEACTIVATE;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid0_get_option_count
 *
 *  Determine the type of Task that is being performed, and return
 *  the number of options that are available for that Task.
 */
static int raid0_get_option_count( task_context_t * task ) {
	int count = 0;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	switch (task->action) {
	case EVMS_Task_Create:
		count = RAID0_CREATE_OPT_COUNT;
		break;

	case EVMS_Task_Expand:
		count = RAID0_EXPAND_OPTION_COUNT;
		break;

	case EVMS_Task_Shrink:
		count = RAID0_SHRINK_OPTION_COUNT;
		break;

	default:
		count = 0;
		break;
	}

	LOG_EXIT_INT(count);
	return count;
}


/* Function: raid0_init_task
 *
 *  Determine the type of Task that is being performed, and set up the
 *  context structure with the appropriate initial values.
 */
static int raid0_init_task( task_context_t * context ) {

	int rc = 0;
	list_anchor_t tmp_list;
	md_volume_t *vol;
	md_member_t *member;
	list_element_t iter;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	// Parameter check.
	if (!context) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	switch (context->action) {
	
	case EVMS_Task_Create:

		context->option_descriptors->count = RAID0_CREATE_OPT_COUNT;

		// Version 1 Superblock Option
		if (md_can_create_sb_1() == TRUE) {
			context->option_descriptors->option[RAID0_CREATE_OPT_SB1_INDEX].flags = 0;
			context->min_selected_objects = 1;
			context->max_selected_objects = MD_SB_1_DISKS;
		} else {
			context->option_descriptors->option[RAID0_CREATE_OPT_SB1_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
			context->min_selected_objects = 1;
			context->max_selected_objects = MD_SB_DISKS;
		}
		context->option_descriptors->option[RAID0_CREATE_OPT_SB1_INDEX].constraint.list = NULL;
		context->option_descriptors->option[RAID0_CREATE_OPT_SB1_INDEX].constraint_type = EVMS_Collection_None;
		context->option_descriptors->option[RAID0_CREATE_OPT_SB1_INDEX].help = NULL;
		context->option_descriptors->option[RAID0_CREATE_OPT_SB1_INDEX].name =
			EngFncs->engine_strdup( RAID0_CREATE_OPT_SB1_NAME );
		context->option_descriptors->option[RAID0_CREATE_OPT_SB1_INDEX].tip =
			EngFncs->engine_strdup( _("Choose Yes if you want to create MD version 1 super block.") );
		context->option_descriptors->option[RAID0_CREATE_OPT_SB1_INDEX].title = EngFncs->engine_strdup( _("Version 1 Super Block") );
		context->option_descriptors->option[RAID0_CREATE_OPT_SB1_INDEX].type = EVMS_Type_Boolean;
		context->option_descriptors->option[RAID0_CREATE_OPT_SB1_INDEX].unit = EVMS_Unit_None;
		context->option_descriptors->option[RAID0_CREATE_OPT_SB1_INDEX].value.b = FALSE;


		// Chunk Size Option
		context->option_descriptors->option[RAID0_CREATE_OPT_CHUNK_SIZE_INDEX].flags = 0;
		SET_POWER2_LIST(context->option_descriptors->option[RAID0_CREATE_OPT_CHUNK_SIZE_INDEX].constraint.list, MD_MIN_CHUNK_SIZE, MD_MAX_CHUNK_SIZE);
		context->option_descriptors->option[RAID0_CREATE_OPT_CHUNK_SIZE_INDEX].constraint_type = EVMS_Collection_List;
		context->option_descriptors->option[RAID0_CREATE_OPT_CHUNK_SIZE_INDEX].help = NULL;
		context->option_descriptors->option[RAID0_CREATE_OPT_CHUNK_SIZE_INDEX].name = EngFncs->engine_strdup(RAID0_CREATE_OPT_CHUNK_SIZE_NAME);
		context->option_descriptors->option[RAID0_CREATE_OPT_CHUNK_SIZE_INDEX].tip = EngFncs->engine_strdup(_("Size of IO to each member of the array (also refered to as stripe size)." ));
		context->option_descriptors->option[RAID0_CREATE_OPT_CHUNK_SIZE_INDEX].title = EngFncs->engine_strdup(_("Chunk Size:" ));
		context->option_descriptors->option[RAID0_CREATE_OPT_CHUNK_SIZE_INDEX].type = EVMS_Type_Unsigned_Int32;
		context->option_descriptors->option[RAID0_CREATE_OPT_CHUNK_SIZE_INDEX].unit = EVMS_Unit_Kilobytes;
		context->option_descriptors->option[RAID0_CREATE_OPT_CHUNK_SIZE_INDEX].value.ui32 = MD_DEFAULT_CHUNK_SIZE;

		// get a list of all valid input disks, segments, and regions.
		EngFncs->get_object_list(DISK | SEGMENT | REGION,
					 DATA_TYPE,
					 NULL,
					 NULL,
					 VALID_INPUT_OBJECT,
					 &tmp_list);

		// move these items to the acceptable objects list.
		md_transfer_list(tmp_list, context->acceptable_objects);
		EngFncs->destroy_list(tmp_list);

		break;

	case EVMS_Task_Expand:
		/* No options on expand */
		context->option_descriptors->count = RAID0_EXPAND_OPTION_COUNT;

		// get a list of all valid input disks, segments, and regions.
		EngFncs->get_object_list(DISK | SEGMENT | REGION,
					 DATA_TYPE,
					 NULL,
					 context->object->disk_group,
					 VALID_INPUT_OBJECT | NO_DISK_GROUP,
					 &tmp_list);



		// remove this region from the list
		EngFncs->remove_thing(tmp_list, context->object);

		/* Remove all parents of this MD region from acceptable list */
		remove_parent_regions_from_list(tmp_list, context->object);

		// move these items to the acceptable objects list.
		md_transfer_list(tmp_list, context->acceptable_objects);
		EngFncs->destroy_list(tmp_list);

		vol = (md_volume_t *) context->object->private_data;

		context->min_selected_objects = 1;
		context->max_selected_objects = MAX_DISKS(vol) - vol->nr_disks;
		break;

	case EVMS_Task_Shrink:
		/* No options on shrink */
		context->option_descriptors->count = RAID0_SHRINK_OPTION_COUNT;
		vol = (md_volume_t *) context->object->private_data;

		LIST_FOR_EACH(vol->members, iter, member) {
			EngFncs->insert_thing(context->acceptable_objects, member->obj, INSERT_AFTER, NULL);
		}

		context->min_selected_objects = 1;
		context->max_selected_objects = vol->nr_disks - 1;
		break;

	default:
		rc = EINVAL;
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid0_set_option
 *
 *  Determine the type of Task that is being performed. Then examine the
 *  desired option (using the index), and verify that the given value is
 *  appropriate. Reset the value if necessary and possible. Adjust other
 *  options as appropriate.
 */
static int raid0_set_option( task_context_t * context,
			     u_int32_t        index,
			     value_t        * value,
			     task_effect_t  * effect ) {
	int rc = 0;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	// Parameter check.
	if (!context || !value || !effect) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	switch (context->action) {
	
	case EVMS_Task_Create:
		switch (index) {
		case RAID0_CREATE_OPT_SB1_INDEX:
			context->option_descriptors->option[index].value.b = value->b;
			if (value->b == TRUE) {
				context->max_selected_objects = MD_SB_1_DISKS;
			} else {
				context->max_selected_objects = MD_SB_DISKS;
			}
			break;

		case RAID0_CREATE_OPT_CHUNK_SIZE_INDEX:
			// Option is in kilobytes
			context->option_descriptors->option[index].value.ui32 = value->ui32;
			break;

		default:
			break;
                }

	default:
		break;
	}
	LOG_EXIT_INT(rc);
	return rc;
}


static int raid0_set_expand_object( task_context_t * context,
			      list_anchor_t declined_objects,
			      task_effect_t * effect )
{
	int rc = 0;
	LOG_ENTRY();

	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * FUNCTION: raid0_set_shrink_object
 *
 * 	Do not allow the user to select all children of the RAID0 region.
 */
static int raid0_set_shrink_object( task_context_t * context,
			      list_anchor_t declined_objects,
			      task_effect_t * effect )
{
	int rc = 0;
	int rc2 = 0;
	md_volume_t *vol = (md_volume_t *)context->object->private_data;
	storage_object_t *obj;
	list_anchor_t my_list = NULL;
	list_anchor_t decline_list = NULL;
	list_element_t li, iter;
	declined_object_t *declined_obj;
	u_int64_t shrink_size;
	md_member_t *member;

	LOG_ENTRY();
	
	
	decline_list = EngFncs->allocate_list();
	my_list = EngFncs->allocate_list();
	if (!decline_list || !my_list) {
		rc = ENOMEM;
		LOG_EXIT_INT(rc);
		return rc;
	}

	if (EngFncs->list_count(context->selected_objects) >= vol->nr_disks) {
		/* Decline all objects */
		md_transfer_list(context->selected_objects, decline_list);
		goto decline_objects;
	}

	/*
	 * Transfer all selected objects to local list (my_list).
	 * If we can shrink, we will transfer the objects back to selected list.
	 * If we cannot shrink, the selected object list will be empty.
	 */
	md_transfer_list(context->selected_objects, my_list);

try_again:
	shrink_size = 0;
	LIST_FOR_EACH(my_list, iter, obj) {
		member = md_volume_find_object(vol, obj);
		if (member) {
			shrink_size += member->data_size;
		} else {
			LOG_MD_BUG();
		}
	}

	/* Ask the engine if it's ok to remove these objects. */
	rc2 = EngFncs->can_shrink_by(context->object, &shrink_size);
	switch (rc2) {
		case 0:
			/* Transfer all remaining objects back to selected objects list. */
			md_transfer_list(my_list, context->selected_objects);
			goto decline_objects;
			break;

		case EAGAIN:
			/*
			 * Remove the last object from the list.
			 * Append this object to the decline list.
			 * If the list is not empty, try again.
			 */
			obj = EngFncs->last_thing(my_list, &li);
			EngFncs->delete_element(li);
			EngFncs->insert_thing(decline_list, obj, INSERT_AFTER, NULL);
			if (EngFncs->list_count(my_list) == 0) {
				/* Decline all objects */
				goto decline_objects;
			}
			goto try_again;
			break;

		default:
			/* Some other error has occurred, decline all objects */
			md_transfer_list(my_list, decline_list);
			break;
	}
	

decline_objects:

	/* Decline objects in decline list */
	LIST_FOR_EACH(decline_list, iter, obj) {
		declined_obj = EngFncs->engine_alloc(sizeof(declined_object_t));
		if (!declined_obj) {
			rc = ENOMEM; /* Set error code, but continue looping */
		} else {
			declined_obj->object = obj;
			declined_obj->reason = EINVAL;
			EngFncs->insert_thing(declined_objects, declined_obj, INSERT_AFTER, NULL);
		}
	}

	EngFncs->destroy_list(decline_list);
	EngFncs->destroy_list(my_list);

	*effect |= EVMS_Effect_Reload_Objects;

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid0_set_objects
 *
 *  Determine the type of task, and then validate that the objects on the
 *  "selected" list are valid for that task. If so, adjust the option
 *  descriptor as appropriate.
 */
static int raid0_set_objects( task_context_t * context,
			      list_anchor_t declined_objects,
			      task_effect_t * effect ) {
	int rc = 0;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	// Parameter check.
	if (!context || !declined_objects || !effect) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	switch (context->action) {
	
	case EVMS_Task_Create:
		// Assume front end has selected valid objects from the
		// acceptable_objects list.  Could add safety checking.
		break;

	case EVMS_Task_Expand:
		raid0_set_expand_object( context, declined_objects, effect);
		break;
	
	case EVMS_Task_Shrink:
		raid0_set_shrink_object( context, declined_objects, effect);
		break;
	
	default:
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid0_get_info
 *
 *  Return MD-specific information about the specified region. If the
 *  name field is set, only return the "extra" information pertaining
 *  to that name.
 */
static int raid0_get_info( storage_object_t       * region,
			   char                   * name,
			   extended_info_array_t ** info_array ) {

	md_volume_t * volume = NULL;
	int           rc= 0;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	// Parameter check
	if (!info_array) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	// Make sure this is an MD region
	if (region->plugin != raid0_plugin) {
		LOG_ERROR("Region %s is not owned by MD RAID1\n", region->name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	volume = region->private_data;

	rc = md_get_info(volume, name, info_array);

	LOG_EXIT_INT(rc);
	return rc;
}


/* Function: raid0_get_plugin_info
 *
 *  Return information about the MD plugin. There is no "extra"
 *  information about MD, so "name" should always be NULL.
 */
static int raid0_get_plugin_info( char                   * name,
				  extended_info_array_t ** info_array ) {

	extended_info_array_t   * info = NULL;
	char                    buffer[50] = {0};
	int                     i = 0;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	// Parameter check
	if (!info_array) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (!name) {
		// Get memory for the info array
		if (!(info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t)*6))) {
			LOG_ERROR("Error allocating memory for info array\n");
			LOG_EXIT_INT(ENOMEM);
			return ENOMEM;
		}

		// Short Name
		info->info[i].name = EngFncs->engine_strdup("ShortName");
		info->info[i].title = EngFncs->engine_strdup(_("Short Name"));
		info->info[i].desc = EngFncs->engine_strdup(_("A short name given to this plug-in"));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(raid0_plugin->short_name);
		i++;

		// Long Name
		info->info[i].name = EngFncs->engine_strdup("LongName");
		info->info[i].title = EngFncs->engine_strdup(_("Long Name"));
		info->info[i].desc = EngFncs->engine_strdup(_("A longer, more descriptive name for this plug-in"));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(raid0_plugin->long_name);
		i++;

		// Plugin Type
		info->info[i].name = EngFncs->engine_strdup("Type");
		info->info[i].title = EngFncs->engine_strdup(_("Plug-in Type"));
		info->info[i].desc = EngFncs->engine_strdup(_("There are various types of plug-ins, each responsible for some kind of storage object or logical volume."));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(_("Region Manager"));
		i++;

		// Plugin Version
		info->info[i].name = EngFncs->engine_strdup("Version");
		info->info[i].title = EngFncs->engine_strdup(_("Plug-in Version"));
		info->info[i].desc = EngFncs->engine_strdup(_("This is the version number of the plug-in."));
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;

		// Required Engine Services Version
		info->info[i].name = EngFncs->engine_strdup("Required_Engine_Version");
		info->info[i].title = EngFncs->engine_strdup(_("Required Engine Services Version"));
		info->info[i].desc = EngFncs->engine_strdup(_("This is the version of the Engine services that this plug-in requires.  "
							      "It will not run on older versions of the Engine services."));
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", raid0_plugin->required_engine_api_version.major, raid0_plugin->required_engine_api_version.minor, raid0_plugin->required_engine_api_version.patchlevel);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;

		// Required Plug-in API Version
		info->info[i].name = EngFncs->engine_strdup("Required_Plugin_Version");
		info->info[i].title = EngFncs->engine_strdup(_("Required Plug-in API Version"));
		info->info[i].desc = EngFncs->engine_strdup(_("This is the version of the Engine plug-in API that this plug-in requires.  "
							      "It will not run on older versions of the Engine plug-in API."));
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", raid0_plugin->required_plugin_api_version.plugin.major, raid0_plugin->required_plugin_api_version.plugin.minor, raid0_plugin->required_plugin_api_version.plugin.patchlevel);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;
	} else {
		LOG_ERROR("No support for extra plugin information about \"%s\"\n", name);
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	info->count = i;
	*info_array = info;
	LOG_EXIT_INT(0);
	return 0;
}

static int raid0_volume_rw_special(
	storage_object_t * region,
	lsn_t lsn,
	sector_count_t count,
	void * buffer,
	int rw )
{
	int rc = 0;
	md_volume_t * volume = (md_volume_t *)region->private_data;
	list_element_t iter;
	md_volume_t *temp;
	md_volume_t *exp_vol;
	md_volume_t *org_vol;
	boolean found;
	md_member_t *saved_member = NULL;
	md_saved_info_t *info;

	LOG_ENTRY();
	
	found = FALSE;
	LIST_FOR_EACH(raid0_expand_shrink_list, iter, temp) {
		if (temp->region == region) {
			found = TRUE;
			break;
		}
	}

	if (found == FALSE) {
		if (rw == 0) {
			memset(buffer, 0x0, count * EVMS_VSECTOR_SIZE);
		}
		LOG_ERROR("Region %s, we could not find orginal volume to redirect I/0,"
			  " returning zero filled buffer.\n",
			  volume->name);
		rc = EIO;
		goto out;
	}

	if (volume->region_mgr_flags & (MD_RAID0_EXPAND_PENDING | MD_RAID0_SHRINK_PENDING)) {
		org_vol = temp;
		rc = raid0_volume_rw(org_vol, lsn, count, buffer, rw);		
		goto out;
	}
	
	if (md_check_for_expand_shrink_in_progress(volume, &saved_member)) {
		info = saved_member->saved_info;
	} else {
		if (rw == 0) {
			memset(buffer, 0x0, count * EVMS_VSECTOR_SIZE);
		}
		LOG_ERROR("No saved superblock info for region %s,"
			  " returning zero filled buffer.\n",
			  volume->name);
		rc = EIO;
		goto out;
	}

	LOG_DEFAULT("Sector mark=%"PRIu64".\n", info->sector_mark);
	
	if (volume->region_mgr_flags & MD_RAID0_UNWIND_EXPANSION_PENDING) {
		/*
		 * The expansion was interrupted.  We now have to handle I/O requests.
		 * Retrieve the sector_mark in the saved superblock.
		 * - All requests for LSN >= sector_mark will go to the original region.
		 * - All requests for LSN + count <= sector_mark will go to the "expanded" region.
		 * - The worse case, LSN < sector_mark and LSN + count > sector_mark,
		 * break up the request.
		 */
		exp_vol = temp;
		if (lsn >= info->sector_mark) {
			/* Request goes to the original region */
			LOG_DEFAULT("Unwind expansion is pending, request to original volume,"
				    " lsn=%"PRIu64", count=%"PRIu64".\n",
				    lsn, count);
			rc = raid0_volume_rw(volume, lsn, count, buffer, rw);
		} else {
			if (lsn + count <= info->sector_mark) {
				/* Request goes to the "expanded" region */
				LOG_DEFAULT("Unwind expansion is pending,"
					    " request to expanded region,"
					    " lsn=%"PRIu64", count=%"PRIu64".\n",
					    lsn, count);		
				rc = raid0_volume_rw(exp_vol, lsn, count, buffer, rw);
			} else { /* Bad news: must break up this request */
				u_int64_t count1, count2;
				
				count1 = info->sector_mark - lsn;
				count2 = count - count1;
				rc = raid0_volume_rw(exp_vol, lsn, count1, buffer, rw);
				rc |= raid0_volume_rw(volume, lsn+count1, count2, buffer, rw);
			}
		}
		goto out;
	}

	if (volume->region_mgr_flags & MD_RAID0_RESUME_SHRINKING_PENDING) {
		/*
		 * The shrinking was interrupted.  We now have to handle I/O requests.
		 * Retrieve the sector_mark in the saved superblock.
		 * - All requests for LSN >= sector_mark will go to the "shrunk" region.
		 * - All requests for LSN + count <= sector_mark will go to the original region.
		 * - The worse case, LSN < sector_mark and LSN + count > sector_mark,
		 * break up the request.
		 */
		org_vol = temp;
		if (lsn >= info->sector_mark) {
			/* Request goes to the "expanded" region */
			LOG_DEFAULT("Resume shrinking is pending,"
				    " request to shrunk region,"
				    " lsn=%"PRIu64", count=%"PRIu64".\n",
				    lsn, count);		
			/* Request goes to the "shrunk" region */
			rc = raid0_volume_rw(volume, lsn, count, buffer, rw);
		} else {
			if (lsn + count <= info->sector_mark) {
				/* Request goes to the original region */
				LOG_DEFAULT("Resume shrinking is pending, request to original volume,"
					    " lsn=%"PRIu64", count=%"PRIu64".\n",
					    lsn, count);
				rc = raid0_volume_rw(org_vol, lsn, count, buffer, rw);
			} else { /* Bad news: must break up this request */
				u_int64_t count1, count2;
				
				count1 = info->sector_mark - lsn;
				count2 = count - count1;
				rc = raid0_volume_rw(org_vol, lsn, count1, buffer, rw);
				rc |= raid0_volume_rw(volume, lsn+count1, count2, buffer, rw);
			}
		}
		goto out;
	}
out:
	LOG_EXIT_INT(rc);
	return rc;

}

/* Function: raid0_read
 *
 *  Perform a logical-to-physical remapping, and send the read down to
 *  the next plugin.
 */
static int raid0_read(
	storage_object_t * region,
	lsn_t              lsn,
	sector_count_t     count,
	void             * buffer )
{
	int                rc = 0;
	md_volume_t      * volume = (md_volume_t *)region->private_data;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	// Parameter check.
	if (!buffer) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (volume->flags & MD_CORRUPT) {
		memset(buffer, 0x0, count * EVMS_VSECTOR_SIZE);
		LOG_ERROR("MD Object %s is corrupt, returning zero filled buffer.\n",
			  volume->name);
		LOG_EXIT_INT(0);
		return 0;
	}

	if ((lsn + count) > region->size) {
		LOG_ERROR("Attempt to read past end of region %s sector=%"PRIu64"\n",
			  volume->name,lsn+count);
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}
	
	rc = md_region_rw(region, lsn, count, buffer, 0);
	if (rc == ENODEV) {
		if (volume->region_mgr_flags &
			(MD_RAID0_EXPAND_PENDING | MD_RAID0_SHRINK_PENDING |
			MD_RAID0_UNWIND_EXPANSION_PENDING | MD_RAID0_RESUME_SHRINKING_PENDING))
		{
			rc = raid0_volume_rw_special(region, lsn, count, buffer, 0);
		} else {
	
			rc = raid0_volume_rw(volume, lsn, count, buffer, 0);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/* Function: raid0_write
 *
 *  Perform a logical-to-physical remapping, and send the write down to
 *  the next plugin.
 */
static int raid0_write( storage_object_t * region,
			lsn_t              lsn,
			sector_count_t     count,
			void             * buffer ) {

	int                rc = 0;
	md_volume_t      * volume = (md_volume_t *)region->private_data;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	// Parameter check.
	if (!buffer) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (volume->flags & MD_CORRUPT) {
		MESSAGE(_("MD Object %s is corrupt.  Writing data is not allowed.\n"),volume->name);
		LOG_EXIT_INT(EIO);
		return EIO;
	}

	if ((lsn + count) > region->size) {
		LOG_ERROR("Attempt to write past end of region %s sector=%"PRIu64"\n",volume->name,lsn+count);
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}
	
	rc = md_region_rw(region, lsn, count, buffer, 1);
	if (rc == ENODEV) {
		if (volume->region_mgr_flags &
			(MD_RAID0_EXPAND_PENDING | MD_RAID0_SHRINK_PENDING |
			MD_RAID0_UNWIND_EXPANSION_PENDING | MD_RAID0_RESUME_SHRINKING_PENDING))
		{
			rc = raid0_volume_rw_special(region, lsn, count, buffer, 1);
		} else {
	
			rc = raid0_volume_rw(volume, lsn, count, buffer, 1);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

static int raid0_get_plugin_functions(storage_object_t *region, function_info_array_t * * functions)
{
	my_plugin = raid0_plugin;
	LOG_ENTRY();
	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

static int raid0_plugin_function(storage_object_t * object,
				 task_action_t      action,
				 list_anchor_t            objects,
				 option_array_t   * options)
{
	my_plugin = raid0_plugin;
	LOG_ENTRY();
	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}


static void free_region (storage_object_t * region)
{
	md_volume_t * volume = (md_volume_t *)region->private_data;
	LOG_ENTRY();
	raid0_free_private_data(volume);
	md_free_volume(volume);
	LOG_EXIT_VOID();
}

/*
 * raid0_backup_metadata
 *
 * Called to write metadata backup.
 */
int raid0_backup_metadata(storage_object_t *region)
{
	md_volume_t *volume;
	int rc=0;
	
	my_plugin = linear_plugin;
	LOG_ENTRY();


	volume = region->private_data;
	if (volume->flags & MD_CORRUPT) {
		rc = ENOSYS;
		goto out;
	}
	volume->commit_flag |= MD_COMMIT_BACKUP_METADATA;
	volume->commit_flag |= MD_COMMIT_DONT_CHECK_ACTIVE;
	volume->flags |= MD_DIRTY;
	rc = md_write_sbs_to_disk(volume); 
	volume->commit_flag &= ~MD_COMMIT_BACKUP_METADATA;
	volume->commit_flag &= ~MD_COMMIT_DONT_CHECK_ACTIVE;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

static void raid0_plugin_cleanup() {

	int rc;
	list_anchor_t raid0_regions_list;
	list_element_t li;
	storage_object_t *region;
	md_volume_t *vol;
	raid0_delay_kill_sector_t *killsect;

	my_plugin = raid0_plugin;
	LOG_ENTRY();

	rc = EngFncs->get_object_list(REGION, DATA_TYPE, raid0_plugin, NULL, 0, &raid0_regions_list);

	if (rc == 0) {
		LIST_FOR_EACH(raid0_regions_list, li, region) {
			free_region(region);
		}
		EngFncs->destroy_list(raid0_regions_list);
	}

	if (raid0_expand_shrink_list) {
		LIST_FOR_EACH(raid0_expand_shrink_list, li, vol) {
			/* Note: we just free the memory allocated for the md volume structure */
			LOG_WARNING("Hmm... Cleaning up %s.\n", vol->name);
			raid0_free_private_data(vol);
			md_free_volume(vol);
		}

		EngFncs->destroy_list(raid0_expand_shrink_list);
	}

	if (raid0_delay_kill_sector_list) {
		LIST_FOR_EACH(raid0_delay_kill_sector_list, li, killsect) {
			LOG_CRITICAL("Hmm... Found delayed kill sector (LSN:%"PRIu64", count:%"PRIu64").\n",
				     killsect->lsn, killsect->count);
			EngFncs->engine_free(killsect);
		}
		EngFncs->destroy_list(raid0_delay_kill_sector_list);
	}
        LOG_EXIT_VOID();
}


/* Function table for the MD RAID0 Region Manager */
static plugin_functions_t raid0_functions = {
	setup_evms_plugin              : raid0_setup_evms_plugin,
	cleanup_evms_plugin            : raid0_plugin_cleanup,
	can_delete                     : raid0_can_delete,
	can_expand                     : raid0_can_expand,
	can_shrink                     : raid0_can_shrink,
	can_replace_child              : raid0_can_replace_child,
	discover                       : raid0_discover,
	create                         : raid0_create,
	delete                         : raid0_delete,
	discard                        : raid0_discard,
	expand                         : raid0_expand,
	shrink                         : raid0_shrink,
	replace_child                  : raid0_replace_child,
	add_sectors_to_kill_list       : raid0_add_sectors_to_kill_list,
	commit_changes                 : raid0_commit_changes,
	can_activate                   : raid0_can_activate_region,
	activate                       : raid0_activate_region,
	can_deactivate                 : raid0_can_deactivate_region,
	deactivate                     : raid0_deactivate_region,
	get_option_count               : raid0_get_option_count,
	init_task                      : raid0_init_task,
	set_option                     : raid0_set_option,
	set_objects                    : raid0_set_objects,
	get_info                       : raid0_get_info,
	get_plugin_info                : raid0_get_plugin_info,
	read                           : raid0_read,
	write                          : raid0_write,
	get_plugin_functions           : raid0_get_plugin_functions,
	plugin_function                : raid0_plugin_function,
	backup_metadata                : raid0_backup_metadata
};



/*
 *  Initialize the local plugin record
 */

plugin_record_t raid0_plugin_record = {
	id:                     SetPluginID(EVMS_OEM_IBM, EVMS_REGION_MANAGER, 6),

	version:                {major:      MAJOR_VERSION,
				 minor:      MINOR_VERSION,
				 patchlevel: PATCH_LEVEL},

	required_engine_api_version: {major:      15,
				      minor:      0,
				      patchlevel: 0},
	required_plugin_api_version: {plugin: {major:      13,
					       minor:      0,
					       patchlevel: 0} },

	short_name:             "MDRaid0RegMgr",
	long_name:              "MD RAID0 Region Manager",
	oem_name:               "IBM",

	functions:              {plugin: &raid0_functions},

	container_functions:    NULL
};

