/*
 *
 *   (C) Copyright IBM Corp. 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdrivelink.so
 *
 *   File: dl_plugin.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "dl_common.h"


static plugin_record_t drivelink_plugin_record;

plugin_record_t              *dl_plugin_record=&drivelink_plugin_record;

struct engine_functions_s    *EngFncs=NULL;


/*
 *  Function:  dl_delete_all_private_data
 *
 *  This routine is called to delete -ALL- the private data
 *  we have allocated during discovery.  This means all the
 *  missing child object's private data ... along with all
 *  the drivelink object private data.
 */
static void  dl_delete_all_private_data( void )
{
        int rc,i;
        storage_object_t *drivelink;
        drivelink_private_data_t *pdata;
        list_anchor_t object_list = NULL;
        list_element_t iter;

        LOG_ENTRY();

        rc = dl_get_drivelink_list(&object_list);
        if (rc==0) {

                LIST_FOR_EACH(object_list, iter, drivelink) {
                        pdata = (drivelink_private_data_t *) drivelink->private_data;
                        if (pdata) {
                                for (i=0; i<pdata->drive_link_count; i++) {
                                        if (dl_isa_missing_child(pdata->drive_link[i].object)==TRUE) {
                                                free(pdata->drive_link[i].object->private_data);
                                                pdata->drive_link[i].object->private_data = NULL;
                                        }
                                }

                                free(pdata);
                                drivelink->private_data = NULL;
                        }
                }

                EngFncs->destroy_list(object_list);
        }

        LOG_EXIT_VOID();
}

/*
 *  Function:  dl_setup
 *
 *  This is the DRIVELINKING setup API.  It is called after the
 *  engine has inspected and verified our plugin record.  Besides
 *  telling us that the engine has loaded our plugin ... it also
 *  is the means by which the engine provides us with its plugin
 *  support functions.
 */
static int dl_setup( engine_functions_t * engine_functions)
{
        EngFncs = engine_functions;
        LOG_ENTRY();
        LOG_EXIT_INT(0);
        return 0;
}


/*
 *  Function:  dl_cleanup
 *
 *  This is the DRIVELINKING cleanup API.  It is called when the engine
 *  is being closed and we are suppose to do any work necessary to
 *  cleanup after ourselves.
 */
static void dl_cleanup( void )
{
        LOG_ENTRY();
        dl_delete_all_private_data();
        LOG_EXIT_VOID();
}


/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                              PLUGIN FUNCTION TABLE                                   +
+                                                                                      +
+--------------------------------------------------------------------------------------*/
static struct plugin_functions_s fft={

        setup_evms_plugin:                      dl_setup,
        cleanup_evms_plugin:                    dl_cleanup,
        can_set_volume:                         dl_can_set_volume,
        can_delete:                             dl_can_delete,
        can_expand:                             dl_can_expand,
        can_expand_by:                          dl_can_expand_by,
        can_shrink:                             dl_can_shrink,
        can_shrink_by:                          dl_can_shrink_by,
        can_replace_child:                      dl_can_replace_child,
        discover:                               dl_discover,
        create:                                 dl_create,
        can_add_feature:                        dl_can_add_feature,
        delete:                                 dl_delete,
        discard:                                dl_discard,
        expand:                                 dl_expand,
        shrink:                                 dl_shrink,
        replace_child:                          dl_replace_child,
        add_sectors_to_kill_list:               dl_add_sectors_to_kill_list,
        read:                                   dl_read,
        write:                                  dl_write,
        set_volume:                             dl_set_volume,
        commit_changes:                         dl_commit_changes,
        can_activate:                           dl_can_activate,
        activate:                               dl_activate,
        can_deactivate:                         dl_can_deactivate,
        deactivate:                             dl_deactivate,
        get_option_count:                       dl_get_option_count,
        init_task:                              dl_init_task,
        set_option:                             dl_set_option,
        set_objects:                            dl_set_objects,
        get_info:                               dl_get_info,
        get_plugin_info:                        dl_get_plugin_info,
        get_plugin_functions:                   dl_get_plugin_functions,
        plugin_function:                        dl_plugin_function,
        backup_metadata:                        dl_backup_metadata,
};


/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                       BUILD AND EXPORT AN EVMS PLUGIN RECORD                         +
+                                                                                      +
+--------------------------------------------------------------------------------------*/

static plugin_record_t drivelink_plugin_record = {

        id:                                 SetPluginID(EVMS_OEM_IBM, EVMS_FEATURE, EVMS_DRIVELINK_FEATURE_ID ),

        version:                            {MAJOR_VERSION, MINOR_VERSION, PATCH_LEVEL},

        required_engine_api_version:        {15, 0, 0},
        required_plugin_api_version:        {plugin: {13, 1, 0} },

        short_name:                         "DriveLink",
        long_name:                          "Drive Linking Feature",
        oem_name:                           "IBM",

        functions:                          {plugin: &fft},

        container_functions:                NULL

};


// Vector of plugin record ptrs that we export for the EVMS Engine.
plugin_record_t *evms_plugin_records[] = {
        &drivelink_plugin_record,
        NULL
};

