/*
 *  Copyright (C) 2001-2002 Hewlett-Packard Co.
 *	Contributed by Stephane Eranian <eranian@hpl.hp.com>
 *
 * This file is part of the ELILO, the EFI Linux boot loader.
 *
 *  ELILO is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  ELILO is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with ELILO; see the file COPYING.  If not, write to the Free
 *  Software Foundation, 59 Temple Place - Suite 330, Boston, MA
 *  02111-1307, USA.
 *
 * Please check out the elilo.txt for complete documentation on how
 * to use this program.
 */
#include <efi.h>
#include <efilib.h>

#include "glue_netfs.h"
#include "fs/netfs.h"
#include "strops.h"

#include "elilo.h"
#include "vars.h"

/*
 * disable this if you only want the default config file (elilo.conf)
 * and not the ip-address based first file attempt
 */

static INTN glue(fileops_t *this, VOID *intf);

/* object exported to fileops */
fileops_fs_t netfs_glue = { NETFS_PROTOCOL  , glue, netfs_install, netfs_uninstall};
	

#define NETFS_DEFAULT_KERNEL		L"vmlinux"
#define NETFS_DEFAULT_CONFIG		L"elilo.conf"
#define NETFS_DEFAULT_SERVER_TYPE	EFI_PXE_BASE_CODE_BOOT_TYPE_REDHAT_BOOT

/*
 * Pxe Discovery protocol layers
 * Layer 0 is used to download the boot loader
 */
#define NETFS_CONFIG_LAYER	1
#define NETFS_KERNEL_LAYER	2


static CHAR16 *hexa=L"0123456789ABCDEF";

static VOID
convert_ip2hex(UINT8 *ip, INTN l, CHAR16 *str)
{
	UINTN i;

	for(i=0; i < l; i++) {
		str[2*i]   = hexa[(ip[i] & 0xf0)>>4];
		str[2*i+1] = hexa[ip[i] & 0x0f];
	}
}

static VOID
convert_ip2decstr(UINT8 *ip, INTN l, CHAR16 *str)
{
	UINTN i, j;
	UINTN v, val;

	for(i=0, j=0; i < l; i++) {
		val = ip[i];
		v = val / 100;
		if (v) {
			str[j++] = '0'+v;
		}
		val = val % 100;
		v = val / 10;
		if (v) {
			str[j++] = '0'+v;
		}
		v = val % 10;
		str[j++] = '0'+v;
		if (i < l-1) str[j++] = '.';
	}
	str[j] = CHAR_NULL;
}


static EFI_STATUS
netfs_setdefaults(VOID *intf, CHAR16 *config, CHAR16 *kname, UINTN maxlen)
{
	netfs_interface_t *netfs = (netfs_interface_t *)intf;
	netfs_info_t info;
	EFI_STATUS status;
	UINT8 *ipaddr;
	UINTN m;
	CHAR16 ip_var[64], str[64];
	UINT8 *ip;

	if (config == NULL || kname == NULL || maxlen < 1) return EFI_INVALID_PARAMETER;

	netfs->netfs_getinfo(netfs, &info);

	m      = info.using_ipv6 ? 16 : 4;
	ipaddr = info.using_ipv6 ? info.cln_ipaddr.v6.Addr: info.cln_ipaddr.v4.Addr;

	convert_ip2decstr(ipaddr, m, ip_var);
	set_var(VAR_NETFS_IPADDR, ip_var);

	ip = info.using_ipv6 ? info.netmask.v6.Addr: info.netmask.v4.Addr;
	convert_ip2decstr(ip, m, str);
	set_var(VAR_NETFS_NETMASK, str);

	ip = info.using_ipv6 ? info.gw_ipaddr.v6.Addr: info.gw_ipaddr.v4.Addr;
	convert_ip2decstr(ip, m, str);
	set_var(VAR_NETFS_GATEWAY, str);

	set_var(VAR_NETFS_HOSTNAME, info.hostname);
	set_var(VAR_NETFS_DOMAINAME, info.domainame);

	if (info.using_pxe) {
		status = netfs->netfs_query_layer(netfs, 0, NETFS_CONFIG_LAYER, maxlen, config);
		if (EFI_ERROR(status)) {
			StrnCpy(config, NETFS_DEFAULT_CONFIG, maxlen-1);
			config[maxlen-1] = CHAR_NULL;
		}

		status = netfs->netfs_query_layer(netfs, 0, NETFS_KERNEL_LAYER, maxlen, kname);
		if (EFI_ERROR(status)) {
			StrnCpy(kname, NETFS_DEFAULT_KERNEL, maxlen-1);
			kname[maxlen-1] = CHAR_NULL;
		}
	} else {
#ifdef ENABLE_MACHINE_SPECIFIC_NETCONFIG
		/*
		 * will try a machine specific file first.
		 * the file is constructed based on the IP(v4) address
		 */
		convert_ip2hex(ipaddr, m, config);

		config[8]  = L'.';
		config[9]  = L'c';
		config[10] = L'o';
		config[11] = L'n';
		config[12] = L'f';
		config[13] = CHAR_NULL;
#else
		StrnCpy(config, NETFS_DEFAULT_CONFIG, maxlen-1);
		config[maxlen-1] = CHAR_NULL;
#endif
		StrnCpy(kname, NETFS_DEFAULT_KERNEL, maxlen-1);
		kname[maxlen-1] = CHAR_NULL;
	}
	return EFI_SUCCESS;
}

static INTN
glue(fileops_t *fp, VOID *intf)
{
	netfs_interface_t *netfs = (netfs_interface_t *)intf;

	/* record underlying interface */
	fp->intf        = intf;

	fp->open        = (fops_open_t)netfs->netfs_open;
	fp->read        = (fops_read_t)netfs->netfs_read;
	fp->close       = (fops_close_t)netfs->netfs_close;
	fp->infosize    = (fops_infosize_t)netfs->netfs_infosize;
	fp->seek        = (fops_seek_t)netfs->netfs_seek;
	fp->setdefaults = (fops_setdefaults_t)netfs_setdefaults;
	fp->intf	= intf;

	/* fill out the name of the underlying file system */
	netfs->netfs_name(netfs, fp->name, FILEOPS_NAME_MAXLEN);

	return 0;
}
