/*******************************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.equinox.internal.p2.ui.admin.dialogs;

import org.eclipse.equinox.internal.p2.ui.admin.ProvAdminUIMessages;
import org.eclipse.equinox.internal.provisional.p2.core.ProvisionException;
import org.eclipse.equinox.internal.provisional.p2.engine.IProfile;
import org.eclipse.equinox.internal.provisional.p2.metadata.IInstallableUnit;
import org.eclipse.equinox.internal.provisional.p2.ui.ProvUI;
import org.eclipse.equinox.internal.provisional.p2.ui.model.InstalledIUElement;
import org.eclipse.equinox.internal.provisional.p2.ui.operations.ProvisioningUtil;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.*;
import org.eclipse.ui.statushandlers.StatusManager;

/**
 * An IUPropertiesGroup is a reusable UI component that displays and edits the 
 * user-oriented properties of an IU. It can be used in 
 * different dialogs that manipulate or define IU's.
 * 
 * @since 3.4
 */
public class IUProfilePropertiesGroup extends IUGroup {

	private Table propertiesTable;

	public IUProfilePropertiesGroup(final Composite parent, InstalledIUElement iuElement, ModifyListener listener) {
		super(parent, iuElement, listener);
	}

	protected Composite createGroupComposite(Composite parent, ModifyListener listener) {
		Composite composite = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.numColumns = 1;
		composite.setLayout(layout);
		GridData data = new GridData();
		data.widthHint = 400;
		composite.setLayoutData(data);

		propertiesTable = new Table(composite, SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);
		data = new GridData(GridData.FILL_BOTH);
		data.grabExcessVerticalSpace = true;
		propertiesTable.setLayoutData(data);
		propertiesTable.setHeaderVisible(true);
		TableColumn nameColumn = new TableColumn(propertiesTable, SWT.NONE);
		nameColumn.setResizable(true);
		nameColumn.setWidth(200);
		TableColumn valueColumn = new TableColumn(propertiesTable, SWT.NONE);
		valueColumn.setResizable(true);
		valueColumn.setWidth(250);
		initializeFields();
		return composite;
	}

	private void initializeFields() {
		if (iuElement == null || !(iuElement instanceof InstalledIUElement)) {
			return;
		}
		String[] propNames = new String[] {IInstallableUnit.PROP_PROFILE_ROOT_IU};
		String[] userPropNames = new String[] {ProvAdminUIMessages.ProfileRootPropertyName};
		for (int i = 0; i < propNames.length; i++) {
			TableItem item = new TableItem(propertiesTable, SWT.NULL);
			IProfile profile = getProfile((InstalledIUElement) iuElement);
			String value = profile == null ? null : profile.getInstallableUnitProperty(getIU(), propNames[i]);
			if (value != null)
				item.setText(new String[] {userPropNames[i], value});
		}
	}

	private IProfile getProfile(InstalledIUElement element) {
		try {
			return ProvisioningUtil.getProfile(element.getProfileId());
		} catch (ProvisionException e) {
			ProvUI.handleException(e, NLS.bind(ProvAdminUIMessages.IUProfilePropertiesGroup_InvalidProfileID, element.getProfileId()), StatusManager.LOG);
			return null;
		}
	}
}
