/*
 * savedump.c - Copies an lkcd raw dump to the file system
 * Created by:	John Daley (John.Daley@hp.com)
 * Minor mods:	Bob Montgomery (bobm@fc.hp.com)
 * Mods:	Troy Heber (troy.heber@hp.com)
 *
 * Copyright 2004 Hewlett-Packard Development Company, L.P.
 * 
 * Adapted from lkcdutils
 * Created by: Matt D. Robinson (yakker@aparity.com)
 * Copyright 2001 Matt D. Robinson (yakker@aparity.com), all rights reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */ 


#define MAX_LINE_SIZE 132

#include "savedump.h"
#include "parseconf.h"
#include "ftp.h"
#include "version.h"

struct confInfo{
	char  *dev;
	char  *dir;
	char  *user;
	char  *pass;
	int   ftp;
	char  *error;
	char  *host;
	char  *path;
	int    port;
};


void usage(char *);
void version(char *);
int check_dev( char *);
int reset_dump( char *);
int erase_dump(char *);
int copy_dump(struct confInfo *, char *);
static int direxists(const char *);

#if _LIBC
# define struct_stat64 struct stat64
#else
# define struct_stat64 struct stat
# define __xstat64(version, path, buf) stat (path, buf)
#endif

static int
kver(void)
{
	/* Check kernel version, assume 2.6 if we can't figure it out. */
	FILE *fp;
	char buf[512];

	if (((fp = fopen("/proc/version", "r")) == NULL)){
		fprintf(stderr, "WARNING: can not read /proc/version\n");
		return 6;
	}

	while (fgets(buf, 512, fp) != NULL){
		if (strstr(buf, " 2.4.")  != NULL)
			return 4;
		else if (strstr(buf, " 2.6.")  != NULL)
			return 6;
	}

	return 6;
}

/* Return nonzero if DIR is an existent directory.  */
static int
direxists (const char *dir)
{
	struct_stat64 buf;
	return __xstat64 (_STAT_VER, dir, &buf) == 0 && S_ISDIR (buf.st_mode);
}


void
usage(char * pname)
{
	fprintf(stderr, "\nUsage:\n");
	fprintf(stderr, "%s [[-r dumpdevice] || [-e dumpdevice]] || [[-d dumpdevice] [-o dumppath]]\n\n", pname);
	fprintf(stderr, "  dumpdevice\n");
	fprintf(stderr, "    The dumpdevice argument is the dump device that holds the raw LKCD dump.\n");
	fprintf(stderr, "    such as: /dev/vmdump\n\n");
	fprintf(stderr, "  dump-path\n");
	fprintf(stderr, "    The dumppath argument is the path to store the resulting dump file to.\n");
	fprintf(stderr, "    It may be a path on a filesystem or a FTP URL in the form.\n");
	fprintf(stderr, "    ftp://user:password@host:port/path where user, password,\n"); 
	fprintf(stderr, "    port & path are optional.\n\n");
	fprintf(stderr, "  Optional Arguments:\n\n");
	fprintf(stderr, "  [-r]\n");
	fprintf(stderr, "    Reset the raw LKCD dump from the dumpdevice, option is exclusive of -e\n\n");
	fprintf(stderr, "  [-e]\n");
	fprintf(stderr, "    Erase the raw LKCD dump from the dumpdevice, option is exclusive of -r\n\n");
	fprintf(stderr, "  [-d]\n");
	fprintf(stderr, "    Specify the dumpdevice overriding DUMPDEV= /etc/dumputils.conf file.\n\n");
	fprintf(stderr, "  [-o]\n");
	fprintf(stderr, "    Specify the dump-path overriding DUMPDIR= in the /etc/dumputils.conf file.\n\n");
	fprintf(stderr, "  EXAMPLES:\n\n");
	fprintf(stderr, "       Default: use /etc/dumputils.conf\n");
	fprintf(stderr, "       $ %s \n", pname);
	fprintf(stderr, "       $ %s -r /dev/{dumpdevice}\n", pname);
	fprintf(stderr, "       $ %s -e /dev/{dumpdevice}\n", pname);
	fprintf(stderr, "       $ %s -d /dev/{dumpdevice}\n", pname); 
	fprintf(stderr, "       $ %s -o /var/log/dump\n", pname); 
	fprintf(stderr, "       $ %s -o ftp://user:pass@none.com/pub/dump\n", pname); 
	fprintf(stderr, "       $ %s -d /dev/{dumpdevice} -o ftp://user:pass@none.com/pub/dump\n", pname); 
}

void 
version(char *pname)
{
#ifdef VERSION
	fprintf(stdout, "%s %s\n", pname, VERSION);
#else
	fprintf(stdout, "%s undefined\n", pname);
#endif 
}


void
percent_done(int sofar, int pages)
{
	unsigned int percent;
	unsigned static int lastpercent;

	percent = (float)sofar / (float)pages * 100.0;
	if (percent == lastpercent)
		return;
	if (percent % 10 == 0) {
		printf("--->%u%%", percent);
		fflush(stdout);
	}

	if(percent >= 100)
		printf("\n\n");

	lastpercent = percent;
	return;
}

int
copy_map(struct confInfo *conf, char *date)
{
	char *smname;
	FILE *fp=NULL;
	int dfd;
	char line[255], *fuser=NULL, *fpass=NULL;
	int res =0;
	int len = 0, plen = 0, flags, err = 1;
	struct utsname unm;

	uname(&unm);
	if(conf->path != NULL)
		plen = strlen(conf->path);
	
	len = plen + strlen(unm.nodename) + (2*strlen(date)) + 8;

	disconnect(err);

	if(conf->ftp){
		if ((smname=(char *)malloc(len)) == NULL){
			perror("FATAL ERROR: Couldn't allocate" 
					"enough memory");
			exit(10);
		}
		if (conf->path == NULL)
			sprintf(smname, "%s_%s/map.%s", unm.nodename, date, date);
		else
			sprintf(smname, "%s/%s_%s/map.%s", conf->path, 
					unm.nodename, date, date);

		if ((fuser=(char *)malloc(strlen(conf->user)+7)) == NULL){
			perror("FATAL ERROR: Couldn't allocate enough memory");
			exit(10);
		}

		if ((fpass=(char *)malloc(strlen(conf->pass)+7)) == NULL){
			perror("FATAL ERROR: Couldn't allocate enough memory");
			exit(10);
		}

		sprintf(fuser, "USER %s", conf->user);
		sprintf(fpass, "PASS %s", conf->pass);

		if (((res = FTPConnect(conf->host, fuser, fpass)) != 0) ){
			fprintf(stderr, "Could not connect to server!\n");
			free(fuser);
			free(fpass);
			free(smname);
			exit(res);
		}

		if ((dfd = do_open(smname)) < 0){
			fprintf(stderr, "Could not create file: %s\n", smname);
			free(fuser);
			free(fpass);
			free(smname);
			disconnect(err);
			return 38;
		}

		/* Make the sure the fd is non-blocking */	
		if ((flags=fcntl(dfd, F_GETFL, 0)) < 0){
			perror("fcntl read failed");
			return 66;
		}

		if (fcntl(dfd, F_SETFL, flags | O_NONBLOCK) < 0){
			perror("fcntl set failed");
			return 66;
		}

		if((fp = fopen("/boot/System.map", "r")) != NULL){
			while (fgets(line,sizeof(line),fp) != NULL)
				if (protect_write(dfd, line, strlen(line)) < 0){
					fprintf(stderr, "Error writing to: %s\n", smname);
					free(fuser);
					free(fpass);
					free(smname);
					disconnect(err);
					return 86;
				}

		}else{
			fprintf(stderr, "Couldn't open /boot/System.map!\n");
			free(fuser);
			free(fpass);
			free(smname);
			disconnect(err);
			return 82;
		}

#ifdef DEBUG
		printf("Wrote System Map: %s\n", smname);
#endif

		fclose(fp);
		free(fuser);
		free(fpass);
		free(smname);
		disconnect(0);

		return 0;

	} else {

		if(conf->dir == NULL){
			if ((smname=(char *)malloc(2*strlen(date) + 5)) 
					== NULL){
				perror("FATAL ERROR: Couldn't allocate" 
						"enough memory");
				exit(10);
			}
			sprintf(smname, "%s/map.%s", date, date);

		}else{
			if ((smname=(char *)malloc(strlen(conf->dir) 
					+ 2*strlen(date) + 5)) == NULL){
				perror("FATAL ERROR: Couldn't allocate" 
						"enough memory");
				exit(10);
			}
			sprintf(smname, "%s/%s/map.%s", conf->dir, date, date);
		}


#ifdef DEBUG
		printf("System Map file name: %s\n", smname);
#endif
		if ((dfd = open(smname, O_CREAT|O_RDWR)) == -1) {
			perror("Error opening destionation System.map file");
			free(smname);
			return 81;
		}

		if((fp = fopen("/boot/System.map", "r")) != NULL){
			while (fgets(line,sizeof(line),fp) != NULL)
			{
				write(dfd, line, strlen(line));
			}
		}else{
			fprintf(stderr, "Couldn't open /boot/System.map\n");
			free(smname);
			close(dfd);
			return 82;
		}

#ifdef DEBUG
		printf("Wrote System.map file: %s\n", smname);
#endif

		fclose(fp);
		close(dfd);
		free(smname);
		return 0;
	}

}

int
check_dev(char *dumpdev)
{
	struct stat statbuf;	

	/* check input file: dumpdev */

	if (stat(dumpdev, &statbuf) == -1) {
		perror("Can't stat dumpdev");
		exit(40);
	}

	if (!(S_ISBLK(statbuf.st_mode))) {
		fprintf(stderr,"%s is not a block device\n", dumpdev);
		exit(41);
	}
#ifdef DEBUG
	else {
		fprintf(stderr, "%s IS a block device\n", dumpdev);
	}
#endif
	return 0;
}

int
copy_dump(struct confInfo *conf, char *date)
{ 
	char *dfname, *pfname, *pagebuf, *headerbuf, *fuser, *fpass;
	uint64_t dha_magic_number[64]; 
	uint32_t page_index = 0;
	int page_nbr = 0, res = 0, pathlen = 0, dirlen = 0, plen = 0;
	int  dumpfd = 0, devicef, ret, flags, err = 1;
	int dump_header_offset, dump_header_size;
	struct utsname unm;
	dump_page_t dp;
	generic_dump_header_t *dh;

	uname(&unm);
	if(conf->path != NULL)
		plen = strlen(conf->path);
	
	dirlen = plen + strlen(unm.nodename) + strlen(date) + 3;
	pathlen = dirlen + strlen(date) + 8;
	
	/* set up filename for output file, and open it up */
	if(conf->ftp){

		printf("Attempting to save dump via ftp...\n");

		if ((dfname=(char *)malloc(dirlen)) == NULL){
			perror("FATAL ERROR: Couldn't allocate" 
					"enough memory");
			exit(10);
		}

		if ((pfname=(char *)malloc(pathlen)) == NULL){
			perror("FATAL ERROR: Couldn't allocate" 
					"enough memory");
			exit(10);
		}

		if(conf->path == NULL){
			sprintf(dfname, "%s_%s", unm.nodename, date);
			sprintf(pfname, "%s/dump.%s", dfname ,date);
		}else{
			sprintf(dfname, "%s/%s_%s", conf->path, unm.nodename,
					date);
			sprintf(pfname, "%s/dump.%s", dfname, date);
		}

		if ((fuser=(char *)malloc(strlen(conf->user)+7)) == NULL){
			perror("FATAL ERROR: Couldn't allocate enough memory");
			exit(10);
		}

		if ((fpass=(char *)malloc(strlen(conf->pass)+7)) == NULL){
			perror("FATAL ERROR: Couldn't allocate enough memory");
			exit(10);
		}

		sprintf(fuser, "USER %s", conf->user);
		sprintf(fpass, "PASS %s", conf->pass);

		if ( ((res = FTPConnect(conf->host, fuser, fpass)) != 0) ){
			fprintf(stderr, "Could not connect to server!\n");
			free(fuser);
			free(fpass);
			free(dfname);
			free(pfname);
			exit(res);
		}

		if(do_mkdir(dfname) !=0)
			fprintf(stderr, "Could not create directory, it may" 
					" already exsist.\n");

		if ((dumpfd = do_open(pfname)) < 0){
			fprintf(stderr, "Could not create file, it may" 
					" already exsist, or you don't have"
					" permission!\n");
			free(fuser);
			free(fpass);
			free(dfname);
			free(pfname);
			disconnect(err);
			return dumpfd;
		}
		
		/* Make the sure the fd is non-blocking */	
		if ((flags=fcntl(dumpfd, F_GETFL, 0)) < 0){
			perror("fcntl read failed");
			return 66;
		}

		if (fcntl(dumpfd, F_SETFL, flags | O_NONBLOCK) < 0){
			perror("fcntl set failed");
			return 66;
		}

		free(fuser);
		free(fpass);
		free(pfname);
		free(dfname);

	}else{
		printf("Attempting to save dump...\n");
		
		if ((dfname=(char *)malloc(strlen(conf->dir) 
					  + (2*strlen(date)) + 9)) == NULL){
			perror("FATAL ERROR: Couldn't allocate enough memory");
			exit(10);
		}
		
		if ((pfname=(char *)malloc(strlen(conf->dir) 
					  + strlen(date) + 2)) == NULL){
			perror("FATAL ERROR: Couldn't allocate enough memory");
			exit(10);
		}
		
		if (!direxists(conf->dir)){
			fprintf(stderr, "DUMPDIR does not exsist: %s:", conf->dir);
			free(dfname);
			free(pfname);
			return 65;
		}

		sprintf(pfname, "%s/%s", conf->dir, date);
			
		if (!direxists(pfname))
			if((mkdir(pfname, 0755) == -1)){
			fprintf(stderr, "Could not create directory %s:", dfname);
			perror("");
			}

		sprintf(dfname, "%s/%s/dump.%s", conf->dir, date, date);

		if ((dumpfd = open(dfname, O_CREAT|O_RDWR|O_TRUNC, 0600)) == -1) {
			fprintf(stderr, "open of dump file %s:", dfname);
			perror("");
			free(dfname);
			free(pfname);
			return 81;
		}
		
		free(dfname);
		free(pfname);
	}

	/* Open device partition with dump on it */

	if((devicef = open(conf->dev, O_RDONLY, 0)) == -1)
	{
		perror("Open dumpdev");
		return 83;
	}
#ifdef DEBUG
	fprintf(stderr,"dumpdev fs: %d\n", devicef);
#endif

	/* fixed offsets:
	*  DUMP_HEADER_OFFSET is the start of the headers on the raw device.a
	*  DUMP_HEADER_SIZE is the size of all headers plus pad before the
	*     page dumps start.
	*/ 

	if(kver() == 4)
		dump_header_offset = 4 * 1024;
	else 
		dump_header_offset = DUMP_HEADER_OFFSET;

	dump_header_size = DUMP_HEADER_SIZE;

	/* Move beyond swap header */

#ifdef DEBUG
	fprintf(stderr,"just before seek, dho=%d\n", dump_header_offset);
#endif
	if (lseek(devicef, dump_header_offset, SEEK_SET) == -1) {
		perror("lseek on dumpdev");
		close(devicef);
		return 84;
	}

	/*  read the whole dump header block (why not?) */

	headerbuf=(char *)malloc(dump_header_size);
	if (headerbuf == NULL) {
		perror("malloc dump header");
		exit(10);
	}

	if (read(devicef, headerbuf, dump_header_size) <  dump_header_size) {
		perror("read dump headers");
		free(headerbuf);
		return 84;
	}

	/*  check the generic parts of the dump header */

	dh = (generic_dump_header_t *)headerbuf;

#ifdef DEBUG
	fprintf(stderr," dh_offset = %d, dh_size = %d, page_size = %d\n",
			dump_header_offset, dh->dh_header_size, 
			dh->dh_dump_page_size);
#endif

	/* Validate magic number */

	if (dh->dh_magic_number != DUMP_MAGIC_NUMBER) {
		fprintf(stderr, "No dump to save\n");
#ifdef DEBUG		
		printf("%lx != %llx\n", dh->dh_magic_number, 
				DUMP_MAGIC_NUMBER);
#endif
		close(devicef);
		return 85;
	}

	/* Find the asm_dump_header */

	/* use memcpy instead of dha = (..._t *)(headerbuf + dh->dh_header_size)
	*  to avoid unaligned access. 
	*/
	memcpy(dha_magic_number, headerbuf + dh->dh_header_size, 
			sizeof(dha_magic_number));


	/* Validate asm magic number */

	if (*dha_magic_number != DUMP_ASM_MAGIC_NUMBER) {
		fprintf(stderr, "BAD MAGIC Asm dump header invalid!\n");
#ifdef DEBUG
		fprintf(stderr, "%p != %llx\n", dha_magic_number, 
				DUMP_ASM_MAGIC_NUMBER);
#endif
		close(devicef);
		return 85;
	}

#ifdef DEBUG
		fprintf(stderr,"dumpfd is: %d\n", dumpfd);
#endif

	/*  write the dump header block 
	*     optionally write just the two headers, and seek to
	*     dump_header_size
	*/ 
	  if (protect_write(dumpfd, headerbuf, dump_header_size) != dump_header_size){
		perror("write dump header");
		close(devicef);
		if(conf->ftp)
			disconnect(err);
		else
			close(dumpfd);

		return 86;
	}

	/*   check if just a header, no pages: */

	if (dh->dh_dump_level == DUMP_LEVEL_HEADER)
	{
		fprintf(stderr, "Dump is nothing but header!\n");
		close(devicef);
		if(conf->ftp)
			disconnect(err);
		else
			close(dumpfd);

		erase_dump(conf->dev);
		return 87;
	}


	/*  Seek past swap and dump headers in input, dump header in output */

	(void)lseek(devicef, dump_header_offset + dump_header_size, SEEK_SET);
	(void)lseek(dumpfd, dump_header_size, SEEK_SET);

	/*  loop to write pages */

	pagebuf=(char *)malloc(dh->dh_dump_page_size);
	page_index=0;

	while (1)
	{
		/* read a page header, then the page (below) */

		if (read(devicef, (char *)&dp, sizeof(dp)) != sizeof(dp))
		{
			perror("\nReading dump page header");
			if(conf->ftp)
				disconnect(err);
			else
				close(dumpfd);

			free(pagebuf);
			close(devicef);
			return 88;
		}


		/* some sanity checks: */

		if ((dp.dp_flags == 0) || (dp.dp_flags >= DUMP_DH_NOT_USED) ||
				((dp.dp_size == 0) 
				 && (!(dp.dp_flags & DUMP_DH_END))))  {
			fprintf(stderr, "\ndump page flags wrong! Maybe" 
					"truncated\n");
#ifdef DEBUG
			fprintf(stderr,
					"dp.dp_flags = 0x%x\ndp.dp_size=0x%x\n"
					"dp.dp_address=0x%lx, page_index=%d\n", 
					dp.dp_flags, dp.dp_size,
					dp.dp_address, page_index);
#endif			

			if(conf->ftp)
				disconnect(err);
			else
				close(dumpfd);

			free(pagebuf);
			free(dh);
			close(devicef);
			erase_dump(conf->dev);
			return 88;
		}

		/* Check that page size isn't too big */
		if (dp.dp_size > dh->dh_dump_page_size) {
			fprintf(stderr, "\npage size too big! May be" 
					"corrupt.\n");
#ifdef DEBUG
			fprintf(stderr,
					"\ndp.dp_size =%u\n," 
					"dh->dh_dump_page_size =%u\n",
					dp.dp_size, dh->dh_dump_page_size);
#endif			
			if(conf->ftp)
				disconnect(err);
			else
				close(dumpfd);

			free(dh);
			free(pagebuf);
			close(devicef);
			erase_dump(conf->dev);
			return 88;
		}

		/* write dump page header */

		if ((ret = protect_write(dumpfd, (char *)&dp, sizeof(dp)))
				< sizeof(dp)) {
			perror("\nWrite dump page header");
			if(conf->ftp)
				disconnect(err);
			else
				close(dumpfd);

			free(dh);
			free(pagebuf);
			close(devicef);
			return 86;
		}

		/*  Check for end of dump. */
		if (dp.dp_flags & DUMP_DH_END) {
			percent_done(dh->dh_num_dump_pages,
					dh->dh_num_dump_pages);

			/* "erase" the dump header by changing the magic
			*  number
			*/

			close(devicef);
			free(dh);
			free(pagebuf);
			if(conf->ftp)
				disconnect(0);
			else
				close(dumpfd);

			erase_dump(conf->dev);
			if (dp.dp_flags & DUMP_DH_TRUNCATED) {
				fprintf(stderr,"\nDump truncated\n");
				return 89;
			} else{

				return 0;
			}
		}

		/*  read in actual page, write it out. */
		if (read(devicef, pagebuf, dp.dp_size) != dp.dp_size) {
			perror("\nReading page data");
			if(conf->ftp)
				disconnect(err);
			else
				close(dumpfd);

			close(devicef);
			free(dh);
			free(pagebuf);
		}

		if ((ret = protect_write(dumpfd, (char *)pagebuf, dp.dp_size))
				!= dp.dp_size) {
			perror("\nWriting page data");
			if(conf->ftp)
				disconnect(err);
			else
				close(dumpfd);

			close(devicef);
			free(dh);
			free(pagebuf);
		}

		/*  update percentage */

		percent_done(page_nbr++, dh->dh_num_dump_pages);
		page_index++;

	} /* end while */

	/* never get here */

} /* end copy dump */

int 
reset_dump(char *ddev)
{
	int fs;
	int offset;
	generic_dump_header_t dhead;

	if (kver() == 4)
		offset = 4 * 1024;
	else 
		offset = DUMP_HEADER_OFFSET;

	/* check ddev, is it block dev? */
	check_dev(ddev);


	if ((fs=open(ddev,O_RDWR, 0)) == -1) {
		perror("reset:open");
		return 83;
	}

#ifdef DEBUG
	fprintf(stderr, "offset=%u, fs=%d, ddev=%s\n", offset, fs, ddev);
#endif

	if (lseek(fs, offset, SEEK_SET) != offset) {
		perror("reset:lseek");
		return 84;
	}

	if (read(fs, (char *)&dhead, sizeof(dhead)) != 
			(ssize_t)sizeof(dhead)) {
		perror("reset: read");
		return 84;
	}

	/* Check to try to ascertain if we're looking at the right thing... */

#ifdef DEBUG
	fprintf(stderr,"dhead.dh_magic_number = %lx\n", dhead.dh_magic_number);
#endif

	if (dhead.dh_magic_number == DUMP_MAGIC_NUMBER) {
		fprintf(stderr, "dump is already good\n");
		close(fs);
		return 0;
	} else if (dhead.dh_magic_number == DUMP_MAGIC_ERASED) {
		dhead.dh_magic_number = DUMP_MAGIC_NUMBER;

		if (lseek(fs, offset, SEEK_SET) != offset) {
			perror("reset:lseek back");
			return 84;
		}

		if (write(fs,(char *)&dhead, sizeof(dhead)) < 
				(ssize_t)sizeof(dhead)) {
			perror("reset: write");
			return 86;
		}
		fprintf(stderr,"dump reset to good\n");
		close(fs);
		return 0;
	} else {
		fprintf(stderr,"Not a dump magic number! Got dump device?\n");
		close(fs);
		return 85;
	}

}  /* end reset_dump */

int
erase_dump(char *ddev)
{
	int fs;
	int offset;
	generic_dump_header_t dhead;

	if (kver() == 4)
		offset = 4 * 1024;
	else 
		offset = DUMP_HEADER_OFFSET;

	/* check ddev, is it block dev? */

	check_dev(ddev);

	if ((fs=open(ddev,O_RDWR, 0)) == -1) {
		perror("erase: open");
		return 83;
	}

#ifdef DEBUG
	fprintf(stderr, "offset=%u, fs=%d, ddev=%s\n", offset, fs, ddev);
#endif

	if (lseek(fs, offset, SEEK_SET) != offset) {
		perror("erase:lseek");
		return 84;
	}

	if (read(fs, (char *)&dhead, sizeof(dhead)) != sizeof(dhead)) {
		perror("erase: read");
		return 84;
	}

	/* Check to try to ascertain if we're looking at the right thing... */

#ifdef DEBUG
	fprintf(stderr,"erase: dhead.dh_magic_number = %lx\n", 
			dhead.dh_magic_number);
	fprintf(stderr,"erase: mn = %llx\n", DUMP_MAGIC_NUMBER);
#endif

	if (dhead.dh_magic_number == DUMP_MAGIC_NUMBER) {
		dhead.dh_magic_number = DUMP_MAGIC_ERASED;

		if (lseek(fs, offset, SEEK_SET) != offset) {
			perror("erase:lseek back");
			return 84;
		}

		if (write(fs,(char *)&dhead, sizeof(dhead)) < 
				(ssize_t)sizeof(dhead)) {
			perror("erase: write");
			return 86;
		}
#ifdef DEBUG
		fprintf(stderr,"Dump erased?\n");
		if (lseek(fs, offset, SEEK_SET) != offset) {
			perror("erase:lseek back");
			return 84;
		}

		if (read(fs, (char *)&dhead, sizeof(dhead)) != sizeof(dhead)) {
			perror("erase: read");
			return 84;
		}
		fprintf(stderr, "dhead.dh_magic_number = %lx on disk\n",
				dhead.dh_magic_number);
#endif
		close(fs);
		return 0;
	} else if (dhead.dh_magic_number == DUMP_MAGIC_ERASED) {
		fprintf(stderr,"dump already erased\n");
		close(fs);
		return 0;
	} else {
		fprintf(stderr,"not a dump\n");
		close(fs);
		return 85;
	}

}  /* end erase_dump */


int
have_dump(char *ddev)
{
	int fs;
	int offset;
	generic_dump_header_t dhead;

	if (kver() == 4)
		offset = 4 * 1024;
	else 
		offset = DUMP_HEADER_OFFSET;

	/* check ddev, is it block dev? */
	check_dev(ddev);

	if ((fs=open(ddev,O_RDWR, 0)) == -1) {
		perror("have: open");
		return 83;
	}

#ifdef DEBUG
	fprintf(stderr, "offset=%u, fs=%d, ddev=%s\n", offset, fs, ddev);
#endif

	if (lseek(fs, offset, SEEK_SET) != offset) {
		perror("have:lseek");
		return 84;
	}

	if (read(fs, (char *)&dhead, sizeof(dhead)) != sizeof(dhead)) {
		perror("have: read");
		return 84;
	}

	/* Check to try to ascertain if we're looking at the right thing... */

#ifdef DEBUG
	fprintf(stderr,"erase: dhead.dh_magic_number = %lx\n", 
			dhead.dh_magic_number);
	fprintf(stderr,"erase: mn = %llx\n", DUMP_MAGIC_NUMBER);
#endif

	if (dhead.dh_magic_number == DUMP_MAGIC_NUMBER) 
		return 1;
	else 
		return 0;

}  /* end have_dump */

struct confInfo *
parseConf(struct confInfo *conf)
{
	int res;
	char *tok;
	char config[] = CONF_FILE;
	struct pconf *parse;

	if ((parse=malloc(sizeof(struct pconf))) == NULL){
		perror("FATAL ERROR: Couldn't allocate enough memory");
		exit(10);
	}


	if(conf->error != NULL)
		free(conf->error);

	parse = parseconf(config);

	if (parse == NULL) {
		fprintf(stderr, "ERROR: Can not find %s\n", CONF_FILE);
		exit(53);
	}

	if (!parse->dumpdev[0]) {
		fprintf(stderr, "FATAL ERROR: no valid DUMPDEV= parameter" 
				"found in: %s\n", config);
		exit(50);
	}else
		conf->dev = strdup(parse->dumpdev);

	if (!parse->dumpdir[0]) {
		fprintf(stderr, "FATAL ERROR: no valid DUMPDIR= parameter" 
				"found in: %s\n", config);
		exit(51);
	}else
		conf->dir = strdup(parse->dumpdir);

	free(parse);

	if((tok=strstr(conf->dir,"ftp://"))){
		conf->ftp = 1;

		if( (res=(parseUrl(conf->dir, &conf->user, &conf->pass, 
				&conf->host, &conf->port, &conf->path)) != 0)){
			conf->error=strdup("ERROR parsing ftp URL");
			exit (res);
		}
	}

	return conf;
}

int
main(int argc, char **argv)
{
	char *ovalue=NULL, *dvalue=NULL, *tok;
	struct stat statbuf;
	struct confInfo *conf;
	int c, res;
	unsigned long option = 0x0;
	char *date;
	struct tm *ltime;
	time_t now;

	if ((date=(char *)malloc(DATE_SIZE)) == NULL){
		perror("FATAL ERROR: Couldn't allocate enough memory");
		exit(10);
	}

	if ((conf=malloc(sizeof(struct confInfo))) == NULL){
		perror("FATAL ERROR: Couldn't allocate enough memory");
		exit(10);
	}
	conf->dev = NULL;
	conf->dir = NULL;
	conf->user = NULL;
	conf->pass = NULL;
	conf->ftp = 0;
	conf->error = NULL;
	conf->host = NULL;
	conf->path = NULL;
	conf->port = PORT;


	/* Check that user is root for security */
	if (getuid() != 0) {
		fprintf(stderr, "Must be root to run this program.\n");
		exit(40);
	}


	opterr = 0;

	while ((c = getopt (argc, argv, "v?r:e:d:o:")) != -1)
		switch (c) {
			case 'd':
				dvalue = optarg;
				option = option | 1;
				break;
			case 'e':
				if ((erase_dump(optarg)) < 0) {
					fprintf(stderr,"erase dump failed\n");
					exit(61);
				} else 
					fprintf(stderr, "dump is erased\n");
				return 0;
			case 'o':
				ovalue = optarg;
				option = option | 2;
				break;
			case 'r':
				if ((reset_dump(optarg)) < 0) {
					fprintf(stderr,"reset dump failed\n");
					exit(62);
				}
				return 0;
			case '?':
				usage(argv[0]);
				exit(0);
			case 'v':
				version(argv[0]);
				exit(0);
			default:
				fprintf(stderr, "Unknown Error, command parsing"
						" in trouble\n");
				exit(63);
		}

	parseConf(conf);

	switch (option) {
		
		/* No overides, break*/
		case 0:
			break;

		/* User override of dump device via cmd line */
		case 1:
			if(conf->dev != NULL)
				free(conf->dev);
			conf->dev = dvalue;
			break;
		
		/* User override of output and dump dev via cmd line. */
		case 3:

			if(conf->dev != NULL)
				free(conf->dev);
			conf->dev = dvalue;

			/* fall throught to get output override */

		/* User override of output directory via cmd line */
		case 2:

			if(conf->user != NULL)
				free(conf->user);
			if(conf->pass != NULL)
				free(conf->pass);
			if(conf->host != NULL)
				free(conf->host);
			if(conf->path != NULL)
				free(conf->path);

			conf->user = NULL;
			conf->pass = NULL;
			conf->host = NULL;
			conf->path = NULL;
			conf->port = PORT;

			if((tok=strstr(ovalue,"ftp://"))){
				conf->ftp = 1;

				if( (res=(parseUrl(ovalue, &conf->user, 
						&conf->pass, &conf->host, 
						&conf->port, &conf->path)) 
							!= 0)){
					fprintf(stderr, "ERROR parsing ftp" 
							"URL\n");
					exit(res);
				}

			}else{
				if(conf->dir != NULL)
					free(conf->dir);
				conf->ftp = 0;
				conf->dir = ovalue;
			}
			break;


		default:
			fprintf(stderr, "Unknown Error, command parsing"
					" in trouble\n");
			exit(63);
	}


	if(conf->ftp){
		if(conf->user == NULL)
			conf->user = strdup("anonymous");
		if(conf->pass == NULL)
			conf->pass = strdup("anonymous@anonymous.com");
	}else{
		/* check output directory: dumpdir */
		if (stat(conf->dir, &statbuf) == -1) {
			perror("Can't stat dumpdir");
			exit(64);
		}

		if (!(S_ISDIR(statbuf.st_mode))) {
			fprintf(stderr,"%s is not a directory\n", conf->dir);
			exit(65);
		}
	}

#ifdef DEBUG
	printf("DEV: %s\n",conf->dev);
	printf("DIR: %s\n",conf->dir);
	printf("FTP: %d\n",conf->ftp);
	printf("HOST: %s\n",conf->host);
	printf("USER: %s\n",conf->user);
	printf("PASS: %s\n",conf->pass);
	printf("PATH: %s\n",conf->path);
	printf("PORT: %d\n",conf->port);
	printf("ERROR: %s\n",conf->error);
#endif

	/* Check input device partition */
	check_dev(conf->dev);

	if ((res = have_dump(conf->dev)) == 0){
		printf("No dump to save\n");
		return 0;
	}else if (res != 1){
		fprintf(stderr, "Error while trying to read and locate dump\n");
		exit (res);
	}

	time(&now);
	ltime = localtime(&now);
	sprintf(date, "%04d%02d%02d%02d%02d", ltime->tm_year+1900, ltime->tm_mon+1, 
			ltime->tm_mday, ltime->tm_hour, ltime->tm_min);


	if ((res=copy_dump(conf, date)) != 0) {
		fprintf(stderr, "Error while copying dump\n");
		exit (res);
	}

	if (res == 50) /* No dump to save */
		return 0;

	printf("Successfully wrote dump\n");

	if ((res=copy_map(conf, date)) != 0) {
		fprintf(stderr, "Error while copying System Map\n");
		exit (res);
	}

	printf("Successfully wrote System.map\n");

	return 0;
} /* end main */
