/*
 * Copyright (C) Dirk Jagdmann <doj@cubic.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/* mm and ms modes are currently really broken. We can rip some code from
linux kernel to improve. */

/* TODO:
 * - lock files in /var/lock
 */

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <termios.h>
#include <unistd.h>

#include "debug.h"
#include "pointer.h"
#include "pointer_internal.h"

static int fh=-1;

static void cleanup()
{
  if(fh>=0)
    close(fh);
}

static int serialgetfd()
{
  return fh;
}

enum {
  MICROSOFT,
  MOUSESYSTEMS,
};

static int mode;

static int serialinit(char *device)
{
  struct termios t;

  /* open device */
  fh=open(device, O_RDONLY | O_NOCTTY);
  if(fh<0)
    {
      printf("could not open %s: %s\n", device, strerror(errno));
      return -1;
    }

  atexit(cleanup);

  tcflush(fh, TCIOFLUSH);

  if(tcgetattr(fh, &t) < 0)
    {
      printf("could not get device attributes\n");
      exit(1);
    }

  switch(mode)
    {
    case MICROSOFT:
      cfsetispeed(&t, B1200);
      cfsetospeed(&t, B1200);
      /* baud rate */
      t.c_cflag &= ~CSIZE;
      /* data bits */
      t.c_cflag |= CS7;
      /* 1 stop bit */
      t.c_cflag &= ~CSTOPB;
      /* enable receiver */
      t.c_cflag |= CREAD;
      /* ignore modem control lines */
      t.c_cflag |= CLOCAL;
      /* set hardware handshake */
      t.c_cflag |= CRTSCTS;
      /* ignore parity errors */
      t.c_iflag |= IGNPAR;
      /* set non-canonical mode */
      t.c_lflag &= ~ICANON;
      /* set read timeout = .1s */
      t.c_cc[VTIME]=1;
      /* set min read bytes */
      t.c_cc[VMIN]=3;
      break;

    case MOUSESYSTEMS:
      cfsetispeed(&t, B1200);
      cfsetospeed(&t, B1200);
      /* baud rate */
      t.c_cflag &= ~CSIZE;
      /* data bits */
      t.c_cflag |= CS8;
      /* 2 stop bits */
      t.c_cflag |= CSTOPB;
      /* enable receiver */
      t.c_cflag |= CREAD;
      /* ignore modem control lines */
      t.c_cflag |= CLOCAL;
      /* set hardware handshake */
      t.c_cflag |= CRTSCTS;
      /* ignore parity errors */
      t.c_iflag |= IGNPAR;
      /* set non-canonical mode */
      t.c_lflag &= ~ICANON;
      /* set read timeout = .1s */
      t.c_cc[VTIME]=1;
      /* set min read bytes */
      t.c_cc[VMIN]=5;
      break;
    }

  if(tcsetattr(fh, 0, &t) < 0)
    {
      printf("could not set device attributes\n");
      exit(1);
    }

  return 0;
}

#if 0
static void printbits(unsigned char b)
{
  printf("%i%i%i%i %i%i%i%i\n",
	 (b&128)?1:0,
	 (b&64) ?1:0,
	 (b&32) ?1:0,
	 (b&16) ?1:0,
	 (b&8)  ?1:0,
	 (b&4)  ?1:0,
	 (b&2)  ?1:0,
	 (b&1)  ?1:0);
}
#endif

/*
 *  Bit     D7      D6      D5      D4      D3      D2      D1      D0
 *
 *  1.      X       1       LB      RB      Y7      Y6      X7      X6
 *  2.      X       0       X5      X4      X3      X2      X1      X0
 *  3.      X       0       Y5      Y4      Y3      Y2      Y1      Y0
 *( 4.      X       0       CB      X       X       X       X       X  )
 */

/* ms protocol currently totally sucks with Logitech M-35 (pilot mouse) */

static void msstatemachine(unsigned char c)
{
  static unsigned char start, x, y;
  enum { WAITFORSTART, COLLECTX, COLLECTY, CHECKE };
  static int state=WAITFORSTART;
  switch(state)
    {
    case CHECKE:
      state=WAITFORSTART;
      if(!(c&64))
	{
	  int B=pointerB() & (~BUT3);
	  if(c&32)
	    B|=BUT3;
	  pointersetB(B);
	  break;
	}
      /* no break */

    case WAITFORSTART:
      if(c&64)
	{
	  start=c;
	  state=COLLECTX;
	}
      break;

    case COLLECTX:
      x=c;
      state=COLLECTY;
      break;

    case COLLECTY:
      y=c;
      state=CHECKE;
      {
	int X=0, Y=0, B=pointerB();

	x&=0x3F;
	y&=0x3F;
	x|=(start&3) <<6;
	y|=(start&12)<<4;

	X+=*((signed char*)&x);
	if(X<-MOUSE_LIMIT)
	  X=-MOUSE_LIMIT;
	else if(X>MOUSE_LIMIT)
	  X=MOUSE_LIMIT;
	pointersetrawX(pointerrawX()+X);

	Y+=*((signed char*)&y);
	if(Y<-MOUSE_LIMIT)
	  Y=-MOUSE_LIMIT;
	else if(Y>MOUSE_LIMIT)
	  Y=MOUSE_LIMIT;
	pointersetrawY(pointerrawY()+Y);

	B&=~(BUT1|BUT2);
	if(start&32)
	  B|=BUT1;
	if(start&16)
	  B|=BUT2;
	pointersetB(B);
      }
      break;
    }
}

static int microsoftpoll()
{
  int i;
  unsigned char buf[16];
  int r=read(fh, buf, sizeof(buf));
  if(r<0)
    {
      PRINTF("microsoftpoll: could not read\n");
      return r;
    }
  for(i=0; i<r; ++i)
    msstatemachine(buf[i]);
  return 0;
}

void microsoftregister(struct driver *drv)
{
  mode=MICROSOFT;
  drv->init=serialinit;
  drv->poll=microsoftpoll;
  drv->getfd=serialgetfd;
}

static int mousesystemspoll()
{
  int X=0, Y=0, B=pointerB();
  int i=0;
  signed char buf[5*16];
  int r=read(fh, buf, sizeof(buf));
  if(r<0)
    {
      PRINTF("mousesystemspoll: could not read\n");
      return -1;
    }

  while(r>=5)
    {
      /*
	Bit     D7      D6      D5      D4      D3      D2      D1      D0

	1.      1       0       0       0       0       LB      CB      RB
	2.      X7      X6      X5      X4      X3      X2      X1      X0
	3.      Y7      Y6      Y5      Y4      Y3      Y4      Y1      Y0
	4.      X7'     X6'     X5'     X4'     X3'     X2'     X1'     X0'
	5.      Y7'     Y6'     Y5'     Y4'     Y3'     Y4'     Y1'     Y0'
      */

      /* find start of sequence */
      if((buf[i]&0xF8)!=0x80)
	{
	  ++i;
	  --r;
	  continue;
	}

      X+=buf[i+1];
      Y+=buf[i+2];
#if 0
      X+=buf[i+3];
      Y+=buf[i+4];
#endif
      B=0;
      if(!(buf[i+0]&4))
	B|=BUT1;
      if(!(buf[i+0]&1))
	B|=BUT2;
      if(!(buf[i+0]&2))
	B|=BUT3;

      i+=5;
      r-=5;
    }

  if(X<-MOUSE_LIMIT)
    X=-MOUSE_LIMIT;
  else if(X>MOUSE_LIMIT)
    X=MOUSE_LIMIT;

  if(Y<-MOUSE_LIMIT)
    Y=-MOUSE_LIMIT;
  else if(Y>MOUSE_LIMIT)
    Y=MOUSE_LIMIT;

  pointersetrawX(pointerrawX()+X);
  pointersetrawY(pointerrawY()+Y);
  pointersetB(B);

  return 0;
}

void mousesystemsregister(struct driver *drv)
{
  mode=MOUSESYSTEMS;
  drv->init=serialinit;
  drv->poll=mousesystemspoll;
  drv->getfd=serialgetfd;
}
