/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2002-2005 Christian Schallhart
 *               2006-2007 model.in.tum.de group
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/stream_test_data_adaptor.t.cpp
 *
 * @brief [LEVEL: beta] testing @ref diagnostics::unittest::Interactive_Test_Data_Adaptor
 *
 * $Id: stream_test_data_adaptor.t.cpp,v 1.6 2005/06/23 09:54:26 esdentem Exp $
 *
 * @author Christian Schallhart
 *
 */
#include <diagnostics/unittest.hpp>

#include <diagnostics/unittest/test_system/stream_test_data_adaptor.hpp>

#include <diagnostics/unittest/test_system_exception.hpp>

// test support
#include <diagnostics/util/dummy_test_data_source.ts.hpp>
#include <diagnostics/unittest/test_system/test_data_adaptor.ts.hpp>

#include <sstream>

#define TEST_COMPONENT_NAME Stream_Test_Data_Adaptor
#define TEST_COMPONENT_NAMESPACE diagnostics::unittest

DIAGNOSTICS_NAMESPACE_BEGIN;
UNITTEST_NAMESPACE_BEGIN;
TEST_NAMESPACE_BEGIN;
TEST_COMPONENT_TEST_NAMESPACE_BEGIN;

/**
 * @brief Testing mode
 */ 
void mode(Test_Data & test_data)
{
    Stream_Test_Data_Adaptor adaptor(::std::cin, ::std::cout);
    TEST_ASSERT(adaptor.backend()==NULL);

    TEST_ASSERT(adaptor.mode()==Stream_Test_Data_Adaptor::MODE_INTERACTIVE);

    adaptor.mode(Stream_Test_Data_Adaptor::MODE_INTERACTIVE);
    TEST_ASSERT(adaptor.mode()==Stream_Test_Data_Adaptor::MODE_INTERACTIVE);

    adaptor.mode(Stream_Test_Data_Adaptor::MODE_ACCEPT);
    TEST_ASSERT(adaptor.mode()==Stream_Test_Data_Adaptor::MODE_ACCEPT);

    adaptor.mode(Stream_Test_Data_Adaptor::MODE_REJECT);
    TEST_ASSERT(adaptor.mode()==Stream_Test_Data_Adaptor::MODE_REJECT);

    adaptor.mode(Stream_Test_Data_Adaptor::MODE_INTERACTIVE);
    TEST_ASSERT(adaptor.mode()==Stream_Test_Data_Adaptor::MODE_INTERACTIVE);
}

/**
 * @brief Testing get and compare when mode()==MODE_REJECT
 */ 
void get_compare_mode_reject(Test_Data & test_data)
{
    Dummy_Test_Data_Source source;
    Stream_Test_Data_Adaptor adaptor(::std::cin, ::std::cout);

    // switch into non-interactive mode
    TEST_ASSERT(adaptor.mode()==Stream_Test_Data_Adaptor::MODE_INTERACTIVE);
    adaptor.mode(Stream_Test_Data_Adaptor::MODE_REJECT);
    TEST_ASSERT(adaptor.mode()==Stream_Test_Data_Adaptor::MODE_REJECT);

    // copy for later comparision
    Dummy_Test_Data_Source::Data_t const data(source.data());
    
    // backend()==NULL after construction
    TEST_ASSERT(adaptor.backend()==NULL);

    // get throws if backend()==NULL
    TEST_THROWING_BLOCK_ENTER;
    adaptor.get("1");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    // compare throws if backend()==NULL
    TEST_THROWING_BLOCK_ENTER;
    adaptor.compare("1","XX");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    // attach and check backend()
    adaptor.attach(&source);
    TEST_ASSERT(adaptor.backend()==&source);

    // get of an existing entry
    TEST_ASSERT(adaptor.get("1")=="A");
    TEST_ASSERT(adaptor.get("2")=="B");

    // get of an non-existing entry
    TEST_THROWING_BLOCK_ENTER;
    adaptor.get("DOES NOT EXIST");
    TEST_THROWING_BLOCK_EXIT(Test_Exception);

    // compare of an existing entry
    TEST_ASSERT(adaptor.compare("1","A"));
    TEST_ASSERT(!adaptor.compare("1","XXX"));

    TEST_ASSERT(adaptor.compare("2","B"));
    TEST_ASSERT(!adaptor.compare("2","XXX"));

    // compare of an non-existing entry
    TEST_THROWING_BLOCK_ENTER;
    adaptor.compare("DOES NOT EXIST","bla");
    TEST_THROWING_BLOCK_EXIT(Test_Exception);

    // compare data of Test_Data_Source
    TEST_ASSERT(data==source.data());
}

/**
 * @brief Testing get and compare when mode()==MODE_ACCEPT
 */ 
void get_compare_mode_accept(Test_Data & test_data)
{
    Dummy_Test_Data_Source source;
    Stream_Test_Data_Adaptor adaptor(::std::cin, ::std::cout);

    // copy for later comparision
    Dummy_Test_Data_Source::Data_t data(source.data());

    // switch into non-interactive mode
    TEST_ASSERT(adaptor.mode()==Stream_Test_Data_Adaptor::MODE_INTERACTIVE);
    adaptor.mode(Stream_Test_Data_Adaptor::MODE_ACCEPT);
    TEST_ASSERT(adaptor.mode()==Stream_Test_Data_Adaptor::MODE_ACCEPT);

    // backend()==NULL after construction
    TEST_ASSERT(adaptor.backend()==NULL);

    // get throws if backend()==NULL
    TEST_THROWING_BLOCK_ENTER;
    adaptor.get("1");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    // compare throws if backend()==NULL
    TEST_THROWING_BLOCK_ENTER;
    adaptor.compare("1","XX");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    // attach and check backend()
    adaptor.attach(&source);
    TEST_ASSERT(adaptor.backend()==&source);

    // get of an existing entry
    TEST_ASSERT(adaptor.get("1")=="A");
    TEST_ASSERT(adaptor.get("2")=="B");

    // get of an non-existing entry
    adaptor.get("DOES NOT EXIST");
    data["DOES NOT EXIST"]="";
    TEST_ASSERT(data==source.data());

    // compare of an existing entry
    TEST_ASSERT(adaptor.compare("1","A"));
    TEST_ASSERT(data==source.data());
    TEST_ASSERT(adaptor.compare("1","XXX"));
    data["1"]="XXX";
    TEST_ASSERT(data==source.data());

    // compare of an existing entry
    TEST_ASSERT(adaptor.compare("2","B"));
    TEST_ASSERT(data==source.data());
    TEST_ASSERT(adaptor.compare("2","XXX"));
    data["2"]="XXX";
    TEST_ASSERT(data==source.data());	

    // compare of an non-existing entry
    adaptor.compare("DOES NOT EXIST YEY","bla");
    data["DOES NOT EXIST YEY"]="bla";
    TEST_ASSERT(data==source.data());
}



/**
 * @brief Testing get and compare when mode()==MODE_INTERACTIVE
 */
void get_compare_mode_interactive(Test_Data & test_data)
{
    // used to control the Stream_Test_Data_Adaptor in the interactive
    // mode. Each character represents a yes/no decision. The decision
    // points are marked with in-feed below.
    ::std::istringstream in("r\nr\nr\nr\na\nr\na\n");
    ::std::ostringstream out;

    Dummy_Test_Data_Source source;
    Stream_Test_Data_Adaptor adaptor(in,out);
    TEST_ASSERT(adaptor.mode()==Stream_Test_Data_Adaptor::MODE_INTERACTIVE);

    // copy for later comparision
    Dummy_Test_Data_Source::Data_t data(source.data());
    
    // backend()==NULL after construction
    TEST_ASSERT(adaptor.backend()==NULL);

    // get throws if backend()==NULL
    TEST_THROWING_BLOCK_ENTER;
    adaptor.get("1");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    // compare throws if backend()==NULL
    TEST_THROWING_BLOCK_ENTER;
    adaptor.compare("1","XX");
    TEST_THROWING_BLOCK_EXIT(Test_System_Exception);

    // attach and check backend()
    adaptor.attach(&source);
    TEST_ASSERT(adaptor.backend()==&source);

    // get of an existing entry
    TEST_ASSERT(adaptor.get("1")=="A");
    TEST_ASSERT(adaptor.get("2")=="B");

    // in-feed 1: n
    // get of an non-existing entry
    TEST_THROWING_BLOCK_ENTER;
    adaptor.get("DOES NOT EXIST");
    TEST_THROWING_BLOCK_EXIT(Test_Exception);

    // in-feed 2: n
    // get of an non-existing entry
    TEST_THROWING_BLOCK_ENTER;
    adaptor.get("DOES NOT EXIST");
    TEST_THROWING_BLOCK_EXIT(Test_Exception);

    // compare of an existing entry
    TEST_ASSERT(adaptor.compare("1","A"));
    TEST_ASSERT(adaptor.compare("2","B"));

    // in-feed 3: n
    // compare of an non-existing entry
    TEST_THROWING_BLOCK_ENTER;
    adaptor.compare("DOES NOT EXIST","bla");
    TEST_THROWING_BLOCK_EXIT(Test_Exception);

    // in-feed 4: n
    // compare of an mismatching entry
    TEST_ASSERT(!adaptor.compare("1","AA"));

    // compare data of Test_Data_Source
    TEST_ASSERT(data==source.data());

    // in-feed 5: y
    // get of an non-existing entry
    TEST_ASSERT(adaptor.get("DOES NOT EXIST YET")=="");

    // modify and compare data of Test_Data_Source
    data["DOES NOT EXIST YET"]="";
    TEST_ASSERT(data==source.data());

    // get of a now existing entry
    TEST_ASSERT(adaptor.get("DOES NOT EXIST YET")=="");

    // in-feed 6: n
    // compare of a mismatching entry
    TEST_ASSERT(!adaptor.compare("DOES NOT EXIST YET","bla"));

    // get of a now existing entry
    TEST_ASSERT(adaptor.get("DOES NOT EXIST YET")=="");

    // compare data of Test_Data_Source
    TEST_ASSERT(data==source.data());

    // in-feed 7: y
    // compare of a mismatching entry
    TEST_ASSERT(adaptor.compare("DOES NOT EXIST YET","bla"));

    // modify and compare data of Test_Data_Source
    data["DOES NOT EXIST YET"]="bla";
    TEST_ASSERT(data==source.data());

    // get of a now existing entry
    TEST_ASSERT(adaptor.get("DOES NOT EXIST YET")=="bla");

    // check end of input stream
    char c;
    in >> c;
    TEST_ASSERT(in.eof());

    // check the output stream
    TEST_ASSERT(test_data.compare("output",out.str()));
}


TEST_COMPONENT_TEST_NAMESPACE_END;
TEST_NAMESPACE_END;
UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

TEST_SUITE_BEGIN;
TEST_NORMAL_CASE(Test_Data_Adaptor_Compliance
		 (new Stream_Test_Data_Adaptor(::std::cin,::std::cout)),
		 LEVEL_PROD);
TEST_NORMAL_CASE(&mode,LEVEL_PROD);
TEST_NORMAL_CASE(&get_compare_mode_interactive,LEVEL_PROD);
TEST_NORMAL_CASE(&get_compare_mode_reject,LEVEL_PROD);
TEST_NORMAL_CASE(&get_compare_mode_accept,LEVEL_PROD);
TEST_SUITE_END;

STREAM_TEST_SYSTEM_MAIN;
// vim:ts=4:sw=4
